/*  job_computediagraminterferences.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_computediagraminterferences.h"
#include "functions.h"
#include "files.h"
#include "filedata.h"
#include "amplitude.h"
#include "diagram.h"
#include "feynmanrules.h"
#include "job_catfiles.h"
#include "equation.h"
#include <ginac/ginac.h>
#include "yamlutils.h"

using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<ComputeDiagramInterferences> dummy;
}

void ComputeDiagramInterferences::read_manual_options(const YAML::Node& node) {
	if (node.FindValue("polarization_orthogonal_vectors")) {
		const YAML::Node& n = node["polarization_orthogonal_vectors"];
		if (n.size() > 0 && n.Type() != YAML::NodeType::Sequence)
			throw runtime_error("node is not a sequence " + position_info(n));
		for (YAML::Iterator i = n.begin(); i != n.end(); ++i) {
			if (i->Type() != YAML::NodeType::Sequence || i->size() != 2)
				throw runtime_error("node is not a 2-element sequence "
						+ position_info(*i));
			int index;
			GiNaC::ex vec;
			const Kinematics* kin = Files::instance()->kinematics();
			GiNaC::lst emom = kin->external_momenta();
			(*i)[0] >> index;
			Reduze::read((*i)[1], vec, emom);
			polarization_orthogonal_vectors_[index] = vec;
		}
	}
}

void ComputeDiagramInterferences::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "polarization_orthogonal_vectors" << Value << BeginSeq;
	const map<int, GiNaC::ex>& p = polarization_orthogonal_vectors_;
	for (map<int, GiNaC::ex>::const_iterator i = p.begin(); i != p.end(); ++i) {
		os << Flow << BeginSeq << i->first << i->second << EndSeq;
	}
}

std::string ComputeDiagramInterferences::get_description() const {
	if (position_1_ > 0 && position_2_ > 0)
		return "compute interference " + result_name_;
	else
		return "compute interferences " + short_filename(output_file_);
}

bool ComputeDiagramInterferences::find_dependencies(
		const set<string>& outothers, list<string>& in, list<string>& out,
		list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);

	// wait for preceding jobs before generating subjobs
	if (outothers.find(get_canonical_filename(diagrams_file_1_))
			!= outothers.end() || outothers.find(get_canonical_filename(
			diagrams_file_2_)) != outothers.end())
		return false;

	bool skip_job = is_readable_file(output_file_) && is_conditional();

	if ((position_1_ <= 0 || position_2_ <= 0) && !skip_job) {
		LOGX("Divide computation into smaller jobs");
		list<Diagram> dias1, dias2;
		read_diagrams(dias1, diagrams_file_1_); // size, names would be enough
		read_diagrams(dias2, diagrams_file_2_); // size, names would be enough

		list<string> partial_files;
		list<Diagram>::const_iterator d1, d2;
		size_t pos1, pos2;
		for (d1 = dias1.begin(), pos1 = 1; d1 != dias1.end(); ++d1, ++pos1) {
			for (d2 = dias2.begin(), pos2 = 1; d2 != dias2.end(); ++d2, ++pos2) {
				string tmp_directory = Files::instance()->get_tmp_directory();
				const string prodfn = tmp_directory
						+ get_filename_without_directory(output_file_) + "_"
						+ d1->name() + "X" + d2->name();
				partial_files.push_back(prodfn);
				ComputeDiagramInterferences* auxjob =
						new ComputeDiagramInterferences;
				auxjob->set_conditional(this->is_conditional());
				auxjob->diagrams_file_1_ = diagrams_file_1_;
				auxjob->diagrams_file_2_ = diagrams_file_2_;
				auxjob->position_1_ = pos1;
				auxjob->position_2_ = pos2;
				auxjob->output_file_ = prodfn;
				auxjob->result_name_ = d1->name() + " X " + d2->name()
						+ " from diagrams in " + short_filename(diagrams_file_1_)
						+ " and " + short_filename(diagrams_file_2_);
				auxjob->polarization_orthogonal_vectors_
						= polarization_orthogonal_vectors_;
				auxjobs.push_back(auxjob);
				LOGX("New job for " << prodfn);
			}
		}
		CatFiles* catjob = new CatFiles(partial_files, output_file_, true);
		catjob->set_conditional(false);
		auxjobs.push_back(catjob);
		LOG("Number of interferences to be calculated : " << partial_files.size());
		// remember: we don't compute anything directly
		// wait for result of aux jobs
		in.push_back(output_file_);
	} else {
		// we are really going to compute something: (1 diag) X (1 diag)*
		in.push_back(diagrams_file_1_);
		in.push_back(diagrams_file_2_);
		out.push_back(output_file_);
	}
	return true;
}

void ComputeDiagramInterferences::run_serial() {
	if (position_1_ <= 0 || position_2_ <= 0) // only metajob ?
		return;

	// really do some computation: multiply 2 diagrams
	LOG("Reading two diagrams");
	Diagram first_diagram = read_diagram(diagrams_file_1_, position_1_);
	Diagram second_diagram = read_diagram(diagrams_file_2_, position_2_);
	LOG("Generating amplitudes");
	const map<int, GiNaC::ex>& p = polarization_orthogonal_vectors_;
	for (map<int, GiNaC::ex>::const_iterator i = p.begin(); i != p.end(); ++i) {
		LOG("  using physical polarizations for field " << i->first);
		first_diagram.set_polarization_orthogonal_vector(i->first, i->second);
		second_diagram.set_polarization_orthogonal_vector(i->first, i->second);
	}
	if (p.empty())
		LOG("  using Feynman polarizations everywhere");
	FeynmanRules* fr = Files::instance()->feynmanrules();
	Amplitude amp1 = first_diagram.get_amplitude(fr);
	Amplitude amp2 = second_diagram.get_amplitude(fr);
	Amplitude amp2_conjugated = amp2.conjugate();

	LOG("Multiplying amplitudes");
	AmplitudeProd ampsquared(amp1, amp2_conjugated, fr);
	LinearCombination lincomb;
	EquationInfo info = ampsquared.calculate(lincomb);

	if (info.valid_equation_) {
		OutFileLinearCombinations out(output_file_.c_str());
		lincomb.set_name(to_safe_variable_name(output_file_));
		out << lincomb;
		out.finalize();
	} else {
		string tmp = output_file_ + ".tmp";
		ofstream out(tmp.c_str());
		if(!out)
			throw runtime_error("Cannot open file '" + tmp + "'");
		out << "equation_" << to_safe_variable_name(output_file_)
				<< "_not_calculated_because_no_sector_has_been_defined" << endl;
		out.close();
		rename(tmp, output_file_);
	}

}

}
