/*  job_findfamily.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_findfamily.h"
#include <vector>
#include <ginac/ginac.h>
#include "kinematics.h"
#include "crossing.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "propagator.h"
#include "files.h"
#include "functions.h"
#include "combinatorics.h"
#include "integralfamily.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

namespace {
JobProxy<FindFamily> dummy;
}

bool FindFamily::find_dependencies(const set<string>& outothers, //
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	out.push_back(output_file_);
	return true;
}

void FindFamily::init() {
	if (output_file_.empty())
		throw runtime_error("output file undefined");
}

std::string FindFamily::get_description() const {
	return "find integral family";
}

void FindFamily::read_manual_options(const YAML::Node& node) {
	using namespace YAML;
	const Kinematics* kin = Files::instance()->kinematics();

	list < string > loop_momenta_str;
	node["loop_momenta"] >> loop_momenta_str;
	loop_momenta_ = create_symbols(loop_momenta_str);

	const Node& props_node = node["propagators"];
	for (Iterator n = props_node.begin(); n != props_node.end(); ++n)
		input_propagators_.push_back(Propagator::read(kin, loop_momenta_, *n));
}

void FindFamily::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "loop_momenta" << Value << loop_momenta_;

	os << Key << "propagators" << Value << BeginSeq;
	for (size_t i = 0; i < input_propagators_.size(); ++i)
		input_propagators_[i].print(os);
	os << EndSeq;
}

list<exmap> find_symmetry_candidates(const vector<ex>& pmom, const lst& loopmom,
		const lst& extmom, const list<exmap>& idxs, unsigned& ntries) {
	using benbear::next_combination;
	// use temporary symbols for rhs of rules
	GiNaC::lst tmploopmom, tmpextmom;
	GiNaC::exmap loop2tmp, ext2tmp, tmp2src;
	provide_alternative_symbols(loopmom, tmploopmom, loop2tmp, tmp2src);
	provide_alternative_symbols(extmom, tmpextmom, ext2tmp, tmp2src);
	list<lst> trafos;
	vector<int> aa(pmom.size()), bb(pmom.size());
	for (size_t i = 0; i < pmom.size(); ++i)
		aa[i] = bb[i] = i;
	if (pmom.size() < loopmom.nops())
		ERROR("too few initial propagators");
	int* af, *am, *al, *bf, *bm, *bl;
	// try to map propagators [af,af+1,...,am) to [bf,bf+1,...,bm)
	// we consider nloop propagators and require the trafo to be fixed by them
	af = &aa[0], am = &aa[0] + loopmom.nops(), al = &aa[0] + pmom.size();
	bf = &bb[0], bm = &bb[0] + loopmom.nops(), bl = &bb[0] + pmom.size();
	benbear::init_combination(af, am, al, true /* min first */);
	benbear::init_combination(bf, bm, bl, true /* min first */);
	unsigned signs = 0; // signs for all target momenta
	// (overall sign of propagator momentum is arbitrary)
	list<exmap>::const_iterator idx;
	for (idx = idxs.begin(); idx != idxs.end(); ++idx) { // crossings
		vector<ex> lmom(pmom.size()), rmom(pmom.size());
		for (size_t i = 0; i < pmom.size(); ++i) {
			lmom[i] = pmom[i].subs(*idx).expand();
			rmom[i] = pmom[i].subs(loop2tmp).expand();
		}
		do { // try all combinations of propagators and overall signs
			++ntries;
			lst sys;
			for (size_t i = 0; i < loopmom.nops(); ++i) {
				bool s = !(signs & (1 << i));
				const ex& ma = lmom[aa[i]];
				const ex& mb = (s ? -rmom[bb[i]] : rmom[bb[i]]);
				sys.append(ma == mb);
			}
			sys = ex_to < lst > (lsolve(sys, loopmom));
			if (sys.nops() == 0)
				continue;
			ex jac = jacobi_determinant(sys, tmploopmom);
			if (abs(jac.expand()) != 1)
				continue;
			for (exmap::const_iterator x = idx->begin(); x != idx->end(); ++x)
				sys.append(x->first == x->second.subs(ext2tmp));
			trafos.push_back(sys);
		} while ((signs = (signs + 1) % (1 << loopmom.nops())) > 0
				|| next_combination(af, am, al) // canonical ordering for [af,am)
				|| next_permutation(bf, bm) || next_combination(bf, bm, bl));
	}
	trafos.sort(ex_is_less());
	trafos.unique(ex_is_equal());
	list < exmap > trafosm;
	for (list<lst>::const_iterator t = trafos.begin(); t != trafos.end(); ++t) {
		exmap tm = equations_to_substitutions(*t, tmp2src, false);
		for (exmap::iterator m = tm.begin(); m != tm.end(); ++m)
			m->second = m->second.subs(tmp2src);
		trafosm.push_back(tm);
	}
	return trafosm;
}

/* alternative strategy:
 * - select nloops independent source propagators, map them to all possible
 *   target props, this fixes the trafo
 *   (note: will not always be better than current version, but much faster
 *    if most propagators are given already)
 * - some source props may be mapped to aux props, therefore consider
 *   various combinations for source props:
 *   select 0, 1, 2, ... , or nmissing props out of all props, they are
 *   assumed to be mapped to aux props, select one set of nloops indep. props
 *   from the remaining ones for each such selection
 *   remove multiples from the list of sets of source propagators
 */

// whether moms forms a complete set of linear combinations of loopmoms and extmoms
bool is_complete(const vector<ex>& moms, const lst& loopmoms) {
	matrix mat(moms.size(), loopmoms.nops());
	for (unsigned int i = 0; i < moms.size(); ++i) {
		for (unsigned int j = 0; j < loopmoms.nops(); ++j) {
			VERIFY(is_a<symbol>(loopmoms.op(j)));
			mat(i, j) = moms[i].diff(ex_to<symbol>(loopmoms.op(j)));
		}
	}
	return mat.determinant() != 0;
}

// whether tuple forms a complete set of linear combinations of loopmoms and extmoms
bool is_complete(const vector<ex>& pmom, const vector<int>& tuple, const lst& loopmom) {
	VERIFY(tuple.size() == loopmom.nops());
	vector<ex> sm(loopmom.nops());
	for (size_t i = 0; i < loopmom.nops(); ++i)
		sm[i] = pmom[tuple[i]];
	return is_complete(sm, loopmom);
}

list<exmap> find_symmetry_candidates_alt(const vector<ex>& pmom, const lst& loopmom,
		const lst& extmom, const list<exmap>& idxs, unsigned maxaux, unsigned& ntries) {
	using benbear::next_combination;
	list<exmap> res;
	size_t nl = loopmom.nops();

	LOG("  finding source propagator sets");
	unsigned maxauxeff = maxaux;
	if (pmom.size() < nl) {
		ERROR("too few propagators given (level 0)");
	} else if (pmom.size() - maxaux < nl) {
		WARNING("too few propagators given (level 1) !");
		maxauxeff = pmom.size() - nl;
	}
	set<vector<ex> > srcmoms;
	map<vector<int>, bool> ok;
	unsigned nsrc = 0;
	int ntoofew = -1;
	for (unsigned naux = 0; naux <= maxauxeff; ++naux) {
		LOG("    considering trafos generating " << naux << " of " << maxaux << " aux propagators");
		// the propagators in [af,af+1,...,am) are mapped to existing props
		vector<int> aa(pmom.size());
		for (size_t i = 0; i < pmom.size(); ++i)
			aa[i] = i;
		int* af, *as, *am, *al;
		af = &aa[0], am = &aa[0] + pmom.size() - naux, al = &aa[0] + pmom.size();
		benbear::init_combination(af, am, al, true /* min first */);
		do {
			as = &aa[0] + nl;
			benbear::init_combination(af, as, am, true /* min first */);
			bool found = false;
			do {
				vector<int> t(&aa[0], &aa[0] + nl);
				map<vector<int>, bool>::iterator c = ok.find(t);
				if (c == ok.end()) { // no cached completeness test result
					c = ok.insert(c, make_pair(t, is_complete(pmom, t, loopmom)));
					if (c->second)
						++nsrc;
				}
				found = c->second;
			} while (!found && next_combination(af, as, am)); // first working set
			if (!found && ntoofew < 0)
				ntoofew = naux;
		} while (benbear::next_combination(af, am, al)); // all combis of props to aux
	}
	LOG("    found " << nsrc << " source combinations for momenta mapping");
	if (ntoofew >= 0) {
		LOG("\nNOTE: Not enough propagators to guarantee optimal family is found");
		LOG("      The problem occurs first for " << ntoofew << " / " << maxaux << " aux propagators.");
		//WARNING("too few propagators for trafos with " << ntoofew << " aux props");
	}

	LOG("  determining transformations");
	GiNaC::lst tmploopmom, tmpextmom;
	GiNaC::exmap loop2tmp, ext2tmp, tmp2src;
	provide_alternative_symbols(loopmom, tmploopmom, loop2tmp, tmp2src);
	provide_alternative_symbols(extmom, tmpextmom, ext2tmp, tmp2src);
	list<lst> trafos;
	unsigned signs = 0; // signs for all target momenta
	// (overall sign of propagator momentum is arbitrary)
	list<exmap>::const_iterator idx;
	ProgressBar pbar(2, "testing combinations:", idxs.size()*nsrc);
	pbar.start();
	for (idx = idxs.begin(); idx != idxs.end(); ++idx) { // crossings
		vector<ex> lmom(pmom.size()), rmom(pmom.size());
		for (size_t i = 0; i < pmom.size(); ++i) {
			lmom[i] = pmom[i].subs(*idx).expand();
			rmom[i] = pmom[i].subs(loop2tmp).expand();
		}
		map<vector<int>, bool>::iterator s;
		for (s = ok.begin(); s != ok.end(); ++s) {
			if (!s->second)
				continue;
			const vector<int>& aa = s->first;
			vector<int> bb(pmom.size());
			for (size_t i = 0; i < pmom.size(); ++i)
				bb[i] = i;
			int* bf, *bm, *bl;
			bf = &bb[0], bm = &bb[0] + nl, bl = &bb[0] + pmom.size();
			benbear::init_combination(bf, bm, bl, true /* min first */);
			do { // try all combinations of propagators and overall signs
				++ntries;
				lst sys;
				for (size_t i = 0; i < nl; ++i) {
					bool s = !(signs & (1 << i));
					const ex& ma = lmom[aa[i]];
					const ex& mb = (s ? -rmom[bb[i]] : rmom[bb[i]]);
					sys.append(ma == mb);
				}
				sys = ex_to < lst > (lsolve(sys, loopmom));
				if (sys.nops() == 0)
					continue;
				ex jac = jacobi_determinant(sys, tmploopmom);
				if (abs(jac.expand()) != 1)
					continue;
				for (exmap::const_iterator x = idx->begin(); x != idx->end();
						++x)
					sys.append(x->first == x->second.subs(ext2tmp));
				trafos.push_back(sys);
			} while ((signs = (signs + 1) % (1 << nl)) > 0
					|| next_permutation(bf, bm) || next_combination(bf, bm, bl));
			pbar.print();
		}
	}
	pbar.end();
	trafos.sort(ex_is_less());
	trafos.unique(ex_is_equal());
	list < exmap > trafosm;
	for (list<lst>::const_iterator t = trafos.begin(); t != trafos.end(); ++t) {
		exmap tm = equations_to_substitutions(*t, tmp2src, false);
		for (exmap::iterator m = tm.begin(); m != tm.end(); ++m)
			m->second = m->second.subs(tmp2src);
		trafosm.push_back(tm);
	}
	return trafosm;
}

bool is_permutation_symmetry(const vector<ex>& mom, const exmap& trafo,
		unsigned max_new_propagators, vector<ex>* new_aux,
		map<int, int>* perm) {
	vector<ex> allmom = mom;
	vector<ex> aux;
	for (size_t i = 0; i < allmom.size(); ++i) {
		ex pip = allmom[i].subs(trafo).expand();
		size_t j;
		for (j = 0; j < allmom.size(); ++j)
			if (pip == allmom[j] || pip == -allmom[j]) {
				if (perm)
					(*perm)[i] = j;
				break;
			}
		if (j == allmom.size()) { // need new aux prop
			pip = pip.compare(-pip) > 0 ? -pip : pip; // minimal version
			aux.push_back(pip);
			allmom.push_back(pip);
		}
		if (aux.size() > max_new_propagators) // not a symmetry
			break;
	}
	if (aux.size() <= max_new_propagators) {
		if (new_aux)
			*new_aux = aux;
		return true;
	}
	return false;
}

exset find_symmetries_and_auxilary_propagators(list<exmap>& trafos,
		const vector<ex>& pmom, const lst& loopmom, const lst& extmom,
		unsigned max_aux, bool force_natural, bool force_noexternal) {
	// sort valid symmetries by decreasing number of induced propagators
	// this might help to test fewer combinations of symmetries
	exset allaux;
	map<unsigned, list<exmap> > trafo_by_numaux;
	lst lemom = add_lst(loopmom, extmom);
	for (list<exmap>::iterator t = trafos.begin(); t != trafos.end(); ++t) {
		vector<ex> aux;
		if (is_permutation_symmetry(pmom, *t, max_aux, &aux, 0)) {
			if (force_noexternal) {
				bool ok = true;
				for (size_t i = 0; ok && i < aux.size(); ++i) {
					for (size_t j = 0; ok && j < extmom.nops(); ++j) {
						VERIFY(is_a<symbol>(extmom.op(j)));
						const symbol l = ex_to<symbol>(extmom.op(j));
						const ex d = aux[i].diff(l);
						if (!(d == 0))
							ok = false;
					}
				}
				if (!ok)
					continue;
			}
			if (force_natural) {
				bool ok = true;
				for (size_t i = 0; ok && i < aux.size(); ++i) {
					for (size_t j = 0; ok && j < lemom.nops(); ++j) {
						VERIFY(is_a<symbol>(lemom.op(j)));
						const symbol l = ex_to<symbol>(lemom.op(j));
						const ex d = aux[i].diff(l);
						if (!(d == 0 || d == 1 || d == -1))
							ok = false;
					}
				}
				if (!ok)
					continue;
			}
			allaux.insert(aux.begin(), aux.end());
			trafo_by_numaux[aux.size()].push_back(*t);
		}
	}
	trafos.clear();
	map<unsigned, list<exmap> >::reverse_iterator tn;
	for (tn = trafo_by_numaux.rbegin(); tn != trafo_by_numaux.rend(); ++tn)
		trafos.splice(trafos.end(), tn->second);
	return allaux;
}

map<int, exset> find_aux_propagator_combination(
		map<ex, list<exmap>, ex_is_less>& symm_of_auxcombi, unsigned max_aux,
		const exset& allaux, const vector<ex>& pmom, const list<exmap>& trafos,
		unsigned& ntries) {
	ntries = 0;
	// try all combinations of auxiliary propagators and determine #symmetries
	if (allaux.size() < max_aux)
		ERROR("could not find enough auxiliary propagators induced by symmetries");

	map<int, exset> auxcombis_of_numsymm;
	vector<ex> allauxvec(allaux.begin(), allaux.end());
	vector<int> aa(allaux.size());
	for (size_t i = 0; i < aa.size(); ++i)
		aa[i] = i;
	int* af, *am, *al;
	af = &aa[0], am = &aa[0] + max_aux, al = &aa[0] + aa.size();
	do {
		++ntries;
		vector<ex> mom = pmom;
		lst auxcombi;
		for (size_t i = 0; i < max_aux; ++i) {
			mom.push_back(allauxvec[aa[i]]);
			auxcombi.append(allauxvec[aa[i]]);
		}
		for (list<exmap>::const_iterator t = trafos.begin(); t != trafos.end();
				++t)
			if (is_permutation_symmetry(mom, *t, 0, 0, 0))
				symm_of_auxcombi[auxcombi].push_back(*t);
		auxcombis_of_numsymm[symm_of_auxcombi[auxcombi].size()].insert(
				auxcombi);
	} while (benbear::next_combination(af, am, al));
	return auxcombis_of_numsymm;
}

map<int, exset> find_aux_propagator_combination_v2(
		map<ex, list<exmap>, ex_is_less>& symm_of_auxcombi, unsigned max_aux,
		const exset& allaux, const vector<ex>& pmom, const list<exmap>& trafos,
		unsigned& ntries) {
	// try all combinations of symmetries and determine auxiliary propagators
	map<int, exset> auxcombis_of_numsymm;
	list < list < exmap > ::const_iterator > trafocombi; // combi to test
	list < exmap > tc; // verified trafo combination
	list < vector<ex> > moms; // momenta incl. aux props
	if (trafos.empty())
		ERROR("could not identify identity transformation");
	trafocombi.push_back(trafos.begin());
	moms.push_back(pmom);
	// a maximal set of symmetries for a aux prop combination
	// (is unique for a complete set of auxiliary propagators)
	ntries = 0;
	do {
		++ntries;
		vector<ex> mom = moms.back();
		unsigned ntot = pmom.size() + max_aux;
		list<list<exmap>::const_iterator>::const_iterator t = --trafocombi.end();
		while (t != trafocombi.end()) {
			vector<ex> a;
			if (!is_permutation_symmetry(mom, **t, ntot - mom.size(), &a, 0))
				break; // doesn't work
			for (size_t i = 0; i < a.size(); ++i)
				mom.push_back(a[i]);
			if (!a.empty()) // found new aux prop(s)
				t = trafocombi.begin(); // recheck all trafos include new props
			else
				++t;
		}
		if (t == trafocombi.end()) { // last combi is working
			tc.push_back(*trafocombi.back());
			moms.push_back(mom);
			list<ex> acl;
			for (size_t i = pmom.size(); i < mom.size(); ++i)
				acl.push_back(mom[i]);
			acl.sort(ex_is_less());
			lst ac(acl);
			if (symm_of_auxcombi[ac].size() < tc.size())
				symm_of_auxcombi[ac] = tc;
			// try one trafo more
			trafocombi.push_back(trafocombi.back());
		}
		// next trafo combi
		while (!trafocombi.empty()) {
			++trafocombi.back();
			if (trafocombi.back() != trafos.end())
				break;
			trafocombi.pop_back();
			moms.pop_back();
			if (!tc.empty())
				tc.pop_back();
		}
	} while (!trafocombi.empty());
	return auxcombis_of_numsymm;
}


void print_solutions_brief(const map<int, exset>& aux,
		const map<ex, list<exmap>, ex_is_less>& symm, unsigned max_aux) {
	for (map<int, exset>::const_iterator n = aux.begin(); n != aux.end(); ++n)
		LOG( "  " << n->first << " symmetries:  " //
				<< n->second.size() << " possible aux. prop. combinations");
}

vector<Propagator> assemble_propagators(const vector<Propagator>& prop,
		const ex& combi) {
	vector<Propagator> allprop(prop);
	for (size_t i = 0; i < combi.nops(); ++i)
		allprop.push_back(Propagator(combi.op(i), combi.op(i), 0));
	return allprop;
}

void discard_invalid_solutions(map<int, exset>& aux,
		map<ex, list<exmap>, ex_is_less>& symm, unsigned max_aux,
		const lst& loopmom, const vector<Propagator>& prop) {
	Kinematics* kin = Files::instance()->kinematics();
	for (map<int, exset>::iterator aa = aux.begin(); aa != aux.end(); ++aa)
		for (exset::iterator a = aa->second.begin(); a != aa->second.end();)
			if (a->nops() != max_aux) { // incomplete set of propagators
				++a;
			} else {
				try {
					// would suffice to test this for "optimal" candidates only:
					IntegralFamily fam(kin, "dummy", loopmom,
							assemble_propagators(prop, *a), 0);
					++a;
				} catch (exception&) {
					aa->second.erase(a++);
				}
			}
}

void discard_incomplete_solutions(map<int, exset>& aux,
		map<ex, list<exmap>, ex_is_less>& symm, unsigned max_aux) {
	for (map<int, exset>::iterator aa = aux.begin(); aa != aux.end(); ++aa)
		for (exset::iterator a = aa->second.begin(); a != aa->second.end();)
			if (a->nops() != max_aux) // incomplete set of propagators
				aa->second.erase(a++);
			else
				++a;
}

void print_solutions(const map<int, exset>& aux,
		const map<ex, list<exmap>, ex_is_less>& symm, unsigned max_aux) {
	LOGX("");
	for (map<int, exset>::const_iterator aa = aux.begin(); aa != aux.end();
			++aa) {
		LOGX("  solutions with " << aa->first << " symmetries:");
		for (exset::const_iterator a = aa->second.begin();
				a != aa->second.end(); ++a) {
			//if (a->nops() != max_aux) // incomplete set of propagators
			//	continue;
			LOGX("    aux " << *a << ":");
			VERIFY(symm.find(*a) != symm.end());
			const list<exmap>& ss = symm.find(*a)->second;
			for (list<exmap>::const_iterator s = ss.begin(); s != ss.end(); ++s)
				LOGX("      " << *s);
		}
	}
}

// extract complete solutions with maximal number of symmetries
void write_optimal_families(const map<int, exset>& aux,
		const map<ex, list<exmap>, ex_is_less>& symm, const vector<ex>& pmom,
		const vector<Propagator>& inprops, unsigned max_aux,
		const lst& loopmom, const string& nameprefix, YAML::Emitter& ye) {
	using namespace YAML;
	ye << BeginMap << Key << "integralfamilies" << Value << BeginSeq;
	const Kinematics* kin = Files::instance()->kinematics();
	int nfam = 0;
	map<int, exset>::const_reverse_iterator aa;
	for (aa = aux.rbegin(); aa != aux.rend() && nfam == 0; ++aa) {
		exset::const_iterator a;
		for (a = aa->second.begin(); a != aa->second.end(); ++a)
			if (a->nops() == max_aux) {
				++nfam;
				LOG("\nGenerating family number " << nfam);
				// find permutations
				vector<ex> fmom = pmom;
				for (size_t i = 0; i < a->nops(); ++i) {
					LOG("  aux propagator: " << a->op(i));
					fmom.push_back(a->op(i));
				}
				list<Permutation> perm;
				list < exmap > ss = symm.find(*a)->second;
				list<exmap>::const_iterator s;
				for (s = ss.begin(); s != ss.end(); ++s) {
					map<int, int> p;
					is_permutation_symmetry(fmom, *s, 0, 0, &p);
					perm.push_back(Permutation(p));
				}
				// generate family and print it
				string name = nameprefix + to_string(nfam);
				vector<Propagator> prop = assemble_propagators(inprops, *a);
				IntegralFamily fam(kin, name, loopmom, prop, 0, &perm);
				fam.print(ye);
				list<Permutation>::const_iterator p;
				for (s = ss.begin(), p = perm.begin(); s != ss.end(); ++s, ++p) {
					if (*p == Permutation())
						continue; // skip identity
					Permutation ps(*p);
					ps.relabel(1); // IntegralFamily numbering convention
					YAML::Emitter se;
					se << ps;
					ye << Newline << Comment(string("permutation ") + se.c_str()
							+ ": " + to_string(*s));
					LOG("  permutation " << se.c_str() << ": " << *s);
				}
				ye << Newline;
			}
	}
	ye << EndSeq << EndMap;
	if (nfam == 0)
		ERROR("could not find any complete set of propagators");
	LOG("\nFound " << nfam << " optimal families.\n");
}

list<exmap> admissable_crossings() {
	list < exmap > idxs;
	const Kinematics* kin = Files::instance()->kinematics();
	list<Crossing> oc =
			Files::instance()->crossings(kin->name())->ordered_crossings();
	oc.push_front(Crossing(kin)); // include identity
	for (list<Crossing>::const_iterator c = oc.begin(); c != oc.end(); ++c)
		if (c->is_equivalent_to_identity())
			idxs.push_back(c->rules_momenta());
	return idxs;
}

void FindFamily::run_serial() {
	LOG("Find a family for a set of propagators");
	const Kinematics* kin = Files::instance()->kinematics();
	GiNaC::lst extmom = kin->independent_external_momenta();
	unsigned nloopmom = loop_momenta_.nops();
	unsigned nextmom = extmom.nops();
	unsigned nsprod = IntegralFamily::num_scalar_products(nloopmom, nextmom);
	if (input_propagators_.size() > nsprod)
		ERROR("too many propagators given");
	unsigned max_aux = nsprod - input_propagators_.size();
	LOG(nloopmom << " loops");
	LOG(input_propagators_.size() << " propagators");
	LOG(max_aux << " auxiliary propagators needed");

	LOG("\nAnalyzing initial set of propagators, finding crossings");
	Timer timer;
	// e.g. some momenta assigned to a specific ("maximal") graph
	vector<ex> pmom; //momenta are expanded, mom. cons. applied
	for (size_t i = 0; i < input_propagators_.size(); ++i) {
		if (input_propagators_[i].squaredmass() != 0)
			ERROR( "find_family does not support massive propagators yet");
		pmom.push_back(input_propagators_[i].momentum().expand());
	}
	// crossings equivalent to identity
	list < exmap > idxs = admissable_crossings();
	LOG("  found " << idxs.size() << " crossings equivalent to identity");

	LOG( "\nInitial transformations: determining all symmetry candidates");
	unsigned ntries = 0;
	list < exmap > trafos;
	if (use_alt_trafo_finder_)
		trafos = find_symmetry_candidates_alt(pmom, loop_momenta_, extmom, idxs,
				max_aux, ntries);
	else
		trafos = find_symmetry_candidates(pmom, loop_momenta_, extmom, idxs,
				ntries);
	LOG( "  found " << trafos.size() << " shifts with " << ntries << " tries");
	LOG("  " << timer.get_wall_time_nice_string());

	LOG("\nFiltering: finding symmetries and auxiliary propagators");
	timer.restart();
	exset allaux = find_symmetries_and_auxilary_propagators(trafos, pmom,
			loop_momenta_, extmom, max_aux, force_natural_momenta_, force_no_external_momenta_);
	LOG("  found " << allaux.size() << " different aux props");
	for (exset::const_iterator a = allaux.begin(); a != allaux.end(); ++a)
		LOGX("    auxprop: " << *a);
	LOG("  found " << trafos.size() << " valid symmetries");
	for (list<exmap>::const_iterator t = trafos.begin(); t != trafos.end(); ++t)
		LOGX("    " << *t);
	LOG("  " << timer.get_wall_time_nice_string());

	LOG("\nAnalyzing combinations of auxiliary propagators");
	timer.restart();
	map<int, exset> aux; // auxcombinations by number of symmetries
	map<ex, list<exmap>, ex_is_less> symm; // by auxcombi
	// _v2 is painfully slow
	aux = find_aux_propagator_combination(symm, max_aux, allaux, pmom,
			trafos, ntries);
	LOG("  tested " << ntries << " combinations of symmetries");
	map<ex, list<exmap>, ex_is_less>::const_iterator s;
	for (s = symm.begin(); s != symm.end(); ++s)
		aux[s->second.size()].insert(s->first);
	LOG("  " << timer.get_wall_time_nice_string());

	LOG("\nPrimary sets of auxiliary propagators (incl. invalid/incomplete sets)");
	print_solutions_brief(aux, symm, max_aux);
	LOG("\nAll candidate sets of auxiliary propagators (incl. uncompleted)");
	// it might be possible to complete some of these sets
	discard_invalid_solutions(aux, symm, max_aux, loop_momenta_,
			input_propagators_);
	print_solutions_brief(aux, symm, max_aux);
	print_solutions(aux, symm, max_aux);
	LOG("\nAll completed sets of auxiliary propagators");
	discard_incomplete_solutions(aux, symm, max_aux);
	print_solutions_brief(aux, symm, max_aux);
	print_solutions(aux, symm, max_aux);

	LOG("\nExtracting optimal solutions");
	YAML::Emitter ye;
	write_optimal_families(aux, symm, pmom, input_propagators_, max_aux, loop_momenta_, "autogenerated", ye);
	ofstream of(output_file_.c_str());
	of << ye.c_str();
	of.close();
}
}
