/*  job_relatesymanzik.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_relatesymanzik.h"
#include "functions.h"
#include "int.h"
#include "sector.h"
#include "filedata.h"
#include "equation.h"
#include "ginac/ginac.h"
#include "ginacutils.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

namespace {
JobProxy<RelateSymanzik> dummy;
}

bool RelateSymanzik::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	in.push_back(input_file_);
	out.push_back(output_file_);
	return true;
}

void RelateSymanzik::init() {
}

std::string RelateSymanzik::get_description() const {
	return "find equivalences from Symanzik polynomials";
}

// canonical permutation of xi
ex find_canonical_label(const ex& u, const ex& f, const ex& p, const lst& xi, int t) {
	ex umin = u;
	ex fmin = f;
	ex pmin = p;
	exmap perm;
	for (lst::const_iterator x = xi.begin(); x != xi.end(); ++x)
		perm[*x] = *x;
	exvector xip(xi.begin(), xi.end());
	xip.resize(t);
	sort(xip.begin(), xip.end(), ex_is_less());
	while (next_permutation(xip.begin(), xip.end(), ex_is_less())) {
		exvector::const_iterator x = xip.begin();
		for (exmap::iterator s = perm.begin(); s != perm.end(); ++s)
			s->second = *(x++);
		int c = 0;
		ex up = u.subs(perm, subs_options::no_pattern);
		if (c == 0 && (c = umin.compare(up)) > 0)
			continue;
		ex fp = f.subs(perm, subs_options::no_pattern);
		if (c == 0 && (c = fmin.compare(fp)) > 0)
			continue;
		ex pp = p.subs(perm, subs_options::no_pattern);
		if (c == 0 && (c = pmin.compare(pp)) > 0)
			continue;
		if (c == 0)
			continue;
		umin = up;
		fmin = fp;
		pmin = pp;
	}
#ifdef NEW_GINAC
    return lst({umin, fmin, pmin});
#else
    return lst(umin, fmin, pmin);
#endif
}

void RelateSymanzik::run_serial() {
	LOG("Reading integrals from file \"" << input_file_ << '\"');
	InFileINTs in(input_file_);
	OutFileEquations out(output_file_);
	set<INT> ints;
	in.get_all(ints);
	in.close();
	LOG("Analyzing " << ints.size() << " integrals");
	map<ex, set<INT>, ex_is_less> int_by_label;
	int maxt = 0;
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i)
		maxt = max(maxt, (int)i->t());
	lst xi = create_symbols("x", maxt);
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i) {
		if (i->s() > 0)
			ERROR("sorry, no support for numerator in " << *i);
		LOGN("  " << *i << flush);
		ex u, f;
		i->get_sector().find_UF_polynomials(xi, u, f);
		// p consists of powers of xi and adjusts the overall minus sign
		ex p = pow(ex(-1), (int)i->r());
		for (unsigned j = 0, k = 0 ; j < i->size(); ++j)
			if (i->v_i(j) > 0)
				p *= pow(xi.op(k++), (int)(i->v_i(j)) - 1);
		LOG("  |  U F p : { " << u << ",  " << f << ",  " << p << " }");
		// canonical permutation of xi
		int_by_label[find_canonical_label(u, f, p, xi, i->t())].insert(*i);
	}
	LOG(int_by_label.size() << " of all integrals found to be different");
	if (int_by_label.size() == ints.size())
		LOG("could not identify any integrals");
	map<ex, set<INT>, ex_is_less >::const_iterator ibl;
	for (ibl = int_by_label.begin(); ibl != int_by_label.end(); ++ibl) {
		const set<INT>& same = ibl->second;
		if (same.size() <= 1)
			continue;
		LOG("found " << same.size() << " to be equal");
		set<INT>::const_iterator i = same.begin();
		INT rhs(*i++);
		for (; i != same.end(); ++i) {
			Identity id;
			id.insert(rhs, -1);
			id.insert(*i, 1);
			out << id;
		}
	}
	out.finalize();
}

} // namespace Reduze
