/*  job_verifypermutationsymmetries.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_verifypermutationsymmetries.h"
#include "functions.h"
#include "files.h"
#include "integralfamily.h"
#include "combinatorics.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<VerifyPermutationSymmetries> dummy;
}

void VerifyPermutationSymmetries::run_serial() {
	LOGX("Verifying permutation symmetries of integral family '" //
			<< integralfamily_ << "'");
	const IntegralFamily* f = Files::instance()->uncrossed_integralfamily(
			integralfamily_);
	const list<Permutation>& perm_syms =
			f->incomplete_permutation_symmetries();
	bool found_error = false;
	list<Permutation>::const_iterator p;
	for (p = perm_syms.begin(); p != perm_syms.end(); ++p) {
		YAML::Emitter ye;
		p->print_to_onebased(ye);
		string ps = ye.c_str();
		LOGX("  verifying " << ps);
		pair<bool, string> res = f->is_valid_perm_sym(*p);
		if (!res.first) {
			LOGX("    FALSE (" << res.second << ")");
			found_error = true;
			if (abort_on_first_error_)
				ERROR("Invalid permutation symmetry: " << ps //
						<< " of integral family " << integralfamily_//
						<< " (" << res.second << ")");
		}
	}
	if (found_error)
		ERROR("Invalid global permutation symmetries detected! See log-file");
	else
		LOGX("All permutation symmetries verified");

	LOGX("");
	if (true) {
		LOGX("Complete permutation symmetries of integral family '" //
				<< integralfamily_ << "'");
		const list<Permutation>& perm_syms = f->permutation_symmetries();
		list<Permutation>::const_iterator p;
		for (p = perm_syms.begin(); p != perm_syms.end(); ++p) {
			YAML::Emitter ye;
			p->print_to_onebased(ye);
			LOGX(ye.c_str());
		}
	}

	string tmp_dummy = dummy_output_file_ + ".tmp";
	ofstream out(tmp_dummy.c_str());
	out << "Dummy file to ensure verification of permutation symmetries"
			<< " of family '" << integralfamily_ << "' has completed" << endl;
	if (!out)
		ERROR("Cannot open file: " << tmp_dummy);
	out.close();
	rename(tmp_dummy, dummy_output_file_);
}

bool VerifyPermutationSymmetries::find_dependencies(
		const set<string>& outothers, //
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	Files* files = Files::instance();
	if (!dummy_output_file_.empty())
		out.push_back(dummy_output_file_);
	string cr_file = files->get_filename_crossings();
	// if a crossing job is available then the perm.verification must wait
	if (outothers.find(get_canonical_filename(cr_file)) != outothers.end())
		in.push_back(cr_file);
	return true;
}

std::string VerifyPermutationSymmetries::get_description() const {
	std::stringstream s;
	s << "verify permutation symmetries of family '" << integralfamily_ << "'";
	return s.str();
}

}
