/*  job_verifysameterms.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_verifysameterms.h"
#include "functions.h"
#include "filedata.h"
#include "equationlist.h"
#include "files.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<VerifySameTerms> dummy;
}

void VerifySameTerms::run_serial() {
	LOG("Verifying same terms in file \"" << input_filename1_
			<< "\" and file \"" << input_filename2_ << "\"");

	LOG("Checking type of input files");
	bool have_eqs = false;
	if (InFileEquations::check_filetype(input_filename1_)
			&& InFileEquations::check_filetype(input_filename2_))
		have_eqs = true;

	LOG("  => files contain " << (have_eqs ? "equations" : "linear combinations"));

	InFileLinearCombinations in1(input_filename1_);
	InFileLinearCombinations in2(input_filename2_);
	size_t s1 = in1.file_size();
	size_t s2 = in2.file_size();
	ProgressBar pbar(2, "comparing " + to_string(s1) + " bytes with "
			+ to_string(s2) + " bytes", s1);
	pbar.start();
	string abort_string = "files \"" + input_filename1_ + "\" and \""
			+ input_filename2_ + "\" don't contain the same terms";
	size_t count = 0;
	LinearCombination lc1, lc2;
	bool b1 = in1.get(lc1);
	bool b2 = in2.get(lc2);
	while (b1 && b2) {
		++count;
		if (have_eqs) {
			Identity id2;
			id2.swap_terms(lc2);
			lc1.replace(id2, true);
			if (!lc1.empty())
				ERROR(abort_string << ":\nmismatch at equation number " << count);
		} else {
			if (lc1.name() != lc2.name())
				LOGX("name mismatch for linear combination number " << count
						<< ": " << lc1.name() << " vs. " << lc2.name());
			if (lc1.LinCombBasic<INT, GiNaC::ex>::operator!=(lc2))
				ERROR(abort_string << ":\nterms mismatch for linear combination number " << count);
		}
		pbar.print(in1.tellg());
		b1 = in1.get(lc1);
		b2 = in2.get(lc2);
	}
	if (b1 != b2)
		ERROR(abort_string << ":\nnot the same number of equations");
	pbar.end();
	in1.close();
	in2.close();
	LOG("OK!");
}

bool VerifySameTerms::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	in.push_back(input_filename1_);
	in.push_back(input_filename2_);
	return true;
}

void VerifySameTerms::init() {
	if (input_filename1_.empty() || input_filename2_.empty())
		ERROR("Input or output file undefined");
}

std::string VerifySameTerms::get_description() const {
	std::stringstream s;
	s << "verify same terms in file \"" << short_filename(input_filename1_)
			<< "\" and file \"" << short_filename(input_filename2_) << "\"";
	return s.str();
}

}
