/*  polarization.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef POLARIZATION_H_
#define POLARIZATION_H_

#ifndef OVERRIDE
#ifdef NEW_GINAC
#define OVERRIDE override
#else
#define OVERRIDE
#endif
#endif

#include <ginac/ginac.h>

namespace Reduze {

class FeynmanRules;

//
// SPECIAL METRIC TENSOR
//

/// special metric tensor: m(a,b,c,d) = g(a,b)*g(c,d) - g(a,d)*g(b,c)
class metric4_tensor: public GiNaC::tensor {
GINAC_DECLARE_REGISTERED_CLASS(metric4_tensor, GiNaC::tensor)
public:
	GiNaC::ex eval_indexed(const GiNaC::basic& i) const OVERRIDE;
	bool contract_with(GiNaC::exvector::iterator self,
			GiNaC::exvector::iterator other, GiNaC::exvector& v) const OVERRIDE;
protected:
	unsigned return_type() const OVERRIDE;
	void do_print(const GiNaC::print_context& c, unsigned level) const;
};
GINAC_DECLARE_UNARCHIVER(metric4_tensor)
;

inline unsigned metric4_tensor::return_type() const {
	return GiNaC::return_types::commutative;
}

/// Lorentz four vector
class four_vector_tensor: public GiNaC::tensor {
GINAC_DECLARE_REGISTERED_CLASS(four_vector_tensor, GiNaC::tensor)
public:
	four_vector_tensor(const GiNaC::ex& vector);
	//bool info(unsigned inf) const;
	GiNaC::ex eval_indexed(const GiNaC::basic& i) const OVERRIDE;
	bool contract_with(GiNaC::exvector::iterator self,
			GiNaC::exvector::iterator other, GiNaC::exvector& v) const OVERRIDE;
	GiNaC::ex get_vector() const;

	size_t nops() const OVERRIDE;
	GiNaC::ex op(size_t i) const OVERRIDE;
	GiNaC::ex & let_op(size_t i) OVERRIDE;

protected:
	unsigned return_type() const OVERRIDE;
	void do_print(const GiNaC::print_context& c, unsigned level) const;
private:
	GiNaC::ex vector_;
};
GINAC_DECLARE_UNARCHIVER(four_vector_tensor)
;

inline unsigned four_vector_tensor::return_type() const {
	return GiNaC::return_types::commutative;
}
//
//
//


/// general polarization: momentum, mass, conjugated
class Polarization: public GiNaC::basic {
GINAC_DECLARE_REGISTERED_CLASS(Polarization, GiNaC::basic)
public:
	Polarization(const GiNaC::ex& momentum, const GiNaC::ex& mass,
			bool conjugated, const GiNaC::ex& orthogonal_vector =
					four_vector_tensor(0));

	virtual ~Polarization();

	/// returns the momentum (a four vector tensor)
	GiNaC::ex momentum() const;
	/// returns the mass
	GiNaC::ex mass() const;
	/// returns the orthogonal vector (a four vector tensor)
	GiNaC::ex orthogonal_vector() const;

	GiNaC::ex momentum_raw() const;
	GiNaC::ex orthogonal_vector_raw() const;

	/// whether they have the same massses, momenta and orthogonal vectors
	bool has_same_mass_mom_ortho(const Polarization& other) const;
	/// whether the polarization is conjugated
	bool is_conjugated() const;

	size_t nops() const OVERRIDE;
	GiNaC::ex op(size_t i) const OVERRIDE;
	GiNaC::ex & let_op(size_t i) OVERRIDE;
	GiNaC::ex conjugate() const OVERRIDE;

	/// returns a unique identifier for the polarization sum '*this * other'
	std::string unique_pol_sum_name(const Polarization& other) const;
	/// identifier for this polarization
	virtual std::string pol_sum_name() const;

protected:
	unsigned return_type() const OVERRIDE;

protected:
	GiNaC::ex momentum_, mass_;
	bool conjugated_;
	GiNaC::ex orthogonal_vector_;
};

inline GiNaC::ex Polarization::momentum() const {
	return momentum_;
}
inline GiNaC::ex Polarization::mass() const {
	return mass_;
}
inline GiNaC::ex Polarization::orthogonal_vector() const {
	return orthogonal_vector_;
}
inline bool Polarization::has_same_mass_mom_ortho(const Polarization& other) const {
	return (mass_.is_equal(other.mass_) && momentum_.is_equal(other.momentum_)
			&& orthogonal_vector_.is_equal(other.orthogonal_vector_));
}
inline bool Polarization::is_conjugated() const {
	return conjugated_;
}
inline unsigned Polarization::return_type() const {
	return GiNaC::return_types::commutative;
}

//
//
//


/// gluon polarization
class GluonPolarization: public Polarization {
GINAC_DECLARE_REGISTERED_CLASS(GluonPolarization, Polarization)
public:
	/// constructs a gluon polarization tensor
	/** orthogonal_vector is any four vector orthogonal to pol.:  (n eps) = 0
	 ** spin sum of \f$(eps_\mu eps^*_\nu)\f$ is controlled by orthogonal_vector n:
	 **   => \f$ -g_{\mu\nu}\f$                                      if n==0,
	 **   => \f$ -g_{\mu\nu} + (k_\mu n_\nu + n_\mu k_\nu)/(k n)
	 **                + n^2 k_\mu k_\nu / (k n)^2 \f$               if n!=0
	 ** multiplied by \f$delta_{a b}\f$
	 **/
	GluonPolarization(const GiNaC::ex& momentum, const GiNaC::ex& mass,
			bool conjugated, const GiNaC::ex& orthogonal_vector);

	GiNaC::ex conjugate() const OVERRIDE;

protected:
	void do_print(const GiNaC::print_context& c, unsigned level = 0) const;
};

//
//
//


/// ghost polarization
class GhostPolarization: public Polarization {
GINAC_DECLARE_REGISTERED_CLASS(GhostPolarization, Polarization)
public:
	GhostPolarization(const GiNaC::ex& momentum, const GiNaC::ex& mass,
			bool conjugated) :
		inherited(momentum, mass, conjugated) {
	}
	GiNaC::ex conjugate() const OVERRIDE;

protected:
	void do_print(const GiNaC::print_context& c, unsigned level = 0) const;
};

//
//
//


/// photon polarization
class PhotonPolarization: public Polarization {
GINAC_DECLARE_REGISTERED_CLASS(PhotonPolarization, Polarization)
public:
	PhotonPolarization(const GiNaC::ex& momentum, const GiNaC::ex& mass,
			bool conjugated) :
		inherited(momentum, mass, conjugated) {
	}
	GiNaC::ex conjugate() const OVERRIDE;

protected:
	void do_print(const GiNaC::print_context& c, unsigned level = 0) const;
};

//
//
//


/// spinor: momentum, mass, u_type = true/false, conjugated = false/true
class Spinor: public Polarization {
GINAC_DECLARE_REGISTERED_CLASS(Spinor, Polarization)
public:
	Spinor(const GiNaC::ex& momentum, const GiNaC::ex& mass, bool u_type,
			bool dirac_conjugated);
	virtual ~Spinor();

	bool is_u() const;
	bool is_v() const;
	bool is_ubar() const;
	bool is_vbar() const;

	GiNaC::ex conjugate() const OVERRIDE;

	virtual std::string pol_sum_name() const OVERRIDE;

protected:
	void do_print(const GiNaC::print_context& c, unsigned level = 0) const;

private:
	bool u_type_;
};

inline bool Spinor::is_u() const {
	return u_type_ && !conjugated_;
}
inline bool Spinor::is_v() const {
	return !u_type_ && !conjugated_;
}
inline bool Spinor::is_ubar() const {
	return u_type_ && conjugated_;
}
inline bool Spinor::is_vbar() const {
	return !u_type_ && conjugated_;
}

//
//
//


/// performs the polarization sum on corresponding polarizations in the products 'va' and 'vb'
GiNaC::ex polarization_sum(const GiNaC::exvector& va,
		const GiNaC::exvector& vb, const FeynmanRules* fr);

GiNaC::ex metric_l_diff_wrapper(const GiNaC::ex& e1, const GiNaC::ex& e2,
		const GiNaC::ex& e3, const GiNaC::ex& e4);

GiNaC::ex four_vector_tensor_wrapper(const GiNaC::ex& p, const GiNaC::ex& mu);

// gluon
GiNaC::ex pol_gluon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu, const GiNaC::ex& a, const GiNaC::ex& ortho);
GiNaC::ex polc_gluon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu, const GiNaC::ex& a, const GiNaC::ex& ortho);

// ghost
GiNaC::ex pol_ghost_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& a);
GiNaC::ex polc_ghost_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& a);

// photon
GiNaC::ex pol_photon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu);
GiNaC::ex polc_photon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu);

GiNaC::ex u_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i);
GiNaC::ex ubar_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i);
GiNaC::ex v_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i);
GiNaC::ex vbar_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i);

} // namespace Reduze

#endif /* POLARIZATION_H_ */
