/*  sectormappings.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "sectormappings.h"
#include "kinematics.h"
#include "int.h"
#include "equationlist.h"
#include "identitygenerator.h"
#include "functions.h"
#include "files.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "graph.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

bool SectorMappings::no_explicit_shift_verification_ = false;

// constructors

SectorMappings::SectorMappings(const IntegralFamily* ic) :
	integralfamily_(ic) {
}

SectorMappings::SectorMappings(const CrossedIntegralFamily* ic) :
	integralfamily_(ic) {
	// don't waste resources copying zero sectors and sectors without a graph:
	// derive this information on-the-fly when needed
	string sn = ic->source_integralfamily()->name();
}

bool SectorMappings::is_zero(const Sector& s) {
	return get_shifted_sector(s).is_obviously_zero();
}

// get equivalent and shifted sectors

INT SectorMappings::get_equivalent_or_unique_zero(const INT& in) {
	// minimal crossing, permutation symmetries
	INT i = Crossing::to_minimal_crossing(in);
	// exploit permutation symmetries and zero sectors of uncrossed family
	pair<INT, Crossing> unx = Crossing::uncross(i);
	// permutation symmetry
	INT inox = unx.first.get_equivalent();
	// map zero integrals to zero sector integral
	INT z = INT(inox.integralfamily(), 0);
	if (z.is_obviously_zero()) { // #loops>0, ID=0 really is zero
		if (inox.is_obviously_zero())
			return z;
		const SectorMappings* m = Files::instance()->sectormappings(
				inox.integralfamily()->name());
		if (m->zero_sectors_.find(inox.id()) != m->zero_sectors_.end())
			return z;
	}
	// all done
	return unx.second.transform(inox);
}

Sector SectorMappings::get_shifted_sector(const Sector& secin, bool recurse) {
	// minimal crossing, permutation symmetry, zero sectors
	INT i = get_equivalent_or_unique_zero(INT(secin));
	// check for explicit shifts of crossed sector (to uncrossed version of same sector)
	if (i.integralfamily()->is_crossed()) {
		const SectorMappings* xm = Files::instance()->sectormappings(
				i.integralfamily()->name());
		map<int, Sector>::const_iterator to =
				xm->sector_relations_.find(i.id());
		if (to != xm->sector_relations_.end()) {
			if (!recurse)
				return get_equivalent_or_unique_zero(INT(to->second)).get_sector();
			return get_shifted_sector(to->second, recurse);
		}
	}
	// use shifts from uncrossed family
	pair<INT, Crossing> unx = Crossing::uncross(i);
	const SectorMappings* m = Files::instance()->sectormappings(
			unx.first.integralfamily()->name());
	int id = unx.first.id();
	map<int, Sector>::const_iterator targetnox = m->sector_relations_.find(id);
	if (targetnox != m->sector_relations_.end()) {
		Sector target = unx.second.transform(targetnox->second);
		if (!recurse)
			return get_equivalent_or_unique_zero(INT(target)).get_sector();
		return get_shifted_sector(target, recurse);
	}
	// nothing to reduce
	return i.get_sector();
}

bool SectorMappings::search_corner_reduction_to_zero(int sector_id,
		bool allow_longtest) const {
	Sector sector(integralfamily_, sector_id);
	LOGXX("Checking whether sector " << sector << " is zero");
	// for tree level integral family, t=0 is not zero
	if (sector.is_obviously_zero())
		return LOGXX("Is trivial zero sector"), true;
	if (sector.is_obviously_non_zero())
		return LOGXX("Is trivial tree level (non-zero) sector"), false;
	if (zero_sectors_.find(sector_id) != zero_sectors_.end())
		return LOGXX("Is already known to be zero"), true;
	INT corner(integralfamily_, sector_id);
	INT equiv = corner.get_equivalent();
	if (equiv != corner) {
		WARNING("Performing zero test for equivalent integral " << equiv
				<< " instead of " << corner);
		corner = equiv;
		sector = Sector(integralfamily_, equiv.id());
	}
	int t = corner.t();
	set<INT> seeds;
	seeds.insert(corner);
	IdentityList idlist;
	IdentityGeneratorOptions options;
	options.set_eliminate_subsectors_by_shift(false); // sectormappings aren't known yet
	options.set_transform_integrals_to_equivalent(true); // default not sufficient
	static IdentityGenerator generator(options); // static gives great speedup here
	generator.generate(seeds, idlist, "ibp");
	idlist.remove_empty_equations();
	if (idlist.size() == 0)
		WARNING("Empty identity list in zero test for sector " << sector_id);
	LOGXX(idlist);
	idlist.reduze();
	if (idlist.gives_reduction_zero(corner))
		return LOGXX("Is zero since corner reduces to zero"), true;
	// increasing s2 can't help for maximal sector
	if (sector_id == integralfamily_->max_sector_id())
		return LOGXX("Treat maximal sector as non-zero"), false;
	// test with more equations, s2 -> s2 + 1
	if (allow_longtest) {
		SeedGeneratorOptions seed_options;
		seed_options.set_reduce_to_equivalent(true);
		SeedGenerator seed_generator(integralfamily_, seed_options);
		set<INT> seeds2;
		seed_generator.generate(sector_id, t, t, 1, 1, seeds2);
		IdentityList idlist2;
		generator.generate(seeds2, idlist2, "ibp");
		idlist2.splice(idlist);
		idlist2.remove_empty_equations();
		idlist2.reduze();
		seeds2.insert(seeds2.begin(), corner);
		if (idlist2.gives_reduction_zero(corner))
			return LOGXX("Is zero since corner reduces to zero (ext)"), true;
	}
	return LOGXX("Not found to reduce to zero"), false;
}

void SectorMappings::get_graphs(std::list<SectorGL>& sectorgraphs,
		const std::set<Sector>& sectors, bool minimize_by_twists,
		bool construct_minimal_graphs, bool verbose) {
	string twistonoff = (minimize_by_twists ? "on" : "off");
	string minonoff = (construct_minimal_graphs ? "on" : "off");
	ProgressBar pbar(2, "finding graphs: (twists " + twistonoff
			+ " / construct minimal " + minonoff + ")", sectors.size());
	if (verbose)
		pbar.start();
	for (set<Sector>::const_iterator s = sectors.begin(); s != sectors.end(); ++s) {
		if (verbose)
			pbar.print();
		SectorGraph g;
		bool det_non_zero;
		if (s->find_graph(g, det_non_zero, construct_minimal_graphs)) {
			if (minimize_by_twists)
				g.minimize_by_twists();
			sectorgraphs.push_back(SectorGL());
			sectorgraphs.back().swap(g); // swapping SectorGraph into SectorGL
		}
	}
	if (verbose)
		pbar.end();
}

/* Construct a connected graph and disconnect vacuum components.
 * The graph can have n1 vertices of degree 1 (external nodes),
 * n3 vertices of degree 3 and possible higher vertices.
 * The maximum number of edges a graph with given loop L can have is maximal
 * if the degrees are small, eg. a 4 vertex can be split in two 3 vertices
 * increasing the number of edges
 *
 *  L = |edges| + 1 - n1 - n3
 *  2 |edges| =  n1 + 3 n3
 *
 *  max |edges| = 3 (L-1) + 2 n1
 *  max |internal edges| = 3 (L-1) + n1
 *
 */

void SectorMappings::find_graphs(std::list<SectorGraph>& sectorgraphs,
		const std::set<Sector>& sectors, bool minimize_by_twists,
		bool construct_minimal_graphs, bool verbose) {

	if (verbose)
		LOG("Finding representing graphs of sectors (twists " //
				<< (minimize_by_twists ? "on" : "off")//
				<< " / construct minimal "//
				<< (construct_minimal_graphs ? "on" : "off") << "):");
	ProgressBar pbar(2, "finding graphs", sectors.size());
	if (verbose)
		pbar.start();

	// maximum number of internal edges of a graph
	int loop = integralfamily_->loop_momenta().nops();
	int ext = integralfamily_->kinematics()->external_momenta().nops();
	int max_internal_edges = ((loop == 0) ? 0 : (3 * (loop - 1) + ext));
	if (loop == 1 && ext == 0) // one self-loop
		max_internal_edges = 1;

	// id of sector --> id of direct subsector for which graphs are already constructed
	map<int, int> subsec_of_sec;

	// id of sector S --> pair (graphs of sector S,  ids of supersectors of S)
	map<int, pair<list<SectorGraph>, set<int> > > graphs_for_supersecs;

	set<Sector>::const_iterator sec;
	for (sec = sectors.begin(); sec != sectors.end(); ++sec) {
		if (verbose)
			pbar.print();
		int secid = sec->id();
		const IntegralFamily* secfam = sec->integralfamily();
		VERIFY(*secfam == *integralfamily_);

		// test whether sector cannot have a graph (to many edges for 3-vertices)
		if (sec->t() > max_internal_edges) {
			sectors_without_graph_.insert(secid);
			continue;
		}

		// test for trivial zero sector
		if (sec->is_obviously_zero()) {
			zero_sectors_.insert(secid);
			continue;
		}

		// test whether a subsector has no graph
		set<int> equisubsec = secfam->get_subsector_equivalents(secid, false);
		set<int>::iterator n = equisubsec.begin();
		for (; n != equisubsec.end(); ++n)
			if (sectors_without_graph_.find(*n) != sectors_without_graph_.end())
				break;
		if (n != equisubsec.end()) {
			sectors_without_graph_.insert(secid);
			continue;
		}

		// find a direct subsector for which the graphs are already constructed
		map<int, int>::iterator subsec = subsec_of_sec.find(secid);
		map<int, pair<list<SectorGraph>, set<int> > >::iterator graphs_for_super;
		graphs_for_super = graphs_for_supersecs.end();
		if (subsec != subsec_of_sec.end())
			graphs_for_super = graphs_for_supersecs.find(subsec->second);

		// construct the graphs from the graphs of a subsector or from scratch
		list<SectorGraph> graphlist;
		bool det_non_zero = true;
		if (graphs_for_super != graphs_for_supersecs.end()) {
			int subsec_id = subsec->second;
			const list<SectorGraph>& subsec_graphs = graphs_for_super->second.first;
			sec->construct_graphs(graphlist, subsec_id, subsec_graphs);
			// clean up to safe memory
			subsec_of_sec.erase(subsec);
			graphs_for_super->second.second.erase(secid);
			if (graphs_for_super->second.second.empty())
				graphs_for_supersecs.erase(graphs_for_super);
		} else {
			sec->construct_graphs(graphlist, det_non_zero);
		}

		// case where no graph exists or sector is trivially zero
		if (graphlist.empty()) {
			if (det_non_zero)
				sectors_without_graph_.insert(secid);
			else
				zero_sectors_.insert(secid); // det = 0
			continue;
		}

		// case where graphs have been found
		SectorGraph secgraph;
		if (construct_minimal_graphs)
			secgraph = Sector::cleanup_and_select_minimal_graph(graphlist);
		else
			secgraph = Sector::cleanup_and_select_first_graph(graphlist);
		if (minimize_by_twists)
			secgraph.minimize_by_twists();
		sectorgraphs.push_back(SectorGraph());
		sectorgraphs.back().swap(secgraph);

		// keep the remaining graphs if needed for graphs of supersectors
		set<int> supersecs; // sectors to be constructed from graphs of *sec
		set<int> all_supersecs = secfam->get_immediate_supersectors(secid);
		set<int>::const_iterator sup;
		for (sup = all_supersecs.begin(); sup != all_supersecs.end(); ++sup) {
			if (subsec_of_sec.find(*sup) == subsec_of_sec.end()
					&& sectors.find(Sector(secfam, *sup)) != sectors.end()) {
				subsec_of_sec[*sup] = secid;
				supersecs.insert(*sup);
			}
		}
		if (!supersecs.empty()) {
			pair<list<SectorGraph>, set<int> >& ref = graphs_for_supersecs[secid];
			ref.first.swap(graphlist);
			ref.second.swap(supersecs);
		}
	}
	if (verbose) {
		pbar.end();
		LOG("Found " << sectorgraphs.size() << " graphs");
	}
}

void SectorMappings::find_sector_shifts(
		std::set<SectorGL>& minimal, //
		const std::set<SectorGL>& sectors, //
		std::map<SectorGL, std::list<SectorGL> >& new_targets_with_equiv,
		bool minimize_target_crossings, //
		bool find_zero_sectors,//
		bool allow_crossings,//
		bool allow_general_crossings,//
		std::map<Sector, std::list<std::map<int, int> > >& cached_node_perm,
		bool verbose) {

	if (verbose)
		LOG("Finding sector shifts:");
	ProgressBar pbar(2, "sector shifts:", sectors.size());
	if (verbose) {
		pbar.start();
		LOGX("");
	}
	int num_shifts = 0;
	//map<Sector, list<map<int, int> > > symmetry_shifts;
	for (set<SectorGL>::const_iterator s = sectors.begin(); s != sectors.end(); ++s) {
		const SectorGL& secgraph = *s;
		if (verbose) {
			pbar.print();
			LOGX("\nFinding shift for sector " << secgraph.sector());
		}
		set<SectorGL>::const_iterator m;
		for (m = minimal.begin(); m != minimal.end(); ++m) {
			const SectorGL& min = *m;
			const IntegralFamily* f = min.sector().integralfamily();
			ASSERT(*f == *f->source_integralfamily()); // min not from crossed integral family
			ASSERT(min.sector() < secgraph.sector()); // min lower than sec
			ASSERT(*secgraph.sector().integralfamily() == *integralfamily_); // sec belong to this integral family
			Crossing crossing(f->kinematics());
			GiNaC::exmap shift;
			GiNaC::ex det(0);
			if (min.label_pair().first == secgraph.label_pair().first) {
				list<map<int, int> > node_perms;
				const list<map<int, int> >* node_perms_p = &node_perms;
				// cache the node permutation if target crossings should be minimized, needed later again for sec.syms.
				if (minimize_target_crossings && min.sectorgraph().num_external_nodes() > 0) {
					map<Sector, list<map<int, int> > >::const_iterator found;
					found = cached_node_perm.find(min.sector());
					if (found == cached_node_perm.end()) {
						cached_node_perm[min.sector()] =
								min.sectorgraph().find_node_symmetry_group();
						found = cached_node_perm.find(min.sector());
					}
					node_perms_p = &(found->second);
				}
				// even though the labels are equal there might be no shift if it involves not allowed crossings
				try {
					det = Graph::find_shift(secgraph.sectorgraph(), secgraph.label_pair(),
							min.sectorgraph(), min.label_pair(), shift, crossing,
							*node_perms_p);
				} catch (exception& exc) {
					LOGX(exc.what());
					continue;
				}
				Crossing inverse_crossing = crossing.inverse();
				string name = inverse_crossing.name_for_crossed_family(f);
				const Sector target = Sector(Files::instance()->integralfamily(
						name), min.sector().id());
				const Sector& source = secgraph.sector();
				if (target < source && det.is_equal(1) && (//
						allow_general_crossings //
								|| crossing.is_identity()//
								|| (allow_crossings
										&& crossing.is_equivalent_to_identity()))) {
					++num_shifts;
					LOGX("Found shift from " << source << " to " << target);
					insert_shift(source.id(), target, shift);
					// add uncrossed to new targets if it is not shifted to another family
					if (*f == *integralfamily_)
						new_targets_with_equiv[min].push_back(secgraph);
					else if (find_zero_sectors && is_zero(min.sector()))
						zero_sectors_.insert(source.id());
					break;
				} else if (verbose) {
					LOGX("Reject shift with |det| = " << det << " from sector "
							<< source << " to " << target);
					LOGX("  shift:    " << shift);
					LOGX("  crossing: " << crossing.name());
				}
			}
		}
		if (m == minimal.end()) { // sec not shifted to any sector in minimal
			if (verbose)
				LOGX("No shift found for sector " << secgraph.sector());
			minimal.insert(minimal.end(), secgraph);
			new_targets_with_equiv[secgraph];
		}
	}
	if (verbose) {
		pbar.end();
		LOG("Found " << num_shifts << " new shifts to lower sectors");
	}
}

void SectorMappings::insert_zero_sectors(const std::set<int>& sector_ids) {
	zero_sectors_.insert(sector_ids.begin(), sector_ids.end());
}

int SectorMappings::find_zero_sectors(
		const std::map<Sector, std::list<Sector> >& sectors_by_rep,
		bool allow_longtest) {
	int tt = -1, col = 1, num_zeros = 0;
	map<Sector, list<Sector> >::const_iterator m;
	LOGN(endl);
	for (m = sectors_by_rep.begin(); m != sectors_by_rep.end(); ++m) {
		ASSERT(*(m->first.integralfamily()) == *integralfamily_);
		const Sector& s = m->first;
		int id = s.id();

		if (zero_sectors_.find(id) != zero_sectors_.end())
			continue;
		set<int> sub = integralfamily_->get_subsector_equivalents(id, false);
		for (set<int>::iterator n = sub.begin(); n != sub.end();)
			if (zero_sectors_.find(*n) != zero_sectors_.end()) {
				sub.erase(n++);
			} else if (Sector(integralfamily(), *n).is_obviously_zero()) {
				zero_sectors_.insert(*n);
				sub.erase(n++);
			} else {
				break; // non-zero sub sectors for id
			}
		// calculate if zero sector (skip test if non-zero subsectors present)
		bool found_zero = false;
		if (sub.empty() && search_corner_reduction_to_zero(id, allow_longtest)) {
			zero_sectors_.insert(id); // is zero
			// include isomorphic zero sectors
			for (list<Sector>::const_iterator z = m->second.begin(); z
					!= m->second.end(); ++z) {
				ASSERT(*(z->integralfamily()) == *integralfamily_);
				zero_sectors_.insert(z->id());
			}
			found_zero = true;
			++num_zeros;
		}
		if (s.t() != tt) {
			if (tt != -1)
				LOGN(endl);
			tt = s.t();
			col = 1;
			LOGN("   t = " << tt << ":\t\b\b\b\b" << flush);
		} else if ((col++) % 60 == 0) {
			LOGN (endl << "\t\t\b\b\b\b" << flush);
		}
		if (sub.empty() && found_zero)
			LOGN("0" << flush);
		else
			LOGN("x" << flush);
	}
	LOG("");
	return num_zeros;
}

void SectorMappings::find_symmetry_shifts(const std::set<SectorGL>& sectors,
		bool examine_twists) {
	ProgressBar pbar(2, "finding symmetry shifts", sectors.size());
	pbar.start();
	bool test_line_graph = false;
	for (set<SectorGL>::const_iterator s = sectors.begin(); s != sectors.end();
			++s) {
		//LOGX("\nFinding symmetry shifts for sector " << s->sector());
		ASSERT(*s->sector().integralfamily() == *integralfamily_);
		list<exmap> shifts;
		s->sectorgraph().find_symmetry_shifts(shifts, examine_twists);
		if (test_line_graph) {
			list<exmap> eshifts;
			s->sectorgraph().find_edge_symmetry_shifts(eshifts, examine_twists);
			VERIFY(shifts.size() == eshifts.size());
		}
		if (!shifts.empty())
			insert_symmetry_shifts(s->sector().id(), shifts);
		LOGX("Found " << shifts.size() << " symmetry shifts for sector " //
				<< s->sector());
		pbar.print();
	}
	pbar.end();
}

bool SectorMappings::find_permutations_for_shift(const Sector& from,
		const Sector& to, const GiNaC::exmap& shift,
		std::vector<int>& perm ) {
	const vector<Propagator>& tp = to.integralfamily()->propagators();
	const vector<Propagator>& fp = from.integralfamily()->propagators();
	map<ex, int, ex_is_less> targets;
	for (int i = 0 ; i < (int)tp.size(); ++i)
		if (((1 << i) & to.id()) != 0)
			targets.insert(make_pair(tp[i], i));
	perm.resize(fp.size());
	int torec = 0;
	for (int i = 0 ; i < (int)fp.size(); ++i) {
		if (((1 << i) & from.id()) != 0) {
			ex shifted = ex(fp[i]).subs(shift).eval();
			map<ex, int, ex_is_less>::iterator t = targets.find(shifted);
			if (t == targets.end())
				return false;
			perm[i] = t->second;
			torec |= (1 << t->second);
		} else {
			perm[i] = -1;
		}
	}
	if (to.id() != torec || from.t() != to.t())
		return false;
	return true;
}

void SectorMappings::verify_shift(int from_id, const Sector& to,
		const GiNaC::exmap& shift) {
	if (no_explicit_shift_verification_)
		return;
	Sector from(integralfamily_, from_id);
	try {
		// from-sectors must be minimal w.r.t. global permutation symmetries.
		int equiv = integralfamily_->get_equivalent(from_id);
		if (equiv != from_id)
			throw string("source sector is equivalent to lower sector ")
					+ to_string(Sector(integralfamily_, equiv))
					+ " by a global permutation symmetry";

		// target sectors must be lower sectors than "from"-sectors.
		if (from < to)
			throw string("source sector lower than target sector");

		// explicitly verify the shift achieves the map 'from' -> 'to'
		vector<int> perm;
		if (!find_permutations_for_shift(from, to, shift, perm))
			throw string("transformation does not achieve the sector mapping");

		// don't allow cut propagators to mix with non-cut propagators
		for (size_t i = 0; i < perm.size(); ++i)
			if (perm[i] >= 0
					&& (from.integralfamily()->is_cut_propagator(i)
					    != to.integralfamily()->is_cut_propagator(perm[i])))
				throw string("shift mixes cut with non-cut propagator");

	} catch (const string& s) {
		throw runtime_error(
				"shift " + to_string(from) + " -> " + to_string(to) + " with "
						+ to_string(shift) + " rejected:\n" + s);
	}
}

void SectorMappings::remove_shift(int from_id) {
	sector_relations_.erase(from_id);
	sector_relation_shifts_.erase(from_id);
}

void SectorMappings::insert_shift(int from_id, const Sector& to,
		const GiNaC::exmap& shift) {
	const Sector from(integralfamily_, from_id);
	verify_shift(from_id, to, shift);

	// Shifts between physical and unphysical sectors are not allowed.
	// Ideally, this check should be in verify_shift(). However, it leads to
	// problems loading the target sector mappings in read().
    // note that crossed families don't store graphless sectors on their own
	Files* f = Files::instance();
	const SectorMappings *fm, *tm;
	fm = f->sectormappings(Crossing::uncross(integralfamily_).first->name());
	tm = f->sectormappings(Crossing::uncross(to.integralfamily()).first->name());
	bool g1 = fm->sectors_without_graph_.find(from.id())
			== fm->sectors_without_graph_.end();
	bool g2 = tm->sectors_without_graph_.find(to.id())
			== tm->sectors_without_graph_.end();
	try {
		if (g1 && !g2)
			throw string("shift from physical to unphysical sector");
		if (!g1 && g2)
			throw string("shift from unphysical to physical sector");
	} catch (const string& s) {
		throw runtime_error(
				"shift " + to_string(from) + " -> " + to_string(to) + " with "
						+ to_string(shift) + " rejected:\n" + s);
	}
	pair<map<int, Sector>::const_iterator, bool> inserted1;
	pair<map<int, GiNaC::exmap>::const_iterator, bool> inserted2;
	inserted1 = sector_relations_.insert(make_pair(from_id, to));
	inserted2 = sector_relation_shifts_.insert(make_pair(from_id, shift));
	ASSERT(inserted1.second == inserted2.second);
	if (!inserted1.second)
		WARNING("Shift from " << from << " to " << to << " already inserted.");
}

void SectorMappings::insert_symmetry_shifts(int sector_id, const std::list<
		GiNaC::exmap>& shifts) {
	list<exmap>::const_iterator symm;
	Sector sec(integralfamily_, sector_id);
	for (symm = shifts.begin(); symm != shifts.end(); ++symm)
		verify_shift(sector_id, sec, *symm);
	pair<map<int, list<GiNaC::exmap> >::const_iterator, bool> inserted;
	inserted = sector_symmetries_.insert(make_pair(sector_id, shifts));
	//if (!inserted.second)
	//	WARNING("Symmetry shifts for sector " << sec << " already inserted.");
}

void SectorMappings::find_shift_targets(std::set<int>& sectors) const {
	int maxid = integralfamily_->max_sector_id();
	set<int> equiv = integralfamily_->get_subsector_equivalents(maxid, true);
	equiv.insert(maxid);
	// exclude sectors for which a shift exists
	for (set<int>::iterator e = equiv.begin(); e != equiv.end(); ++e)
		if (sector_relations_.find(*e) == sector_relations_.end()) // no shift found
			sectors.insert(*e);
}

void SectorMappings::find_shift_targets_with_graph(
		std::list<SectorGL>& shift_targets, bool minimize_by_twists,
		bool construct_minimal_graphs) const {

	// find sectors of this family with graph and no shift
	LOG("Finding shift targets (with graph and canonical label):");

	set<Sector> sectors;
	set<int> secs;
	find_shift_targets(secs);
	for (set<int>::iterator s = secs.begin(); s != secs.end(); ++s)
		if (sectors_without_graph_.find(*s) == sectors_without_graph_.end())
			sectors.insert(Sector(integralfamily_, *s));

	// construct the graphs
	get_graphs(shift_targets, sectors, minimize_by_twists,
			construct_minimal_graphs, true);
}

size_t SectorMappings::find_shift_targets_with_graph(std::map<int, std::map<
		int, std::list<SectorGL> > >& secbytbyloop, int l_min, int l_max,
		bool minimize_by_twists, bool construct_minimal_graphs) {
	//LOG("Finding sectors and their graphs to match with diagrams:");
	size_t num_secs = 0;
	Files* files = Files::instance();
	list<IntegralFamily*> ics = files->integralfamilies();
	list<IntegralFamily*>::const_iterator ic;
	for (ic = ics.begin(); ic != ics.end(); ++ic) {
		int loop = (*ic)->loop_momenta().nops();
		if ((*ic)->is_crossed() || loop < l_min || loop > l_max)
			continue;
		string name = (*ic)->name();
		const SectorMappings* m = files->sectormappings(name);
		list<SectorGL> targets;
		LOG("Finding graphs of shift target sectors from integral family " << name << ":");
		m->find_shift_targets_with_graph(targets, minimize_by_twists,
				construct_minimal_graphs);
		num_secs += targets.size();
		map<int, list<SectorGL> >& secgraphsbyt = secbytbyloop[loop];
		list<SectorGL>::iterator s; // \todo avoid copy
		for (s = targets.begin(); s != targets.end(); ++s) {
			list<SectorGL>& secgraphs = secgraphsbyt[s->sector().t()];
			secgraphs.push_back(SectorGL());
			secgraphs.back().swap(*s);
		}
	}
	return num_secs;
}

/// \todo: idea for improvement:
// show separate t graph: 4->3->2->1->0 vertically
// and show sectors at appropriate vert. position
// similarly to timeline example on http://www.graphviz.org/pdf/dotguide.pdf
// page 16/17
void SectorMappings::print_sectortree_dot(const std::set<Sector>& secs,
		const std::string& filename) {
	string filename_tmp = filename + ".tmp";
	ofstream file(filename_tmp.c_str());

	// header
	file << "digraph SectorRelations {\n";
	file << "size = \"8.3,11.7\"\n";
	file << "ratio = \"1.5\"\n";
	file << "center = \"true\"\n";

	set<Sector> allsecs = secs;
	set<Sector>::const_iterator s;
	for (s = secs.begin(); s != secs.end(); ++s) {
		INT ie(SectorMappings::get_equivalent_or_unique_zero(INT(*s)));
		Sector equiv = ie.get_sector();
		if (equiv == *s)
			continue;
		allsecs.insert(equiv);
		file//
				<< s->integralfamily()->name()//
				<< s->id()//
				<< " -> "//
				<< equiv.integralfamily()->name()//
				<< equiv.id()//
				<< " [color=green];\n";
	}
	for (s = secs.begin(); s != secs.end(); ++s) {
		set<Sector> sub = SectorSelection::find_prerequisite_sectors(*s, true);
		allsecs.insert(sub.begin(), sub.end());
	}

	// sectors are vertices, relations are directed edges
	map<Sector, set<Sector> > relations;
	for (s = allsecs.begin(); s != allsecs.end(); ++s)
		relations[*s] = SectorSelection::find_prerequisite_sectors(*s, false);

	map<Sector, set<Sector> >::const_iterator r;
	for (r = relations.begin(); r != relations.end(); ++r) {
		for (s = r->second.begin(); s != r->second.end(); ++s)
			file//
					<< r->first.integralfamily()->name() << r->first.id()//
					<< " -> "//
					<< s->integralfamily()->name()//
					<< s->id()//
					<< " [color=blue];\n";
	}
	for (s = allsecs.begin(); s != allsecs.end(); ++s)
		file << s->integralfamily()->name()//
				<< s->id()//
				<< " [label=\"" << *s
		/*<< " " << s->t()*/<< "\"];\n";

	// trailer
	file << "}\n";

	file.close();
	rename(filename_tmp, filename);
}

void SectorMappings::read_sector_relations(const YAML::Node& node) {
	using namespace YAML;
	for (Iterator r = node.begin(); r != node.end(); ++r) {
		int from_equiv_id;
		r.first() >> from_equiv_id;
		Sector from_equiv_sec(integralfamily_, from_equiv_id);
		Sector to_equiv_sec(r.second()[0]);

		bool new_notation = true;
		if (r.second().size() == 3) {
			new_notation = false;
			int from_id;
			r.second()[1] >> from_id;
			Sector from_sec(integralfamily_, from_id);
			VERIFY(from_sec == from_equiv_sec);
		} else {
			VERIFY(r.second().size() == 2);
		}

		const Node& shift_node = new_notation ? r.second()[1] : r.second()[2];
		exmap shift;
		lst from_momenta = integralfamily_->get_all_momenta();
		lst to_momenta = to_equiv_sec.integralfamily()->get_all_momenta();
		for (Iterator s = shift_node.begin(); s != shift_node.end(); ++s) {
			string lhs, rhs;
			(*s)[0] >> lhs;
			(*s)[1] >> rhs;
			shift[ex(lhs, from_momenta)] = ex(rhs, to_momenta);
		}
		try {
		    verify_shift(from_equiv_id, to_equiv_sec, shift);
		} catch (exception& e) {
			ERROR(string("error reading sector_relations:\n") + e.what());
		}
		sector_relations_.insert(make_pair(from_equiv_id, to_equiv_sec));
		sector_relation_shifts_.insert(make_pair(from_equiv_id, shift));
	}
}

void SectorMappings::read(const YAML::Node& node) {
	using namespace YAML;
	using namespace GiNaC;
	string name;
	node["name"] >> name;
	string required = integralfamily_->name();
	if (name != required)
		throw runtime_error("Name " + name + " does not match " + required);

	const Node& z_node = node["zero_sectors"];
	for (Iterator n = z_node.begin(); n != z_node.end(); ++n)
		for (Iterator s = n.second().begin(); s != n.second().end(); ++s) {
			int id;
			*s >> id;
			zero_sectors_.insert(id);
		}

	if (node.FindValue("sectors_without_graph")) {
		const Node& ng_node = node["sectors_without_graph"];
		for (Iterator n = ng_node.begin(); n != ng_node.end(); ++n)
			for (Iterator s = n.second().begin(); s != n.second().end(); ++s) {
				int id;
				*s >> id;
				sectors_without_graph_.insert(id);
			}
	}

	const Node& rel_node = node["sector_relations"];
	read_sector_relations(rel_node);

	if (node.FindValue("crossed_sector_relations")) {
		const Node& xrel_node = node["crossed_sector_relations"];
		for (Iterator r = xrel_node.begin(); r != xrel_node.end(); ++r) {
			string xname;
			r.first() >> xname;
			string xfamname = integralfamily()->name() + xname;
			SectorMappings* xm = Files::instance()->sectormappings(xfamname);
			xm->read_sector_relations(r.second());
		}
	}

	if (node.FindValue("sector_symmetries")) {
		const Node& sym_node = node["sector_symmetries"];
		const lst& symbols = integralfamily_->get_all_momenta();
		for (Iterator r = sym_node.begin(); r != sym_node.end(); ++r) {
			int id;
			r.first() >> id;
			Sector sector(integralfamily_, id);
			list<exmap>& id_shifts = sector_symmetries_[id]; // allow merging
			for (Iterator s = r.second().begin(); s != r.second().end(); ++s) {
				exmap shift;
				Reduze::read(*s, shift, symbols);
				try {
				    verify_shift(id, sector, shift);
				} catch (exception& e) {
					ERROR(string("error reading sector_symmetries:\n") + e.what());
				}
				id_shifts.push_back(shift);
			}
			// drop duplicates when merging (remove for normal operation ?)
			id_shifts.sort(exmap_is_less());
			id_shifts.unique();
		}
	}
}

void write_sector_relations(YAML::Emitter& os,
		const map<int, Sector>& relations, const map<int, exmap>& shifts) {
	using namespace YAML;
	os << BeginMap;
	map<int, GiNaC::exmap>::const_iterator r;
	for (r = shifts.begin(); r != shifts.end(); ++r) {
		int from_equiv = r->first;
		const exmap& shift = r->second;
		map<int, Sector>::const_iterator equiv = relations.find(from_equiv);
		VERIFY(equiv != relations.end());
		const Sector& to_equiv = equiv->second;
		os << Key << from_equiv << Value << Flow << BeginSeq;
		os << to_equiv;
		os << BeginSeq;
		for (exmap::const_iterator sr = shift.begin(); sr != shift.end(); ++sr)
			os << BeginSeq << sr->first << sr->second << EndSeq;
		os << EndSeq;
		os << EndSeq;
	}
	os << EndMap;
}

YAML::Emitter & operator<<(YAML::Emitter& os, const SectorMappings& m) {
	using namespace YAML;
	using namespace GiNaC;
	const IntegralFamily* ic = m.integralfamily();
	os << BeginMap;

	os << Key << "name" << Value << ic->name();

	os << Key << "zero_sectors" << Value << BeginMap;
	map<int, set<int> > z_by_t; // collect by common t
	set<int>::const_iterator z;
	for (z = m.zero_sectors_.begin(); z != m.zero_sectors_.end(); ++z)
		z_by_t[ic->get_t(*z)].insert(*z);
	map<int, set<int> >::iterator i;
	for (i = z_by_t.begin(); i != z_by_t.end(); ++i) {
		stringstream ss;
		ss << "t=" << i->first;
		os << Key << ss.str() << Value << Flow << BeginSeq;
		for (set<int>::iterator s = i->second.begin(); s != i->second.end(); ++s)
			os << *s;
		os << EndSeq;
	}
	os << EndMap;

	os << Key << "sectors_without_graph" << Value << BeginMap;
	map<int, set<int> > ng_by_t; // collect by common t
	set<int>::const_iterator seci;
	for (seci = m.sectors_without_graph_.begin(); seci
			!= m.sectors_without_graph_.end(); ++seci)
		ng_by_t[ic->get_t(*seci)].insert(*seci);
	//map<int, set<int> >::iterator i;
	for (i = ng_by_t.begin(); i != ng_by_t.end(); ++i) {
		stringstream ss;
		ss << "t=" << i->first;
		os << Key << ss.str() << Value << Flow << BeginSeq;
		for (set<int>::iterator s = i->second.begin(); s != i->second.end(); ++s)
			os << *s;
		os << EndSeq;
	}
	os << EndMap;

	os << Key << "sector_relations" << Value;
	write_sector_relations(os, m.sector_relations_, m.sector_relation_shifts_);

	os << Key << "crossed_sector_relations" << Value << BeginMap;
	list<Crossing> xs = Files::instance()->crossings(
			m.integralfamily()->kinematics()->name())->ordered_crossings();
	for (list<Crossing>::const_iterator x = xs.begin(); x != xs.end(); ++x) {
		string fn = x->name_for_crossed_family(m.integralfamily());
		SectorMappings* xm = Files::instance()->sectormappings(fn);
		if (!xm->sector_relations_.empty()) {
			os << Key << x->name() << Value;
			write_sector_relations(os, xm->sector_relations_,
					xm->sector_relation_shifts_);
		}
	}
	os << EndMap;

	os << Key << "sector_symmetries" << Value << BeginMap;
	const map<int, list<GiNaC::exmap> >& syms = m.sector_symmetries_;
	map<int, list<GiNaC::exmap> >::const_iterator it;
	list<GiNaC::exmap>::const_iterator it2;
	for (it = syms.begin(); it != syms.end(); ++it) {
		os << Key << it->first << Value;
		os << BeginSeq;
		for (it2 = it->second.begin(); it2 != it->second.end(); ++it2) {
			os << Flow << *it2;
		}
		os << EndSeq;
	}
	os << EndMap;

	os << EndMap;
	return os;
}

void SectorMappings::verify_is_equivalent(const SectorMappings& other) const {
	try {
		if (integralfamily()->name() != other.integralfamily()->name())
			throw runtime_error("integral family names differ");
		if (zero_sectors() != other.zero_sectors())
			throw runtime_error("zero sectors differ");
		if (sectors_without_graph() != other.sectors_without_graph())
			throw runtime_error("sectors without graph differ");
		const map<int, Sector>& secrel1 = sector_relations();
		const map<int, Sector>& secrel2 = other.sector_relations();
		if (secrel1 != secrel2)
			throw runtime_error("sector relations differ");

		const map<int, list<GiNaC::exmap> >& sym1 = sector_symmetries();
		const map<int, list<GiNaC::exmap> >& sym2 = other.sector_symmetries();
		if (sym1.size() != sym2.size())
			throw runtime_error("not the same number of sector symmetries");
		map<int, list<GiNaC::exmap> >::const_iterator it1 = sym1.begin(), it2 =
				sym2.begin();
		int lauf = 0;
		for (; it1 != sym1.end(); ++it1, ++it2) {
			++lauf;
			if (it1->first != it2->first || it1->second.size()
					!= it2->second.size())
				throw runtime_error("sector relation mismatch at position "
						+ to_string(lauf));
		}
	} catch (exception& e) {
		throw runtime_error(integralfamily_->name() + ": " + e.what());
	}
}

}
// namespace Reduze
