/*  shiftoperators.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "shiftoperators.h"
#include "propagator.h"
#include "functions.h"

using namespace std;

namespace Reduze {

// OP


ostream& operator<<(ostream& strm, const OP& in) {
	strm << "OP[" << in.n_ << "]";
	return strm;
}

void OPPROD::unique() {
	// sort
	prod_.sort();
	// remove identity
	OP id;
	for (list<OP>::iterator op = prod_.begin(); op != prod_.end();) {
		if (*op == id)
			op = prod_.erase(op);
		else if (id < *op)
			break;
		else
			++op;
	}
	// remove OP(-n)*OP(n)
	for (list<OP>::iterator it1 = prod_.begin(); it1 != prod_.end(); ++it1) {
		list<OP>::iterator it2 = it1;
		for (++it2; it2 != prod_.end(); ++it2) {
			if (*it1 == it2->inverse()) {
				it2 = prod_.erase(it2);
				it1 = prod_.erase(it1);
				it2 = it1;
			}
			if (it2 == prod_.end())
				break;
		}
		if (it1 == prod_.end())
			break;
	}

	if (size() == 0)
		*this = OPPROD();
}

bool OPPROD::operator ==(const OPPROD& other) const {
	if (size() != other.size())
		return false;
	list<OP>::const_iterator it1 = prod_.begin(), it2 = other.prod_.begin();
	for (; it1 != prod_.end(); ++it1, ++it2)
		if (*it1 != *it2)
			return false;
	return true;
}

bool OPPROD::operator<(const OPPROD& other) const {
	return prod_ < other.prod_;
}

OPPROD& OPPROD::operator*=(const OPPROD& other) {
	prod_.insert(prod_.begin(), other.prod_.begin(), other.prod_.end());
	unique();
	return *this;
}

OPPROD OPPROD::inverse() const {
	OPPROD res;
	for (list<OP>::const_reverse_iterator o = prod_.rbegin(); o != prod_.rend(); ++o)
		res.prod_.push_back(o->inverse());
	res.unique(); // TODO: avoid this
	return res;
}

OPPROD operator*(const OPPROD & a, const OPPROD & b) {
	OPPROD tmp = a;
	tmp *= b;
	return tmp;
}

OPPROD pow(const OPPROD& a, int i) {
	OPPROD res;
	if (!a.prod_.empty() || i == 0)
		res.prod_.clear(); // remove unity operator
	for (list<OP>::const_iterator o = a.prod_.begin(); o != a.prod_.end(); ++o)
		for (size_t j = 0 ; j < (size_t) abs(i) ; ++j)
			res.prod_.push_back(*o);
	return (i < 0 ? res.inverse() : res);
}

ostream& operator<<(ostream& strm, const OPPROD& in) {
	list<OP>::const_iterator it = in.prod_.begin();
	while (it != in.prod_.end()) {
		strm << *it;
		++it;
		if (it != in.prod_.end()) {
			strm << ".";
		}
	}
	return strm;
}

// OPSUM

OPSUM::OPSUM(const OPPROD& p, const GiNaC::ex coeff) {
	insert(p, coeff);
}

void OPSUM::insert(const OPPROD& i, const GiNaC::ex& e) {
	using GiNaC::wild;
	if (e.has(Propagator(wild(1), wild(2), wild(3))))
		throw runtime_error(
				"Coefficient of the shift operator products are not allowed to contain propagators");
	pair<map<OPPROD, GiNaC::ex>::iterator, bool> info = sum_.insert(make_pair(
			i, e));
	if (!info.second)
		info.first->second = normal(info.first->second + e);
	if (info.first->second.is_zero())
		sum_.erase(info.first);
}

OPSUM& OPSUM::operator+=(const OPSUM& other) {
	map<OPPROD, GiNaC::ex>::const_iterator it;
	for (it = other.sum_.begin(); it != other.sum_.end(); ++it)
		insert(it->first, it->second);
	return *this;
}

OPSUM& OPSUM::operator-=(const OPSUM& other) {
	map<OPPROD, GiNaC::ex>::const_iterator it;
	for (it = other.sum_.begin(); it != other.sum_.end(); ++it)
		insert(it->first, -it->second);
	return *this;
}

OPSUM& OPSUM::operator*=(const OPSUM& other) {
	OPSUM neu;
	neu.sum_.swap(sum_);
	map<OPPROD, GiNaC::ex>::const_iterator it1, it2;
	for (it1 = neu.sum_.begin(); it1 != neu.sum_.end(); ++it1)
		for (it2 = other.sum_.begin(); it2 != other.sum_.end(); ++it2)
			insert(it1->first * it2->first, normal(it1->second * it2->second));
	return *this;
}

OPSUM operator+(const OPSUM & a, const OPSUM & b) {
	OPSUM tmp = a;
	tmp += b;
	return tmp;
}

OPSUM operator-(const OPSUM & a, const OPSUM & b) {
	OPSUM tmp = a;
	tmp -= b;
	return tmp;
}

OPSUM operator*(const OPSUM & a, const OPSUM & b) {
	OPSUM tmp = a;
	tmp *= b;
	return tmp;
}

OPSUM pow(const OPSUM& a, int i) {
	if (i < 0) {
		VERIFY(a.sum_.size() == 1);
		const pair<OPPROD, GiNaC::ex>& t = *a.sum_.begin();
		return OPSUM(pow(t.first, i), pow(t.second, i));
	} else if (i == 0) {
		return OPSUM(OPPROD(OP()), 1);
	} else if (i == 1) {
		return a;
	} else { // i > 1
		return pow(a, i-1)*a;
	}
}

ostream& operator<<(ostream& strm, const OPSUM& in) {
	map<OPPROD, GiNaC::ex>::const_iterator it = in.sum_.begin();
	if (it == in.sum_.end()) {
		strm << "0";
	}
	while (it != in.sum_.end()) {
		strm << "(" << it->second << ")*";
		strm << it->first;
		++it;
		if (it != in.sum_.end())
			strm << " + ";
	}
	return strm;
}

} // namespace Reduze


