/*  yamlutils.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef YAMLUTILS_H_
#define YAMLUTILS_H_

#include "../yaml/include/yaml.h" // forward decl. not enough for template impl.
#include <ginac/ginac.h> // GiNaC::lst is typedef, requires work to forward decl.
namespace Reduze {

/// throws if the node is not a map or not a non-empty sequence
void verify_is_valid_map_or_non_empty_sequence(const YAML::Node& n);

// often used YAML operators
std::string position_info(const YAML::Node& n);
YAML::Emitter& operator <<(YAML::Emitter& os, const GiNaC::ex& e);
void operator >>(const YAML::Node& node, GiNaC::symbol& s);
/// read an already known symbol
void read(const YAML::Node& node, GiNaC::symbol& s, const GiNaC::lst& symbols);

void read(const YAML::Node& node, GiNaC::ex& e, const GiNaC::lst& symbols);

YAML::Emitter& operator <<(YAML::Emitter& os, const GiNaC::lst& e);

YAML::Emitter& operator <<(YAML::Emitter& os, const GiNaC::exmap& m);
void read(const YAML::Node& node, GiNaC::exmap& m, const GiNaC::lst& symbols);

/// reads a list of symbols or expressions (appends to m)
template<class T>
void read(const YAML::Node& node, GiNaC::lst& m, const GiNaC::lst& symbols);

/// reads an unsigned integer
void read(const YAML::Node& node, unsigned& n);

/// reads a vector
void read(const YAML::Node& node, std::vector<int>& v);
/// reads a matrix
void read(const YAML::Node& node, std::vector<std::vector<int> >& v);

template<class T>
void operator >>(const YAML::Node& node, std::set<T>& l) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw std::runtime_error(
				"node is not a sequence " + position_info(node));
	l.clear();
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		T s;
		*it >> s;
		l.insert(s);
	}
}

template<class T>
void operator >>(const YAML::Node& node, std::multiset<T>& s) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw std::runtime_error(
				"node is not a sequence " + position_info(node));
	std::multiset<T> tmp;
	for (unsigned i = 0; i < node.size(); ++i) {
		T value;
		node[i] >> value;
		tmp.insert(value);
	}
	tmp.swap(s);
}

template<class T>
void operator >>(const YAML::Node& node, std::list<T>& l) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw std::runtime_error(
				"node is not a sequence " + position_info(node));
	l.clear();
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		T value;
		*it >> value;
		l.push_back(value);
	}
}

void operator >>(const YAML::Node& node, std::map<int, std::string>& l);

/// reads a list of some type from a YAML specification
/** if prototype is non-zero, operator>> will be used to read,
 ** if prototype is zero, the objects constructor will be used */
template<class T>
void read_proto_ctor(const YAML::Node& node, std::list<T>& l,
		const T* prototype) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw std::runtime_error(
				"node is not a sequence " + position_info(node));
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it)
		if (prototype != 0) {
			T value(*prototype);
			*it >> value;
			l.push_back(value);
		} else {
			T value(*it);
			l.push_back(value);
		}
}

/// reads a list of some type from a YAML specification
/** prototype will be used for initial settings */
template<class T>
void read_proto(const YAML::Node& node, std::list<T>& l, const T& prototype) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw std::runtime_error(
				"node is not a sequence " + position_info(node));
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		T value(prototype);
		*it >> value;
		l.push_back(value);
	}
}

//

}// namespace Reduze


// YAML provides Emitter operators only for the stl-container: vector, list, set, map
// we add multiset
namespace YAML {

template<typename T>
inline Emitter& operator <<(Emitter& emitter, const std::multiset<T>& v) {
	emitter << BeginSeq;
	typename std::multiset<T>::const_iterator it;
	for (it = v.begin(); it != v.end(); ++it)
		emitter << *it;
	emitter << EndSeq;
	return emitter;
}

} // namespace YAML

#endif /* YAMLUTILS_H_ */
