/************************************************************************
 * Copyright (c) 2023 Advanced Micro Devices, Inc.
 * ***********************************************************************/

#pragma once

#include "rocblas.hpp"
#include "rocsolver/rocsolver.h"
#include "rocsparse.hpp"

template <typename T>
ROCSOLVER_KERNEL void rf_splitLU_kernel(const rocblas_int n,
                                        const rocblas_int nnzM,
                                        rocblas_int* Mp,
                                        rocblas_int* Mi,
                                        T* Mx,
                                        rocblas_int* Lp,
                                        rocblas_int* Li,
                                        T* Lx,
                                        rocblas_int* Up,
                                        rocblas_int* Ui,
                                        T* Ux,
                                        rocblas_int* work)
{
    rocblas_int tid = hipThreadIdx_x;

    rocblas_int* const nzLp = work;
    rocblas_int* const nzUp = work + n;

    // -------------------------------------------------
    // 1st pass to determine number of non-zeros per row
    // and set up Lp and Up
    // -------------------------------------------------
    rocblas_int i, j;
    rocblas_int irow, icol, istart, iend;
    __shared__ rocblas_int nnzL;
    __shared__ rocblas_int nnzU;

    for(irow = tid; irow < n; irow += hipBlockDim_x)
    {
        istart = Mp[irow];
        iend = Mp[irow + 1];

        for(i = istart; i < iend; i++)
        {
            icol = Mi[i];
            if(icol >= irow)
                break;
        }

        nzLp[irow] = i - istart + 1; // add 1 for unit diagonal
        nzUp[irow] = iend - i;
    }
    __syncthreads();

    if(tid == 0)
    {
        nnzL = 0;
        nnzU = 0;

        for(i = 0; i < n; i++)
        {
            Lp[i] = nnzL;
            Up[i] = nnzU;
            nnzL += nzLp[i];
            nnzU += nzUp[i];
        };

        Lp[n] = nnzL;
        Up[n] = nnzU;
    };
    __syncthreads();

    // ------------------------------------
    // 2nd pass to populate Li, Lx, Ui, Ux
    // ------------------------------------
    for(irow = tid; irow < n; irow += hipBlockDim_x)
    {
        istart = Mp[irow];
        iend = Mp[irow + 1];

        for(i = istart; i < iend; i++)
        {
            icol = Mi[i];
            if(icol < irow)
            {
                // element in L
                j = Lp[irow] + (i - istart);

                Li[j] = icol;
                Lx[j] = Mx[i];
            }
            else
            {
                // element in U
                j = Up[irow + 1] - (iend - i);

                Ui[j] = icol;
                Ux[j] = Mx[i];
            }
        }
    }

    // -----------------------------
    // set unit diagonal entry in L
    // -----------------------------
    for(irow = tid; irow < n; irow += hipBlockDim_x)
    {
        j = Lp[irow + 1] - 1;
        Li[j] = irow;
        Lx[j] = T(1);
    };
}

template <typename T>
void rocsolver_csrrf_splitlu_getMemorySize(const rocblas_int n,
                                           const rocblas_int nnzT,
                                           size_t* size_work)
{
    // if quick return, no need of workspace
    if(n == 0 || nnzT == 0)
    {
        *size_work = 0;
        return;
    }

    // space to store the number of non-zeros per row in L and U
    *size_work = sizeof(rocblas_int) * 2 * n;
}

template <typename T>
rocblas_status rocsolver_csrrf_splitlu_argCheck(rocblas_handle handle,
                                                const rocblas_int n,
                                                const rocblas_int nnzT,
                                                rocblas_int* ptrT,
                                                rocblas_int* indT,
                                                T valT,
                                                rocblas_int* ptrL,
                                                rocblas_int* indL,
                                                T valL,
                                                rocblas_int* ptrU,
                                                rocblas_int* indU,
                                                T valU)
{
    // order is important for unit tests:

    // 1. invalid/non-supported values
    // N/A
    if(handle == nullptr)
    {
        return rocblas_status_invalid_handle;
    };

    // 2. invalid size
    if(n < 0 || nnzT < 0)
        return rocblas_status_invalid_size;

    // skip pointer check if querying memory size
    if(rocblas_is_device_memory_size_query(handle))
        return rocblas_status_continue;

    // 3. invalid pointers
    if(!ptrL || !ptrU || !ptrT || (nnzT && (!indT || !valT || !indU || !valU))
       || ((n || nnzT) && (!indL || !valL)))
        return rocblas_status_invalid_pointer;

    return rocblas_status_continue;
}

template <typename T, typename U>
rocblas_status rocsolver_csrrf_splitlu_template(rocblas_handle handle,
                                                const rocblas_int n,
                                                const rocblas_int nnzT,
                                                rocblas_int* ptrT,
                                                rocblas_int* indT,
                                                U valT,
                                                rocblas_int* ptrL,
                                                rocblas_int* indL,
                                                U valL,
                                                rocblas_int* ptrU,
                                                rocblas_int* indU,
                                                U valU,
                                                rocblas_int* work)
{
    ROCSOLVER_ENTER("csrrf_splitlu", "n:", n, "nnzT:", nnzT);

    // quick return
    if(n == 0)
        return rocblas_status_success;

    hipStream_t stream;
    ROCBLAS_CHECK(rocblas_get_stream(handle, &stream));

    // quick return with matrix zero
    if(nnzT == 0)
    {
        // set ptrU = 0
        rocblas_int blocks = n / BS1 + 1;
        dim3 grid(blocks, 1, 1);
        dim3 threads(BS1, 1, 1);
        ROCSOLVER_LAUNCH_KERNEL(reset_info, grid, threads, 0, stream, ptrU, n + 1, 0);
        ROCSOLVER_LAUNCH_KERNEL(reset_info, grid, threads, 0, stream, ptrL, n + 1, 0, 1);
        ROCSOLVER_LAUNCH_KERNEL(reset_info, grid, threads, 0, stream, indL, n, 0, 1);
        ROCSOLVER_LAUNCH_KERNEL(reset_info, grid, threads, 0, stream, valL, n, 1);

        return rocblas_status_success;
    }

    rocblas_int const nthreads = BS1;
    rocblas_int const nblocks = 1;
    ROCSOLVER_LAUNCH_KERNEL(rf_splitLU_kernel<T>, dim3(nblocks), dim3(nthreads), 0, stream, n, nnzT,
                            ptrT, indT, valT, ptrL, indL, valL, ptrU, indU, valU, work);

    return rocblas_status_success;
}
