#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "info.h"

struct crw_property properties[crw_items_last];
unsigned char *buffer = NULL;
off_t info_block_start_offset = -8;
off_t info_block_end_offset = -4;
off_t info_block_offset_size = 4;
off_t thumb_block_start_offset = -18;
off_t thumb_block_size_offset = -22;
off_t thumb_block_offset_size = 4;

struct {
   int camera;
   int lens;
   int exposure;
   int parameters;
   int cf;
   int flash;
   int misc;
   int experimental;
   char *thumbnail;
   char *filename;
} options;
   

void init_property_array( void );
int read_information_from_crw( const char *filename );
void read_info(void);
void process_info(void);
void output_info(void);
void parse_parameters( int, char**);
void print_help(char *);
int extract_thumbnail(void);

int main(int argc, char **argv)
{

   options.camera = 0;
   options.lens = 0;
   options.exposure = 0;
   options.parameters = 0;
   options.cf = 0;
   options.flash = 0;
   options.misc = 0;
   options.experimental = 0;
   options.filename = NULL;
   options.thumbnail = NULL;

   parse_parameters( argc, argv );
   if( options.filename == NULL )
      print_help(argv[0]);
   if( options.camera == 0 && options.lens == 0 && options.exposure == 0 &&
	           options.parameters == 0 && options.cf == 0 && 
		   options.flash == 0 && options.misc == 0 && 
		   options.experimental == 0 ) 
      options.exposure = 1;
   if( read_information_from_crw( options.filename ) == -1 ) {
      fprintf( stderr, "Cannot read from %s. Aborting\n", options.filename );
      exit(-1);
   }
   if( options.thumbnail ) {
      if( extract_thumbnail() == -1 ) {
	 fprintf( stderr, "Error writing thumbnail file %s. Aborting\n",
	       options.thumbnail);
	 exit(-1);
      }
   }
 
   init_property_array();
   read_info();
   process_info();
   output_info();

   free(buffer);
   exit(0);
}

void print_help(char *progname)
{
   fprintf( stderr, "Usage: %s [-a] [-e] [-c] [-l] [-p] [-cf] [-m] [-f] [-x] [-t [thumbnailfilename]] filename\n",
	    progname );
   fprintf( stderr, "-a        show all information, except experimental\n" );
   fprintf( stderr, "-e        show exposure information (shutter, f-stop etc)\n" );
   fprintf( stderr, "-c        show camera information (firmware, serial etc)\n" );
   fprintf( stderr, "-l        show lens information (focal length, focussed range etc)\n" );
   fprintf( stderr, "-p        show parameters used (saturation, sharpness etc)\n" );
   fprintf( stderr, "-m        show misc. information (whitepoint, AF mode etc)\n" );
   fprintf( stderr, "-f        show flash information (flash mode etc)\n" );
   fprintf( stderr, "-cf       show custom function settings\n" );
   fprintf( stderr, "-t        extract the thumbnail to the optional filename.
	                       If no filename for the thumbnail is given, the
			       name of the CRW-file is used with a .jpg
			       extension\n" );
   fprintf( stderr, "-x        show experimental information (don't rely on it)\n" );
   exit(0);
}

void parse_parameters( int argc, char **argv )
{
int i;
int thumb = 0;

   for( i = 1; i < argc; ++i ) {
      if( strcmp( argv[i], "-h" ) == 0 || strcmp( argv[i], "--help") == 0 ) {
	 print_help(argv[0]);
      } else if( strcmp( argv[i], "-a" ) == 0 ) {
	 options.exposure = options.camera = options.lens =
	    options.parameters = options.cf = options.flash = options.misc = 1;
      } else if( strcmp( argv[i], "-e" ) == 0 ) {
	 options.exposure = 1;
      } else if( strcmp( argv[i], "-c" ) == 0 ) {
	 options.camera = 1;
      } else if( strcmp( argv[i], "-l" ) == 0 ) {
	 options.lens = 1;
      } else if( strcmp( argv[i], "-p" ) == 0 ) {
	 options.parameters = 1;
      } else if( strcmp( argv[i], "-cf" ) == 0 ) {
	 options.cf = 1;
      } else if( strcmp( argv[i], "-f" ) == 0 ) {
	 options.flash = 1;
      } else if( strcmp( argv[i], "-m" ) == 0 ) {
	 options.misc = 1;
      } else if( strcmp( argv[i], "-x" ) == 0 ) {
	 options.experimental = 1;
      } else if( strcmp( argv[i], "-t" ) == 0 ) {
	 if( i < argc - 2 && *argv[i+1] != '-' ) {
	    ++i;
	    options.thumbnail = strdup( argv[i] );  
	 } else {
	    thumb = 1;
	 }
      } else {
	 options.filename = strdup( argv[i] );
      }
   }
  
   if( options.filename == NULL ) 
      return;

   if( thumb ) { /* now we have the filename to process, so derive 
		    the thumbnails filename accordingly */
      char *dud = strrchr( options.filename, '.' );
      char *file = strrchr( options.filename, '/' );
      file = (file != NULL) ? file + 1 : options.filename;
	 
      options.thumbnail = (char*)malloc( (strlen(file) + 5) * sizeof(char));
      strncpy( options.thumbnail, file, dud - file );
      *(options.thumbnail + (dud - file)) = 0;
      strcat( options.thumbnail, ".jpg" );
   }
}

void output_info(void)
{
   printf( "Timestamp: %s", ctime( &properties[timestamp].value ) );
   printf( "File format: %s\n", properties[fileformat].string_value ); 
   if( options.camera ) {
      printf( "Camera owner: %s\n", properties[owner].string_value ); 
      printf( "Camera vendor: %s\n", properties[vendor].string_value ); 
      printf( "Camera model: %s\n", properties[camera_model].string_value ); 
      printf( "Firmware version: %s\n", properties[camera_firmware].string_value );
      printf( "Camera serial no.: %lu\n", properties[camera_serial].value );
   }
   if( options.exposure ) {
      printf( "Shutter speed: %s\n", properties[shutter_speed].string_value );
      printf( "Bulb duration: %lu\n", properties[bulb_duration].value );
      printf( "Aperature: %s\n", properties[aperature].string_value );
      printf( "ISO speed: %s\n", properties[iso].string_value );
      printf( "Exposure compensation: %s\n", 
	       properties[exposure_compensation].string_value );
   } 
   if( options.misc ) {
      printf( "Autofocus mode: %s\n", properties[autofocus_mode].string_value );
      printf( "Drive mode: %s\n", properties[driver_mode].string_value );
      printf( "Photo series: %s\n", properties[driver_mode].string_value );
      printf( "# in series: %hu\n", 
	       (unsigned short int)properties[photo_in_sequence].value );
      printf( "Shutter delay: %4.2f\n", properties[shutter_delay].value * 0.1 );
      printf( "White point mode: %s\n", properties[white_point].string_value );
      printf( "Orientation: %s\n", properties[orientation].string_value );
      printf( "Rotation: %s\n", properties[rotation].string_value );
      printf( "Number in AEB sequence: %hu\n", 
	    (unsigned short int)properties[aeb_in_sequence].value );
      printf( "AEB bracket value: %s\n", 
	    properties[aeb_bracket_value].string_value );
   }
   if( options.lens ) {
      printf( "Focused distance, lower:" );
      if( properties[focus_distance_lower].value == 0xffff )
	 printf( " infinity\n" );
      else 
	 printf( " %8.3f m\n", properties[focus_distance_lower].value * 0.01);

      printf( "Focused distance, upper:" );
      if( properties[focus_distance_upper].value == 0xffff )
	 printf( " infinity\n" );
      else 
	 printf( " %8.3f m\n", properties[focus_distance_upper].value * 0.01);

      printf( "Zoom range, lower: %hu mm\n", 
	    (unsigned short int)properties[zoom_range_lower].value );
      printf( "Zoom range, upper: %hu mm\n", 
	    (unsigned short int)properties[zoom_range_upper].value );
      printf( "Used focal length: %hu mm\n", 
	    (unsigned short int)properties[focal_length_used].value );
   }
   if( options.flash ) {
      printf( "Flash mode: %s\n", properties[flash_mode].string_value );
      printf( "Flash compensation: %s\n", 
	    properties[flash_compensation].string_value );
   }
   if( options.parameters ) {
      printf( "Contrast (Parameter): %hd\n", 
	    (short int)properties[parameter_contrast].value );
      printf( "Saturation (Parameter): %hd\n", 
	    (short int)properties[parameter_saturation].value );
      printf( "Sharpness (Parameter): %hd\n", 
	    (short int)properties[parameter_sharpness].value );
      printf( "Color tone (Parameter): %hd\n", 
	    (short int)properties[parameter_colortone].value );
   }
   if( options.cf ) {
      printf( "CF 2: %hu\n", 
	    (unsigned short int)(properties[cf_2].value & 0xff) );
      printf( "CF 3: %hu\n", 
	    (unsigned short int)(properties[cf_3].value & 0xff) );
      printf( "CF 4: %hu\n", 
	    (unsigned short int)(properties[cf_4].value & 0xff) );
      printf( "CF 5: %hu\n", 
	    (unsigned short int)(properties[cf_5].value & 0xff) );
      printf( "CF 6: %hu\n", 
	    (unsigned short int)(properties[cf_6].value & 0xff) );
      printf( "CF 7: %hu\n", 
	    (unsigned short int)(properties[cf_7].value & 0xff) );
      printf( "CF 8: %hu\n",
	    (unsigned short int)(properties[cf_8].value & 0xff) );
      printf( "CF 9: %hu\n", 
	    (unsigned short int)(properties[cf_9].value & 0xff) );
      printf( "CF 10: %hu\n", 
	    (unsigned short int)(properties[cf_10].value & 0xff) );
      printf( "CF 11: %hu\n", 
	    (unsigned short int)(properties[cf_11].value & 0xff) );
      printf( "CF 12: %hu\n", 
	    (unsigned short int)(properties[cf_12].value & 0xff) );
      printf( "CF 13: %hu\n", 
	    (unsigned short int)(properties[cf_13].value & 0xff) );
      printf( "CF 14: %hu\n", 
	    (unsigned short int)(properties[cf_14].value & 0xff) );
      printf( "CF 15: %hu\n", 
	    (unsigned short int)(properties[cf_15].value & 0xff) );
   }
   if( options.experimental ) {
      printf( "Gamma(?): %hu\n", 
	    (unsigned short int) properties[x_gamma].value );
      printf( "Color temparature(?): %hu\n",
	    (unsigned short int) properties[x_colortemp].value );
      printf( "LensID(?): %lu\n",
	    (unsigned long int) properties[x_lensid].value );
   }
}

void process_info(void)
{
char *string_buffer = (char*)malloc(80 * sizeof(char) );

   switch( properties[shutter_speed].value ) {
      case s_4000: sprintf( string_buffer, "1/4000" ); break;
      case s_3200: sprintf( string_buffer, "1/3200" ); break;
      case s_3000: sprintf( string_buffer, "1/3000" ); break;
      case s_2500: sprintf( string_buffer, "1/2500" ); break;
      case s_2000: sprintf( string_buffer, "1/2000" ); break;
      case s_1600: sprintf( string_buffer, "1/1600" ); break;
      case s_1500: sprintf( string_buffer, "1/1500" ); break;
      case s_1250: sprintf( string_buffer, "1/1250" ); break;
      case s_1000: sprintf( string_buffer, "1/1000" ); break;
      case s_800: sprintf( string_buffer, "1/800" ); break;
      case s_750: sprintf( string_buffer, "1/750" ); break;
      case s_640: sprintf( string_buffer, "1/640" ); break;
      case s_500: sprintf( string_buffer, "1/500" ); break;
      case s_400: sprintf( string_buffer, "1/400" ); break;
      case s_350: sprintf( string_buffer, "1/350" ); break;
      case s_320: sprintf( string_buffer, "1/320" ); break;
      case s_250: sprintf( string_buffer, "1/250" ); break;
      case s_200: sprintf( string_buffer, "1/200" ); break;
      case s_180: sprintf( string_buffer, "1/180" ); break;
      case s_160: sprintf( string_buffer, "1/160" ); break;
      case s_125: sprintf( string_buffer, "1/125" ); break;
      case s_100: sprintf( string_buffer, "1/100" ); break;
      case s_90: sprintf( string_buffer, "1/90" ); break;
      case s_80: sprintf( string_buffer, "1/80" ); break;
      case s_60_1:
      case s_60_2: sprintf( string_buffer, "1/60" ); break;
      case s_50: sprintf( string_buffer, "1/50" ); break;
      case s_45: sprintf( string_buffer, "1/45" ); break;
      case s_40: sprintf( string_buffer, "1/40" ); break;
      case s_30: sprintf( string_buffer, "1/30" ); break;
      case s_25: sprintf( string_buffer, "1/25" ); break;
      case s_20_1:
      case s_20_2: sprintf( string_buffer, "1/20" ); break;
      case s_15: sprintf( string_buffer, "1/15" ); break;
      case s_13: sprintf( string_buffer, "1/13" ); break;
      case s_10_1: 
      case s_10_2: sprintf( string_buffer, "1/10" ); break;
      case s_8: sprintf( string_buffer, "1/8" ); break;
      case s_6_1:
      case s_6_2: sprintf( string_buffer, "1/6" ); break;
      case s_4: sprintf( string_buffer, "1/4" ); break;
      case s0_3_1:
      case s0_3_2: sprintf( string_buffer, "0.3" ); break;
      case s0_4: sprintf( string_buffer, "0.4" ); break;
      case s0_5: sprintf( string_buffer, "0.5" ); break;
      case s0_6: sprintf( string_buffer, "0.6" ); break;
      case s0_7: sprintf( string_buffer, "0.7" ); break;
      case s0_8: sprintf( string_buffer, "0.8" ); break;
      case s1_0_1: 
      case s1_0_2: sprintf( string_buffer, "1" ); break;
      case s1_3: sprintf( string_buffer, "1.3" ); break;
      case s1_5: sprintf( string_buffer, "1.5" ); break;
      case s1_6: sprintf( string_buffer, "1.6" ); break;
      case s2: sprintf( string_buffer, "2" ); break;
      case s2_5: sprintf( string_buffer, "2.5" ); break;
      case s3: sprintf( string_buffer, "3" ); break;
      case s3_2: sprintf( string_buffer, "3.2" ); break;
      case s4: sprintf( string_buffer, "4" ); break;
      case s5: sprintf( string_buffer, "5" ); break;
      case s6_1:
      case s6_2: sprintf( string_buffer, "6" ); break;
      case s8: sprintf( string_buffer, "8" ); break;
      case s10_1:
      case s10_2: sprintf( string_buffer, "10" ); break;
      case s13: sprintf( string_buffer, "13" ); break;
      case s15: sprintf( string_buffer, "15" ); break;
      case s20_1:
      case s20_2: sprintf( string_buffer, "20" ); break;
      case s25: sprintf( string_buffer, "25" ); break;
      case s30: sprintf( string_buffer, "30" ); break;
      case sbulb: sprintf( string_buffer, "Bulb" ); break;
      default: fprintf( stderr, "Unknown shutter speed value %hu.\n Please report the value and the actual speed. Thank you\n", (unsigned short int)properties[shutter_speed].value ); 
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[shutter_speed].string_value = strdup( string_buffer );
  
   switch( properties[aperature].value ) {
      case fnone: sprintf( string_buffer, "No lens attached?" ); break;
      case f1_0: sprintf( string_buffer, "f 1.0" ); break;
      case f1_1: sprintf( string_buffer, "f 1.1" ); break;
      case f1_2: sprintf( string_buffer, "f 1.2" ); break;
      case f1_4: sprintf( string_buffer, "f 1.4" ); break;
      case f1_6: sprintf( string_buffer, "f 1.6" ); break;
      case f1_8: sprintf( string_buffer, "f 1.8" ); break;
      case f2_0: sprintf( string_buffer, "f 2.0" ); break;
      case f2_2: sprintf( string_buffer, "f 2.2" ); break;
      case f2_5: sprintf( string_buffer, "f 2.5" ); break;
      case f2_8: sprintf( string_buffer, "f 2.8" ); break;
      case f3_2: sprintf( string_buffer, "f 3.2" ); break;
      case f3_5_1:
      case f3_5_2: sprintf( string_buffer, "f 3.5" ); break;
      case f4_0: sprintf( string_buffer, "f 4.0" ); break;
      case f4_5_1:
      case f4_5_2: sprintf( string_buffer, "f 4.5" ); break;
      case f5_0: sprintf( string_buffer, "f 5.0" ); break;
      case f5_6: sprintf( string_buffer, "f 5.6" ); break;
      case f6_3: sprintf( string_buffer, "f 6.3" ); break;
      case f6_7: sprintf( string_buffer, "f 6.7" ); break;
      case f7_1: sprintf( string_buffer, "f 7.1" ); break;
      case f8_0: sprintf( string_buffer, "f 8.0" ); break;
      case f9_0: sprintf( string_buffer, "f 9.0" ); break;
      case f9_5: sprintf( string_buffer, "f 9.5" ); break;
      case f10: sprintf( string_buffer, "f 10" ); break;
      case f11: sprintf( string_buffer, "f 11" ); break;
      case f13_1:
      case f13_2: sprintf( string_buffer, "f 13" ); break;
      case f14: sprintf( string_buffer, "f 14" ); break;
      case f16: sprintf( string_buffer, "f 16" ); break;
      case f18: sprintf( string_buffer, "f 18" ); break;
      case f19: sprintf( string_buffer, "f 19" ); break;
      case f20: sprintf( string_buffer, "f 20" ); break;
      case f22: sprintf( string_buffer, "f 22" ); break;
      case f25: sprintf( string_buffer, "f 25" ); break;
      case f27: sprintf( string_buffer, "f 27" ); break;
      case f29: sprintf( string_buffer, "f 29" ); break;
      case f32: sprintf( string_buffer, "f 32" ); break;
      case f36: sprintf( string_buffer, "f 36" ); break;
      case f38: sprintf( string_buffer, "f 38" ); break;
      case f40: sprintf( string_buffer, "f 40" ); break;
      case f45: sprintf( string_buffer, "f 45" ); break;
      case f51: sprintf( string_buffer, "f 51" ); break;
      case f54: sprintf( string_buffer, "f 54" ); break;
      case f57: sprintf( string_buffer, "f 57" ); break;
      case f64: sprintf( string_buffer, "f 64" ); break;
      case f72: sprintf( string_buffer, "f 72" ); break;
      case f76: sprintf( string_buffer, "f 76" ); break;
      case f81: sprintf( string_buffer, "f 81" ); break;
      case f91: sprintf( string_buffer, "f 91" ); break;
      default: fprintf( stderr, "Unknown aparature value %hu.\n Please report the value and the actual aparature. Thank you\n", (unsigned short int)properties[aperature].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[aperature].string_value = strdup( string_buffer );

   switch( properties[iso].value ) {
      case iso100: sprintf( string_buffer, "100" ); break;
      case iso200: sprintf( string_buffer, "200" ); break;
      case iso400: sprintf( string_buffer, "400" ); break;
      case iso800: sprintf( string_buffer, "800" ); break;
      case iso1000: sprintf( string_buffer, "1000" ); break;
      default: fprintf( stderr, "Unknown ISO value %hu.\n Please report the value and the actual ISO number. Thank you\n", (unsigned short int)properties[iso].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[iso].string_value = strdup( string_buffer );
   
   switch( properties[creative_modes].value ) {
      case cm_adep: sprintf( string_buffer, "A-DEP" ); break;
      case cm_m: sprintf( string_buffer, "M" ); break;
      case cm_av: sprintf( string_buffer, "Av" ); break;
      case cm_tv: sprintf( string_buffer, "Tv" ); break;
      case cm_p: sprintf( string_buffer, "P" ); break;
      case cm_none: sprintf( string_buffer, "None" ); break;
      default: fprintf( stderr, "Unknown creative mode value %hu.\n Please report the value and the actual mode. Thank you\n", (unsigned short int)properties[creative_modes].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[creative_modes].string_value = strdup( string_buffer );
   
   switch( properties[autofocus_mode].value ) {
      case af_single: sprintf( string_buffer, "Single shot" ); break;
      case af_servo: sprintf( string_buffer, "AI Servo" ); break;
      case af_mf: sprintf( string_buffer, "Manual Focus" ); break;
      default: fprintf( stderr, "Unknown autofocus value %hu.\n Please report the value and the actual autofocus method used. Thank you\n", (unsigned short int)properties[autofocus_mode].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[autofocus_mode].string_value = strdup( string_buffer );

   switch( properties[metering_mode].value ) {
      case mm_spot: sprintf( string_buffer, "Spot metering" ); break;
      case mm_center: sprintf( string_buffer, "Center weight" ); break;
      case mm_multi: sprintf( string_buffer, "Evaluative" ); break;
      default: fprintf( stderr, "Unknown metering mode value %hu.\n Please report the value and the actual metering mode used. Thank you\n", (unsigned short int)properties[metering_mode].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[metering_mode].string_value = strdup( string_buffer );

   switch( properties[driver_mode].value ) {
      case ps_single: sprintf( string_buffer, "Single shot" ); break;
      case ps_series: sprintf( string_buffer, "Continuous" ); break;
      default: fprintf( stderr, "Unknown driver mode value %hu.\n Please report the value and the actual driver mode used. Thank you\n", (unsigned short int)properties[driver_mode].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[driver_mode].string_value = strdup( string_buffer );

   switch( properties[white_point].value ) {
      case wp_auto: sprintf( string_buffer, "Auto" ); break;
      case wp_sun: sprintf( string_buffer, "Sun" ); break;
      case wp_cloud: sprintf( string_buffer, "Cloudy" ); break;
      case wp_tungsten: sprintf( string_buffer, "Tungsten" ); break;
      case wp_fluorescent: sprintf( string_buffer, "Fluorescent" ); break;
      case wp_flash: sprintf( string_buffer, "Flash" ); break;
      case wp_manual: sprintf( string_buffer, "Manual" ); break;
      default: fprintf( stderr, "Unknown white point mode value %hu.\n Please report the value and the actual white point mode used. Thank you\n", (unsigned short int)properties[white_point].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[white_point].string_value = strdup( string_buffer );
   
   switch( properties[orientation].value ) {
      case o_landscape: sprintf( string_buffer, "Landscape" ); break;
      case o_portrait: sprintf( string_buffer, "Portrait" ); break;
      default: fprintf( stderr, "Unknown orientation value %hu.\n Please report the value and the actual orientation used. Thank you\n", (unsigned short int)properties[orientation].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[orientation].string_value = strdup( string_buffer );

   switch( properties[rotation].value ) {
      case r_0: sprintf( string_buffer, "  0 degrees" ); break;
      case r_90: sprintf( string_buffer, " 90 degrees" ); break;
      case r_270: sprintf( string_buffer, "270 degrees" ); break;
      default: fprintf( stderr, "Unknown rotation value %hu.\n Please report the value and the actual rotation value used. Thank you\n", (unsigned short int)properties[rotation].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[rotation].string_value = strdup( string_buffer );

   switch( properties[flash_mode].value ) {
      case fm_none: sprintf( string_buffer, "None" ); break;
      case fm_normal: sprintf( string_buffer, "Normal" ); break;
      case fm_redeye: sprintf( string_buffer, "Red-eye reduction" ); break;
      default: fprintf( stderr, "Unknown flash mode value %hu.\n Please report the value and the actual flash mode used. Thank you\n", (unsigned short int)properties[flash_mode].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[flash_mode].string_value = strdup( string_buffer );

   switch( properties[flash_compensation].value ) {
      case ev_2_0: sprintf( string_buffer, "-2.0 EV" ); break;
      case ev_1_6: sprintf( string_buffer, "-1.66 EV" ); break;
      case ev_1_5: sprintf( string_buffer, "-1.5 EV" ); break;
      case ev_1_3: sprintf( string_buffer, "-1.33 EV" ); break;
      case ev_1_0: sprintf( string_buffer, "-1.0 EV" ); break;
      case ev_0_6: sprintf( string_buffer, "-0.66 EV" ); break;
      case ev_0_5: sprintf( string_buffer, "-0.5 EV" ); break;
      case ev_0_3: sprintf( string_buffer, "-0.33 EV" ); break;
      case ev0: sprintf( string_buffer, "0 EV" ); break;
      case ev0_3: sprintf( string_buffer, "+0.33 EV" ); break;
      case ev0_5: sprintf( string_buffer, "+0.5 EV" ); break;
      case ev0_6: sprintf( string_buffer, "+0.66 EV" ); break;
      case ev1_0: sprintf( string_buffer, "+1.0 EV" ); break;
      case ev1_3: sprintf( string_buffer, "+1.33 EV" ); break;
      case ev1_5: sprintf( string_buffer, "+1.5 EV" ); break;
      case ev1_6: sprintf( string_buffer, "+1.66 EV" ); break;
      case ev2_0: sprintf( string_buffer, "+2.0 EV" ); break;
      default: fprintf( stderr, "Unknown flash compensation value %hu.\n Please report the value and the actual flash compensation used. Thank you\n", (unsigned short int)properties[flash_compensation].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[flash_compensation].string_value = strdup( string_buffer );

   switch( properties[exposure_compensation].value ) {
      case ev_2_0: sprintf( string_buffer, "-2.0 EV" ); break;
      case ev_1_6: sprintf( string_buffer, "-1.66 EV" ); break;
      case ev_1_5: sprintf( string_buffer, "-1.5 EV" ); break;
      case ev_1_3: sprintf( string_buffer, "-1.33 EV" ); break;
      case ev_1_0: sprintf( string_buffer, "-1.0 EV" ); break;
      case ev_0_6: sprintf( string_buffer, "-0.66 EV" ); break;
      case ev_0_5: sprintf( string_buffer, "-0.5 EV" ); break;
      case ev_0_3: sprintf( string_buffer, "-0.33 EV" ); break;
      case ev0: sprintf( string_buffer, "0 EV" ); break;
      case ev0_3: sprintf( string_buffer, "+0.33 EV" ); break;
      case ev0_5: sprintf( string_buffer, "+0.5 EV" ); break;
      case ev0_6: sprintf( string_buffer, "+0.66 EV" ); break;
      case ev1_0: sprintf( string_buffer, "+1.0 EV" ); break;
      case ev1_3: sprintf( string_buffer, "+1.33 EV" ); break;
      case ev1_5: sprintf( string_buffer, "+1.5 EV" ); break;
      case ev1_6: sprintf( string_buffer, "+1.66 EV" ); break;
      case ev2_0: sprintf( string_buffer, "+2.0 EV" ); break;
      default: fprintf( stderr, "Unknown exposure compensation value %hu.\n Please report the value and the actual exposure compensation used. Thank you\n", (unsigned short int)properties[exposure_compensation].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[exposure_compensation].string_value = strdup( string_buffer );

   switch( properties[aeb_bracket_value].value ) {
      case ev_2_0: sprintf( string_buffer, "-2.0 EV" ); break;
      case ev_1_6: sprintf( string_buffer, "-1.66 EV" ); break;
      case ev_1_5: sprintf( string_buffer, "-1.5 EV" ); break;
      case ev_1_3: sprintf( string_buffer, "-1.33 EV" ); break;
      case ev_1_0: sprintf( string_buffer, "-1.0 EV" ); break;
      case ev_0_6: sprintf( string_buffer, "-0.66 EV" ); break;
      case ev_0_5: sprintf( string_buffer, "-0.5 EV" ); break;
      case ev_0_3: sprintf( string_buffer, "-0.33 EV" ); break;
      case ev0: sprintf( string_buffer, "0 EV" ); break;
      case ev0_3: sprintf( string_buffer, "+0.33 EV" ); break;
      case ev0_5: sprintf( string_buffer, "+0.5 EV" ); break;
      case ev0_6: sprintf( string_buffer, "+0.66 EV" ); break;
      case ev1_0: sprintf( string_buffer, "+1.0 EV" ); break;
      case ev1_3: sprintf( string_buffer, "+1.33 EV" ); break;
      case ev1_5: sprintf( string_buffer, "+1.5 EV" ); break;
      case ev1_6: sprintf( string_buffer, "+1.66 EV" ); break;
      case ev2_0: sprintf( string_buffer, "+2.0 EV" ); break;
      default: fprintf( stderr, "Unknown AEB bracket value %hu.\n Please report the value and the actual bracket value used. Thank you\n", (unsigned short int)properties[aeb_bracket_value].value );
	       sprintf( string_buffer, "Unknown" ); break;
   }
   properties[aeb_bracket_value].string_value = strdup( string_buffer );

   free(string_buffer);
}

void read_info(void)
{
enum crw_items i;

   for( i = 0; i < crw_items_last; ++i ) {
      if( properties[i].type == crw_short ) {
	 properties[i].value = *((short int*)(buffer + properties[i].offset));
      } else if( properties[i].type == crw_long ) {
	 properties[i].value = *((long int*)(buffer + properties[i].offset));
      } else if( properties[i].type == crw_string ) {
	 properties[i].string_value = 
	    (char*)malloc( sizeof(char) * properties[i].len );
	 strncpy( properties[i].string_value, 
	          buffer+properties[i].offset, 
		  properties[i].len );
	 properties[i].string_value[ properties[i].len - 1 ] = 0;
      } else {
	 fprintf( stderr, "internal error. aborting\n" );
	 exit(-1);
      }
   }
}

int read_information_from_crw( const char *filename )
{
int fd = open( filename, O_RDONLY );
long int offset_buffer;
off_t info_block_start = 0;
off_t info_block_end = 0;

   if( fd == -1 ) 
      return -1;

   lseek( fd, info_block_start_offset, SEEK_END );
   if( read( fd, &offset_buffer, 4 ) == -1 )
      return -1;
   info_block_start = 26 + (off_t)offset_buffer;

   lseek( fd, info_block_end_offset, SEEK_END );
   if( read( fd, &offset_buffer, 4 ) == -1 )
      return -1;
   info_block_end = 26 + (off_t)offset_buffer;
  
   buffer = (char*)malloc( sizeof(char) * (info_block_end - info_block_start));
   lseek( fd, info_block_start, SEEK_SET );
   if( read( fd, buffer, sizeof(char) * (info_block_end - info_block_start)) == -1 )
      return -1;
   close(fd);
   return 0;
}

int extract_thumbnail( void )
{
long int offset_buffer;
off_t thumb_block_start = 0;
ssize_t thumb_block_size = 0;
char *thumb_buffer = (char*)malloc( sizeof(char) * 8192 );
ssize_t bytes_read = 0;
int thumbfd;
int fd = open( options.filename, O_RDONLY );

   if( fd == -1 ) 
      return -1;

   lseek( fd, thumb_block_start_offset, SEEK_END );
   if( read( fd, &offset_buffer, 4 ) == -1 )
      return -1;
   thumb_block_start = 26 + (off_t)offset_buffer;

   lseek( fd, thumb_block_size_offset, SEEK_END );
   if( read( fd, &offset_buffer, 4 ) == -1 )
      return -1;
   thumb_block_size = (off_t)offset_buffer;

   thumbfd = open( options.thumbnail, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR );
   if( thumbfd == -1 ) 
      return -1;
   lseek( fd, thumb_block_start, SEEK_SET );
   while( thumb_block_size > 0 ) {
      if( ( bytes_read = read( fd, thumb_buffer, 8192) ) == -1 )
	 return -1;
      if( bytes_read == 0 ) {
	 fprintf( stderr, "Premature eof reached while extracting thumbnail.\n");
	 break;
      }
      write( thumbfd, thumb_buffer, bytes_read );
      thumb_block_size -= bytes_read;
   }
   close( thumbfd );
   close(fd);
   free( thumb_buffer );
   return 0;
}

void init_property_array( void ) 
{
enum crw_items i;
   
   for( i = 0; i < crw_items_last; ++i ) {
      properties[i].len = 2;
      properties[i].value = -1;
      properties[i].string_value = NULL;
      properties[i].type = crw_short;
   }

   properties[timestamp].offset = 0x0000; properties[timestamp].len = 4;
   properties[timestamp].type = crw_long;
   properties[fileformat].offset = 0x0178; properties[fileformat].len = 32;
   properties[fileformat].type = crw_string;
   properties[owner].offset = 0x01d2; properties[owner].len = 32;
   properties[owner].type = crw_string;
   properties[vendor].offset = 0x01f2; properties[vendor].len = 6;
   properties[vendor].type = crw_string;
   properties[camera_model].offset = 0x01f8; properties[camera_model].len = 14;
   properties[camera_model].type = crw_string;
   properties[camera_firmware].offset = 0x0212; properties[camera_firmware].len = 32;
   properties[camera_firmware].type = crw_string;
   properties[camera_serial].offset = 0x0236; properties[camera_serial].len = 4;
   properties[camera_serial].type = crw_long;
   properties[shutter_speed].offset = 0x02b0; 
   properties[bulb_duration].offset = 0x02b4; properties[bulb_duration].len = 4;
   properties[bulb_duration].type = crw_long;
   properties[aperature].offset = 0x02ae; 
   properties[iso].offset = 0x0288; 
   properties[creative_modes].offset = 0x02e2; 
   properties[autofocus_mode].offset = 0x02c8;
   properties[auto_manual_focus1].offset = 0x02e0;
   properties[auto_manual_focus2].offset = 0x02a0;
   properties[metering_mode].offset = 0x02dc;
   properties[photo_in_sequence].offset = 0x0296; 
   properties[shutter_delay].offset = 0x02be; 
   properties[driver_mode].offset = 0x02c4; 
   properties[white_point].offset = 0x0292;
   properties[exposure_compensation].offset = 0x0290; 
   properties[focus_distance_lower].offset = 0x02aa; 
   properties[focus_distance_upper].offset = 0x02ac;
   properties[zoom_range_lower].offset = 0x02e8;
   properties[zoom_range_upper].offset = 0x02ea;
   properties[focal_length_used].offset = 0x152e;
   properties[flash_compensation].offset = 0x02a2;
   properties[parameter_contrast].offset = 0x02d4;
   properties[parameter_saturation].offset = 0x02d6;
   properties[parameter_sharpness].offset = 0x02d8;
   properties[parameter_colortone].offset = 0x030e;
   properties[flash_mode].offset = 0x02c2; 
   properties[orientation].offset = 0x15f4; 
   properties[rotation].offset = 0x000c; 
   properties[aeb_in_sequence].offset = 0x02a4; 
   properties[aeb_bracket_value].offset = 0x02a6; 
   properties[cf_2].offset = 0x0f1c;  
   properties[cf_3].offset = 0x0f1e; 
   properties[cf_4].offset = 0x0f20;
   properties[cf_5].offset = 0x0f22;  
   properties[cf_6].offset = 0x0f24; 
   properties[cf_7].offset = 0x0f26;
   properties[cf_8].offset = 0x0f28;  
   properties[cf_9].offset = 0x0f2a; 
   properties[cf_10].offset = 0x0f2c;
   properties[cf_11].offset = 0x0f2e;
   properties[cf_12].offset = 0x0f30;
   properties[cf_13].offset = 0x0f32;
   properties[cf_14].offset = 0x0f34;
   properties[cf_15].offset = 0x0f36; 
   /* experimental values */
   properties[x_lensid].offset = 0x02ec; properties[x_lensid].len = 4;
   properties[x_lensid].type = crw_long;
   properties[x_colortemp].offset = 0x14da;
   properties[x_gamma].offset = 0x02b2; 

}
