/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <glib.h>

#include <libraw1394/raw1394.h>

#include "sbp2main.h"


typedef gint (*ReadWriteFunc) (raw1394handle_t  handle,
			       nodeid_t         node,
			       nodeaddr_t       address,
			       gsize            length,
			       quadlet_t       *buffer,
			       gulong           tag);

struct _SBP2Tag
{
  guint              ref_count;

  SBP2TagProcessFunc process;
  SBP2TagFreeFunc    free;

  gpointer           data;
};


static gboolean raw_activity    (GIOChannel        *source,
				 GIOCondition       condition,
				 gpointer           data);

static gint     tag_handler     (raw1394handle_t    handle,
				 gulong             the_tag,
				 raw1394_errcode_t  errcode);

static void     read_write_data (ReadWriteFunc      read_write_func,
				 raw1394handle_t    handle,
				 gsize              chunk_size,
				 nodeid_t           node,
				 SBP2Pointer       *pointer,
				 gpointer           data,
				 gsize              length,
				 SBP2Tag           *tag);


guint
sbp2_watch_handle (raw1394handle_t  handle,
		   GMainContext    *context)
{
  gint        fd;
  GIOChannel *channel;
  GSource    *source;
  guint       id;

  g_return_val_if_fail (handle != NULL, 0);
  g_return_val_if_fail (context != NULL, 0);

  raw1394_set_tag_handler (handle, tag_handler);

  fd = raw1394_get_fd (handle);

  channel = g_io_channel_unix_new (fd);

  g_io_channel_set_encoding (channel, NULL, NULL);
  g_io_channel_set_buffered (channel, FALSE);

  source = g_io_create_watch (channel, G_IO_IN | G_IO_PRI);
  g_io_channel_unref (channel);

  g_source_set_priority (source, G_PRIORITY_HIGH_IDLE);

  g_source_set_callback (source, (GSourceFunc) raw_activity, handle, NULL);
  id = g_source_attach (source, context);

  return id;
}

SBP2Tag *
sbp2_tag_new (SBP2TagProcessFunc process,
              SBP2TagFreeFunc    free,
	      gpointer           data)
{
  SBP2Tag *tag;

  tag = g_new (SBP2Tag, 1);

  tag->ref_count = 0;

  tag->process = process;
  tag->free    = free;
  tag->data    = data;

  return tag;
}

void
sbp2_write_data (raw1394handle_t  handle,
                 gsize            chunk_size,
		 nodeid_t         node,
		 SBP2Pointer     *pointer,
		 gpointer         data,
		 gsize            length)
{
  SBP2Tag *tag;

  tag = sbp2_tag_new (NULL, g_free, data);

  read_write_data (raw1394_start_write, handle, chunk_size, node, pointer,
                   data, length, tag);
}

void
sbp2_read_data (raw1394handle_t  handle,
                gsize            chunk_size,
		nodeid_t         node,
		SBP2Pointer     *pointer,
		gpointer         data,
		gsize            length,
		SBP2Tag         *tag)
{
  read_write_data (raw1394_start_read, handle, chunk_size, node, pointer,
                   data, length, tag);
}

static gboolean
raw_activity (GIOChannel   *channel,
              GIOCondition  condition,
	      gpointer      data)
{
  raw1394handle_t handle = data;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (handle != NULL, FALSE);

  raw1394_loop_iterate (handle);

  return TRUE;
}

static gint
tag_handler (raw1394handle_t   handle,
	     gulong            the_tag,
	     raw1394_errcode_t errcode)
{
  SBP2Tag *tag = (SBP2Tag *) the_tag;

  g_return_val_if_fail (handle != NULL, -1);
  g_return_val_if_fail (tag != NULL, -1);
  g_return_val_if_fail (tag->ref_count > 0, -1);

  tag->ref_count--;

  if (tag->ref_count == 0)
    {
      gboolean free_data = TRUE;

      if (errcode != 0 && tag->process)
	free_data = tag->process (tag->data);

      if (free_data && tag->free)
	tag->free (tag->data);

      g_free (tag);
    }

  return 0;
}

static void
read_write_data (ReadWriteFunc    read_write_func,
		 raw1394handle_t  handle,
		 gsize            chunk_size,
		 nodeid_t         node,
		 SBP2Pointer     *pointer,
		 gpointer         data,
		 gsize            length,
		 SBP2Tag         *tag)
{
  nodeaddr_t  address;
  guint8     *buffer;
  gsize       left;

  g_return_if_fail (handle != NULL);
  g_return_if_fail (pointer != NULL);
  g_return_if_fail (data != NULL);
  g_return_if_fail (length > 0);
  g_return_if_fail (tag != NULL);

  if (chunk_size == 0)
    chunk_size = length;

  address = sbp2_pointer_to_1394_address (pointer);

  buffer = data;
  left = length;

  while (left > chunk_size)
    {
      tag->ref_count++;

      read_write_func (handle, node, address, chunk_size,
		       (quadlet_t *) buffer, (gulong) tag);

      buffer += chunk_size;
      address += chunk_size;

      left -= chunk_size;
    }

  tag->ref_count++;

  read_write_func (handle, node, address, left,
		   (quadlet_t *) buffer, (gulong) tag);
}
