/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <string.h>

#include <glib.h>

#include <libraw1394/raw1394.h>
#include <libraw1394/csr.h>

#include "sbp2byteswap.h"
#include "sbp2configrom.h"
#include "sbp2csr.h"
#include "sbp2manager.h"
#include "sbp2main.h"
#include "sbp2raw1394.h"


enum
{
  FUNC_LOGIN_REQUEST        = 0x0,
  FUNC_QUERY_LOGINS_REQUEST = 0x1,
  FUNC_RECONNECT_REQUEST    = 0x3,
  FUNC_SET_PASSWORD_REQUEST = 0x4,
  FUNC_LOGOUT_REQUEST       = 0x7,
  FUNC_ABORT_TASK_REQUEST   = 0xb,
  FUNC_ABORT_TASK_SET       = 0xc,
  FUNC_LOGICAL_UNIT_RESET   = 0xe,
  FUNC_TARGET_RESET_REQUEST = 0xf
};


typedef struct raw1394_arm_request ARMRequest;
typedef struct raw1394_arm_request_response ARMReqResp;

typedef struct _SBP2ManagementORB SBP2ManagementORB;
typedef struct _ORBData ORBData;

struct _SBP2ManagementORB
{
  guint32     reserved1;
  guint32     reserved2;
  guint32     reserved3;
  guint32     reserved4;
#if G_BYTE_ORDER == G_BIG_ENDIAN
  guint       notify    : 1;
  guint       req_fmt   : 2;
  guint       reserved5 : 9;
  guint       function  : 4;
  guint       reserved6 : 16;
#else
  guint       reserved6 : 16;
  guint       function  : 4;
  guint       reserved5 : 9;
  guint       req_fmt   : 2;
  guint       notify    : 1;
#endif
  guint32     reserved7;
  SBP2Pointer status_FIFO;
};

struct _ORBData
{
  SBP2Manager       *manager;

  nodeid_t           node;
  SBP2Pointer        pointer;

  SBP2ManagementORB  orb;
};

struct _SBP2Manager
{
  raw1394handle_t               handle;
  guint                         id;

  struct raw1394_arm_reqhandle  arm_management_agent;

  SBP2ManagerFuncs             *funcs;
  gpointer                      user_data;
};


static gint     management_agent  (raw1394handle_t    handle,
                                   ARMReqResp        *arm_req_resp,
                                   guint              requested_length,
                                   gpointer           manager,
                                   guint8             request_type);

static void     fetch_orb         (SBP2Manager       *manager,
                                   nodeid_t           node,
				   SBP2Pointer       *pointer);
static gboolean process_orb       (gpointer           data);

static gboolean login             (SBP2ManagerAction *action);
static gboolean query_logins      (SBP2ManagerAction *action);
static gboolean reconnect         (SBP2ManagerAction *action);
static gboolean logout            (SBP2ManagerAction *action);

static gint     bus_reset_handler (raw1394handle_t    handle,
                                   guint              generation);


SBP2Manager *
sbp2_manager_new (SBP2ManagerFuncs *funcs,
                  GMainContext     *context,
                  gint              port,
                  gpointer          user_data)
{
  SBP2Manager     *manager = NULL;
  raw1394handle_t  handle = NULL;
  gint             ret;

  handle = sbp2_raw1394_handle ();

  if (!handle)
    goto fail;

  if (raw1394_set_port (handle, port) != 0)
    goto fail;

  if (!sbp2_setup_config_rom (handle))
    goto fail;

  manager = g_new (SBP2Manager, 1);

  manager->arm_management_agent.arm_callback = management_agent;
  manager->arm_management_agent.pcontext = manager;

  ret = raw1394_arm_register (handle,
                              CSR_REGISTER_BASE + SBP2_CSR_MANANGEMENT_AGENT,
			      8, NULL,
			      (gulong) &manager->arm_management_agent,
			      RAW1394_ARM_READ | RAW1394_ARM_WRITE,
			      RAW1394_ARM_WRITE,
			      0);
  if (ret < 0)
    goto fail;

  raw1394_set_userdata (handle, manager);
  raw1394_set_bus_reset_handler (handle, bus_reset_handler);

  manager->handle = handle;
  manager->id = sbp2_watch_handle (handle, context);

  manager->funcs = funcs;
  manager->user_data = user_data;

  return manager;

fail:
  g_free (manager);
  raw1394_destroy_handle (handle);

  return NULL;
}

void
sbp2_manager_destroy (SBP2Manager *manager)
{
  g_return_if_fail (manager != NULL);

  g_source_remove (manager->id);
  
  raw1394_destroy_handle (manager->handle);

  g_free (manager);
}

void
sbp2_manager_write_status (SBP2Manager *manager,
                           nodeid_t     node,
			   SBP2Pointer *pointer,
			   SBP2Status  *status)
{
  gsize length;

  g_return_if_fail (manager != NULL);
  g_return_if_fail (pointer != NULL);
  g_return_if_fail (status != NULL);

  length = (status->len + 1) * 4;

  sbp2_cpu_to_be32_data (status, length);
  sbp2_write_data (manager->handle, 0, node, pointer, status, length);
}

void
sbp2_manager_reset_bus (SBP2Manager *manager)
{
  g_return_if_fail (manager != NULL);

  raw1394_reset_bus (manager->handle);
}

static gint
management_agent (raw1394handle_t  handle,
                  ARMReqResp      *arm_req_resp,
		  guint            requested_length,
		  gpointer         manager,
		  guint8           request_type)
{
  ARMRequest  *request;
  SBP2Pointer *pointer;

  g_return_val_if_fail (manager != NULL, -1);

  request = arm_req_resp->request;

  pointer = (SBP2Pointer *) request->buffer;
  sbp2_be32_to_cpu_data (pointer, sizeof (*pointer));

  fetch_orb (manager, request->source_nodeid, pointer);

  return 0;
}

static void
fetch_orb (SBP2Manager *manager,
           nodeid_t     node,
	   SBP2Pointer *pointer)
{
  ORBData *orb_data;
  SBP2Tag *tag;

  g_return_if_fail (manager != NULL);
  g_return_if_fail (pointer != NULL);

  orb_data = g_new (ORBData, 1);

  orb_data->manager = manager;

  orb_data->node = node;
  memcpy (&orb_data->pointer, pointer, sizeof (orb_data->pointer));

  tag = sbp2_tag_new (process_orb, g_free, orb_data);

  sbp2_read_data (manager->handle, 0, node, pointer,
		  &orb_data->orb, sizeof (orb_data->orb),
		  tag);
}

static gboolean
process_orb (gpointer data)
{
  ORBData           *orb_data = data;
  nodeid_t           node;
  SBP2Pointer       *pointer;
  SBP2ManagementORB *orb;
  SBP2Manager       *manager;
  SBP2Status        *status;
  SBP2ManagerAction  action;
  gboolean           ret;

  g_return_val_if_fail (orb_data != NULL, FALSE);

  manager = orb_data->manager;

  node = orb_data->node;
  pointer = &orb_data->pointer;

  orb = &orb_data->orb;

  sbp2_be32_to_cpu_data (orb, sizeof (*orb));

  status = g_new0 (SBP2Status, 1);

  sbp2_status_fill_pointer (status, pointer);

  status->len = 1;
  status->src = SBP2_STATUS_ORIGIN_NULL;
  status->resp = SBP2_STATUS_RESP_REQUEST_COMPLETE;

  action.manager   = manager;
  action.node      = node;
  action.pointer   = pointer;
  action.status    = status;
  action.orb       = orb;
  action.user_data = manager->user_data;

  switch (orb->function)
    {
    case FUNC_LOGIN_REQUEST:
      ret = login (&action);
      break;

    case FUNC_QUERY_LOGINS_REQUEST:
      ret = query_logins (&action);
      break;

    case FUNC_RECONNECT_REQUEST:
      ret = reconnect (&action);
      break;

    case FUNC_LOGOUT_REQUEST:
      ret = logout (&action);
      break;

    case FUNC_SET_PASSWORD_REQUEST:
    case FUNC_ABORT_TASK_REQUEST:
    case FUNC_ABORT_TASK_SET:
    case FUNC_LOGICAL_UNIT_RESET:
    case FUNC_TARGET_RESET_REQUEST:
    default:
      status->resp = SBP2_STATUS_RESP_ILLEGAL_REQUEST;
      ret = TRUE;
      break; 
    }

  if (ret)
    sbp2_manager_write_status (manager, node, &orb->status_FIFO, status);
  else
    g_free (status);

  return TRUE;
}

static gboolean
login (SBP2ManagerAction *action)
{
  SBP2LoginORB      *orb = action->orb;
  SBP2LoginResponse *resp;
  SBP2Manager       *manager = action->manager;
  gint               length;

  resp = g_new0 (SBP2LoginResponse, 1);

  if (!manager->funcs->login (action, orb, resp))
    {
      g_free (resp);
      return FALSE;
    }

  length = resp->length;

  sbp2_cpu_to_be32_data (resp, length);
  sbp2_write_data (manager->handle, 0, action->node, &orb->login_resp,
		   resp, length);

  return TRUE;
}

static gboolean
query_logins (SBP2ManagerAction *action)
{
  SBP2QueryLoginsORB      *orb = action->orb;
  SBP2QueryLoginsResponse *resp;
  SBP2Manager             *manager = action->manager;
  gsize                    length;

  resp = g_new0 (SBP2QueryLoginsResponse, 1);

  if (!manager->funcs->query_logins (action, orb, resp))
    {
      g_free (resp);
      return FALSE;
    }

  length = 4; /* TODO: use resp->length */

  sbp2_cpu_to_be32_data (resp, length);
  sbp2_write_data (manager->handle, 0, action->node, &orb->query_resp,
		   resp, length);

  return TRUE;
}

static gboolean
reconnect (SBP2ManagerAction *action)
{
  SBP2ReconnectORB *orb = action->orb;
  SBP2Manager      *manager = action->manager;

  return manager->funcs->reconnect (action, orb);
}

static gboolean
logout (SBP2ManagerAction *action)
{
  SBP2LogoutORB *orb = action->orb;
  SBP2Manager   *manager = action->manager;

  return manager->funcs->logout (action, orb);
}

static gint
bus_reset_handler (raw1394handle_t handle,
		   guint           generation)
{
  SBP2Manager *manager;

  g_return_val_if_fail (handle != NULL, -1);

  raw1394_update_generation (handle, generation);

  manager = raw1394_get_userdata (handle);

  g_return_val_if_fail (manager != NULL, -1);

  manager->funcs->bus_reset (manager, manager->user_data);

  return 0;
}
