/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <unistd.h>
#include <signal.h>

#include <glib.h>

#include "app.h"
#include "cleanup.h"
#include "util.h"


static gboolean do_cleanup      (GIOChannel   *channel,
                                 GIOCondition  condition,
				 EndpointApp  *app);

static void     trigger_cleanup (gint          signum);


static gint cleanup_signum = SIGTERM;
static gint cleanup_fd = -1;


gboolean
cleanup_register (EndpointApp *app,
		  CleanupFunc  cleanup)
{
  gint        cpipe[2];
  GIOChannel *channel;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (cleanup != NULL, FALSE);

  cleanup_unregister (app);

  if (pipe (cpipe) < 0)
    return FALSE;

  app->cleanup = cleanup;

  cleanup_fd = cpipe[1];

  channel = util_io_channel_new (cpipe[0]);

  util_io_add_watch (app->context, channel,
                     G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
		     (GIOFunc) do_cleanup, app);

  signal (SIGINT,  trigger_cleanup);
  signal (SIGQUIT, trigger_cleanup);
  signal (SIGABRT, trigger_cleanup);
  signal (SIGTERM, trigger_cleanup);
 
  return TRUE; 
}

void
cleanup_unregister (EndpointApp *app)
{
  g_return_if_fail (app != NULL);

  if (app->cleanup)
    {
      signal (SIGINT,  SIG_DFL);
      signal (SIGQUIT, SIG_DFL);
      signal (SIGABRT, SIG_DFL);
      signal (SIGTERM, SIG_DFL);

      app->cleanup = NULL;

      g_return_if_fail (cleanup_fd > 0);

      close (cleanup_fd);
      cleanup_fd = -1;
    }
}

static gboolean
do_cleanup (GIOChannel   *channel,
	    GIOCondition  condition,
            EndpointApp  *app)
{
  CleanupFunc cleanup;
  gint        signum;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->cleanup != NULL, FALSE);

  signum = cleanup_signum;

  signal (signum, SIG_IGN);

  cleanup = app->cleanup;
  cleanup (app, signum);

  app->context = NULL;
  g_main_loop_quit (app->loop);

  signal (signum, SIG_DFL);

  return FALSE;
}

static void
trigger_cleanup (gint signum)
{
  cleanup_signum = signum;
  write (cleanup_fd, "C", 1);
}
