/* $Id: pktrecord.c,v 1.2 2004/09/22 15:15:52 iscjonm Exp $
 *
 * Copyright (C) 2004 The Trustees of the University of Pennsylvania
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* utilities for reading saved packets out of a packet trace
   file. Currently we can handle:
     1761-compliant (Solaris snoop v2)
     libpcap/tcpdump
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/uio.h>
#include <inttypes.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_PCAP_H
#include <pcap.h>
#endif

#include "pktrecord.h"

struct prec_handle {
  int filetype;			/* which type of packet record */
  unsigned char *hdr_data;	/* pointer to the file header */

#ifdef HAVE_PCAP_H
  pcap_t *pcap_handle;		/* for tcpdump/pcap packet records */
  struct pcap_pkthdr *pcap_hdr;
#endif
};

int snoopv2_init(struct prec_handle **phout) {
  unsigned char *pkthdr = NULL;
  unsigned int version_num;
  struct prec_handle *ph = NULL;

  if (phout == NULL) goto err;
  *phout = NULL;

  if ((pkthdr = malloc(16)) == NULL) {
    fprintf(stderr,"%s:%d: out of memory\n",__FILE__,__LINE__);
    goto err;
  }
  if (fread(pkthdr,16,1,stdin) < 1) goto err;
  if ((ph = malloc(sizeof(struct prec_handle))) == NULL) {
    fprintf(stderr,"%s:%d: out of memory\n",__FILE__,__LINE__);
    goto err;
  }
  memset(ph, 0, sizeof(struct prec_handle));
  ph->hdr_data = pkthdr;

  /* see if this is really a snoopv2 file */
  /* see RFC 1761 for further details */
  if ((char)pkthdr[0] != 's' || (char)pkthdr[1] != 'n' ||
      (char)pkthdr[2] != 'o' || (char)pkthdr[3] != 'o' ||
      (char)pkthdr[4] != 'p' || (char)pkthdr[5] != '\0' ||
      (char)pkthdr[6] != '\0' || (char)pkthdr[7] != '\0') goto err;
  memcpy(&version_num,&(pkthdr[8]),4);
  version_num = ntohl(version_num);
  if (version_num != 2) goto err;
  
  *phout = ph;
  return 1;

 err:
  if (pkthdr != NULL) free(pkthdr);
  if (ph != NULL) free(ph);
  return 0;
}

int snoopv2_is_eth(struct prec_handle *ph) {
  unsigned int dlink_type;

  if (ph == NULL || ph->hdr_data == NULL) return 0;
  memcpy(&dlink_type, &(ph->hdr_data[12]), 4);
  
  /* "4" is the datalink type for Ethernet */
  return(ntohl(dlink_type) == 4);
}

int snoopv2_next(struct prec_handle *ph, struct pktrecord *p) {
  int tmpint;
  unsigned int record_len;

  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  p->orig_len = ntohl(tmpint);
  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  p->included_len = ntohl(tmpint);
  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  record_len = ntohl(tmpint);
  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  /* cum_drops not used */
  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  p->ts_secs = ntohl(tmpint);
  if (fread(&tmpint,4,1,stdin) < 1) return 0;
  p->ts_usecs = ntohl(tmpint);

  if (p->pkt_data != NULL) free(p->pkt_data);
  if ((p->pkt_data = malloc(record_len - 24)) == NULL) return 0;
  if (fread(p->pkt_data,record_len - 24,1,stdin) < 1)
    return 0;

  return 1;
}

int snoopv2_close(struct prec_handle *ph) {
  return 1;			/* nothing to do */
}

#ifdef HAVE_PCAP_H
int tcpdump_init(struct prec_handle **phout) {
  struct prec_handle *ph = NULL;
  char errbuf[PCAP_ERRBUF_SIZE];

  if (phout == NULL) goto err;
  *phout = NULL;
  if ((ph = malloc(sizeof(struct prec_handle))) == NULL) goto err;
  memset(ph, 0, sizeof(struct prec_handle));

  /* a file argument of "-" means stdin */
  if ((ph->pcap_handle = pcap_open_offline("-",errbuf)) == NULL)
    goto err;

#ifndef HAVE_PCAP_NEXT_EX
  if ((ph->pcap_hdr = malloc(sizeof(struct pcap_pkthdr))) == NULL)
    goto err;
#endif

  *phout = ph;
  return 1;

 err:
  if (ph != NULL) free(ph);
  return 0;
}

int tcpdump_is_eth(struct prec_handle *ph) {
  if (ph == NULL || ph->pcap_handle == NULL) return 0;
  return (pcap_datalink(ph->pcap_handle) == DLT_EN10MB);
}

int tcpdump_next(struct prec_handle *ph, struct pktrecord *p) {
  if (ph == NULL || p == NULL || ph->pcap_handle == NULL) return 0;
  
#ifdef HAVE_PCAP_NEXT_EX
  if (pcap_next_ex(ph->pcap_handle, &(ph->pcap_hdr),
		   &(p->pkt_data)) != 1) return 0;
#else
  /* older version of libpcap */
  p->pkt_data = (unsigned char *)pcap_next(ph->pcap_handle, ph->pcap_hdr);
  if (p->pkt_data == NULL) return 0;
#endif
  p->orig_len = ph->pcap_hdr->len;
  p->included_len = ph->pcap_hdr->caplen;
  p->ts_secs = ph->pcap_hdr->ts.tv_sec;
  p->ts_usecs = ph->pcap_hdr->ts.tv_usec;
  
  return 1;
}

int tcpdump_close(struct prec_handle *ph) {
  pcap_close(ph->pcap_handle);
#ifndef HAVE_PCAP_NEXT_EX
  if (ph->pcap_hdr != NULL) free(ph->pcap_hdr);
#endif
  return 1;
}
#endif

/* Our basic strategy is to loop through these file types until one of
   the init functions succeeds, doing a rewind() on stdin after failed
   attempts */

struct prec_type {
  char *name;
  int (*init)(struct prec_handle **hout);
  int (*is_eth)(struct prec_handle *ph);
  int (*next_pkt)(struct prec_handle *ph, struct pktrecord *p);
  int (*close)(struct prec_handle *ph);
};

struct prec_type prec_types[] = {
  { "RFC 1761 (snoop v2)",
    snoopv2_init, snoopv2_is_eth, snoopv2_next, snoopv2_close },
#ifdef HAVE_PCAP_H
  { "libpcap/tcpdump",
    tcpdump_init, tcpdump_is_eth, tcpdump_next, tcpdump_close },
#endif
  /* prec_init expects this NULL entry to be here to find the end of
     the array */
  { NULL, NULL, NULL, NULL, NULL },
};

int prec_init(struct prec_handle **phout) {
  int i;
#ifdef HAVE_PCAP_H
  unsigned int magic;
  unsigned int tcpdump_magic = 0xa1b2c3d4;
  unsigned int tcpdump_magic_swapped = 0xd4c3b2a1;
  unsigned int patched_tcpdump_magic = 0xa1b2cd34;
  unsigned int patched_tcpdump_magic_swapped = 0x34cdb2a1;
#endif
  unsigned char word[4];
  int filetype = -1;

  /* let's have a peek at the first four bytes */
  for(i=0; i<4; i++) {
    if ((char)(word[i] = fgetc(stdin)) == EOF) return 0;
  }
  
  /* check for snoop v2 */
  if ((char)word[0] == 's' && (char)word[1] == 'n' &&
      (char)word[2] == 'o' && (char)word[3] == 'o') {
    filetype = 0;
  }
#ifdef HAVE_PCAP_H
  else {
    /* check for libpcap format, in either byte order */
    memcpy(&magic, word, 4);
    if (magic == tcpdump_magic || magic == patched_tcpdump_magic ||
	magic == tcpdump_magic_swapped ||
	magic == patched_tcpdump_magic_swapped) {
      filetype = 1;
    }
  }
#endif

  /* put back what we peeked at */
  for(i=3; i>=0; i--) {
    if ((char)(ungetc(word[i],stdin)) == EOF) return 0;
  }

  /* didn't find anything we recognized */
  if (filetype == -1) return 0;

  if ((*(prec_types[filetype].init))(phout)) {
      (*phout)->filetype = filetype;
      fprintf(stderr,"Looks like a(n) %s packet trace\n",
	      prec_types[filetype].name);
      return 1;
  } else {
    return 0;
  }
}

/* for these next three, just call the filetype-specific function */

int prec_is_eth(struct prec_handle *ph) {
  if (ph == NULL) return 0;
  return (*(prec_types[ph->filetype].is_eth))(ph);
}

int prec_next_packet(struct prec_handle *ph, struct pktrecord *p) {
  if (ph == NULL || p == NULL) return 0;
  return (*(prec_types[ph->filetype].next_pkt))(ph,p);
}

int prec_close(struct prec_handle *ph) {
  if (ph == NULL) return 0;
  return (*(prec_types[ph->filetype].close))(ph);
}
