"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdeProjectService = void 0;
const SIDE_EFFECT_PROJECTS_LIMIT = 20;
const SIDE_EFFECT_FILES_LIMIT = 30;
// Spreading expiration time to avoid closing multiple files at once
const SIDE_EFFECT_EXPIRATION_TIME_MIN = 5 * 60 * 1000; // 5 mins
const SIDE_EFFECT_EXPIRATION_TIME_MAX = 10 * 60 * 1000; // 10 mins
class IdeProjectService {
    constructor(ts, projectService) {
        this.ts = ts;
        this.projectService = projectService;
        this.openedFilesByIde = new Set();
        this.openedProjectsByIde = new Set();
        this.openedFilesBySideEffect = new Map();
        this.openedProjectsBySideEffect = new Map();
        this.fallbackToOldEvalOutsideOfImportGraph = process.argv.slice(2).indexOf("--noFallbackToOldEvalOutsideOfImportGraph") < 0;
        let thisService = this;
        const openClientFileWithNormalizedPath = projectService.openClientFileWithNormalizedPath;
        const closeClientFile = projectService.closeClientFile;
        const closeExternalProject = projectService.closeExternalProject;
        const openExternalProject = projectService.openExternalProject;
        if (openClientFileWithNormalizedPath) {
            this.originalOpenClientFileWithNormalizedPath = openClientFileWithNormalizedPath.bind(projectService);
            projectService.openClientFileWithNormalizedPath = function (fileName, _fileContent, _scriptKind, _hasMixedContent, _projectRootPath) {
                thisService.fileOpenedByIde(fileName);
                return openClientFileWithNormalizedPath.apply(projectService, arguments);
            };
        }
        else {
            this.originalOpenClientFileWithNormalizedPath = () => {
                throw new Error("Unsupported version of compiler plugin");
            };
        }
        if (closeClientFile) {
            this.originalCloseClientFile = closeClientFile.bind(projectService);
            projectService.closeClientFile = function (uncheckedFileName) {
                thisService.fileClosedByIde(uncheckedFileName);
                closeClientFile.apply(projectService, arguments);
            };
        }
        else {
            this.originalCloseClientFile = () => {
                throw new Error("Unsupported version of compiler plugin");
            };
        }
        if (openExternalProject) {
            projectService.openExternalProject = function (proj) {
                thisService.projectOpenedByIde(proj.projectFileName);
                openExternalProject.apply(projectService, arguments);
            };
        }
        if (closeExternalProject) {
            this.originalCloseExternalProject = closeExternalProject.bind(projectService);
            projectService.closeExternalProject = function (uncheckedFileName) {
                thisService.projectClosedByIde(uncheckedFileName);
                closeExternalProject.apply(projectService, arguments);
            };
        }
        else {
            this.originalCloseExternalProject = () => {
                throw new Error("Unsupported version of compiler plugin");
            };
        }
    }
    closeClientFileSafely(fileName) {
        let normalizedFileName = this.ts.server.toNormalizedPath(fileName);
        this.fileClosedByIde(normalizedFileName);
        this.closeSafely(normalizedFileName);
    }
    getProjectAndSourceFile(fileName, projectFileName) {
        var _a, _b, _c, _d;
        let normalizedProjectFileName = projectFileName ? this.ts.server.toNormalizedPath(projectFileName) : undefined;
        let normalizedFileName = this.ts.server.toNormalizedPath(fileName);
        let project = this.findOrOpenProject(normalizedProjectFileName);
        let sourceFile = (_b = (_a = project === null || project === void 0 ? void 0 : project.getLanguageService(true)) === null || _a === void 0 ? void 0 : _a.getProgram()) === null || _b === void 0 ? void 0 : _b.getSourceFile(normalizedFileName);
        if (projectFileName && !sourceFile && this.fallbackToOldEvalOutsideOfImportGraph) {
            let error = new Error("File outside of import graph");
            error.isFileOutsideOfImportGraphError = true;
            throw error;
        }
        if (!project || !sourceFile) {
            // Try to fallback to default project for the file
            project = this.findOrOpenDefaultProjectForFile(normalizedFileName);
            sourceFile = (_d = (_c = project === null || project === void 0 ? void 0 : project.getLanguageService(true)) === null || _c === void 0 ? void 0 : _c.getProgram()) === null || _d === void 0 ? void 0 : _d.getSourceFile(fileName);
        }
        return { project, sourceFile };
    }
    tryOpenProject(projectFileName) {
        if (this.internalTryOpenProject(projectFileName)) {
            this.projectOpenedByIde(projectFileName);
            return true;
        }
        return false;
    }
    closeSafely(fileName) {
        const wasOpened = this.projectService.openFiles.has(fileName)
            || (!this.ts.sys.useCaseSensitiveFileNames && this.projectService.openFiles.has(fileName.toLowerCase()));
        if (wasOpened) {
            this.projectService.logger.info(`IdeProjectService.closeSafely: closing file - ${fileName}`);
            this.originalCloseClientFile(fileName);
        }
        else {
            this.projectService.logger.info(`IdeProjectService.closeSafely: file is no longer open - ${fileName}`);
        }
    }
    findOrOpenProject(projectFileName) {
        if (!projectFileName)
            return undefined;
        let project = this.projectService.findProject(projectFileName);
        if (!project) {
            this.openProjectBySideEffect(projectFileName);
            project = this.projectService.findProject(projectFileName);
        }
        else {
            this.usedAsSideEffect('project', projectFileName);
        }
        return project;
    }
    openProjectBySideEffect(projectFileName) {
        this.projectService.logger.info(`IdeProjectService.openProjectBySideEffect: opening project by side effect - ${projectFileName}`);
        if (this.internalTryOpenProject(projectFileName)) {
            this.openedAsSideEffect('project', projectFileName);
            this.cleanupOpenedProjectsBySideEffect();
        }
    }
    findOrOpenDefaultProjectForFile(fileName) {
        let project = this.projectService.getDefaultProjectForFile(fileName, false);
        if (!project) {
            this.openFileBySideEffect(fileName);
            project = this.projectService.getDefaultProjectForFile(fileName, true);
        }
        else {
            this.usedAsSideEffect('file', fileName);
        }
        return project;
    }
    openFileBySideEffect(fileName) {
        if (this.openedFilesByIde.has(fileName)) {
            throw new Error(`IdeProjectService.openFileBySideEffect: File ${fileName} is already opened by IDE`);
        }
        this.projectService.logger.info(`IdeProjectService.openFileBySideEffect: opening file by side effect - ${fileName}`);
        this.originalOpenClientFileWithNormalizedPath(fileName);
        this.openedAsSideEffect('file', fileName);
        this.cleanupOpenedFilesBySideEffect();
    }
    cleanupOpenedProjectsBySideEffect() {
        for (let projectFileName of getNamesToClose(this.openedProjectsBySideEffect, SIDE_EFFECT_PROJECTS_LIMIT)) {
            this.closeProjectBySideEffect(projectFileName);
            this.openedProjectsBySideEffect.delete(projectFileName);
        }
    }
    cleanupOpenedFilesBySideEffect() {
        for (let fileName of getNamesToClose(this.openedFilesBySideEffect, SIDE_EFFECT_FILES_LIMIT)) {
            this.closeFileBySideEffect(fileName);
            this.openedFilesBySideEffect.delete(fileName);
        }
    }
    closeProjectBySideEffect(projectFileName) {
        var _a, _b, _c;
        if (this.openedProjectsByIde.has(projectFileName)) {
            this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: not closing project - project is opened by ide - ${projectFileName}`);
            return;
        }
        let project = this.projectService.findProject(projectFileName);
        if (!project) {
            this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: project already closed - ${projectFileName}`);
            return;
        }
        for (let [openFile] of this.projectService.openFiles) {
            if (openFile && ((_c = (_b = (_a = this.projectService.getScriptInfo(openFile)) === null || _a === void 0 ? void 0 : _a.containingProjects) === null || _b === void 0 ? void 0 : _b.indexOf(project)) !== null && _c !== void 0 ? _c : -1) >= 0) {
                this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: not closing project ${projectFileName} - in use by ${openFile}`);
                return;
            }
        }
        this.projectService.logger.info(`IdeProjectService.closeProjectBySideEffect: closing project - ${projectFileName}`);
        this.originalCloseExternalProject(projectFileName);
    }
    closeFileBySideEffect(fileName) {
        if (this.openedFilesByIde.has(fileName)) {
            this.projectService.logger.info(`IdeProjectService.closeFileBySideEffect: not closing file - file is opened by ide - ${fileName}`);
            return;
        }
        this.projectService.logger.info(`IdeProjectService.closeFileBySideEffect: trying to close file by side effect - ${fileName}`);
        this.closeSafely(fileName);
    }
    openedAsSideEffect(entityKind, fileName) {
        let map = entityKind == 'project' ? this.openedProjectsBySideEffect : this.openedFilesBySideEffect;
        let now = new Date().getTime();
        // use delete to move the entry to the end of the map
        map.delete(fileName);
        map.set(fileName, now + randomizedExpirationTime());
    }
    usedAsSideEffect(entityKind, fileName) {
        let map = entityKind == 'project' ? this.openedProjectsBySideEffect : this.openedFilesBySideEffect;
        // use delete to move the entry to the end of the map
        if (map.delete(fileName)) {
            let now = new Date().getTime();
            map.set(fileName, now + randomizedExpirationTime());
        }
    }
    fileOpenedByIde(fileName) {
        this.openedFilesByIde.add(fileName);
    }
    fileClosedByIde(uncheckedFileName) {
        let normalizedFileName = this.ts.server.toNormalizedPath(uncheckedFileName);
        this.openedFilesByIde.delete(normalizedFileName);
    }
    projectClosedByIde(projectName) {
        let normalizedProjectName = this.ts.server.toNormalizedPath(projectName);
        this.openedProjectsByIde.delete(normalizedProjectName);
    }
    projectOpenedByIde(projectName) {
        let normalizedProjectName = this.ts.server.toNormalizedPath(projectName);
        this.openedProjectsByIde.add(normalizedProjectName);
    }
    internalTryOpenProject(projectFileName) {
        const tsConfigPath = this.ts.server.toNormalizedPath(projectFileName);
        const _projectService = this.projectService;
        if (_projectService.createAndLoadConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
            _projectService.externalProjectToConfiguredProjectMap.set(tsConfigPath, [tsConfigPath]);
            const project = _projectService.createAndLoadConfiguredProject(tsConfigPath, `Creating own configured project in external project: ${tsConfigPath}`);
            project.updateGraph();
            if (project.addExternalProjectReference) {
                project.addExternalProjectReference();
            }
            return true;
        }
        if (_projectService.createConfiguredProject && _projectService.externalProjectToConfiguredProjectMap) {
            // Like in ProjectService.openExternalProject()
            const project = _projectService.createConfiguredProject(tsConfigPath, `Creating configured project in external project: ${tsConfigPath}`);
            if (!this.projectService.getHostPreferences().lazyConfiguredProjectsFromExternalProject) {
                project.updateGraph();
            }
            _projectService.externalProjectToConfiguredProjectMap.set(tsConfigPath, new Set([project]));
            return true;
        }
        return false;
    }
}
exports.IdeProjectService = IdeProjectService;
function randomizedExpirationTime() {
    return SIDE_EFFECT_EXPIRATION_TIME_MIN + Math.floor(Math.random() * (SIDE_EFFECT_EXPIRATION_TIME_MAX - SIDE_EFFECT_EXPIRATION_TIME_MIN));
}
function getNamesToClose(map, limit) {
    if (map.size > limit) {
        let allEntries = [...map.entries()];
        // oldest entries are at the beginning of the map
        return allEntries
            // close a little bit more to allow for expiration code to trigger next time
            .slice(0, map.size - limit + 2)
            .map(entry => entry[0]);
    }
    else {
        const toClose = [];
        const now = new Date().getTime();
        for (let [projectFileName, expiration] of map) {
            if (expiration < now) {
                toClose.push(projectFileName);
            }
        }
        return toClose;
    }
}
//# sourceMappingURL=ide-project-service.js.map