/*
 * twin_squares.c --- twin squares package
 *
 * Copyright (c) 2001, 2002 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2002-12-31 01:23:20 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"twin_squares.h"
#include	"pcpj.h"

/*--------------------------------------------------------------------------*/

typedef struct
{
    square_t		first, second ;
    bool_t		orthogonal ;
    piece_type_t	type ;
    colour_t		colour ;
} one_twin_t ;

typedef struct
{
    one_twin_t		twins[ 32 ];
    unsigned int	nb_twins ;
} twin_squares_t ;

/*--------------------------------------------------------------------------*/

static twin_squares_t	Twin_Squares ;
square_t		No_Twin_Square = 0xFE ;

/*--------------------------------------------------------------------------*/

/* #define	DEBUG_TWINS */
#ifdef DEBUG_TWINS
static void	print_twin_squares();
#endif /* DEBUG_TWINS */

/*--------------------------------------------------------------------------*/

void
create_twin_squares(
    const piece_t*	pieces,
    unsigned int	nb_pieces )
{
    unsigned int	nb_twins = 0 ;
    unsigned int	i ;
    one_twin_t*		twins = Twin_Squares.twins ;

    for ( i = 0 ; i < nb_pieces ; i++ )
    {
	square_delta_t	deltaX, deltaY ;
	column_t	depX, 	arrX   ;
	row_t		depY, 	arrY   ;
	piece_type_t	type   = pieces[ i ].typePiece ;
	colour_t	colour = pieces[ i ].camp ;
	/* caseArrivee may not be up to date, so we have to use the expression
	 * below to obtain the destination square.
	 */
	square_t	destination = pieces[ i ].destination[
	    pieces[ i ].nbDestination - 1 ];

	if ( pieces[ i ].castling != NO_CASTLING )
	{
	    if ( type == ROI && pieces[ i ].nbDestination == 1
		 && ( ( colour == BLANC && NbCoupsBlancsRestants < 2 )
		      || ( colour == NOIR && NbCoupsNoirsRestants < 2 ) ) )
	    {
		twins[ nb_twins ].first      = pieces[ i ].caseInitiale ;
		twins[ nb_twins ].second     = destination ;
		twins[ nb_twins ].orthogonal = TRUE ;
		twins[ nb_twins ].type       = type ;
		twins[ nb_twins ].colour     = colour ;
		nb_twins++ ;
	    }
	    continue ;
	}

	if ( type == CAVALIER
	     || pieces[ i ].pieceCapturante != PIECE_PAS_CAPTUREE )
	{
	    continue ;
	}
	if ( ( type == PION && pieces[ i ].distance[ 0 ] > 0 )
	     || ( pieces[ i ].nbDestination == 1
		  && pieces[ i ].distance[ 0 ] == 1
		  && ( ( colour == BLANC && NbCoupsBlancsRestants == 0 )
		       || ( colour == NOIR && NbCoupsNoirsRestants == 0 ) ) ) )
	{
	    /* There are potentially twin squares. */
	}
	else
	{
	    continue ;
	}

	/* Eliminates promoted pawns except if they are on the promotion square
	 * and may never have moved.
	 */
	if ( type == PION
	     && pieces[ i ].casePromotion != CASE_PAS_DE_PROMOTION
	     && ( pieces[ i ].nbDestination > 1
		  || ( colour == BLANC && NbCoupsBlancsRestants > 1 )
		  || ( colour == NOIR  && NbCoupsNoirsRestants  > 1 ) ) )
	{
	    continue ;
	}


	depX = column( pieces[ i ].caseInitiale );
	depY = row( pieces[ i ].caseInitiale ) >> 4 ;
	arrX = column( destination );
	arrY = row( destination ) >> 4 ;
	deltaX = depX > arrX ? depX - arrX : arrX - depX ;
	deltaY = depY > arrY ? depY - arrY : arrY - depY ;

	if ( ( type != PION && type != ROI )
	     || deltaY == 1
	     || deltaX == deltaY
	     || ( type == PION
		  && deltaX == 0
		  && pieces[ i ].nbDestination == 1
		  && ( ( colour == BLANC && NbResteACapturerNoires < 2 )
		       || ( colour == NOIR
			    && NbResteACapturerBlanches < 2 ) ) ) )
	{
	    twins[ nb_twins ].first      = pieces[ i ].caseInitiale ;
	    twins[ nb_twins ].second     = destination ;
	    twins[ nb_twins ].orthogonal = ( deltaX != deltaY ) ? TRUE : FALSE ;
	    twins[ nb_twins ].type     	 = type ;
	    twins[ nb_twins ].colour     = colour ;
	    nb_twins++ ;
	}
    }

    Twin_Squares.nb_twins = nb_twins ;

#ifdef DEBUG_TWINS
    writeAssociation( "" );
    print_twin_squares();
#endif /* DEBUG_TWINS */
}

square_t
find_twin_square(
    square_t		square,
    bool_t		orthogonal,
    piece_type_t	type,
    colour_t		colour )
{
    unsigned int	i ;
    one_twin_t*		twins = Twin_Squares.twins ;
    for ( i = 0 ; i < Twin_Squares.nb_twins ; i++ )
    {
	if ( twins[ i ].orthogonal == orthogonal && twins[ i ].type != type )
	{
	    if ( square == twins[ i ].first )
	    {
		return twins[ i ].second ;
	    }
	    else if ( square == twins[ i ].second )
	    {
		return twins[ i ].first;
	    }
	}
    }

    return No_Twin_Square ;
}

#ifdef DEBUG_TWINS

static void
print_twin_squares()
{
    int		i ;
    for ( i = 0 ; i < Twin_Squares.nb_twins ; i++ )
    {
	fprintf( MainFD,
		 "Twins %d : 0x%02x - 0x%02x (%s)\n",
		 i + 1,
		 Twin_Squares.twins[ i ].first,
		 Twin_Squares.twins[ i ].second,
		 Twin_Squares.twins[ i ].orthogonal ? "ortho" : "diag" );
    }
}

#endif /* DEBUG_TWINS */
