"""
Custom user model for use in tests.

"""

# SPDX-License-Identifier: BSD-3-Clause

from django.contrib.auth.base_user import AbstractBaseUser
from django.contrib.auth.models import UserManager
from django.contrib.auth.validators import UnicodeUsernameValidator
from django.core.mail import send_mail
from django.db import models
from django.utils import timezone
from django.utils.translation import gettext_lazy as _


class CustomUser(AbstractBaseUser):
    """
    A model for use in testing django-registration's support for custom user
    models. Do not use this in your own projects.

    """

    EMAIL_FIELD = "email"
    REQUIRED_FIELDS = ["email"]
    USERNAME_FIELD = "username"

    age = models.IntegerField(blank=True, null=True)
    username = models.CharField(
        _("username"),
        max_length=150,
        unique=True,
        help_text=_("Required. 150 characters or fewer."),
        validators=[UnicodeUsernameValidator()],
        error_messages={"unique": _("A user with that username already exists.")},
    )
    first_name = models.CharField(_("first name"), max_length=30, blank=True)
    last_name = models.CharField(_("last name"), max_length=150, blank=True)
    email = models.EmailField(_("email address"), blank=True)
    is_staff = models.BooleanField(
        _("staff status"),
        default=False,
        help_text=_("Whether the user can log into the admin."),
    )
    is_active = models.BooleanField(
        _("active"),
        default=True,
        help_text=_("Whether this user should be treated as active."),
    )
    date_joined = models.DateTimeField(_("date joined"), default=timezone.now)

    objects = UserManager()

    def email_user(self, subject, message, from_email=None, **kwargs):
        """
        Send an email to this user.

        """
        send_mail(subject, message, from_email, [self.email], **kwargs)
