/*
 * ipt.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>

#include "util.h"
#include "ipt.h"
#include "curs_util.h"

#include "libiptc/libiptc.h"

extern void quit(const char *);

extern struct iptc_s *  ipt_sel;
extern  struct ipt_s *   iptf_sel;

struct iptc_s *  ipt_sel = NULL;
struct ipt_s *   iptf_sel = NULL;

enum {
    mi_nat,
    mi_mangle,
    mi_filter
} m_i_table;

static iptc_handle_t handle;

void
calc_ipt(void)
{
    const char *this;
    struct iptc_s *chain;
    struct ipt_s *filter;

    handle = iptc_init("filter");

	if (!handle)
        quit("Can't initialize iptables table");

    for (this = iptc_first_chain(&handle);
         this;
         this = iptc_next_chain(&handle)) {

        const struct ipt_entry *i;

        chain = ipt_chains;
        while (chain) {
            if (!strcmp(chain->name, this))
                goto found;
            chain = chain->next;
        }

        if ( !(chain = (struct iptc_s *) calloc(1, sizeof(struct iptc_s))) )
            quit("Out of memory!\n");
        strncpy(chain->name, this, sizeof(chain->name));

        if (ipt_chains)
            ipt_chains->prev = chain;

        chain->next = ipt_chains;
        ipt_chains = chain;

found:

        i = iptc_first_rule(this, &handle);
        while (i) {

            const char *tn = iptc_get_target(i, &handle);

            if (!tn)
                continue;

            filter = chain->filters;
            while (filter) {

                if (!memcmp(&i->ip, &filter->e.ip, sizeof(struct ipt_ip)) &&
                    !strncmp(tn, filter->name, 12))
                        goto found_;

                filter = filter->next;
            }

            if ( !(filter = (struct ipt_s *) calloc(1, sizeof(struct ipt_s))) )
                quit("Out of memory!\n");
            strncpy(filter->name, tn, sizeof(filter->name));

            memcpy(&filter->e.ip, &i->ip, sizeof(struct ipt_ip));

            if (chain->filters)
                chain->filters->prev = filter;

            filter->next = chain->filters;
            chain->filters = filter;

found_:

            filter->e.counters.bcnt = i->counters.bcnt;
            filter->e.counters.pcnt = i->counters.pcnt;

            if (!filter->bytes_old)
                filter->bytes_old = i->counters.bcnt;

            if (!filter->bt)
                filter->bt = time(0);

            if ( (time(0) - filter->bt) >= 3) {

                filter->r_bytes += (i->counters.bcnt - filter->bytes_old);
                filter->r_bytes /= ( (time(0) - filter->bt) + 2);

                filter->bytes_old = i->counters.bcnt;
                filter->bt = time(0);
            }

            i = iptc_next_rule(i, &handle);
        }
    }
}

void
print_ipt(void)
{
    struct iptc_s *chain = ipt_chains;
    struct ipt_s *f;
    int i=3;
    struct protoent *pe;
    char addrbuf[INET6_ADDRSTRLEN], addrbuf2[INET6_ADDRSTRLEN], *u;
    float r;

    if (LINES < 7) {
        print_full_line("Need at least 7 lines");
        return;
    }

#define IT(mode, str) {           \
    if (m_i_table == mode)        \
        attrset(A_REVERSE);       \
    printw(str);                  \
    if (m_i_table == mode)        \
        attroff(A_REVERSE);       \
    printw(" ");                  \
}
    
    IT(mi_nat, "(n)at");
    IT(mi_mangle, "(m)angle");
    IT(mi_filter, "(f)ilter");

#undef IT

    NEXT_ROW;
    hline(ACS_HLINE, COLS-1);

    while (chain) {

        if (row >= LINES-2)
            break;

        if (!ipt_sel)
            ipt_sel = chain;

        NEXT_ROW;

        print_full_line("");

        move(row,0);

        if (ipt_sel == chain)
            attrset(A_REVERSE);

        printw("%-12s", chain->name);

        if (ipt_sel == chain)
            attroff(A_REVERSE);

        chain = chain->next;
    }

    if (ipt_sel) {
        f = ipt_sel->filters;
        while (f) {

            if (i >= LINES-4)
                break;
            
            move(i++, 14);

            if (!iptf_sel)
                iptf_sel = f;

            if (f == iptf_sel)
                attrset(A_REVERSE);

            pe = getprotobynumber(f->e.ip.proto);

            if (f->e.ip.smsk.s_addr == 0L)
                strcpy(addrbuf, "*");
            else {
                memset(addrbuf, 0, sizeof(addrbuf));
                resolvebyaddr(addrbuf, sizeof(addrbuf), (char *) &f->e.ip.src,
                    sizeof(f->e.ip.src), AF_INET);
            }

            if (f->e.ip.smsk.s_addr == 0L)
                strcpy(addrbuf2, "*");
            else {
                memset(addrbuf2, 0, sizeof(addrbuf2));
                resolvebyaddr(addrbuf2, sizeof(addrbuf2), (char *) &f->e.ip.dst,
                    sizeof(f->e.ip.dst), AF_INET);
            }

            r = sumup(f->r_bytes, &u);

            printw("%-12s %c%5s %c%15s %c%15s %7.2f %s",
                f->name,
                f->e.ip.invflags & IPT_INV_PROTO ? '!' : ' ',
                pe->p_name,
                f->e.ip.invflags & IPT_INV_SRCIP ? '!' : ' ',
                addrbuf, 
                f->e.ip.invflags & IPT_INV_DSTIP ? '!' : ' ',
                addrbuf2,
                r, u);

            if (f == iptf_sel)
                attroff(A_REVERSE);

            f = f->next;
        }
    }

    for (row=i; row < LINES-4; row++)
        mvhline(row, 13, ' ', COLS-13);
    
    if (iptf_sel) {
        f = iptf_sel;

        move(LINES-3, 13);

        printw("Packets: %12llu  Bytes: %12llu", f->e.counters.pcnt, f->e.counters.bcnt);

    }

    mvvline(3, 12, ACS_VLINE, LINES-3);

    mvhline(LINES-4, 13, ACS_HLINE, COLS-13);
}

int
ipt_handle_input(int ch)
{
    switch (ch) {
        case KEY_UP:
            if (ipt_sel) {
                if (ipt_sel == ipt_chains)
                    for ( ;ipt_sel->next; ipt_sel = ipt_sel->next);
                else
                    ipt_sel = ipt_sel->prev;
                iptf_sel = NULL;
            }
            break;

        case KEY_DOWN:
            if (ipt_sel) {
                ipt_sel = ipt_sel->next;
                iptf_sel = NULL;
            }
            break;

        case KEY_LEFT:
            if (iptf_sel && ipt_sel) {
                if (iptf_sel == ipt_sel->filters)
                    for ( ; iptf_sel->next; iptf_sel = iptf_sel->next);
                else
                    iptf_sel = iptf_sel->prev;
            }
            break;

        case KEY_RIGHT:
            if (iptf_sel)
                iptf_sel = iptf_sel->next;
    }

    print_top("IPTables statistics (this sucks)");
    print_ipt();

    return 0;
}
