/* File:      xsb_hook.P -- mechanism for adding hooks to xsb
** Author(s): kifer
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1999
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: xsb_hook.P,v 1.7 2004/04/09 00:02:16 dwarren Exp $
** 
*/


:- compiler_options([xpp_on]).
#include "psc_defs.h"

%% Hooks
:- thread_shared 
	xsb_exit_hook(_),
	xsb_assert_hook(_),
	xsb_intercept_assert_hook(_),
	xsb_retract_hook(_),
	xsb_intercept_retractall_hook(_),
	xsb_before_prompt_hook(_),
	xsb_undefined_predicate_hook(_).

:- mode_on_success(xsb_hook_class_active(+)).
:- mode_on_success(xsb_exit_hook(?)).
:- mode_on_success(xsb_assert_hook(?)).
:- mode_on_success(xsb_intercept_assert_hook(?)).
:- mode_on_success(xsb_retract_hook(?)).
:- mode_on_success(xsb_intercept_retractall_hook(?)).
:- mode_on_success(xsb_before_prompt_hook(?)).
:- mode_on_success(xsb_undefined_predicate_hook(?)).

:- dynamic
	xsb_hook_class_active(_),
	xsb_exit_hook(_),
	xsb_assert_hook(_),
	xsb_intercept_assert_hook(_),
	xsb_retract_hook(_),
	xsb_intercept_retractall_hook(_),
	xsb_before_prompt_hook(_),
	xsb_undefined_predicate_hook(_).
:- dynamic rename_dyn_pred(_,_).

xsb_hook_class(xsb_exit_hook(_),exit).
xsb_hook_class(xsb_assert_hook(_),assert).
xsb_hook_class(xsb_intercept_assert_hook(_),assert).
xsb_hook_class(xsb_retract_hook(_),retract).
xsb_hook_class(xsb_intercept_retractall_hook(_),retract).
xsb_hook_class(xsb_before_prompt_hook(_),prompt).
xsb_hook_class(xsb_undefined_predicate_hook(_),undefined_predicate).

%% A hook is of the form hookname(HookPredicate)
%% hookname(_) just lists the callable hooks of the given kind.
%% add_xsb_hook adds hooks to hookname(_) and remove_xsb_hook(_) removes them.

%% call_xsb_hook(hookname(_)) goes over all the hooks stored in hookname(_) 
%% and calls them. This is supposed to call only propositional hook predicates.
%%
%% call_xsb_hook(hookhame(_),ArgList) calls the hooks by substituting ArgList 
%% for the first argument of the hook. The hook predicates that populate 
%% such a hook type must all be 1-ary predicates.

%% Hook handling

:- mode add_xsb_hook(?).
add_xsb_hook(Hook) :-
	call(Hook), !.  % hook is already there
add_xsb_hook(Hook) :-
	(   callable(Hook),
	    Hook =.. [HookName, Call]
	->  ( term_psc(Call,Psc),
		psc_type(Psc,Type),
		Type > T_ORDI
	     ->	(xsb_hook_class(Hook,HookType)
		 -> (xsb_hook_class_active(HookType)
		     ->	true
		     ;	assert(xsb_hook_class_active(HookType),1,1)
		    )
		 ;  abort(['ADD_XSB_HOOK: undefined hook class for ',HookName])
		),
		assert(Hook,1,1)
	     ;	abort(['ADD_XSB_HOOK: undefined hook ',Call,' in ',HookName])
	    )
	;   abort(['ADD_XSB_HOOK: ',Hook,' must be a callable literal'])
	).

:- mode remove_xsb_hook(?).
remove_xsb_hook(Hook) :-
	(   callable(Hook)
	 -> retractall(Hook),
	    (xsb_hook_class(Hook,HookType)
	     ->	(xsb_hook_class(NHook,HookType),
		 call_c(NHook)
		 -> true
		 ;  retractall(xsb_hook_class_active(HookType))
		)
	     ;	abort(['REMOVE_XSB_HOOK: Unrecognized hook: ',Hook])
	    )
	;   abort(['REMOVE_XSB_HOOK: ',Hook,' must be a callable literal'])
	).

:- mode set_rename_dyn_pred(?,?).
set_rename_dyn_pred(From,To) :-
    retractall(rename_dyn_pred(From,_)),
    assert(rename_dyn_pred(From,To),1,1).

:- mode reset_rename_dyn_pred(?).
reset_rename_dyn_pred(From) :-
    retractall(rename_dyn_pred(From,_)).

%% call_xsb_hook(hookname(X)) calls all "hookname" hooks one by one.
%% Hook must be of the form hookName(_)
call_one_hook(Hook) :-
	call(Hook),
	Hook =.. [HookName, HookPredicate],
	(   atom(HookName)
	->  (   callable(HookPredicate)
	    ->  (   term_psc(HookPredicate,Psc),
		    psc_type(Psc,Type),
		    Type > T_ORDI
		->	call_c(HookPredicate)
		;       abort(['CALL_XSB_HOOK: ',
			       HookPredicate,' undefined hook'])
		)
	    ;	abort(['CALL_XSB_HOOK: ',HookPredicate,
		       ' must be a callable literal'])
	    )
	;   abort(['CALL_XSB_HOOK: ',HookName, ' is not the name of a hook'])
	).


%% executes all hooks of a given type
:- mode call_xsb_hook(?,+).
call_xsb_hook(Hook,HookType) :-
	(HookType == all
	 ->	call_xsb_hook_all(Hook)
	 ; HookType = until_true
	 ->	once(call_one_hook(Hook))
	).

call_xsb_hook_all(Hook) :- call_one_hook(Hook), fail.
call_xsb_hook_all(_Hook).


%% Execute hooks with arguments
%% Assume that any hook accepts exactly one argument 
%% (pass a list, if more is needed)
call_one_hook(Hook,ArgList) :- 
	call(Hook),
	Hook =.. [HookName, HookPredicate],
	(   atom(HookName)
	->  (   callable(HookPredicate)
	    ->  substituteArgs(HookName,HookPredicate,ArgList,HookedUpCall),
		call_c(HookedUpCall)
	    ;	abort(['CALL_XSB_HOOK: ',HookPredicate,
		       ' must be a callable literal'])
	    )
	;   abort(['CALL_XSB_HOOK: ',HookName, ' is not the name of a hook'])
	).

:- mode call_xsb_hook(?,?,+).
call_xsb_hook(Hook,ArgList,HookType) :-
    (HookType == all
	 ->	call_xsb_hook_all(Hook,ArgList)
	 ; HookType == until_true
	 ->	once(call_one_hook(Hook,ArgList))
	).

call_xsb_hook_all(Hook,ArgList) :- call_one_hook(Hook,ArgList), fail.
call_xsb_hook_all(_Hook,_ArgList).

%% Substitute ArgList for the first arg of the hook predicate to be called.
%% Do all the necessary checks for arity and such
substituteArgs(HookName,HookPredicate,ArgList,HookedUpCall) :-
	term_psc(HookPredicate,Psc),
	(   psc_type(Psc,Type),
	    Type =< T_ORDI
	->  abort(['CALL_XSB_HOOK: undefined hook, ',HookPredicate])
	;   psc_arity(Psc,Arity), Arity =\= 1
	->  abort(['Hook ', HookedUpCall, ' of ', HookName,
		   ' must have arity 1'])
	;   term_new(Psc,HookedUpCall),
	    arg(1,HookedUpCall,ArgList)
	).




