/* SPDX-FileCopyrightText: 2024 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import React, {useEffect} from 'react';
import {useNavigate} from 'react-router';
import Gmp from 'gmp/gmp';
import Filter from 'gmp/models/filter';
import Permission from 'gmp/models/permission';
import {TARGET_CREDENTIAL_NAMES} from 'gmp/models/target';
import Task, {USAGE_TYPE} from 'gmp/models/task';
import {isDefined} from 'gmp/utils/identity';
import Badge from 'web/components/badge/Badge';
import {
  AlterableIcon,
  NoteIcon,
  OverrideIcon,
  ReportIcon,
  ResultIcon,
  TaskIcon,
} from 'web/components/icon';
import ExportIcon from 'web/components/icon/ExportIcon';
import ListIcon from 'web/components/icon/ListIcon';
import ManualIcon from 'web/components/icon/ManualIcon';
import Divider from 'web/components/layout/Divider';
import IconDivider from 'web/components/layout/IconDivider';
import Layout from 'web/components/layout/Layout';
import PageTitle from 'web/components/layout/PageTitle';
import DetailsLink from 'web/components/link/DetailsLink';
import Link from 'web/components/link/Link';
import {
  NO_RELOAD,
  USE_DEFAULT_RELOAD_INTERVAL,
  USE_DEFAULT_RELOAD_INTERVAL_ACTIVE,
} from 'web/components/loading/Reload';
import Tab from 'web/components/tab/Tab';
import TabLayout from 'web/components/tab/TabLayout';
import TabList from 'web/components/tab/TabList';
import TabPanel from 'web/components/tab/TabPanel';
import TabPanels from 'web/components/tab/TabPanels';
import Tabs from 'web/components/tab/Tabs';
import TabsContainer from 'web/components/tab/TabsContainer';
import InfoTable from 'web/components/table/InfoTable';
import TableBody from 'web/components/table/TableBody';
import TableCol from 'web/components/table/TableCol';
import TableData from 'web/components/table/TableData';
import TableRow from 'web/components/table/TableRow';
import EntitiesTab from 'web/entity/EntitiesTab';
import EntityPage from 'web/entity/EntityPage';
import EntityPermissions, {
  EntityPermissionsProps,
} from 'web/entity/EntityPermissions';
import {OnDownloadedFunc} from 'web/entity/hooks/useEntityDownload';
import CloneIcon from 'web/entity/icon/CloneIcon';
import EditIcon from 'web/entity/icon/EditIcon';
import TrashIcon from 'web/entity/icon/TrashIcon';
import {goToDetails, goToList} from 'web/entity/navigation';
import EntityTags from 'web/entity/Tags';
import withEntityContainer, {
  permissionsResourceFilter,
} from 'web/entity/withEntityContainer';
import useTranslation from 'web/hooks/useTranslation';
import NewIconMenu from 'web/pages/tasks/icons/NewIconMenu';
import TaskIconWithSync from 'web/pages/tasks/icons/TaskIconWithSync';
import TaskImportReportIcon from 'web/pages/tasks/icons/TaskImportReportIcon';
import TaskScheduleIcon from 'web/pages/tasks/icons/TaskScheduleIcon';
import TaskStopIcon from 'web/pages/tasks/icons/TaskStopIcon';
import TaskComponent from 'web/pages/tasks/TaskComponentComponent';
import TaskDetails from 'web/pages/tasks/TaskDetails';
import TaskStatus from 'web/pages/tasks/TaskStatus';
import {
  selector as notesSelector,
  loadEntities as loadNotes,
} from 'web/store/entities/notes';
import {
  selector as overridesSelector,
  loadEntities as loadOverrides,
} from 'web/store/entities/overrides';
import {
  selector as permissionsSelector,
  loadEntities as loadPermissions,
} from 'web/store/entities/permissions';
import {
  selector as taskSelector,
  loadEntity as loadTask,
} from 'web/store/entities/tasks';
import {renderYesNo} from 'web/utils/Render';
import {formattedUserSettingShortDate} from 'web/utils/userSettingTimeDateFormatters';
import withComponentDefaults from 'web/utils/withComponentDefaults';

interface ToolBarIconsProps {
  entity: Task;
  links?: boolean;
  notes?: Array<unknown>;
  overrides?: Array<unknown>;
  onContainerTaskCreateClick?: () => void | Promise<void>;
  onReportImportClick?: (value: Task) => void | Promise<void>;
  onTaskCloneClick?: (value: Task) => void | Promise<void>;
  onTaskCreateClick?: () => void | Promise<void>;
  onTaskDeleteClick?: (value: Task) => void | Promise<void>;
  onTaskDownloadClick?: () => void | Promise<void>;
  onTaskEditClick?: (value: Task) => void | Promise<void>;
  onTaskResumeClick?: (value: Task) => void | Promise<void>;
  onTaskStartClick?: (value: Task) => void | Promise<void>;
  onTaskStopClick?: (value: Task) => void | Promise<void>;
}

interface DetailsProps {
  entity: Task;
  links?: boolean;
}

interface TaskDetailsPageProps {
  entity: Task;
  permissions: Permission[];
  onChanged?: () => void;
  onDownloaded?: OnDownloadedFunc;
  onError?: (error: unknown) => void;
}

export const ToolBarIcons = ({
  entity,
  links,
  notes = [],
  overrides = [],
  onTaskDeleteClick,
  onTaskCloneClick,
  onTaskDownloadClick,
  onTaskEditClick,
  onReportImportClick,
  onTaskCreateClick,
  onContainerTaskCreateClick,
  onTaskStartClick,
  onTaskStopClick,
  onTaskResumeClick,
}: ToolBarIconsProps) => {
  const [_] = useTranslation();
  const {
    current_report: currentReport,
    last_report: lastReport,
    report_count: reportCount,
    result_count: resultCount,
  } = entity;
  return (
    <Divider margin="10px">
      <IconDivider align={['start', 'start']}>
        <ManualIcon
          anchor="managing-tasks"
          page="scanning"
          title={_('Help: Tasks')}
        />
        <ListIcon page="tasks" title={_('Task List')} />
        {entity.isAlterable() && !entity.isNew() && (
          <AlterableIcon
            title={_(
              'This is an Alterable Task. Reports may not relate to ' +
                'current Scan Config or Target!',
            )}
          />
        )}
      </IconDivider>

      <IconDivider>
        <NewIconMenu
          onNewClick={onTaskCreateClick}
          onNewContainerClick={onContainerTaskCreateClick}
        />
        <CloneIcon entity={entity} name="task" onClick={onTaskCloneClick} />
        <EditIcon entity={entity} name="task" onClick={onTaskEditClick} />
        <TrashIcon entity={entity} name="task" onClick={onTaskDeleteClick} />
        <ExportIcon
          title={_('Export Task as XML')}
          value={entity}
          onClick={onTaskDownloadClick}
        />
      </IconDivider>

      <IconDivider>
        {isDefined(entity.schedule) && (
          <TaskScheduleIcon links={links} schedule={entity.schedule} />
        )}
        <TaskIconWithSync
          task={entity}
          type="start"
          onClick={onTaskStartClick}
        />

        <TaskImportReportIcon task={entity} onClick={onReportImportClick} />

        <TaskStopIcon task={entity} onClick={onTaskStopClick} />

        {!entity.isContainer() && (
          <TaskIconWithSync
            task={entity}
            type="resume"
            onClick={onTaskResumeClick}
          />
        )}
      </IconDivider>

      <Divider margin="10px">
        <IconDivider>
          {isDefined(currentReport) && (
            <DetailsLink
              id={currentReport.id}
              title={_('Current Report for Task {{- name}} from {{- date}}', {
                name: entity.name as string,
                date: formattedUserSettingShortDate(
                  currentReport.scan_start,
                ) as string,
              })}
              type="report"
            >
              <ReportIcon />
            </DetailsLink>
          )}

          {!isDefined(currentReport) && isDefined(lastReport) && (
            <DetailsLink
              id={lastReport.id}
              title={_('Last Report for Task {{- name}} from {{- date}}', {
                name: entity.name as string,
                date: formattedUserSettingShortDate(
                  lastReport.scan_start,
                ) as string,
              })}
              type="report"
            >
              <ReportIcon />
            </DetailsLink>
          )}

          <Link
            filter={'task_id=' + entity.id}
            title={_('Total Reports for Task {{- name}}', {
              name: entity.name as string,
            })}
            to="reports"
          >
            <Badge content={reportCount?.total ?? 0}>
              <ReportIcon />
            </Badge>
          </Link>
        </IconDivider>

        <Link
          filter={'task_id=' + entity.id}
          title={_('Results for Task {{- name}}', {
            name: entity.name as string,
          })}
          to="results"
        >
          <Badge content={resultCount}>
            <ResultIcon />
          </Badge>
        </Link>

        <IconDivider>
          <Link
            filter={'task_id=' + entity.id}
            title={_('Notes for Task {{- name}}', {
              name: entity.name as string,
            })}
            to="notes"
          >
            <Badge content={notes.length}>
              <NoteIcon />
            </Badge>
          </Link>

          <Link
            filter={'task_id=' + entity.id}
            title={_('Overrides for Task {{- name}}', {
              name: entity.name as string,
            })}
            to="overrides"
          >
            <Badge content={overrides.length}>
              <OverrideIcon />
            </Badge>
          </Link>
        </IconDivider>
      </Divider>
    </Divider>
  );
};

const Details = ({entity, links}: DetailsProps) => {
  const [_] = useTranslation();
  return (
    <Layout flex="column">
      <InfoTable>
        <colgroup>
          <TableCol width="10%" />
          <TableCol width="90%" />
        </colgroup>
        <TableBody>
          <TableRow>
            <TableData>{_('Name')}</TableData>
            <TableData>{entity.name}</TableData>
          </TableRow>

          <TableRow>
            <TableData>{_('Comment')}</TableData>
            <TableData>{entity.comment}</TableData>
          </TableRow>

          <TableRow>
            <TableData>{_('Alterable')}</TableData>
            <TableData>{renderYesNo(entity.isAlterable())}</TableData>
          </TableRow>

          <TableRow>
            <TableData>{_('Status')}</TableData>
            <TableData>
              <TaskStatus task={entity} />
            </TableData>
          </TableRow>
        </TableBody>
      </InfoTable>

      <TaskDetails entity={entity} links={links} />
    </Layout>
  );
};

const TaskDetailsPage = ({
  entity,
  permissions = [],
  onChanged,
  onDownloaded,
  onError,
  ...props
}: TaskDetailsPageProps) => {
  const navigate = useNavigate();
  const [_] = useTranslation();

  useEffect(() => {
    if (isDefined(entity) && entity.usageType !== USAGE_TYPE.scan) {
      void navigate('/audit/' + entity.id, {replace: true});
    }
  }, [entity, navigate]);

  return (
    <TaskComponent
      onCloneError={onError}
      onCloned={goToDetails('task', navigate)}
      onContainerCreated={goToDetails('task', navigate)}
      onContainerSaved={onChanged}
      onCreated={goToDetails('task', navigate)}
      onDeleteError={onError}
      onDeleted={goToList('tasks', navigate)}
      onDownloadError={onError}
      onDownloaded={onDownloaded}
      onReportImported={onChanged}
      onResumeError={onError}
      onResumed={onChanged}
      onSaved={onChanged}
      onStartError={onError}
      onStarted={onChanged}
      onStopError={onError}
      onStopped={onChanged}
    >
      {({
        clone,
        create,
        createContainer,
        delete: deleteFunc,
        download,
        edit,
        start,
        stop,
        resume,
        reportImport,
      }) => (
        <EntityPage
          {...props}
          entity={entity}
          sectionIcon={<TaskIcon size="large" />}
          title={_('Task')}
          // @ts-expect-error
          toolBarIcons={ToolBarIcons}
          onChanged={onChanged}
          onContainerTaskCreateClick={createContainer}
          onError={onError}
          onReportImportClick={reportImport}
          onTaskCloneClick={clone}
          onTaskCreateClick={create}
          onTaskDeleteClick={deleteFunc}
          onTaskDownloadClick={download}
          onTaskEditClick={edit}
          onTaskResumeClick={resume}
          onTaskStartClick={start}
          onTaskStopClick={stop}
        >
          {() => {
            return (
              <React.Fragment>
                <PageTitle
                  title={_('Task: {{name}}', {name: entity.name as string})}
                />
                <TabsContainer flex="column" grow="1">
                  <TabLayout align={['start', 'end']} grow="1">
                    <TabList align={['start', 'stretch']}>
                      <Tab>{_('Information')}</Tab>
                      <EntitiesTab entities={entity.userTags}>
                        {_('User Tags')}
                      </EntitiesTab>
                      <EntitiesTab entities={permissions}>
                        {_('Permissions')}
                      </EntitiesTab>
                    </TabList>
                  </TabLayout>

                  <Tabs>
                    <TabPanels>
                      <TabPanel>
                        <Details entity={entity} />
                      </TabPanel>
                      <TabPanel>
                        <EntityTags
                          entity={entity}
                          onChanged={onChanged}
                          onError={onError}
                        />
                      </TabPanel>
                      <TabPanel>
                        <TaskPermissions
                          entity={entity}
                          permissions={permissions}
                          onChanged={onChanged}
                          onDownloaded={onDownloaded}
                          onError={onError}
                        />
                      </TabPanel>
                    </TabPanels>
                  </Tabs>
                </TabsContainer>
              </React.Fragment>
            );
          }}
        </EntityPage>
      )}
    </TaskComponent>
  );
};

export const TaskPermissions = withComponentDefaults<
  EntityPermissionsProps<Task>
>({
  relatedResourcesLoaders: [
    ({entity}: {entity: Task}) =>
      isDefined(entity.alerts)
        ? Promise.resolve([...entity.alerts])
        : Promise.resolve([]),
    ({entity}: {entity: Task}) => {
      const resources = [entity.config, entity.scanner, entity.schedule].filter(
        isDefined,
      );
      return Promise.resolve(resources);
    },
    ({entity, gmp}: {entity: Task; gmp: Gmp}) => {
      if (isDefined(entity.target)) {
        // @ts-expect-error
        return gmp.target.get(entity.target).then(response => {
          const target = response.data;
          const resources = [target];

          for (const name of ['port_list', ...TARGET_CREDENTIAL_NAMES]) {
            const cred = target[name];
            if (isDefined(cred)) {
              resources.push(cred);
            }
          }
          return resources;
        });
      }
      return Promise.resolve([]);
    },
  ],
})(EntityPermissions<Task>);

const taskIdFilter = (id: string) => Filter.fromString('task_id=' + id).all();

const mapStateToProps = (rootState: unknown, {id}: {id: string}) => {
  const permSel = permissionsSelector(rootState);
  const notesSel = notesSelector(rootState);
  const overridesSel = overridesSelector(rootState);
  return {
    notes: notesSel.getEntities(taskIdFilter(id)),
    overrides: overridesSel.getEntities(taskIdFilter(id)),
    permissions: permSel.getEntities(permissionsResourceFilter(id)),
  };
};

const load = (gmp: Gmp) => {
  const loadTaskFunc = loadTask(gmp);
  const loadPermissionsFunc = loadPermissions(gmp);
  const loadNotesFunc = loadNotes(gmp);
  const loadOverridesFunc = loadOverrides(gmp);
  return (id: string) => dispatch =>
    Promise.all([
      dispatch(loadTaskFunc(id)),
      dispatch(loadPermissionsFunc(permissionsResourceFilter(id))),
      dispatch(loadNotesFunc(taskIdFilter(id))),
      dispatch(loadOverridesFunc(taskIdFilter(id))),
    ]);
};

export const reloadInterval = ({entity}: {entity: Task}) => {
  if (!isDefined(entity) || entity.isContainer()) {
    return NO_RELOAD;
  }
  return entity.isActive()
    ? USE_DEFAULT_RELOAD_INTERVAL_ACTIVE
    : USE_DEFAULT_RELOAD_INTERVAL;
};

export default withEntityContainer('task', {
  load,
  entitySelector: taskSelector,
  mapStateToProps,
  // @ts-expect-error
  reloadInterval,
})(TaskDetailsPage);
