/** @defgroup seed */
// @{

#ifndef __SEED_H__
#define __SEED_H__

//STL
#include <vector>
#include <queue>
#include <stack>
#include <list>
#include <algorithm>
//STD
#include <assert.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <string>
//STR
using namespace std;
#include "macro.h"

   
/**
 * @class seed
 * 
 * @brief represents a subset seed (can also handle a list of positions where the seed is meant to apply)
 */

class seed {
 public:

  /** @name  seed(int nbcyclespos, int maxcyclepos) 
   *  @brief build a seed of given span but full of jokers
   *  @param nbcyclepos is the number of positions in a cycle where the seed can be applied (0 = all positions),
   *  @param maxcyclepos is the size of the cycle.
   */
 
 seed(int nbcyclepos = 0, int maxcyclepos = 1):_seedNbCyclePos(nbcyclepos),_seedMaxCyclePos(maxcyclepos) {

    if (_seedNbCyclePos > _seedMaxCyclePos) {
      ERROR("Seed Cycle Pos","cannot set " << _seedNbCyclePos << " positions in a cycle of size " << _seedMaxCyclePos);
    }

    _span          = gv_minspan;
    _seedMotif_int = new int[_span];
    _swap_from = 0;
    _swap_to   = 0;
    _swap_pos_index = -1;
    _swap_pos_old_value = 0;

    for(int i=0; i<_span; i++)
      _seedMotif_int[i] = 0;

    if (gv_signature_flag) {
      checksignature();
      setsignature();
    }
    
    if (_seedNbCyclePos > 0) {
      _seedCyclePos_int = new int[_seedNbCyclePos];
      for(int i=0; i<_seedNbCyclePos; i++)
	_seedCyclePos_int[i] = i;
    } else {
      _seedCyclePos_int = NULL;
    }
        
    if (gv_nbruns > 0){
      random();
    } else {      
      next();
    }

  }


  /** @name  seed(string  str_desc) 
   *  @brief build a seed according to a descriptor 
   *  @param desc is a string on ['0'-'9''A'-'Z''a'-'z']* or [(#[0-9]*)*]
   *         as "90A" or "#9#0#10" in equivalent form.
   *         if the Bsymbols option is selected, then the seed must respect 
   *         the new sysmbols definition (eg for "-spaced", "##-##-###" ...)
   */
  
  //@{
  
  seed(string str_desc) {
    /* seed span */
    {
      int pos = 0;
      _span = 0;
      while(pos < (int)str_desc.length()) {
	convert(str_desc, &pos);
	_span++;
      }
      if (_span == 0) {
	ERROR("Single Seed Parsing","empty string provided");
      }
    }

    _seedMotif_int = new int[_span];
    _swap_from = 0;
    _swap_to   = 0;
    _swap_pos_index     = -1;
    _swap_pos_old_value = 0;
    _seedCyclePos_int   = NULL;
    _seedNbCyclePos     = 0;
    _seedMaxCyclePos    = 1;
    /* seed shape */
    {
      int pos = 0;
      for(int i = 0; i < _span ; i++)
	_seedMotif_int[i] = convert(str_desc, &pos);
    }
  }

  //@}

 /** @name  ~seed() 
   *  @brief delete a seed and associed structures (cyclepos and seed motif)
   */  
  ~seed(){
    if  (_seedMotif_int){
      delete _seedMotif_int;
      _seedMotif_int = NULL;
    }
    if (_seedCyclePos_int) {
      delete _seedCyclePos_int;
      _seedCyclePos_int = NULL;
    }
  };


  /** @name Get functions
   *  @brief these methods are provided to get information from current seed 
   */
  
  //@{

  /** @brief gives the integer representation of the seed
   *  @return an integer table of lenght span, each element must 
   *          be in the range 0 .. |B|-1
   */
  
  int  *   table()   {return _seedMotif_int;};


  /** @brief gives the span of the seed
   *  @return the span of the seed
   */
  int      span()    {return _span;};


  /** @brief gives the integer positions where the seed should match
   *  @return an integer table of lenght nbpos(), each element must 
   *          be in the range 0..(maxpos()-1)
   */

  int *    pos()    {return _seedCyclePos_int;};

  /** @brief gives the number of positions where the seed should match (on a cycle)
   *  @return the number of positions
   */

  int    nbpos()    {return _seedNbCyclePos;};

  /** @brief gives the size of the cycle
   *  @return the size of the cycle
   */

  int   maxpos()    {return _seedMaxCyclePos;};
        

  bool  cycled()    {return _seedCyclePos_int != NULL;};


  /** @brief check if the seed is symetric
   *  @return true or false
   */
  bool symetric()  {
    for (int i=0;i<_span/2;i++)
      if (_seedMotif_int[i] != _seedMotif_int[_span-1-i])
	return false;
    return true;
  }
  
  
  /** @brief check if the seed is equal to another one
   *  @return true or false
   */

  bool equal(seed * other) {
    if (_span != other->span())
      return false;
    
    for(int i=0; i<_span; i++)
      if (_seedMotif_int[i] != other->_seedMotif_int[i])
	return false;
    /* FIXME : do the same on position restricted seeds ?? */
    return true;
  }

  /** @brief gives the current seed shape (a string
   *         in [0-9A-Za-z]* or (#[0-9]*)* with a possible position restricted part ":1,3,5/7"
   *  @return a string object that represents this seed.
   */

  string str() {
    ostringstream outs;
    
    for (int i = 0 ; i < _span ; i++) {
      int c = this->_seedMotif_int[i];
      outs << rconvert(c);
    }
    if ( _seedCyclePos_int ) {
      outs << ":";
      
      /* convert to "seed begin position" */ 
      vector<int> cp = vector<int>(0);
      for (int p=0; p<_seedNbCyclePos; p++)
	cp.push_back(((_seedCyclePos_int[p] - (_span%_seedMaxCyclePos) + _seedMaxCyclePos)%_seedMaxCyclePos) + 1);
      sort(cp.begin(),cp.end());

      for (int p=0;p<_seedNbCyclePos;p++) {
	if (p>0)
	  outs << ",";
	outs << (cp[p]);
      }
      outs << "/" << (_seedMaxCyclePos);
    }
    return outs.str();
  }
  
  
  //@}

  
  /** @brief compute the weight of a given seed
   *  @return the weight as a double value
   */
  double weight() {
    double weight = 0.0;
    for(int i=0; i<_span; i++)
      weight += gv_bsel_weight[_seedMotif_int[i]];
    return weight;
  }
  
  /** @brief compute seed selectivity according to the weight (bernoulli model estimation)
   *  @return the selectivity as a double value
   */
  double selectivityFromWeight() {
    return exp(weight()*log(gv_bsel_minprob));
  }


  /** @brief check if the current seed is acceptable : 
   * this implies a correct weight  (according to gloval variables gv_minweight and gv_maxweight), 
   * and a symetric shape when requested.
   *  @return true when the current seed is acceptable
   */
  bool acceptable() {
    if (!gv_vectorized_flag && 
	(
	 gv_bsel_weight[_seedMotif_int[0]]       <= (1e-32) || 
	 gv_bsel_weight[_seedMotif_int[_span-1]] <= (1e-32) 
	 )
	) {
      return false;
    }
    
    if (gv_symetric)
      if (!symetric())
	return false;

    double w = weight();
    
    if (gv_weight_interval && (w < gv_minweight || w > gv_maxweight))
      return false;
    


    return true;
  }
  

  /** @brief check if the signature is realisable according to the span : stop the program otherwise
   */
  
  void checksignature() {
    int sumsignature = 0;
    for(int i = 0 ; i < gv_seed_alphabet_size ; i++)
      sumsignature += gv_signature[i];      
    if (sumsignature > gv_minspan ){
      ERROR("Invalid Signature","size of the signature is " << sumsignature << " and does not fit inside minimal span " << gv_minspan);
    }
  }

  /** @brief set the very first seed signature
   */

  void setsignature() {
    int pos = 0;
    for(int b = gv_seed_alphabet_size-1 ; b >= 0 && pos < _span ; b--){
      int lettersignature =  gv_signature[b];
      while(lettersignature > 0) { // fill with b
	_seedMotif_int[pos++] = b;
	lettersignature--;
      }
    }
    while (pos < _span) { // fill remaining part with jokers
      _seedMotif_int[pos++] = 0;
    }
  }



  /** @brief set all the elements of position >= first_pos that are <= b in the seed in the decreasing order
   *  @brief this function is used to enumerate signatures and inner used.
   *  @param first_pos gives the first position to be sorted
   *  @param b gives the maximal value that can be sorted (all others remain unchanged)
   *  @see next()
   */

  void reorder(int first_pos,int b) {
    int last_bm = -1;
    for(int i = first_pos ; i < _span ; i++) {
      if (_seedMotif_int[i] == b && last_bm >= 0) {
	//swap elements
	int e = _seedMotif_int[i];
	_seedMotif_int[i] = _seedMotif_int[last_bm];
	_seedMotif_int[last_bm] = e;
	i = last_bm - 1;
	last_bm = -1;
      } else {
	if (_seedMotif_int[i] < b && last_bm == -1)
	  last_bm = i;
      }
    }
  }


  /** @brief generates the next seed 
   *  @brief this function does an enumerative generation of the acceptable seeds (and the cycle pos when required), 
   *         with or without signature constraits
   *  @return 1 if next seed has been generated, 0 if all the seeds in the set have already been generated
   *  @see acceptable()
   *  @see setsignature(),checksignature()
   */

  int next() {
    /* A) positions move first */
    if ( _seedCyclePos_int ) {
      int i=_seedNbCyclePos-1;
      int j=_seedMaxCyclePos-1;
      while (i>=0 && _seedCyclePos_int[i] == j) {
	i--;
	j--;
      }
      if (i<0) {
	for(int j=0; j<_seedNbCyclePos; j++)
	  _seedCyclePos_int[j] = j;
	goto nextseed;
      } else {
	_seedCyclePos_int[i]++;
	for (int j=i+1; j<_seedNbCyclePos; j++) {
	  _seedCyclePos_int[j] = _seedCyclePos_int[j-1] + 1;
	}
	return 1;
      }
    }  

  nextseed:

    /* B) seed move second */
    if (gv_signature_flag) {
      
    start_sign:
      /* 1) non trivial signature enumeration */
      for (int b = 1; b < gv_seed_alphabet_size ; b++ ){
	
	int first_b_from_right_to_left = -1;
	int last_bm_before_b           = -1; 
	
	for (int j=_span-1; j >= 0; j--) {
	  if ( _seedMotif_int[j] <  b ) {
	    last_bm_before_b = j; 
	  } else if ( _seedMotif_int[j] == b && last_bm_before_b >= 0 ) {
	    first_b_from_right_to_left = j ;
	    break;
	  }
	}
	if (first_b_from_right_to_left >= 0) {
	  //swap elements 
	  {
	    int e = b;
	    _seedMotif_int[first_b_from_right_to_left] =  _seedMotif_int[last_bm_before_b];
	    _seedMotif_int[last_bm_before_b] = e;
	  }
	  //reorder elements b and <b after this swap position
	  reorder(last_bm_before_b,b);

	  //reorder the elements <b (only at position where such elements are)
	  for (int bl = b-1 ; bl >= 0 ; bl-- )
	    reorder(0,bl);
	  while(!acceptable())
	    goto start_sign;
	  return 1;
	}
      }
      
      // span changing
      if (_span < gv_maxspan) {
	_span ++;
	delete _seedMotif_int;
	_seedMotif_int = new int[_span];
	setsignature();
	while(!acceptable())
	  goto start_sign;
	return 1;
      } else {
	return 0;
      }


    } else {
      
      /* 2) complete enumeration */
    start:
      int j=_span-1;
      
      while(_seedMotif_int[j] == (gv_seed_alphabet_size-1)){
	j--;
	if (j < 0) {
	  if (_span < gv_maxspan) {
	    _span ++;
	    delete _seedMotif_int;
	    _seedMotif_int = new int[_span];
	    goto reset;
	  } else {
	    return 0;
	  }
	}
      }
      
      _seedMotif_int[j]++;
    reset:
      j++;
      for(int i = j ; i < _span ; i++) {
	_seedMotif_int[i] = 0;
      }
      
      while(!acceptable()){
	goto start;
      }
      return 1;
    }/* 2) */
    
  }
  
  
  


  /** @brief this function generates a random seed (and random positions) according to gv_minspan and gv_maxspan 
   *
   *  @return 1 all the time
   */
  int random(){
    
    /* A) select the seed shape */

    if (gv_signature_flag) {


      /* 1) signature random generation */
      // check signature
      int sumsignature = 0;
      for(int i = 0 ; i < gv_seed_alphabet_size ; i++)
	sumsignature += gv_signature[i];      
      if (sumsignature > gv_minspan ){
	ERROR("Invalid Signature","size of the signature is " << sumsignature << " and does not fit inside minimal span " << gv_minspan);
      }
      
      // select a new span 
      int newspan = rand()%(gv_maxspan - gv_minspan + 1) + gv_minspan;
      if (newspan != _span) {
	_span = newspan;
	delete _seedMotif_int;
	_seedMotif_int = new int[_span];
      }
       
      
      // "fill and swap" randomization : 

      // fill
      setsignature();
      // swap
    swap:
      for(int i= 0 ; i < _span ; i++){
	int j = rand()%_span;
	int letter_tmp    = _seedMotif_int[i];
	_seedMotif_int[i] = _seedMotif_int[j];
	_seedMotif_int[j] = letter_tmp;
      }
      
     
      while(!acceptable())
	goto swap; /* FIXME : can loop infinitely if a bad signature is given */

    } else {

    start:

      /* 2) random  enumeration */
      int newspan = rand()%(gv_maxspan - gv_minspan + 1) + gv_minspan;
      if (newspan != _span) {
	_span = newspan;
	delete _seedMotif_int;
	_seedMotif_int = new int[_span];
      }

    refill:
      for(int i = 0 ; i < _span ; i++) {
	_seedMotif_int[i] = rand() % gv_seed_alphabet_size;
      }



      /* heuristic weight trick to modify the seed to reach the reasonable weight */
      if (gv_weight_interval) {

	double w = weight();
	
	if ((w < gv_minweight) || ( w > gv_maxweight)) {
	  int pos_shuffle[_span];	  

	  /* set a shuffle of seed positions to be considered */
	  for(int i=0;i<_span;i++)
	    pos_shuffle[i] = i;
	  for(int i=1;i<_span;i++) {
	    int p = rand()%(i);
	    pos_shuffle[i] = pos_shuffle[p];
	    pos_shuffle[p] = i;
	  }

	  /* increase weight loop */
	  while (w < gv_minweight && (_seedMotif_int[pos_shuffle[_span-1]] < gv_seed_alphabet_size - 1)) {
	    for(int i = 0 ; i < _span ; i++) {
	      int p = pos_shuffle[i];
	      if (_seedMotif_int[p] < gv_seed_alphabet_size - 1) {
		int      bold = _seedMotif_int[p]++;
		double  delta = gv_bsel_weight[_seedMotif_int[p]] - gv_bsel_weight[bold];
		w  += delta;
		if (w >= gv_minweight)
		  goto eoup;
	      }
	    }
	  }

	eoup:

	  /* set a shuffle of seed positions to be considered */
	  for(int i=0;i<_span;i++)
	    pos_shuffle[i] = i;
	  for(int i=1;i<_span;i++) {
	    int p = rand()%(i);
	    pos_shuffle[i] = pos_shuffle[p];
	    pos_shuffle[p] = i;
	  }

	  /* decrease weight loop */
	  while (w > gv_maxweight && _seedMotif_int[pos_shuffle[_span-1]] > 0) {
	    for(int i = 0 ; i < _span ; i++) {
	      int p = pos_shuffle[i];
	      if (_seedMotif_int[p] > 0) {
		int      bold = _seedMotif_int[p]--;
		double  delta = gv_bsel_weight[_seedMotif_int[p]] - gv_bsel_weight[bold];
		w  += delta;

		if (w <= gv_minweight)
		  goto eodown;
	      }
	    }
	  }	  
	eodown:;	  
	}
      }

      while(!acceptable()){
	if (!(rand()%500)){
          goto start;
	} else {
          goto refill;
        }
      }
    }


    /* B) select the seed pos */
    if ( _seedCyclePos_int ) {
      for(int i=0; i<_seedNbCyclePos; i++) {
      re:	
	_seedCyclePos_int[i] = rand()%_seedMaxCyclePos;	  
	for(int j=0; j<i; j++)
	  if (_seedCyclePos_int[j] == _seedCyclePos_int[i]) /* FIXME : do this more efficiently ... */
	    goto re;
      }
    }

    return 1;
  }





  /** @brief does a swap for the hill climbing function (swap cycle positions then pair of seeds elements, one at a time)
   *  @return 1 if the next swap has been generated, 0 if all possible swaps have already been enumerated
   */
  int nextswap() {

    if (gv_swap_choice%2) {
      
      /* A) swap seed positions first*/
      if ( _seedCyclePos_int ) {
	
	if (_swap_pos_index >= _seedNbCyclePos)
	  goto nextswapseed;
	
	if (_swap_pos_index >= 0) {
	nxinc:
	  _seedCyclePos_int[_swap_pos_index]++;
	  for(int i=0;i<_seedNbCyclePos;i++)
	    if ((i != _swap_pos_index && _seedCyclePos_int[_swap_pos_index] == _seedCyclePos_int[i]) || _seedCyclePos_int[_swap_pos_index] == _swap_pos_old_value)
	      goto nxinc;
	  
	  if (_seedCyclePos_int[_swap_pos_index] >= _seedMaxCyclePos) {	  
	    _seedCyclePos_int[_swap_pos_index] = _swap_pos_old_value;
	    _swap_pos_index++;	
	    if (_swap_pos_index >= _seedNbCyclePos)
	      goto nextswapseed;
	    _swap_pos_old_value = _seedCyclePos_int[_swap_pos_index];
	    _seedCyclePos_int[_swap_pos_index] = -1;
	    goto nxinc;
	  }
	} else {
	  _swap_pos_index = 0;
	  _swap_pos_old_value = _seedCyclePos_int[_swap_pos_index];
	  _seedCyclePos_int[_swap_pos_index] = -1;
	  goto nxinc;
	}
	return 1;
      }
      
    nextswapseed:    
      if (gv_symetric)
	return 0;
      
      /* B) swap seed elements */
      /* reverse previous swap */
      if ( _swap_from < _span && _swap_to < _span) {
	int letter_tmp             = _seedMotif_int[_swap_from];
	_seedMotif_int[_swap_from] = _seedMotif_int[_swap_to];
	_seedMotif_int[_swap_to]   = letter_tmp;
      }

      /* move _swap...  positions */
      do {
	_swap_to++;
	if (_swap_to >= _span) {
	  _swap_from++;
	  if (_swap_from >= _span - 1) {
	    return 0;
	  }
	  _swap_to = _swap_from + 1;
	}
      } while(_seedMotif_int[_swap_from] == _seedMotif_int[_swap_to]);

      /* swap */
      if ( _swap_from < _span && _swap_to < _span) {
	int letter_tmp             = _seedMotif_int[_swap_from];
	_seedMotif_int[_swap_from] = _seedMotif_int[_swap_to];
	_seedMotif_int[_swap_to]   = letter_tmp;
      }    
      if (!acceptable()) {
	goto nextswapseed;
      }
      


    } else {



    nextswapseed_2:
      if (gv_symetric)
	goto nextpos_2;
      
      /* A) swap seed elements */    
      /* reverse previous swap */
      if ( _swap_from < _span && _swap_to < _span) {
	int letter_tmp             = _seedMotif_int[_swap_from];
	_seedMotif_int[_swap_from] = _seedMotif_int[_swap_to];
	_seedMotif_int[_swap_to]   = letter_tmp;
      }
      
      /* move _swap...  positions */
      do {
	_swap_to++;
	if (_swap_to >= _span) {
	  _swap_from++;    
	  if (_swap_from >= _span - 1) {
	    goto nextpos_2;
	  }
	  _swap_to = _swap_from + 1;
	}
      } while(_seedMotif_int[_swap_from] == _seedMotif_int[_swap_to]);
      
      /* swap */
      if ( _swap_from < _span && _swap_to < _span) {
	int letter_tmp             = _seedMotif_int[_swap_from];
	_seedMotif_int[_swap_from] = _seedMotif_int[_swap_to];
	_seedMotif_int[_swap_to]   = letter_tmp;
      }    
      if (!acceptable()) {
	goto nextswapseed_2;
      }
      return 1;

    nextpos_2:
      /* B) swap seed positions first*/
      if ( _seedCyclePos_int ) {
	
	if (_swap_pos_index >= _seedNbCyclePos) {
	  return 0;
	}
	if (_swap_pos_index >= 0) {
	nxinc_2:
	  _seedCyclePos_int[_swap_pos_index]++;
	  for(int i=0;i<_seedNbCyclePos;i++)
	    if ((i != _swap_pos_index && _seedCyclePos_int[_swap_pos_index] == _seedCyclePos_int[i]) || _seedCyclePos_int[_swap_pos_index] == _swap_pos_old_value)
	      goto nxinc_2;
	  
	  if (_seedCyclePos_int[_swap_pos_index] >= _seedMaxCyclePos) {	  
	    _seedCyclePos_int[_swap_pos_index] = _swap_pos_old_value;
	    _swap_pos_index++;	
	    if (_swap_pos_index >= _seedNbCyclePos) {
	      return 0;
	    }
	    _swap_pos_old_value = _seedCyclePos_int[_swap_pos_index];
	    _seedCyclePos_int[_swap_pos_index] = -1;
	    goto nxinc_2;
	  }
	} else {
	  _swap_pos_index = 0;
	  _swap_pos_old_value = _seedCyclePos_int[_swap_pos_index];
	  _seedCyclePos_int[_swap_pos_index] = -1;
	  goto nxinc_2;
	}
      } else {
	return 0;
      }
    }
    return 1;
  }


  
  /** @brief reset the swap for the hill climbing function, all swap will then be processed from the begining
   */
  void resetswap() {
    _swap_from      = 0;
    _swap_to        = 0;
    _swap_pos_index = -1;
    _swap_pos_old_value = 0;
    if (_seedCyclePos_int) {
      // shuffle positions without changing their values
      for (int i = 0; i < _seedNbCyclePos; i++) {
	int j = rand()%_seedNbCyclePos;
	int t                = _seedCyclePos_int[i];
	_seedCyclePos_int[i] = _seedCyclePos_int[j];
	_seedCyclePos_int[j] = t;
      }
    }
  }


  /** @brief fix definitely the swap (reverse it)
   */
  void setswap() {
    /* swap */
    if (!gv_symetric && _swap_from < _span && _swap_to < _span) {
      int letter_tmp             = _seedMotif_int[_swap_from];
      _seedMotif_int[_swap_from] = _seedMotif_int[_swap_to];
      _seedMotif_int[_swap_to]   = letter_tmp;
    }
    /* swap_pos */
    if (_swap_pos_index >= 0 && _swap_pos_index < _seedNbCyclePos) {
      _seedCyclePos_int[_swap_pos_index] = _swap_pos_old_value;
    }
    /* reset after swaping */
    resetswap();
  }






  /** @brief set cycle properties of the seed 
   *  @param cycle_pos is a vector of positions where the seed is allowed to match
   *  @param cycle_size is the maximal position where a position can be set
   */
  void setCyclePos(vector<int> cycle_pos, int cycle_size) {
    _seedCyclePos_int   = new int[cycle_pos.size()];
    for(int i=0;i<(int)cycle_pos.size();i++)
      _seedCyclePos_int[i] = cycle_pos[i];
    _seedNbCyclePos     = cycle_pos.size();
    _seedMaxCyclePos    = cycle_size;
  }
  
  
  /** @brief output a seed object
   *  @param os is the outputstream
   *  @param s is the seed to output
   *  @return the outputsteam
   *  @see seed
   */
  friend ostream& operator<<(ostream& os, seed& s);


 protected:
  /// Current seed motif represented as an int array {0,1,2, ...} of lenght "_span"
  int  * _seedMotif_int;
  /// Fixed seed span 
  int _span;
  /// Swapped char cursor (from) 
  int _swap_from;
  /// Swapped char cursor (to)
  int _swap_to;


  /// Current seed positions to be set (NULL if applied on all positions)
  int * _seedCyclePos_int;
  /// Size of the "_seedCyclePos_int" table
  int _seedNbCyclePos;
  /// Max value in the  "_seedCyclePos_int" table
  int _seedMaxCyclePos;

  /// Swapped pos index (0 if no swap)
  int _swap_pos_index;
  /// Swapped old value (0 if no swap)
  int _swap_pos_old_value;



  /** 
   * @brief convert a char into a number
   * @brief '0' -> 0, '1' -> 1 ... '9' -> 9, 'A' -> 10 ... 'Z' -> 35, 'a' -> 36 ...
   * @param s is the string where conversion is currently processed
   * @param pos is the first char inside s where the conversion has to be done
   * @return the converted value (integer that represents a seed letter) and increments i to the next symbol
   */
  int convert(string & s, int * pos){    

    const char * c = s.c_str();
    c += (*pos);
    
    if (gv_bsymbols_flag) {
      for (int i=0 ; i < gv_seed_alphabet_size ; i++)
	if (gv_bsymbols_array[i] == *c) {
	  (*pos)++;
	  return i;
	}
      cerr << "* Error : invalid char \'" << c << "\' inside the seed descriptor" << endl;
      exit(-1);
    } else {
      
      if (*c >= '0' && *c <= '9') {
	if (*c - '0' >= gv_seed_alphabet_size) {
	  cerr << "* Error : out of bound char \'" << c << "\' inside the seed descriptor" << endl;
	  exit(-1);
	}
	(*pos)++;
	return *c - '0';
      } else if (*c >= 'A' && *c <= 'Z') {
	if (*c - 'A' + 10 >= gv_seed_alphabet_size) {
	  cerr << "* Error : out of bound char \'" << c << "\' inside the seed descriptor" << endl;
	  exit(-1);
	}
	(*pos)++;
	return *c - 'A' + 10;
      } else if (*c >= 'a' && *c <= 'z') {
	if (*c - 'a' + 36 >= gv_seed_alphabet_size) {
	  cerr << "* Error : out of bound char \'" << c << "\' inside the seed descriptor" << endl;
	  exit(-1);
	}
	(*pos)++;
	return *c - 'a' + 36;
      } else if (*c == '#') {
	int code = 0;
	(*pos)++;
	c++;
	while (*c >= '0' && *c <= '9') {
	  code = code * 10 + (*c - '0');
	  (*pos)++;
	  c++;
	}
	if (code >= 0 && code < gv_seed_alphabet_size) {
	  return code;
	} else {
	  cerr << "* Error : out of bound code \'#" << code << "\' inside the seed descriptor" << endl;
	  exit(-1);
	}
      } else {
	  cerr << "* Error : out of bound char \'" << *c << "\' inside the seed descriptor" << endl;
	exit(-1);
	return 0;
      }
    }
    return 0;
  }
  
  /// convert a number into a char
  string rconvert(int i){
    if (gv_bsymbols_flag) {
      return string(1,gv_bsymbols_array[i]);
    } else {
      if (gv_seed_alphabet_size <= 62) {
	if (i < 10)
	  return string(1,(char)i + '0');
	else
	  if (i < 36)
	    return string(1,(char)i - 10 + 'A');
	  else
	    return string(1,(char)i - 36 + 'a');
      } else {
	string s = "";
	while (i) {
	  char c = '0' + (i%10);
	  i /= 10;
	  s = string(1,c) + s;
	}
	return string("#" + s);
      }
    }
  }

};

#endif

// @}
