// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package fp

import (
	"crypto/rand"
	"encoding/json"
	"fmt"
	"math/big"
	"math/bits"

	mrand "math/rand"

	"testing"

	"github.com/leanovate/gopter"
	ggen "github.com/leanovate/gopter/gen"
	"github.com/leanovate/gopter/prop"

	"github.com/stretchr/testify/require"
)

// -------------------------------------------------------------------------------------------------
// benchmarks
// most benchmarks are rudimentary and should sample a large number of random inputs
// or be run multiple times to ensure it didn't measure the fastest path of the function

var benchResElement Element

func BenchmarkElementSelect(b *testing.B) {
	var x, y Element
	x.SetRandom()
	y.SetRandom()

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Select(i%3, &x, &y)
	}
}

func BenchmarkElementSetRandom(b *testing.B) {
	var x Element
	x.SetRandom()

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		_, _ = x.SetRandom()
	}
}

func BenchmarkElementSetBytes(b *testing.B) {
	var x Element
	x.SetRandom()
	bb := x.Bytes()
	b.ResetTimer()

	for i := 0; i < b.N; i++ {
		benchResElement.SetBytes(bb[:])
	}

}

func BenchmarkElementMulByConstants(b *testing.B) {
	b.Run("mulBy3", func(b *testing.B) {
		benchResElement.SetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy3(&benchResElement)
		}
	})
	b.Run("mulBy5", func(b *testing.B) {
		benchResElement.SetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy5(&benchResElement)
		}
	})
	b.Run("mulBy13", func(b *testing.B) {
		benchResElement.SetRandom()
		b.ResetTimer()
		for i := 0; i < b.N; i++ {
			MulBy13(&benchResElement)
		}
	})
}

func BenchmarkElementInverse(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b.ResetTimer()

	for i := 0; i < b.N; i++ {
		benchResElement.Inverse(&x)
	}

}

func BenchmarkElementButterfly(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		Butterfly(&x, &benchResElement)
	}
}

func BenchmarkElementExp(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b1, _ := rand.Int(rand.Reader, Modulus())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Exp(x, b1)
	}
}

func BenchmarkElementDouble(b *testing.B) {
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Double(&benchResElement)
	}
}

func BenchmarkElementAdd(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Add(&x, &benchResElement)
	}
}

func BenchmarkElementSub(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Sub(&x, &benchResElement)
	}
}

func BenchmarkElementNeg(b *testing.B) {
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Neg(&benchResElement)
	}
}

func BenchmarkElementDiv(b *testing.B) {
	var x Element
	x.SetRandom()
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Div(&x, &benchResElement)
	}
}

func BenchmarkElementFromMont(b *testing.B) {
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.fromMont()
	}
}

func BenchmarkElementSquare(b *testing.B) {
	benchResElement.SetRandom()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Square(&benchResElement)
	}
}

func BenchmarkElementSqrt(b *testing.B) {
	var a Element
	a.SetUint64(4)
	a.Neg(&a)
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Sqrt(&a)
	}
}

func BenchmarkElementMul(b *testing.B) {
	x := Element{
		7358459907925294924,
		14414180951914241931,
		16619482658146888203,
		760736596725344926,
		12753071240931896792,
		13425190760400245818,
		12591714441439252728,
		15325516497554583360,
		5301152003049442834,
		35368377961363834,
	}
	benchResElement.SetOne()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Mul(&benchResElement, &x)
	}
}

func BenchmarkElementCmp(b *testing.B) {
	x := Element{
		7358459907925294924,
		14414180951914241931,
		16619482658146888203,
		760736596725344926,
		12753071240931896792,
		13425190760400245818,
		12591714441439252728,
		15325516497554583360,
		5301152003049442834,
		35368377961363834,
	}
	benchResElement = x
	benchResElement[0] = 0
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		benchResElement.Cmp(&x)
	}
}

func TestElementCmp(t *testing.T) {
	var x, y Element

	if x.Cmp(&y) != 0 {
		t.Fatal("x == y")
	}

	one := One()
	y.Sub(&y, &one)

	if x.Cmp(&y) != -1 {
		t.Fatal("x < y")
	}
	if y.Cmp(&x) != 1 {
		t.Fatal("x < y")
	}

	x = y
	if x.Cmp(&y) != 0 {
		t.Fatal("x == y")
	}

	x.Sub(&x, &one)
	if x.Cmp(&y) != -1 {
		t.Fatal("x < y")
	}
	if y.Cmp(&x) != 1 {
		t.Fatal("x < y")
	}
}
func TestElementIsRandom(t *testing.T) {
	for i := 0; i < 50; i++ {
		var x, y Element
		x.SetRandom()
		y.SetRandom()
		if x.Equal(&y) {
			t.Fatal("2 random numbers are unlikely to be equal")
		}
	}
}

func TestElementIsUint64(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	properties.Property("reduce should output a result smaller than modulus", prop.ForAll(
		func(v uint64) bool {
			var e Element
			e.SetUint64(v)

			if !e.IsUint64() {
				return false
			}

			return e.Uint64() == v
		},
		ggen.UInt64(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementNegZero(t *testing.T) {
	var a, b Element
	b.SetZero()
	for a.IsZero() {
		a.SetRandom()
	}
	a.Neg(&b)
	if !a.IsZero() {
		t.Fatal("neg(0) != 0")
	}
}

// -------------------------------------------------------------------------------------------------
// Gopter tests
// most of them are generated with a template

const (
	nbFuzzShort = 20
	nbFuzz      = 100
)

// special values to be used in tests
var staticTestValues []Element

func init() {
	staticTestValues = append(staticTestValues, Element{}) // zero
	staticTestValues = append(staticTestValues, One())     // one
	staticTestValues = append(staticTestValues, rSquare)   // r²
	var e, one Element
	one.SetOne()
	e.Sub(&qElement, &one)
	staticTestValues = append(staticTestValues, e) // q - 1
	e.Double(&one)
	staticTestValues = append(staticTestValues, e) // 2

	{
		a := qElement
		a[0]--
		staticTestValues = append(staticTestValues, a)
	}
	staticTestValues = append(staticTestValues, Element{0})
	staticTestValues = append(staticTestValues, Element{0, 0})
	staticTestValues = append(staticTestValues, Element{1})
	staticTestValues = append(staticTestValues, Element{0, 1})
	staticTestValues = append(staticTestValues, Element{2})
	staticTestValues = append(staticTestValues, Element{0, 2})

	{
		a := qElement
		a[9]--
		staticTestValues = append(staticTestValues, a)
	}
	{
		a := qElement
		a[9]--
		a[0]++
		staticTestValues = append(staticTestValues, a)
	}

	{
		a := qElement
		a[9] = 0
		staticTestValues = append(staticTestValues, a)
	}

}

func TestElementReduce(t *testing.T) {
	testValues := make([]Element, len(staticTestValues))
	copy(testValues, staticTestValues)

	for i := range testValues {
		s := testValues[i]
		expected := s
		reduce(&s)
		_reduceGeneric(&expected)
		if !s.Equal(&expected) {
			t.Fatal("reduce failed: asm and generic impl don't match")
		}
	}

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := genFull()

	properties.Property("reduce should output a result smaller than modulus", prop.ForAll(
		func(a Element) bool {
			b := a
			reduce(&a)
			_reduceGeneric(&b)
			return a.smallerThanModulus() && a.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementEqual(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("x.Equal(&y) iff x == y; likely false for random pairs", prop.ForAll(
		func(a testPairElement, b testPairElement) bool {
			return a.element.Equal(&b.element) == (a.element == b.element)
		},
		genA,
		genB,
	))

	properties.Property("x.Equal(&y) if x == y", prop.ForAll(
		func(a testPairElement) bool {
			b := a.element
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementBytes(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("SetBytes(Bytes()) should stay constant", prop.ForAll(
		func(a testPairElement) bool {
			var b Element
			bytes := a.element.Bytes()
			b.SetBytes(bytes[:])
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementInverseExp(t *testing.T) {
	// inverse must be equal to exp^-2
	exp := Modulus()
	exp.Sub(exp, new(big.Int).SetUint64(2))

	invMatchExp := func(a testPairElement) bool {
		var b Element
		b.Set(&a.element)
		a.element.Inverse(&a.element)
		b.Exp(b, exp)

		return a.element.Equal(&b)
	}

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}
	properties := gopter.NewProperties(parameters)
	genA := gen()
	properties.Property("inv == exp^-2", prop.ForAll(invMatchExp, genA))
	properties.TestingRun(t, gopter.ConsoleReporter(false))

	parameters.MinSuccessfulTests = 1
	properties = gopter.NewProperties(parameters)
	properties.Property("inv(0) == 0", prop.ForAll(invMatchExp, ggen.OneConstOf(testPairElement{})))
	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func mulByConstant(z *Element, c uint8) {
	var y Element
	y.SetUint64(uint64(c))
	z.Mul(z, &y)
}

func TestElementMulByConstants(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	implemented := []uint8{0, 1, 2, 3, 5, 13}
	properties.Property("mulByConstant", prop.ForAll(
		func(a testPairElement) bool {
			for _, c := range implemented {
				var constant Element
				constant.SetUint64(uint64(c))

				b := a.element
				b.Mul(&b, &constant)

				aa := a.element
				mulByConstant(&aa, c)

				if !aa.Equal(&b) {
					return false
				}
			}

			return true
		},
		genA,
	))

	properties.Property("MulBy3(x) == Mul(x, 3)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(3)

			b := a.element
			b.Mul(&b, &constant)

			MulBy3(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("MulBy5(x) == Mul(x, 5)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(5)

			b := a.element
			b.Mul(&b, &constant)

			MulBy5(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("MulBy13(x) == Mul(x, 13)", prop.ForAll(
		func(a testPairElement) bool {
			var constant Element
			constant.SetUint64(13)

			b := a.element
			b.Mul(&b, &constant)

			MulBy13(&a.element)

			return a.element.Equal(&b)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementLegendre(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("legendre should output same result than big.Int.Jacobi", prop.ForAll(
		func(a testPairElement) bool {
			return a.element.Legendre() == big.Jacobi(&a.bigint, Modulus())
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementBitLen(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("BitLen should output same result than big.Int.BitLen", prop.ForAll(
		func(a testPairElement) bool {
			return a.element.fromMont().BitLen() == a.bigint.BitLen()
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementButterflies(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("butterfly0 == a -b; a +b", prop.ForAll(
		func(a, b testPairElement) bool {
			a0, b0 := a.element, b.element

			_butterflyGeneric(&a.element, &b.element)
			Butterfly(&a0, &b0)

			return a.element.Equal(&a0) && b.element.Equal(&b0)
		},
		genA,
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementLexicographicallyLargest(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("element.Cmp should match LexicographicallyLargest output", prop.ForAll(
		func(a testPairElement) bool {
			var negA Element
			negA.Neg(&a.element)

			cmpResult := a.element.Cmp(&negA)
			lResult := a.element.LexicographicallyLargest()

			if lResult && cmpResult == 1 {
				return true
			}
			if !lResult && cmpResult != 1 {
				return true
			}
			return false
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

}

func TestElementAdd(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Add: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Add(&a.element, &b.element)
			a.element.Add(&a.element, &b.element)
			b.element.Add(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Add: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Add(&a.element, &b.element)

				var d, e big.Int
				d.Add(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Add(&a.element, &r)
				d.Add(&a.bigint, &rb).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Add: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Add(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Add(&a, &b)
				d.Add(&aBig, &bBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Add failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSub(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Sub: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Sub(&a.element, &b.element)
			a.element.Sub(&a.element, &b.element)
			b.element.Sub(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Sub: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Sub(&a.element, &b.element)

				var d, e big.Int
				d.Sub(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Sub(&a.element, &r)
				d.Sub(&a.bigint, &rb).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Sub: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Sub(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Sub(&a, &b)
				d.Sub(&aBig, &bBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Sub failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementMul(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Mul: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Mul(&a.element, &b.element)
			a.element.Mul(&a.element, &b.element)
			b.element.Mul(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Mul: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Mul(&a.element, &b.element)

				var d, e big.Int
				d.Mul(&a.bigint, &b.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Mul(&a.element, &r)
				d.Mul(&a.bigint, &rb).Mod(&d, Modulus())

				// checking generic impl against asm path
				var cGeneric Element
				_mulGeneric(&cGeneric, &a.element, &r)
				if !cGeneric.Equal(&c) {
					// need to give context to failing error.
					return false
				}

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Mul: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Mul(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	properties.Property("Mul: assembly implementation must be consistent with generic one", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			c.Mul(&a.element, &b.element)
			_mulGeneric(&d, &a.element, &b.element)
			return c.Equal(&d)
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Mul(&a, &b)
				d.Mul(&aBig, &bBig).Mod(&d, Modulus())

				// checking asm against generic impl
				var cGeneric Element
				_mulGeneric(&cGeneric, &a, &b)
				if !cGeneric.Equal(&c) {
					t.Fatal("Mul failed special test values: asm and generic impl don't match")
				}

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Mul failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementDiv(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Div: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Div(&a.element, &b.element)
			a.element.Div(&a.element, &b.element)
			b.element.Div(&d, &b.element)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Div: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Div(&a.element, &b.element)

				var d, e big.Int
				d.ModInverse(&b.bigint, Modulus())
				d.Mul(&d, &a.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Div(&a.element, &r)
				d.ModInverse(&rb, Modulus())
				d.Mul(&d, &a.bigint).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Div: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Div(&a.element, &b.element)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Div(&a, &b)
				d.ModInverse(&bBig, Modulus())
				d.Mul(&d, &aBig).Mod(&d, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Div failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementExp(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genB := gen()

	properties.Property("Exp: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b testPairElement) bool {
			var c, d Element
			d.Set(&a.element)

			c.Exp(a.element, &b.bigint)
			a.element.Exp(a.element, &b.bigint)
			b.element.Exp(d, &b.bigint)

			return a.element.Equal(&b.element) && a.element.Equal(&c) && b.element.Equal(&c)
		},
		genA,
		genB,
	))

	properties.Property("Exp: operation result must match big.Int result", prop.ForAll(
		func(a, b testPairElement) bool {
			{
				var c Element

				c.Exp(a.element, &b.bigint)

				var d, e big.Int
				d.Exp(&a.bigint, &b.bigint, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}

			// fixed elements
			// a is random
			// r takes special values
			testValues := make([]Element, len(staticTestValues))
			copy(testValues, staticTestValues)

			for i := range testValues {
				r := testValues[i]
				var d, e, rb big.Int
				r.BigInt(&rb)

				var c Element
				c.Exp(a.element, &rb)
				d.Exp(&a.bigint, &rb, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					return false
				}
			}
			return true
		},
		genA,
		genB,
	))

	properties.Property("Exp: operation result must be smaller than modulus", prop.ForAll(
		func(a, b testPairElement) bool {
			var c Element

			c.Exp(a.element, &b.bigint)

			return c.smallerThanModulus()
		},
		genA,
		genB,
	))

	specialValueTest := func() {
		// test special values against special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			for j := range testValues {
				b := testValues[j]
				var bBig, d, e big.Int
				b.BigInt(&bBig)

				var c Element
				c.Exp(a, &bBig)
				d.Exp(&aBig, &bBig, Modulus())

				if c.BigInt(&e).Cmp(&d) != 0 {
					t.Fatal("Exp failed special test values")
				}
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSquare(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Square: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Square(&a.element)
			a.element.Square(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Square: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Square(&a.element)

			var d, e big.Int
			d.Mul(&a.bigint, &a.bigint).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Square: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Square(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Square(&a)

			var d, e big.Int
			d.Mul(&aBig, &aBig).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Square failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementInverse(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Inverse: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Inverse(&a.element)
			a.element.Inverse(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Inverse: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Inverse(&a.element)

			var d, e big.Int
			d.ModInverse(&a.bigint, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Inverse: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Inverse(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Inverse(&a)

			var d, e big.Int
			d.ModInverse(&aBig, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Inverse failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementSqrt(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Sqrt: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			b := a.element

			b.Sqrt(&a.element)
			a.element.Sqrt(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Sqrt: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Sqrt(&a.element)

			var d, e big.Int
			d.ModSqrt(&a.bigint, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Sqrt: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Sqrt(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Sqrt(&a)

			var d, e big.Int
			d.ModSqrt(&aBig, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Sqrt failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementDouble(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Double: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Double(&a.element)
			a.element.Double(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Double: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Double(&a.element)

			var d, e big.Int
			d.Lsh(&a.bigint, 1).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Double: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Double(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Double(&a)

			var d, e big.Int
			d.Lsh(&aBig, 1).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Double failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementNeg(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Neg: having the receiver as operand should output the same result", prop.ForAll(
		func(a testPairElement) bool {

			var b Element

			b.Neg(&a.element)
			a.element.Neg(&a.element)
			return a.element.Equal(&b)
		},
		genA,
	))

	properties.Property("Neg: operation result must match big.Int result", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Neg(&a.element)

			var d, e big.Int
			d.Neg(&a.bigint).Mod(&d, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA,
	))

	properties.Property("Neg: operation result must be smaller than modulus", prop.ForAll(
		func(a testPairElement) bool {
			var c Element
			c.Neg(&a.element)
			return c.smallerThanModulus()
		},
		genA,
	))

	specialValueTest := func() {
		// test special values
		testValues := make([]Element, len(staticTestValues))
		copy(testValues, staticTestValues)

		for i := range testValues {
			a := testValues[i]
			var aBig big.Int
			a.BigInt(&aBig)
			var c Element
			c.Neg(&a)

			var d, e big.Int
			d.Neg(&aBig).Mod(&d, Modulus())

			if c.BigInt(&e).Cmp(&d) != 0 {
				t.Fatal("Neg failed special test values")
			}
		}
	}

	properties.TestingRun(t, gopter.ConsoleReporter(false))
	specialValueTest()

}

func TestElementFixedExp(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	var (
		_bLegendreExponentElement *big.Int
		_bSqrtExponentElement     *big.Int
	)

	_bLegendreExponentElement, _ = new(big.Int).SetString("93319e6079afb1fe0d0ba780eb955ad47e6c63aebce963a72cbb4d6cdded17c0a953607d6f52485c6d4faf41fabe24bf07442876ded203ebdae73d5c1ce1129e9b4de988a3fb9e6ba48b7522b80006", 16)
	const sqrtExponentElement = "24cc67981e6bec7f8342e9e03ae556b51f9b18ebaf3a58e9cb2ed35b377b45f02a54d81f5bd492171b53ebd07eaf892fc1d10a1db7b480faf6b9cf57073844a7a6d37a6228fee79ae922dd48ae0001"
	_bSqrtExponentElement, _ = new(big.Int).SetString(sqrtExponentElement, 16)

	genA := gen()

	properties.Property(fmt.Sprintf("expBySqrtExp must match Exp(%s)", sqrtExponentElement), prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.expBySqrtExp(c)
			d.Exp(d, _bSqrtExponentElement)
			return c.Equal(&d)
		},
		genA,
	))

	properties.Property("expByLegendreExp must match Exp(93319e6079afb1fe0d0ba780eb955ad47e6c63aebce963a72cbb4d6cdded17c0a953607d6f52485c6d4faf41fabe24bf07442876ded203ebdae73d5c1ce1129e9b4de988a3fb9e6ba48b7522b80006)", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.expByLegendreExp(c)
			d.Exp(d, _bLegendreExponentElement)
			return c.Equal(&d)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementHalve(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	var twoInv Element
	twoInv.SetUint64(2)
	twoInv.Inverse(&twoInv)

	properties.Property("z.Halve must match z / 2", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.Halve()
			d.Mul(&d, &twoInv)
			return c.Equal(&d)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func combineSelectionArguments(c int64, z int8) int {
	if z%3 == 0 {
		return 0
	}
	return int(c)
}

func TestElementSelect(t *testing.T) {
	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := genFull()
	genB := genFull()
	genC := ggen.Int64() //the condition
	genZ := ggen.Int8()  //to make zeros artificially more likely

	properties.Property("Select: must select correctly", prop.ForAll(
		func(a, b Element, cond int64, z int8) bool {
			condC := combineSelectionArguments(cond, z)

			var c Element
			c.Select(condC, &a, &b)

			if condC == 0 {
				return c.Equal(&a)
			}
			return c.Equal(&b)
		},
		genA,
		genB,
		genC,
		genZ,
	))

	properties.Property("Select: having the receiver as operand should output the same result", prop.ForAll(
		func(a, b Element, cond int64, z int8) bool {
			condC := combineSelectionArguments(cond, z)

			var c, d Element
			d.Set(&a)
			c.Select(condC, &a, &b)
			a.Select(condC, &a, &b)
			b.Select(condC, &d, &b)
			return a.Equal(&b) && a.Equal(&c) && b.Equal(&c)
		},
		genA,
		genB,
		genC,
		genZ,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementSetInt64(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("z.SetInt64 must match z.SetString", prop.ForAll(
		func(a testPairElement, v int64) bool {
			c := a.element
			d := a.element

			c.SetInt64(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, ggen.Int64(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementSetInterface(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()
	genInt := ggen.Int
	genInt8 := ggen.Int8
	genInt16 := ggen.Int16
	genInt32 := ggen.Int32
	genInt64 := ggen.Int64

	genUint := ggen.UInt
	genUint8 := ggen.UInt8
	genUint16 := ggen.UInt16
	genUint32 := ggen.UInt32
	genUint64 := ggen.UInt64

	properties.Property("z.SetInterface must match z.SetString with int8", prop.ForAll(
		func(a testPairElement, v int8) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt8(),
	))

	properties.Property("z.SetInterface must match z.SetString with int16", prop.ForAll(
		func(a testPairElement, v int16) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt16(),
	))

	properties.Property("z.SetInterface must match z.SetString with int32", prop.ForAll(
		func(a testPairElement, v int32) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt32(),
	))

	properties.Property("z.SetInterface must match z.SetString with int64", prop.ForAll(
		func(a testPairElement, v int64) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt64(),
	))

	properties.Property("z.SetInterface must match z.SetString with int", prop.ForAll(
		func(a testPairElement, v int) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genInt(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint8", prop.ForAll(
		func(a testPairElement, v uint8) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint8(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint16", prop.ForAll(
		func(a testPairElement, v uint16) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint16(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint32", prop.ForAll(
		func(a testPairElement, v uint32) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint32(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint64", prop.ForAll(
		func(a testPairElement, v uint64) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint64(),
	))

	properties.Property("z.SetInterface must match z.SetString with uint", prop.ForAll(
		func(a testPairElement, v uint) bool {
			c := a.element
			d := a.element

			c.SetInterface(v)
			d.SetString(fmt.Sprintf("%v", v))

			return c.Equal(&d)
		},
		genA, genUint(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))

	{
		assert := require.New(t)
		var e Element
		r, err := e.SetInterface(nil)
		assert.Nil(r)
		assert.Error(err)

		var ptE *Element
		var ptB *big.Int

		r, err = e.SetInterface(ptE)
		assert.Nil(r)
		assert.Error(err)
		ptE = new(Element).SetOne()
		r, err = e.SetInterface(ptE)
		assert.NoError(err)
		assert.True(r.IsOne())

		r, err = e.SetInterface(ptB)
		assert.Nil(r)
		assert.Error(err)

	}
}

func TestElementNegativeExp(t *testing.T) {
	t.Parallel()

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("x⁻ᵏ == 1/xᵏ", prop.ForAll(
		func(a, b testPairElement) bool {

			var nb, d, e big.Int
			nb.Neg(&b.bigint)

			var c Element
			c.Exp(a.element, &nb)

			d.Exp(&a.bigint, &nb, Modulus())

			return c.BigInt(&e).Cmp(&d) == 0
		},
		genA, genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementNewElement(t *testing.T) {
	assert := require.New(t)

	t.Parallel()

	e := NewElement(1)
	assert.True(e.IsOne())

	e = NewElement(0)
	assert.True(e.IsZero())
}

func TestElementBatchInvert(t *testing.T) {
	assert := require.New(t)

	t.Parallel()

	// ensure batchInvert([x]) == invert(x)
	for i := int64(-1); i <= 2; i++ {
		var e, eInv Element
		e.SetInt64(i)
		eInv.Inverse(&e)

		a := []Element{e}
		aInv := BatchInvert(a)

		assert.True(aInv[0].Equal(&eInv), "batchInvert != invert")

	}

	// test x * x⁻¹ == 1
	tData := [][]int64{
		{-1, 1, 2, 3},
		{0, -1, 1, 2, 3, 0},
		{0, -1, 1, 0, 2, 3, 0},
		{-1, 1, 0, 2, 3},
		{0, 0, 1},
		{1, 0, 0},
		{0, 0, 0},
	}

	for _, t := range tData {
		a := make([]Element, len(t))
		for i := 0; i < len(a); i++ {
			a[i].SetInt64(t[i])
		}

		aInv := BatchInvert(a)

		assert.True(len(aInv) == len(a))

		for i := 0; i < len(a); i++ {
			if a[i].IsZero() {
				assert.True(aInv[i].IsZero(), "0⁻¹ != 0")
			} else {
				assert.True(a[i].Mul(&a[i], &aInv[i]).IsOne(), "x * x⁻¹ != 1")
			}
		}
	}

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("batchInvert --> x * x⁻¹ == 1", prop.ForAll(
		func(tp testPairElement, r uint8) bool {

			a := make([]Element, r)
			if r != 0 {
				a[0] = tp.element

			}
			one := One()
			for i := 1; i < len(a); i++ {
				a[i].Add(&a[i-1], &one)
			}

			aInv := BatchInvert(a)

			assert.True(len(aInv) == len(a))

			for i := 0; i < len(a); i++ {
				if a[i].IsZero() {
					if !aInv[i].IsZero() {
						return false
					}
				} else {
					if !a[i].Mul(&a[i], &aInv[i]).IsOne() {
						return false
					}
				}
			}
			return true
		},
		genA, ggen.UInt8(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementFromMont(t *testing.T) {

	t.Parallel()
	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	genA := gen()

	properties.Property("Assembly implementation must be consistent with generic one", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			d := a.element
			c.fromMont()
			_fromMontGeneric(&d)
			return c.Equal(&d)
		},
		genA,
	))

	properties.Property("x.fromMont().toMont() == x", prop.ForAll(
		func(a testPairElement) bool {
			c := a.element
			c.fromMont().toMont()
			return c.Equal(&a.element)
		},
		genA,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementJSON(t *testing.T) {
	assert := require.New(t)

	type S struct {
		A Element
		B [3]Element
		C *Element
		D *Element
	}

	// encode to JSON
	var s S
	s.A.SetString("-1")
	s.B[2].SetUint64(42)
	s.D = new(Element).SetUint64(8000)

	encoded, err := json.Marshal(&s)
	assert.NoError(err)
	// we may need to adjust "42" and "8000" values for some moduli; see Text() method for more details.
	formatValue := func(v int64) string {
		var a big.Int
		a.SetInt64(v)
		a.Mod(&a, Modulus())
		const maxUint16 = 65535
		var aNeg big.Int
		aNeg.Neg(&a).Mod(&aNeg, Modulus())
		if aNeg.Uint64() != 0 && aNeg.Uint64() <= maxUint16 {
			return "-" + aNeg.Text(10)
		}
		return a.Text(10)
	}
	expected := fmt.Sprintf("{\"A\":%s,\"B\":[0,0,%s],\"C\":null,\"D\":%s}", formatValue(-1), formatValue(42), formatValue(8000))
	assert.Equal(expected, string(encoded))

	// decode valid
	var decoded S
	err = json.Unmarshal([]byte(expected), &decoded)
	assert.NoError(err)

	assert.Equal(s, decoded, "element -> json -> element round trip failed")

	// decode hex and string values
	withHexValues := "{\"A\":\"-1\",\"B\":[0,\"0x00000\",\"0x2A\"],\"C\":null,\"D\":\"8000\"}"

	var decodedS S
	err = json.Unmarshal([]byte(withHexValues), &decodedS)
	assert.NoError(err)

	assert.Equal(s, decodedS, " json with strings  -> element  failed")

}

type testPairElement struct {
	element Element
	bigint  big.Int
}

func gen() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var g testPairElement

		g.element = Element{
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
		}
		if qElement[9] != ^uint64(0) {
			g.element[9] %= (qElement[9] + 1)
		}

		for !g.element.smallerThanModulus() {
			g.element = Element{
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
				genParams.NextUint64(),
			}
			if qElement[9] != ^uint64(0) {
				g.element[9] %= (qElement[9] + 1)
			}
		}

		g.element.BigInt(&g.bigint)
		genResult := gopter.NewGenResult(g, gopter.NoShrinker)
		return genResult
	}
}

func genRandomFq(genParams *gopter.GenParameters) Element {
	var g Element

	g = Element{
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
		genParams.NextUint64(),
	}

	if qElement[9] != ^uint64(0) {
		g[9] %= (qElement[9] + 1)
	}

	for !g.smallerThanModulus() {
		g = Element{
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
		}
		if qElement[9] != ^uint64(0) {
			g[9] %= (qElement[9] + 1)
		}
	}

	return g
}

func genFull() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		a := genRandomFq(genParams)

		var carry uint64
		a[0], carry = bits.Add64(a[0], qElement[0], carry)
		a[1], carry = bits.Add64(a[1], qElement[1], carry)
		a[2], carry = bits.Add64(a[2], qElement[2], carry)
		a[3], carry = bits.Add64(a[3], qElement[3], carry)
		a[4], carry = bits.Add64(a[4], qElement[4], carry)
		a[5], carry = bits.Add64(a[5], qElement[5], carry)
		a[6], carry = bits.Add64(a[6], qElement[6], carry)
		a[7], carry = bits.Add64(a[7], qElement[7], carry)
		a[8], carry = bits.Add64(a[8], qElement[8], carry)
		a[9], _ = bits.Add64(a[9], qElement[9], carry)

		genResult := gopter.NewGenResult(a, gopter.NoShrinker)
		return genResult
	}
}

func genElement() gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		a := genRandomFq(genParams)
		genResult := gopter.NewGenResult(a, gopter.NoShrinker)
		return genResult
	}
}

func (z *Element) matchVeryBigInt(aHi uint64, aInt *big.Int) error {
	var modulus big.Int
	var aIntMod big.Int
	modulus.SetInt64(1)
	modulus.Lsh(&modulus, (Limbs+1)*64)
	aIntMod.Mod(aInt, &modulus)

	slice := append(z[:], aHi)

	return bigIntMatchUint64Slice(&aIntMod, slice)
}

// TODO: Phase out in favor of property based testing
func (z *Element) assertMatchVeryBigInt(t *testing.T, aHi uint64, aInt *big.Int) {

	if err := z.matchVeryBigInt(aHi, aInt); err != nil {
		t.Error(err)
	}
}

// bigIntMatchUint64Slice is a test helper to match big.Int words against a uint64 slice
func bigIntMatchUint64Slice(aInt *big.Int, a []uint64) error {

	words := aInt.Bits()

	const steps = 64 / bits.UintSize
	const filter uint64 = 0xFFFFFFFFFFFFFFFF >> (64 - bits.UintSize)
	for i := 0; i < len(a)*steps; i++ {

		var wI big.Word

		if i < len(words) {
			wI = words[i]
		}

		aI := a[i/steps] >> ((i * bits.UintSize) % 64)
		aI &= filter

		if uint64(wI) != aI {
			return fmt.Errorf("bignum mismatch: disagreement on word %d: %x ≠ %x; %d ≠ %d", i, uint64(wI), aI, uint64(wI), aI)
		}
	}

	return nil
}

func TestElementInversionApproximation(t *testing.T) {
	var x Element
	for i := 0; i < 1000; i++ {
		x.SetRandom()

		// Normally small elements are unlikely. Here we give them a higher chance
		xZeros := mrand.Int() % Limbs //#nosec G404 weak rng is fine here
		for j := 1; j < xZeros; j++ {
			x[Limbs-j] = 0
		}

		a := approximate(&x, x.BitLen())
		aRef := approximateRef(&x)

		if a != aRef {
			t.Error("Approximation mismatch")
		}
	}
}

func TestElementInversionCorrectionFactorFormula(t *testing.T) {
	const kLimbs = k * Limbs
	const power = kLimbs*6 + invIterationsN*(kLimbs-k+1)
	factorInt := big.NewInt(1)
	factorInt.Lsh(factorInt, power)
	factorInt.Mod(factorInt, Modulus())

	var refFactorInt big.Int
	inversionCorrectionFactor := Element{
		inversionCorrectionFactorWord0,
		inversionCorrectionFactorWord1,
		inversionCorrectionFactorWord2,
		inversionCorrectionFactorWord3,
		inversionCorrectionFactorWord4,
		inversionCorrectionFactorWord5,
		inversionCorrectionFactorWord6,
		inversionCorrectionFactorWord7,
		inversionCorrectionFactorWord8,
		inversionCorrectionFactorWord9,
	}
	inversionCorrectionFactor.toBigInt(&refFactorInt)

	if refFactorInt.Cmp(factorInt) != 0 {
		t.Error("mismatch")
	}
}

func TestElementLinearComb(t *testing.T) {
	var x Element
	var y Element

	for i := 0; i < 1000; i++ {
		x.SetRandom()
		y.SetRandom()
		testLinearComb(t, &x, mrand.Int63(), &y, mrand.Int63()) //#nosec G404 weak rng is fine here
	}
}

// Probably unnecessary post-dev. In case the output of inv is wrong, this checks whether it's only off by a constant factor.
func TestElementInversionCorrectionFactor(t *testing.T) {

	// (1/x)/inv(x) = (1/1)/inv(1) ⇔ inv(1) = x inv(x)

	var one Element
	var oneInv Element
	one.SetOne()
	oneInv.Inverse(&one)

	for i := 0; i < 100; i++ {
		var x Element
		var xInv Element
		x.SetRandom()
		xInv.Inverse(&x)

		x.Mul(&x, &xInv)
		if !x.Equal(&oneInv) {
			t.Error("Correction factor is inconsistent")
		}
	}

	if !oneInv.Equal(&one) {
		var i big.Int
		oneInv.BigInt(&i) // no montgomery
		i.ModInverse(&i, Modulus())
		var fac Element
		fac.setBigInt(&i) // back to montgomery

		var facTimesFac Element
		facTimesFac.Mul(&fac, &Element{
			inversionCorrectionFactorWord0,
			inversionCorrectionFactorWord1,
			inversionCorrectionFactorWord2,
			inversionCorrectionFactorWord3,
			inversionCorrectionFactorWord4,
			inversionCorrectionFactorWord5,
			inversionCorrectionFactorWord6,
			inversionCorrectionFactorWord7,
			inversionCorrectionFactorWord8,
			inversionCorrectionFactorWord9,
		})

		t.Error("Correction factor is consistently off by", fac, "Should be", facTimesFac)
	}
}

func TestElementBigNumNeg(t *testing.T) {
	var a Element
	aHi := negL(&a, 0)
	if !a.IsZero() || aHi != 0 {
		t.Error("-0 != 0")
	}
}

func TestElementBigNumWMul(t *testing.T) {
	var x Element

	for i := 0; i < 1000; i++ {
		x.SetRandom()
		w := mrand.Int63() //#nosec G404 weak rng is fine here
		testBigNumWMul(t, &x, w)
	}
}

func TestElementVeryBigIntConversion(t *testing.T) {
	xHi := mrand.Uint64() //#nosec G404 weak rng is fine here
	var x Element
	x.SetRandom()
	var xInt big.Int
	x.toVeryBigIntSigned(&xInt, xHi)
	x.assertMatchVeryBigInt(t, xHi, &xInt)
}

type veryBigInt struct {
	asInt big.Int
	low   Element
	hi    uint64
}

// genVeryBigIntSigned if sign == 0, no sign is forced
func genVeryBigIntSigned(sign int) gopter.Gen {
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var g veryBigInt

		g.low = Element{
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
			genParams.NextUint64(),
		}

		g.hi = genParams.NextUint64()

		if sign < 0 {
			g.hi |= signBitSelector
		} else if sign > 0 {
			g.hi &= ^signBitSelector
		}

		g.low.toVeryBigIntSigned(&g.asInt, g.hi)

		genResult := gopter.NewGenResult(g, gopter.NoShrinker)
		return genResult
	}
}

func TestElementMontReduce(t *testing.T) {

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	gen := genVeryBigIntSigned(0)

	properties.Property("Montgomery reduction is correct", prop.ForAll(
		func(g veryBigInt) bool {
			var res Element
			var resInt big.Int

			montReduce(&resInt, &g.asInt)
			res.montReduceSigned(&g.low, g.hi)

			return res.matchVeryBigInt(0, &resInt) == nil
		},
		gen,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElementMontReduceMultipleOfR(t *testing.T) {

	parameters := gopter.DefaultTestParameters()
	if testing.Short() {
		parameters.MinSuccessfulTests = nbFuzzShort
	} else {
		parameters.MinSuccessfulTests = nbFuzz
	}

	properties := gopter.NewProperties(parameters)

	gen := ggen.UInt64()

	properties.Property("Montgomery reduction is correct", prop.ForAll(
		func(hi uint64) bool {
			var zero, res Element
			var asInt, resInt big.Int

			zero.toVeryBigIntSigned(&asInt, hi)

			montReduce(&resInt, &asInt)
			res.montReduceSigned(&zero, hi)

			return res.matchVeryBigInt(0, &resInt) == nil
		},
		gen,
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestElement0Inverse(t *testing.T) {
	var x Element
	x.Inverse(&x)
	if !x.IsZero() {
		t.Fail()
	}
}

// TODO: Tests like this (update factor related) are common to all fields. Move them to somewhere non-autogen
func TestUpdateFactorSubtraction(t *testing.T) {
	for i := 0; i < 1000; i++ {

		f0, g0 := randomizeUpdateFactors()
		f1, g1 := randomizeUpdateFactors()

		for f0-f1 > 1<<31 || f0-f1 <= -1<<31 {
			f1 /= 2
		}

		for g0-g1 > 1<<31 || g0-g1 <= -1<<31 {
			g1 /= 2
		}

		c0 := updateFactorsCompose(f0, g0)
		c1 := updateFactorsCompose(f1, g1)

		cRes := c0 - c1
		fRes, gRes := updateFactorsDecompose(cRes)

		if fRes != f0-f1 || gRes != g0-g1 {
			t.Error(i)
		}
	}
}

func TestUpdateFactorsDouble(t *testing.T) {
	for i := 0; i < 1000; i++ {
		f, g := randomizeUpdateFactors()

		if f > 1<<30 || f < (-1<<31+1)/2 {
			f /= 2
			if g <= 1<<29 && g >= (-1<<31+1)/4 {
				g *= 2 //g was kept small on f's account. Now that we're halving f, we can double g
			}
		}

		if g > 1<<30 || g < (-1<<31+1)/2 {
			g /= 2

			if f <= 1<<29 && f >= (-1<<31+1)/4 {
				f *= 2 //f was kept small on g's account. Now that we're halving g, we can double f
			}
		}

		c := updateFactorsCompose(f, g)
		cD := c * 2
		fD, gD := updateFactorsDecompose(cD)

		if fD != 2*f || gD != 2*g {
			t.Error(i)
		}
	}
}

func TestUpdateFactorsNeg(t *testing.T) {
	var fMistake bool
	for i := 0; i < 1000; i++ {
		f, g := randomizeUpdateFactors()

		if f == 0x80000000 || g == 0x80000000 {
			// Update factors this large can only have been obtained after 31 iterations and will therefore never be negated
			// We don't have capacity to store -2³¹
			// Repeat this iteration
			i--
			continue
		}

		c := updateFactorsCompose(f, g)
		nc := -c
		nf, ng := updateFactorsDecompose(nc)
		fMistake = fMistake || nf != -f
		if nf != -f || ng != -g {
			t.Errorf("Mismatch iteration #%d:\n%d, %d ->\n %d -> %d ->\n %d, %d\n Inputs in hex: %X, %X",
				i, f, g, c, nc, nf, ng, f, g)
		}
	}
	if fMistake {
		t.Error("Mistake with f detected")
	} else {
		t.Log("All good with f")
	}
}

func TestUpdateFactorsNeg0(t *testing.T) {
	c := updateFactorsCompose(0, 0)
	t.Logf("c(0,0) = %X", c)
	cn := -c

	if c != cn {
		t.Error("Negation of zero update factors should yield the same result.")
	}
}

func TestUpdateFactorDecomposition(t *testing.T) {
	var negSeen bool

	for i := 0; i < 1000; i++ {

		f, g := randomizeUpdateFactors()

		if f <= -(1<<31) || f > 1<<31 {
			t.Fatal("f out of range")
		}

		negSeen = negSeen || f < 0

		c := updateFactorsCompose(f, g)

		fBack, gBack := updateFactorsDecompose(c)

		if f != fBack || g != gBack {
			t.Errorf("(%d, %d) -> %d -> (%d, %d)\n", f, g, c, fBack, gBack)
		}
	}

	if !negSeen {
		t.Fatal("No negative f factors")
	}
}

func TestUpdateFactorInitialValues(t *testing.T) {

	f0, g0 := updateFactorsDecompose(updateFactorIdentityMatrixRow0)
	f1, g1 := updateFactorsDecompose(updateFactorIdentityMatrixRow1)

	if f0 != 1 || g0 != 0 || f1 != 0 || g1 != 1 {
		t.Error("Update factor initial value constants are incorrect")
	}
}

func TestUpdateFactorsRandomization(t *testing.T) {
	var maxLen int

	//t.Log("|f| + |g| is not to exceed", 1 << 31)
	for i := 0; i < 1000; i++ {
		f, g := randomizeUpdateFactors()
		lf, lg := abs64T32(f), abs64T32(g)
		absSum := lf + lg
		if absSum >= 1<<31 {

			if absSum == 1<<31 {
				maxLen++
			} else {
				t.Error(i, "Sum of absolute values too large, f =", f, ",g =", g, ",|f| + |g| =", absSum)
			}
		}
	}

	if maxLen == 0 {
		t.Error("max len not observed")
	} else {
		t.Log(maxLen, "maxLens observed")
	}
}

func randomizeUpdateFactor(absLimit uint32) int64 {
	const maxSizeLikelihood = 10
	maxSize := mrand.Intn(maxSizeLikelihood) //#nosec G404 weak rng is fine here

	absLimit64 := int64(absLimit)
	var f int64
	switch maxSize {
	case 0:
		f = absLimit64
	case 1:
		f = -absLimit64
	default:
		f = int64(mrand.Uint64()%(2*uint64(absLimit64)+1)) - absLimit64 //#nosec G404 weak rng is fine here
	}

	if f > 1<<31 {
		return 1 << 31
	} else if f < -1<<31+1 {
		return -1<<31 + 1
	}

	return f
}

func abs64T32(f int64) uint32 {
	if f >= 1<<32 || f < -1<<32 {
		panic("f out of range")
	}

	if f < 0 {
		return uint32(-f)
	}
	return uint32(f)
}

func randomizeUpdateFactors() (int64, int64) {
	var f [2]int64
	b := mrand.Int() % 2 //#nosec G404 weak rng is fine here

	f[b] = randomizeUpdateFactor(1 << 31)

	//As per the paper, |f| + |g| \le 2³¹.
	f[1-b] = randomizeUpdateFactor(1<<31 - abs64T32(f[b]))

	//Patching another edge case
	if f[0]+f[1] == -1<<31 {
		b = mrand.Int() % 2 //#nosec G404 weak rng is fine here
		f[b]++
	}

	return f[0], f[1]
}

func testLinearComb(t *testing.T, x *Element, xC int64, y *Element, yC int64) {

	var p1 big.Int
	x.toBigInt(&p1)
	p1.Mul(&p1, big.NewInt(xC))

	var p2 big.Int
	y.toBigInt(&p2)
	p2.Mul(&p2, big.NewInt(yC))

	p1.Add(&p1, &p2)
	p1.Mod(&p1, Modulus())
	montReduce(&p1, &p1)

	var z Element
	z.linearComb(x, xC, y, yC)
	z.assertMatchVeryBigInt(t, 0, &p1)
}

func testBigNumWMul(t *testing.T, a *Element, c int64) {
	var aHi uint64
	var aTimes Element
	aHi = aTimes.mulWNonModular(a, c)

	assertMulProduct(t, a, c, &aTimes, aHi)
}

func updateFactorsCompose(f int64, g int64) int64 {
	return f + g<<32
}

var rInv big.Int

func montReduce(res *big.Int, x *big.Int) {
	if rInv.BitLen() == 0 { // initialization
		rInv.SetUint64(1)
		rInv.Lsh(&rInv, Limbs*64)
		rInv.ModInverse(&rInv, Modulus())
	}
	res.Mul(x, &rInv)
	res.Mod(res, Modulus())
}

func (z *Element) toVeryBigIntUnsigned(i *big.Int, xHi uint64) {
	z.toBigInt(i)
	var upperWord big.Int
	upperWord.SetUint64(xHi)
	upperWord.Lsh(&upperWord, Limbs*64)
	i.Add(&upperWord, i)
}

func (z *Element) toVeryBigIntSigned(i *big.Int, xHi uint64) {
	z.toVeryBigIntUnsigned(i, xHi)
	if signBitSelector&xHi != 0 {
		twosCompModulus := big.NewInt(1)
		twosCompModulus.Lsh(twosCompModulus, (Limbs+1)*64)
		i.Sub(i, twosCompModulus)
	}
}

func assertMulProduct(t *testing.T, x *Element, c int64, result *Element, resultHi uint64) big.Int {
	var xInt big.Int
	x.toBigInt(&xInt)

	xInt.Mul(&xInt, big.NewInt(c))

	result.assertMatchVeryBigInt(t, resultHi, &xInt)
	return xInt
}

func approximateRef(x *Element) uint64 {

	var asInt big.Int
	x.toBigInt(&asInt)
	n := x.BitLen()

	if n <= 64 {
		return asInt.Uint64()
	}

	modulus := big.NewInt(1 << 31)
	var lo big.Int
	lo.Mod(&asInt, modulus)

	modulus.Lsh(modulus, uint(n-64))
	var hi big.Int
	hi.Div(&asInt, modulus)
	hi.Lsh(&hi, 31)

	hi.Add(&hi, &lo)
	return hi.Uint64()
}
