pub const ext = @import("ext.zig");
const gstmpegts = @This();

const std = @import("std");
const compat = @import("compat");
const gst = @import("gst1");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
/// Event Information Table (ATSC)
pub const AtscEIT = extern struct {
    /// The source id
    f_source_id: u16,
    /// The protocol version
    f_protocol_version: u8,
    /// Events
    f_events: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_eit_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_eit_get_type;
};

/// An ATSC EIT Event
pub const AtscEITEvent = extern struct {
    /// The event id
    f_event_id: u16,
    /// The start time
    f_start_time: u32,
    /// The etm location
    f_etm_location: u8,
    /// The length in seconds
    f_length_in_seconds: u32,
    /// the titles
    f_titles: ?*glib.PtrArray,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_eit_event_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_eit_event_get_type;
};

/// Extended Text Table (ATSC)
pub const AtscETT = extern struct {
    f_ett_table_id_extension: u16,
    /// The protocol version
    f_protocol_version: u16,
    /// The etm id
    f_etm_id: u32,
    /// List of texts
    f_messages: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_ett_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_ett_get_type;
};

/// Master Guide Table (A65)
pub const AtscMGT = extern struct {
    /// The protocol version
    f_protocol_version: u8,
    /// The numbers of subtables
    f_tables_defined: u16,
    /// the tables
    f_tables: ?*glib.PtrArray,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_mgt_new() *gstmpegts.AtscMGT;
    pub const new = gst_mpegts_atsc_mgt_new;

    extern fn gst_mpegts_atsc_mgt_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_mgt_get_type;
};

/// Source from a `GstMpegtsAtscMGT`
pub const AtscMGTTable = extern struct {
    /// `gstmpegts.AtscMGTTableType`
    f_table_type: u16,
    /// The packet ID
    f_pid: u16,
    /// The version number
    f_version_number: u8,
    f_number_bytes: u32,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_mgt_table_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_mgt_table_get_type;
};

pub const AtscMultString = extern struct {
    /// The ISO639 language code
    f_iso_639_langcode: [4]u8,
    f_segments: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_mult_string_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_mult_string_get_type;
};

/// Region Rating Table (A65)
pub const AtscRRT = extern struct {
    /// The protocol version
    f_protocol_version: u8,
    /// the names
    f_names: ?*glib.PtrArray,
    /// the number of dimensions defined for this rating table
    f_dimensions_defined: u8,
    /// A set of dimensions
    f_dimensions: ?*glib.PtrArray,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_rrt_new() *gstmpegts.AtscRRT;
    pub const new = gst_mpegts_atsc_rrt_new;

    extern fn gst_mpegts_atsc_rrt_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_rrt_get_type;
};

pub const AtscRRTDimension = extern struct {
    /// the names
    f_names: ?*glib.PtrArray,
    /// whether the ratings represent a graduated scale
    f_graduated_scale: c_int,
    /// the number of values defined for this dimension
    f_values_defined: u8,
    /// set of values
    f_values: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_rrt_dimension_new() *gstmpegts.AtscRRTDimension;
    pub const new = gst_mpegts_atsc_rrt_dimension_new;

    extern fn gst_mpegts_atsc_rrt_dimension_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_rrt_dimension_get_type;
};

pub const AtscRRTDimensionValue = extern struct {
    /// the abbreviated ratings
    f_abbrev_ratings: ?*glib.PtrArray,
    /// the ratings
    f_ratings: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_rrt_dimension_value_new() *gstmpegts.AtscRRTDimensionValue;
    pub const new = gst_mpegts_atsc_rrt_dimension_value_new;

    extern fn gst_mpegts_atsc_rrt_dimension_value_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_rrt_dimension_value_get_type;
};

/// System Time Table (A65)
pub const AtscSTT = extern struct {
    /// The protocol version
    f_protocol_version: u8,
    /// The system time
    f_system_time: u32,
    /// The GPS to UTC offset
    f_gps_utc_offset: u8,
    f_ds_status: c_int,
    /// The day of month
    f_ds_dayofmonth: u8,
    /// The hour
    f_ds_hour: u8,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,
    /// The UTC date and time
    f_utc_datetime: ?*gst.DateTime,

    extern fn gst_mpegts_atsc_stt_new() *gstmpegts.AtscSTT;
    pub const new = gst_mpegts_atsc_stt_new;

    extern fn gst_mpegts_atsc_stt_get_datetime_utc(p_stt: *AtscSTT) *gst.DateTime;
    pub const getDatetimeUtc = gst_mpegts_atsc_stt_get_datetime_utc;

    extern fn gst_mpegts_atsc_stt_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_stt_get_type;
};

/// A string segment
pub const AtscStringSegment = extern struct {
    /// The compression type
    f_compression_type: u8,
    /// The mode
    f_mode: u8,
    /// The size of compressed data
    f_compressed_data_size: u8,
    /// The compressed data
    f_compressed_data: ?*u8,
    f_cached_string: ?[*:0]u8,

    extern fn gst_mpegts_atsc_string_segment_get_string(p_seg: *AtscStringSegment) [*:0]const u8;
    pub const getString = gst_mpegts_atsc_string_segment_get_string;

    extern fn gst_mpegts_atsc_string_segment_set_string(p_seg: *AtscStringSegment, p_string: [*:0]u8, p_compression_type: u8, p_mode: u8) c_int;
    pub const setString = gst_mpegts_atsc_string_segment_set_string;

    extern fn gst_mpegts_atsc_string_segment_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_string_segment_get_type;
};

/// Represents both:
///   Terrestrial Virtual Channel Table (A65)
///   Cable Virtual Channel Table (A65)
pub const AtscVCT = extern struct {
    /// The transport stream
    f_transport_stream_id: u16,
    /// The protocol version
    f_protocol_version: u8,
    /// sources
    f_sources: ?*glib.PtrArray,
    /// descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_vct_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_vct_get_type;
};

/// Source from a `gstmpegts.AtscVCT`, can be used both for TVCT and CVCT tables
pub const AtscVCTSource = extern struct {
    /// The short name of a source
    f_short_name: ?[*:0]u8,
    /// The major channel number
    f_major_channel_number: u16,
    /// The minor channel number
    f_minor_channel_number: u16,
    /// The modulation mode
    f_modulation_mode: u8,
    /// The carrier frequency
    f_carrier_frequency: u32,
    /// The transport stream ID
    f_channel_TSID: u16,
    /// The program number (see `gstmpegts.PatProgram`)
    f_program_number: u16,
    /// The ETM location
    f_ETM_location: u8,
    /// is access controlled
    f_access_controlled: c_int,
    /// is hidden
    f_hidden: c_int,
    /// is path select, CVCT only
    f_path_select: c_int,
    /// is out of band, CVCT only
    f_out_of_band: c_int,
    /// is hide guide
    f_hide_guide: c_int,
    /// The service type
    f_service_type: u8,
    /// The source id
    f_source_id: u16,
    /// an array of `gstmpegts.Descriptor`
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_atsc_vct_source_get_type() usize;
    pub const getGObjectType = gst_mpegts_atsc_vct_source_get_type;
};

/// Table 110: Audio Preselection Descriptor (ETSI EN 300 468 v1.16.1)
pub const AudioPreselectionDescriptor = extern struct {
    /// 5-bit
    f_preselection_id: u8,
    /// 3-bit field
    f_audio_rendering_indication: u8,
    /// visually impaired
    f_audio_description: c_int,
    f_spoken_subtitles: c_int,
    f_dialogue_enhancement: c_int,
    f_interactivity_enabled: c_int,
    f_language_code_present: c_int,
    f_text_label_present: c_int,
    /// indicates if this PID conveys a complete audio programme
    f_multi_stream_info_present: c_int,
    f_future_extension: c_int,
    /// NULL terminated ISO 639 language code.
    f_language_code: ?[*:0]u8,
    f_message_id: u8,
};

/// DVB Bouquet Association Table (EN 300 468)
pub const BAT = extern struct {
    f_descriptors: ?*glib.PtrArray,
    f_streams: ?*glib.PtrArray,

    extern fn gst_mpegts_bat_get_type() usize;
    pub const getGObjectType = gst_mpegts_bat_get_type;
};

pub const BATStream = extern struct {
    f_transport_stream_id: u16,
    f_original_network_id: u16,
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_bat_stream_get_type() usize;
    pub const getGObjectType = gst_mpegts_bat_stream_get_type;
};

/// Cable Delivery System Descriptor (EN 300 468 v.1.13.1)
pub const CableDeliverySystemDescriptor = extern struct {
    /// the frequency in Hz (Hertz)
    f_frequency: u32,
    /// the outer FEC scheme used
    f_outer_fec: gstmpegts.CableOuterFECScheme,
    /// Modulation scheme used
    f_modulation: gstmpegts.ModulationType,
    /// Symbol rate (in symbols per second)
    f_symbol_rate: u32,
    /// inner FEC scheme used
    f_fec_inner: gstmpegts.DVBCodeRate,

    extern fn gst_mpegts_dvb_cable_delivery_system_descriptor_free(p_source: *CableDeliverySystemDescriptor) void;
    pub const free = gst_mpegts_dvb_cable_delivery_system_descriptor_free;

    extern fn gst_mpegts_dvb_cable_delivery_system_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_cable_delivery_system_descriptor_get_type;
};

pub const ComponentDescriptor = extern struct {
    f_stream_content: u8,
    f_component_type: u8,
    f_component_tag: u8,
    f_language_code: ?[*:0]u8,
    f_text: ?[*:0]u8,

    extern fn gst_mpegts_component_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_component_descriptor_get_type;
};

pub const Content = extern struct {
    f_content_nibble_1: gstmpegts.ContentNibbleHi,
    f_content_nibble_2: u8,
    f_user_byte: u8,

    extern fn gst_mpegts_content_get_type() usize;
    pub const getGObjectType = gst_mpegts_content_get_type;
};

pub const DVBLinkageDescriptor = extern struct {
    /// the transport id
    f_transport_stream_id: u16,
    /// the original network id
    f_original_network_id: u16,
    /// the service id
    f_service_id: u16,
    /// the type which `linkage_data` has
    f_linkage_type: gstmpegts.DVBLinkageType,
    f_linkage_data: ?*anyopaque,
    /// the length for `private_data_bytes`
    f_private_data_length: u8,
    /// additional data bytes
    f_private_data_bytes: ?*u8,

    extern fn gst_mpegts_dvb_linkage_descriptor_free(p_source: *DVBLinkageDescriptor) void;
    pub const free = gst_mpegts_dvb_linkage_descriptor_free;

    extern fn gst_mpegts_dvb_linkage_descriptor_get_event(p_desc: *const DVBLinkageDescriptor) *const gstmpegts.DVBLinkageEvent;
    pub const getEvent = gst_mpegts_dvb_linkage_descriptor_get_event;

    extern fn gst_mpegts_dvb_linkage_descriptor_get_extended_event(p_desc: *const DVBLinkageDescriptor) *const glib.PtrArray;
    pub const getExtendedEvent = gst_mpegts_dvb_linkage_descriptor_get_extended_event;

    extern fn gst_mpegts_dvb_linkage_descriptor_get_mobile_hand_over(p_desc: *const DVBLinkageDescriptor) *const gstmpegts.DVBLinkageMobileHandOver;
    pub const getMobileHandOver = gst_mpegts_dvb_linkage_descriptor_get_mobile_hand_over;

    extern fn gst_mpegts_dvb_linkage_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_linkage_descriptor_get_type;
};

pub const DVBLinkageEvent = extern struct {
    f_target_event_id: u16,
    f_target_listed: c_int,
    f_event_simulcast: c_int,

    extern fn gst_mpegts_dvb_linkage_event_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_linkage_event_get_type;
};

pub const DVBLinkageExtendedEvent = extern struct {
    f_target_event_id: u16,
    f_target_listed: c_int,
    f_event_simulcast: c_int,
    f_link_type: u8,
    f_target_id_type: u8,
    f_original_network_id_flag: c_int,
    f_service_id_flag: c_int,
    f_user_defined_id: u16,
    f_target_transport_stream_id: u16,
    f_target_original_network_id: u16,
    f_target_service_id: u16,

    extern fn gst_mpegts_dvb_linkage_extended_event_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_linkage_extended_event_get_type;
};

pub const DVBLinkageMobileHandOver = extern struct {
    f_hand_over_type: gstmpegts.DVBLinkageHandOverType,
    f_origin_type: c_int,
    f_network_id: u16,
    f_initial_service_id: u16,

    extern fn gst_mpegts_dvb_linkage_mobile_hand_over_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_linkage_mobile_hand_over_get_type;
};

pub const DVBParentalRatingItem = extern struct {
    f_country_code: ?[*:0]u8,
    f_rating: u8,

    extern fn gst_mpegts_dvb_parental_rating_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_parental_rating_item_get_type;
};

pub const DVBServiceListItem = extern struct {
    /// the id of a service
    f_service_id: u16,
    /// the type of a service
    f_type: gstmpegts.DVBServiceType,

    extern fn gst_mpegts_dvb_service_list_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_service_list_item_get_type;
};

pub const DataBroadcastDescriptor = extern struct {
    /// the data broadcast id
    f_data_broadcast_id: u16,
    /// the component tag
    f_component_tag: u8,
    f_length: u8,
    /// the selector byte field
    f_selector_bytes: ?*u8,
    /// language of `text`
    f_language_code: ?[*:0]u8,
    /// description of data broadcast
    f_text: ?[*:0]u8,

    extern fn gst_mpegts_dvb_data_broadcast_descriptor_free(p_source: *DataBroadcastDescriptor) void;
    pub const free = gst_mpegts_dvb_data_broadcast_descriptor_free;

    extern fn gst_mpegts_dvb_data_broadcast_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_data_broadcast_descriptor_get_type;
};

/// These are the base descriptor types and methods.
///
/// For more details, refer to the ITU H.222.0 or ISO/IEC 13818-1 specifications
/// and other specifications mentioned in the documentation.
pub const Descriptor = extern struct {
    /// the type of descriptor
    f_tag: u8,
    /// the extended type (if `descriptor_tag` is 0x7f)
    f_tag_extension: u8,
    /// the length of the descriptor content (excluding tag/length field)
    f_length: u8,
    /// the full descriptor data (including tag, extension, length). The first
    /// two bytes are the `tag` and `length`.
    f_data: ?*u8,
    f__gst_reserved: [4]*anyopaque,

    /// Creates a `gstmpegts.Descriptor` with custom `tag` and `data`
    extern fn gst_mpegts_descriptor_from_custom(p_tag: u8, p_data: [*]const u8, p_length: usize) *gstmpegts.Descriptor;
    pub const fromCustom = gst_mpegts_descriptor_from_custom;

    /// Creates a `gstmpegts.Descriptor` with custom `tag`, `tag_extension` and `data`
    extern fn gst_mpegts_descriptor_from_custom_with_extension(p_tag: u8, p_tag_extension: u8, p_data: [*]const u8, p_length: usize) *gstmpegts.Descriptor;
    pub const fromCustomWithExtension = gst_mpegts_descriptor_from_custom_with_extension;

    /// Creates a `gstmpegts.Descriptor` to be a `GST_MTS_DESC_DVB_NETWORK_NAME`,
    /// with the network name `name`. The data field of the `gstmpegts.Descriptor`
    /// will be allocated, and transferred to the caller.
    extern fn gst_mpegts_descriptor_from_dvb_network_name(p_name: [*:0]const u8) *gstmpegts.Descriptor;
    pub const fromDvbNetworkName = gst_mpegts_descriptor_from_dvb_network_name;

    /// Fills a `gstmpegts.Descriptor` to be a `GST_MTS_DESC_DVB_SERVICE`.
    /// The data field of the `gstmpegts.Descriptor` will be allocated,
    /// and transferred to the caller.
    extern fn gst_mpegts_descriptor_from_dvb_service(p_service_type: gstmpegts.DVBServiceType, p_service_name: ?[*:0]const u8, p_service_provider: ?[*:0]const u8) *gstmpegts.Descriptor;
    pub const fromDvbService = gst_mpegts_descriptor_from_dvb_service;

    extern fn gst_mpegts_descriptor_from_dvb_subtitling(p_lang: [*:0]const u8, p_type: u8, p_composition: u16, p_ancillary: u16) *gstmpegts.Descriptor;
    pub const fromDvbSubtitling = gst_mpegts_descriptor_from_dvb_subtitling;

    /// Creates a `GST_MTS_DESC_ISO_639_LANGUAGE` `gstmpegts.Descriptor` with
    /// a single language
    extern fn gst_mpegts_descriptor_from_iso_639_language(p_language: [*:0]const u8) *gstmpegts.Descriptor;
    pub const fromIso639Language = gst_mpegts_descriptor_from_iso_639_language;

    /// Creates a `GST_MTS_DESC_REGISTRATION` `gstmpegts.Descriptor`
    extern fn gst_mpegts_descriptor_from_registration(p_format_identifier: [*:0]const u8, p_additional_info: ?[*]u8, p_additional_info_length: usize) *gstmpegts.Descriptor;
    pub const fromRegistration = gst_mpegts_descriptor_from_registration;

    extern fn gst_mpegts_descriptor_parse_audio_preselection_dump(p_source: *gstmpegts.AudioPreselectionDescriptor) void;
    pub const parseAudioPreselectionDump = gst_mpegts_descriptor_parse_audio_preselection_dump;

    extern fn gst_mpegts_descriptor_parse_audio_preselection_free(p_source: *gstmpegts.AudioPreselectionDescriptor) void;
    pub const parseAudioPreselectionFree = gst_mpegts_descriptor_parse_audio_preselection_free;

    /// Frees `desc`
    extern fn gst_mpegts_descriptor_free(p_desc: *Descriptor) void;
    pub const free = gst_mpegts_descriptor_free;

    /// Parses out a list of audio preselection from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_audio_preselection_list(p_descriptor: *const Descriptor, p_list: **glib.PtrArray) c_int;
    pub const parseAudioPreselectionList = gst_mpegts_descriptor_parse_audio_preselection_list;

    /// Extracts the Conditional Access information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_ca(p_descriptor: *Descriptor, p_ca_system_id: *u16, p_ca_pid: *u16, p_private_data: ?[*]*const u8, p_private_data_size: ?*usize) c_int;
    pub const parseCa = gst_mpegts_descriptor_parse_ca;

    /// Extracts the cable delivery system information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_cable_delivery_system(p_descriptor: *const Descriptor, p_res: *gstmpegts.CableDeliverySystemDescriptor) c_int;
    pub const parseCableDeliverySystem = gst_mpegts_descriptor_parse_cable_delivery_system;

    /// Extracts the bouquet name from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_bouquet_name(p_descriptor: *const Descriptor, p_bouquet_name: ?*[*:0]u8) c_int;
    pub const parseDvbBouquetName = gst_mpegts_descriptor_parse_dvb_bouquet_name;

    /// Extracts ca id's from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_ca_identifier(p_descriptor: *const Descriptor, p_list: **glib.Array) c_int;
    pub const parseDvbCaIdentifier = gst_mpegts_descriptor_parse_dvb_ca_identifier;

    /// Extracts the DVB component information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_component(p_descriptor: *const Descriptor, p_res: **gstmpegts.ComponentDescriptor) c_int;
    pub const parseDvbComponent = gst_mpegts_descriptor_parse_dvb_component;

    /// Extracts the DVB content information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_content(p_descriptor: *const Descriptor, p_content: **glib.PtrArray) c_int;
    pub const parseDvbContent = gst_mpegts_descriptor_parse_dvb_content;

    /// Parses out the data broadcast from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_data_broadcast(p_descriptor: *const Descriptor, p_res: **gstmpegts.DataBroadcastDescriptor) c_int;
    pub const parseDvbDataBroadcast = gst_mpegts_descriptor_parse_dvb_data_broadcast;

    /// Parses out the data broadcast id from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_data_broadcast_id(p_descriptor: *const Descriptor, p_data_broadcast_id: *u16, p_id_selector_bytes: [*]*u8, p_len: *u8) c_int;
    pub const parseDvbDataBroadcastId = gst_mpegts_descriptor_parse_dvb_data_broadcast_id;

    /// Extracts the DVB extended event information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_extended_event(p_descriptor: *const Descriptor, p_res: **gstmpegts.ExtendedEventDescriptor) c_int;
    pub const parseDvbExtendedEvent = gst_mpegts_descriptor_parse_dvb_extended_event;

    /// Parses out a list of frequencies from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_frequency_list(p_descriptor: *const Descriptor, p_offset: *c_int, p_list: **glib.Array) c_int;
    pub const parseDvbFrequencyList = gst_mpegts_descriptor_parse_dvb_frequency_list;

    /// Extracts the DVB linkage information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_linkage(p_descriptor: *const Descriptor, p_res: **gstmpegts.DVBLinkageDescriptor) c_int;
    pub const parseDvbLinkage = gst_mpegts_descriptor_parse_dvb_linkage;

    /// Parses out the multilingual bouquet name from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_multilingual_bouquet_name(p_descriptor: *const Descriptor, p_bouquet_name_items: **glib.PtrArray) c_int;
    pub const parseDvbMultilingualBouquetName = gst_mpegts_descriptor_parse_dvb_multilingual_bouquet_name;

    /// Parses out the multilingual component from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_multilingual_component(p_descriptor: *const Descriptor, p_component_tag: *u8, p_component_description_items: **glib.PtrArray) c_int;
    pub const parseDvbMultilingualComponent = gst_mpegts_descriptor_parse_dvb_multilingual_component;

    /// Parses out the multilingual network name from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_multilingual_network_name(p_descriptor: *const Descriptor, p_network_name_items: **glib.PtrArray) c_int;
    pub const parseDvbMultilingualNetworkName = gst_mpegts_descriptor_parse_dvb_multilingual_network_name;

    /// Parses out the multilingual service name from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_multilingual_service_name(p_descriptor: *const Descriptor, p_service_name_items: **glib.PtrArray) c_int;
    pub const parseDvbMultilingualServiceName = gst_mpegts_descriptor_parse_dvb_multilingual_service_name;

    /// Parses out the dvb network name from the `descriptor`:
    extern fn gst_mpegts_descriptor_parse_dvb_network_name(p_descriptor: *const Descriptor, p_name: *[*:0]u8) c_int;
    pub const parseDvbNetworkName = gst_mpegts_descriptor_parse_dvb_network_name;

    /// Extracts the DVB parental rating information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_parental_rating(p_descriptor: *const Descriptor, p_rating: **glib.PtrArray) c_int;
    pub const parseDvbParentalRating = gst_mpegts_descriptor_parse_dvb_parental_rating;

    /// Parses out the private data specifier from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_private_data_specifier(p_descriptor: *const Descriptor, p_private_data_specifier: *u32, p_private_data: ?[*]*u8, p_length: ?*u8) c_int;
    pub const parseDvbPrivateDataSpecifier = gst_mpegts_descriptor_parse_dvb_private_data_specifier;

    /// Parses out the scrambling mode from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_scrambling(p_descriptor: *const Descriptor, p_scrambling_mode: *gstmpegts.DVBScramblingModeType) c_int;
    pub const parseDvbScrambling = gst_mpegts_descriptor_parse_dvb_scrambling;

    /// Extracts the dvb service information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_service(p_descriptor: *const Descriptor, p_service_type: ?*gstmpegts.DVBServiceType, p_service_name: ?*[*:0]u8, p_provider_name: ?*[*:0]u8) c_int;
    pub const parseDvbService = gst_mpegts_descriptor_parse_dvb_service;

    /// Parses out a list of services from the `descriptor`:
    extern fn gst_mpegts_descriptor_parse_dvb_service_list(p_descriptor: *const Descriptor, p_list: **glib.PtrArray) c_int;
    pub const parseDvbServiceList = gst_mpegts_descriptor_parse_dvb_service_list;

    /// Extracts the DVB short event information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_short_event(p_descriptor: *const Descriptor, p_language_code: ?*[*:0]u8, p_event_name: ?*[*:0]u8, p_text: ?*[*:0]u8) c_int;
    pub const parseDvbShortEvent = gst_mpegts_descriptor_parse_dvb_short_event;

    /// Extracts the component tag from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_stream_identifier(p_descriptor: *const Descriptor, p_component_tag: *u8) c_int;
    pub const parseDvbStreamIdentifier = gst_mpegts_descriptor_parse_dvb_stream_identifier;

    /// Parses out the stuffing bytes from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_stuffing(p_descriptor: *const Descriptor, p_stuffing_bytes: **u8) c_int;
    pub const parseDvbStuffing = gst_mpegts_descriptor_parse_dvb_stuffing;

    /// Extracts the DVB subtitling informatio from specific table id in `descriptor`.
    ///
    /// Note: Use `gst_tag_get_language_code` if you want to get the the
    /// ISO 639-1 language code from the returned ISO 639-2 one.
    extern fn gst_mpegts_descriptor_parse_dvb_subtitling_idx(p_descriptor: *const Descriptor, p_idx: c_uint, p_lang: *[*:0]u8, p_type: ?*u8, p_composition_page_id: ?*u16, p_ancillary_page_id: ?*u16) c_int;
    pub const parseDvbSubtitlingIdx = gst_mpegts_descriptor_parse_dvb_subtitling_idx;

    extern fn gst_mpegts_descriptor_parse_dvb_subtitling_nb(p_descriptor: *const Descriptor) c_uint;
    pub const parseDvbSubtitlingNb = gst_mpegts_descriptor_parse_dvb_subtitling_nb;

    /// Parses out the DVB-T2 delivery system from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_dvb_t2_delivery_system(p_descriptor: *const Descriptor, p_res: **gstmpegts.T2DeliverySystemDescriptor) c_int;
    pub const parseDvbT2DeliverySystem = gst_mpegts_descriptor_parse_dvb_t2_delivery_system;

    /// Parses teletext number `idx` in the `descriptor`. The language is in ISO639 format.
    extern fn gst_mpegts_descriptor_parse_dvb_teletext_idx(p_descriptor: *const Descriptor, p_idx: c_uint, p_language_code: ?*[*:0]u8, p_teletext_type: ?*gstmpegts.DVBTeletextType, p_magazine_number: ?*u8, p_page_number: ?*u8) c_int;
    pub const parseDvbTeletextIdx = gst_mpegts_descriptor_parse_dvb_teletext_idx;

    /// Find the number of teletext entries in `descriptor`
    extern fn gst_mpegts_descriptor_parse_dvb_teletext_nb(p_descriptor: *const Descriptor) c_uint;
    pub const parseDvbTeletextNb = gst_mpegts_descriptor_parse_dvb_teletext_nb;

    /// Extracts the iso 639-2 language information from `descriptor`.
    ///
    /// Note: Use `gst_tag_get_language_code` if you want to get the the
    /// ISO 639-1 language code from the returned ISO 639-2 one.
    extern fn gst_mpegts_descriptor_parse_iso_639_language(p_descriptor: *const Descriptor, p_res: **gstmpegts.ISO639LanguageDescriptor) c_int;
    pub const parseIso639Language = gst_mpegts_descriptor_parse_iso_639_language;

    /// Extracts the iso 639-2 language information from specific table id in `descriptor`.
    ///
    /// Note: Use `gst_tag_get_language_code` if you want to get the the
    /// ISO 639-1 language code from the returned ISO 639-2 one.
    extern fn gst_mpegts_descriptor_parse_iso_639_language_idx(p_descriptor: *const Descriptor, p_idx: c_uint, p_lang: *[*:0]u8, p_audio_type: ?*gstmpegts.Iso639AudioType) c_int;
    pub const parseIso639LanguageIdx = gst_mpegts_descriptor_parse_iso_639_language_idx;

    extern fn gst_mpegts_descriptor_parse_iso_639_language_nb(p_descriptor: *const Descriptor) c_uint;
    pub const parseIso639LanguageNb = gst_mpegts_descriptor_parse_iso_639_language_nb;

    /// Extracts the logical channels from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_logical_channel(p_descriptor: *const Descriptor, p_res: *gstmpegts.LogicalChannelDescriptor) c_int;
    pub const parseLogicalChannel = gst_mpegts_descriptor_parse_logical_channel;

    /// Parses out the metadata descriptor from the `descriptor`.
    ///
    /// See ISO/IEC 13818-1:2018 Section 2.6.60 and 2.6.61 for details.
    /// metadata_application_format is provided in Table 2-82. metadata_format is
    /// provided in Table 2-85.
    extern fn gst_mpegts_descriptor_parse_metadata(p_descriptor: *const Descriptor, p_res: **gstmpegts.MetadataDescriptor) c_int;
    pub const parseMetadata = gst_mpegts_descriptor_parse_metadata;

    /// Extracts the metadata STD descriptor from `descriptor`.
    ///
    /// See ISO/IEC 13818-1:2018 Section 2.6.62 and 2.6.63 for details.
    extern fn gst_mpegts_descriptor_parse_metadata_std(p_descriptor: *const Descriptor, p_metadata_input_leak_rate: *u32, p_metadata_buffer_size: *u32, p_metadata_output_leak_rate: *u32) c_int;
    pub const parseMetadataStd = gst_mpegts_descriptor_parse_metadata_std;

    /// Extracts the Registration information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_registration(p_descriptor: *Descriptor, p_registration_id: *u32, p_additional_info: ?[*]*u8, p_additional_info_length: ?*usize) c_int;
    pub const parseRegistration = gst_mpegts_descriptor_parse_registration;

    /// Extracts the satellite delivery system information from `descriptor`.
    extern fn gst_mpegts_descriptor_parse_satellite_delivery_system(p_descriptor: *const Descriptor, p_res: *gstmpegts.SatelliteDeliverySystemDescriptor) c_int;
    pub const parseSatelliteDeliverySystem = gst_mpegts_descriptor_parse_satellite_delivery_system;

    /// Parses out the terrestrial delivery system from the `descriptor`.
    extern fn gst_mpegts_descriptor_parse_terrestrial_delivery_system(p_descriptor: *const Descriptor, p_res: *gstmpegts.TerrestrialDeliverySystemDescriptor) c_int;
    pub const parseTerrestrialDeliverySystem = gst_mpegts_descriptor_parse_terrestrial_delivery_system;

    extern fn gst_mpegts_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_descriptor_get_type;
};

/// a multilingual bouquet name entry
pub const DvbMultilingualBouquetNameItem = extern struct {
    /// the ISO 639 language code
    f_language_code: ?[*:0]u8,
    /// the bouquet name
    f_bouquet_name: ?[*:0]u8,

    extern fn gst_mpegts_dvb_multilingual_bouquet_name_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_multilingual_bouquet_name_item_get_type;
};

pub const DvbMultilingualComponentItem = extern struct {
    /// the ISO 639 language code
    f_language_code: ?[*:0]u8,
    /// the component description
    f_description: ?[*:0]u8,

    extern fn gst_mpegts_dvb_multilingual_component_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_multilingual_component_item_get_type;
};

/// a multilingual network name entry
pub const DvbMultilingualNetworkNameItem = extern struct {
    /// the ISO 639 language code
    f_language_code: ?[*:0]u8,
    /// the network name
    f_network_name: ?[*:0]u8,

    extern fn gst_mpegts_dvb_multilingual_network_name_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_multilingual_network_name_item_get_type;
};

/// a multilingual service name entry
pub const DvbMultilingualServiceNameItem = extern struct {
    /// the ISO 639 language code
    f_language_code: ?[*:0]u8,
    /// the provider name
    f_provider_name: ?[*:0]u8,
    /// the service name
    f_service_name: ?[*:0]u8,

    extern fn gst_mpegts_dvb_multilingual_service_name_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_dvb_multilingual_service_name_item_get_type;
};

/// Event Information Table (EN 300 468)
pub const EIT = extern struct {
    f_transport_stream_id: u16,
    f_original_network_id: u16,
    f_segment_last_section_number: u8,
    f_last_table_id: u8,
    f_actual_stream: c_int,
    f_present_following: c_int,
    /// List of events
    f_events: ?*glib.PtrArray,

    extern fn gst_mpegts_eit_get_type() usize;
    pub const getGObjectType = gst_mpegts_eit_get_type;
};

/// Event from a `GstMpegtsEIT`
pub const EITEvent = extern struct {
    f_event_id: u16,
    f_start_time: ?*gst.DateTime,
    f_duration: u32,
    f_running_status: gstmpegts.RunningStatus,
    f_free_CA_mode: c_int,
    /// List of descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_eit_event_get_type() usize;
    pub const getGObjectType = gst_mpegts_eit_event_get_type;
};

/// Extended Event Descriptor (EN 300 468 v.1.13.1)
pub const ExtendedEventDescriptor = extern struct {
    f_descriptor_number: u8,
    f_last_descriptor_number: u8,
    /// NULL terminated language code.
    f_language_code: ?[*:0]u8,
    /// the `gstmpegts.ExtendedEventItem`
    f_items: ?*glib.PtrArray,
    f_text: ?[*:0]u8,

    extern fn gst_mpegts_extended_event_descriptor_free(p_source: *ExtendedEventDescriptor) void;
    pub const free = gst_mpegts_extended_event_descriptor_free;

    extern fn gst_mpegts_extended_event_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_extended_event_descriptor_get_type;
};

pub const ExtendedEventItem = extern struct {
    f_item_description: ?[*:0]u8,
    f_item: ?[*:0]u8,

    extern fn gst_mpegts_extended_event_item_get_type() usize;
    pub const getGObjectType = gst_mpegts_extended_event_item_get_type;
};

pub const ISO639LanguageDescriptor = extern struct {
    f_nb_language: c_uint,
    f_language: [64][*:0]u8,
    f_audio_type: [64]gstmpegts.Iso639AudioType,

    extern fn gst_mpegts_iso_639_language_descriptor_free(p_desc: *ISO639LanguageDescriptor) void;
    pub const descriptorFree = gst_mpegts_iso_639_language_descriptor_free;

    extern fn gst_mpegts_iso_639_language_get_type() usize;
    pub const getGObjectType = gst_mpegts_iso_639_language_get_type;
};

pub const LogicalChannel = extern struct {
    f_service_id: u16,
    f_visible_service: c_int,
    f_logical_channel_number: u16,

    extern fn gst_mpegts_logical_channel_get_type() usize;
    pub const getGObjectType = gst_mpegts_logical_channel_get_type;
};

pub const LogicalChannelDescriptor = extern struct {
    f_nb_channels: c_uint,
    f_channels: [64]gstmpegts.LogicalChannel,

    extern fn gst_mpegts_logical_channel_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_logical_channel_descriptor_get_type;
};

/// The metadata descriptor specifies parameters of a metadata service carried in an MPEG-2 Transport Stream (or Program Stream). The descriptor is included in the PMT in the descriptor loop for the elementary stream that carries the
/// metadata service. The descriptor specifies the format of the associated metadata, and contains the value of the
/// metadata_service_id to identify the metadata service to which the metadata descriptor applies.
///
/// Note that this structure does not include all of the metadata_descriptor items, and will need extension to support DSM-CC and private data.
/// See ISO/IEC 13818-1:2018 Section 2.6.60 and Section 2.6.61 for more information.
pub const MetadataDescriptor = extern struct {
    /// specifies the application responsible for defining usage, syntax and semantics
    f_metadata_application_format: u16,
    /// indicates the format and coding of the metadata
    f_metadata_format: gstmpegts.MetadataFormat,
    /// format identifier (equivalent to registration descriptor), for example 0x4B4C4641 ('KLVA') to indicate SMPTE 336 KLV.
    f_metadata_format_identifier: u32,
    /// metadata service to which this metadata descriptor applies, typically 0x00
    f_metadata_service_id: u8,
    /// decoder flags, see ISO/IEC 13818-1:2018 Table 2-88.
    f_decoder_config_flags: u8,
    /// true if stream associated with this descriptor is in an ISO/IEC 13818-6 data or object carousel.
    f_dsm_cc_flag: c_int,

    extern fn gst_mpegts_metadata_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_metadata_descriptor_get_type;
};

/// Network Information Table (ISO/IEC 13818-1 / EN 300 468)
pub const NIT = extern struct {
    /// Whether this NIT corresponds to the actual stream
    f_actual_network: c_int,
    /// ID of the network that this NIT describes
    f_network_id: u16,
    /// the global descriptors
    f_descriptors: ?*glib.PtrArray,
    /// the streams
    f_streams: ?*glib.PtrArray,

    /// Allocates and initializes a `gstmpegts.NIT`.
    extern fn gst_mpegts_nit_new() *gstmpegts.NIT;
    pub const new = gst_mpegts_nit_new;

    extern fn gst_mpegts_nit_get_type() usize;
    pub const getGObjectType = gst_mpegts_nit_get_type;
};

pub const NITStream = extern struct {
    f_transport_stream_id: u16,
    f_original_network_id: u16,
    f_descriptors: ?*glib.PtrArray,

    /// Allocates and initializes a `gstmpegts.NITStream`
    extern fn gst_mpegts_nit_stream_new() *gstmpegts.NITStream;
    pub const new = gst_mpegts_nit_stream_new;

    extern fn gst_mpegts_nit_stream_get_type() usize;
    pub const getGObjectType = gst_mpegts_nit_stream_get_type;
};

/// Extra buffer metadata describing the PES Metadata context.
/// This is based on the Metadata AU cell header in
/// ISO/IEC 13818-1:2018 Section 2.12.4.
///
/// AU_cell_data_length is not provided, since it matches the length of the buffer
pub const PESMetadataMeta = extern struct {
    /// parent `gst.Meta`
    f_meta: gst.Meta,
    /// metadata service identifier
    f_metadata_service_id: u8,
    /// bit flags, see spec for details
    f_flags: u8,

    /// Gets the global `gst.MetaInfo` describing the `gstmpegts.PESMetadataMeta` meta.
    extern fn gst_mpegts_pes_metadata_meta_get_info() *const gst.MetaInfo;
    pub const getInfo = gst_mpegts_pes_metadata_meta_get_info;
};

/// Program Map Table (ISO/IEC 13818-1). Provides the mappings between program
/// numbers and the program elements that comprise them.
///
/// The program_number is contained in the subtable_extension field of the
/// container `gstmpegts.Section`.
pub const PMT = extern struct {
    /// PID of the stream containing the PCR for this program.
    f_pcr_pid: u16,
    /// The program to which this PMT is applicable.
    f_program_number: u16,
    /// Array of `gstmpegts.Descriptor`
    f_descriptors: ?*glib.PtrArray,
    /// Array of `gstmpegts.PMTStream`
    f_streams: ?*glib.PtrArray,

    /// Allocates and initializes a new `gstmpegts.PMT`. `gstmpegts.PMTStream` can be
    /// added to the streams array, and global PMT `gstmpegts.Descriptor` to the
    /// descriptors array.
    extern fn gst_mpegts_pmt_new() *gstmpegts.PMT;
    pub const new = gst_mpegts_pmt_new;

    extern fn gst_mpegts_pmt_get_type() usize;
    pub const getGObjectType = gst_mpegts_pmt_get_type;
};

/// An individual stream definition of a `gstmpegts.PMT`.
pub const PMTStream = extern struct {
    /// the type of stream. See `gstmpegts.StreamType`
    f_stream_type: u8,
    /// the PID of the stream
    f_pid: u16,
    /// the descriptors of the
    /// stream
    f_descriptors: ?*glib.PtrArray,

    /// Allocates and initializes a new `gstmpegts.PMTStream`.
    extern fn gst_mpegts_pmt_stream_new() *gstmpegts.PMTStream;
    pub const new = gst_mpegts_pmt_stream_new;

    extern fn gst_mpegts_pmt_stream_get_type() usize;
    pub const getGObjectType = gst_mpegts_pmt_stream_get_type;
};

/// A program entry from a Program Association Table (ITU H.222.0, ISO/IEC 13818-1).
pub const PatProgram = extern struct {
    /// the program number
    f_program_number: u16,
    /// the network of program map PID
    f_network_or_program_map_PID: u16,

    /// Allocates a new `gstmpegts.PatProgram`.
    extern fn gst_mpegts_pat_program_new() *gstmpegts.PatProgram;
    pub const new = gst_mpegts_pat_program_new;

    extern fn gst_mpegts_pat_program_get_type() usize;
    pub const getGObjectType = gst_mpegts_pat_program_get_type;
};

pub const SCTESIT = extern struct {
    f_encrypted_packet: c_int,
    f_encryption_algorithm: u8,
    f_pts_adjustment: u64,
    f_cw_index: u8,
    f_tier: u16,
    f_splice_command_length: u16,
    f_splice_command_type: gstmpegts.SCTESpliceCommandType,
    f_splice_time_specified: c_int,
    f_splice_time: u64,
    f_splices: ?*glib.PtrArray,
    f_descriptors: ?*glib.PtrArray,
    /// When encrypted, or when encountering an unknown command type,
    /// we may still want to pass the sit through.
    f_fully_parsed: c_int,
    /// When the SIT was constructed by the application, splice times
    /// are in running_time and must be translated before packetizing.
    f_is_running_time: c_int,

    /// Allocates and initializes a `gstmpegts.SCTESIT`.
    extern fn gst_mpegts_scte_sit_new() *gstmpegts.SCTESIT;
    pub const new = gst_mpegts_scte_sit_new;

    extern fn gst_mpegts_scte_sit_get_type() usize;
    pub const getGObjectType = gst_mpegts_scte_sit_get_type;
};

/// Per-PID splice information.
pub const SCTESpliceComponent = extern struct {
    /// the elementary PID stream containing the Splice Point
    f_tag: u8,
    /// Whether `splice_time` was specified
    f_splice_time_specified: c_int,
    /// the presentation time of the signaled splice event
    f_splice_time: u64,
    /// The UTC time of the signaled splice event
    f_utc_splice_time: u32,

    /// Allocates and initializes a `gstmpegts.SCTESpliceComponent`.
    extern fn gst_mpegts_scte_splice_component_new(p_tag: u8) *gstmpegts.SCTESpliceComponent;
    pub const new = gst_mpegts_scte_splice_component_new;

    extern fn gst_mpegts_scte_splice_component_get_type() usize;
    pub const getGObjectType = gst_mpegts_scte_splice_component_get_type;
};

pub const SCTESpliceEvent = extern struct {
    f_insert_event: c_int,
    f_splice_event_id: u32,
    f_splice_event_cancel_indicator: c_int,
    f_out_of_network_indicator: c_int,
    f_program_splice_flag: c_int,
    f_duration_flag: c_int,
    f_splice_immediate_flag: c_int,
    f_program_splice_time_specified: c_int,
    f_program_splice_time: u64,
    /// The UTC time of the signaled splice event
    f_utc_splice_time: u32,
    /// Per-PID splice time information
    f_components: ?*glib.PtrArray,
    f_break_duration_auto_return: c_int,
    f_break_duration: u64,
    f_unique_program_id: u16,
    f_avail_num: u8,
    f_avails_expected: u8,

    /// Allocates and initializes a `gstmpegts.SCTESpliceEvent`.
    extern fn gst_mpegts_scte_splice_event_new() *gstmpegts.SCTESpliceEvent;
    pub const new = gst_mpegts_scte_splice_event_new;

    extern fn gst_mpegts_scte_splice_event_get_type() usize;
    pub const getGObjectType = gst_mpegts_scte_splice_event_get_type;
};

/// Service Description Table (EN 300 468)
pub const SDT = extern struct {
    /// Network ID of the originating delivery system
    f_original_network_id: u16,
    /// True if the table describes this transport stream
    f_actual_ts: c_int,
    /// ID of this transport stream
    f_transport_stream_id: u16,
    /// List of services
    f_services: ?*glib.PtrArray,

    /// Allocates and initializes a `gstmpegts.SDT`.
    extern fn gst_mpegts_sdt_new() *gstmpegts.SDT;
    pub const new = gst_mpegts_sdt_new;

    extern fn gst_mpegts_sdt_get_type() usize;
    pub const getGObjectType = gst_mpegts_sdt_get_type;
};

pub const SDTService = extern struct {
    /// The program number this table belongs to
    f_service_id: u16,
    /// EIT schedule information is present in this transport stream
    f_EIT_schedule_flag: c_int,
    /// EIT present/following information is present in this transport stream
    f_EIT_present_following_flag: c_int,
    /// Status of this service
    f_running_status: gstmpegts.RunningStatus,
    /// True if one or more streams is controlled by a CA system
    f_free_CA_mode: c_int,
    /// List of descriptors
    f_descriptors: ?*glib.PtrArray,

    /// Allocates and initializes a `gstmpegts.SDTService`.
    extern fn gst_mpegts_sdt_service_new() *gstmpegts.SDTService;
    pub const new = gst_mpegts_sdt_service_new;

    extern fn gst_mpegts_sdt_service_get_type() usize;
    pub const getGObjectType = gst_mpegts_sdt_service_get_type;
};

/// Selection Information Table (EN 300 468)
pub const SIT = extern struct {
    /// List of descriptors
    f_descriptors: ?*glib.PtrArray,
    /// List of services
    f_services: ?*glib.PtrArray,

    extern fn gst_mpegts_sit_get_type() usize;
    pub const getGObjectType = gst_mpegts_sit_get_type;
};

/// SIT Service entry
pub const SITService = extern struct {
    /// The Program number this table belongs to
    f_service_id: u16,
    /// Status of this service
    f_running_status: gstmpegts.RunningStatus,
    /// List of descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_sit_service_get_type() usize;
    pub const getGObjectType = gst_mpegts_sit_service_get_type;
};

/// Satellite Delivery System Descriptor (EN 300 468 v.1.13.1)
pub const SatelliteDeliverySystemDescriptor = extern struct {
    /// the frequency in kHz (kiloHertz)
    f_frequency: u32,
    /// the orbital position in degrees
    f_orbital_position: f32,
    /// If `TRUE`, the satellite is in the eastern part of the orbit,
    /// else in the western part.
    f_west_east: c_int,
    /// The polarization of the transmitted signal
    f_polarization: gstmpegts.SatellitePolarizationType,
    /// Roll-off factor used in DVB-S2
    f_roll_off: gstmpegts.SatelliteRolloff,
    /// modulation system, `TRUE` if DVB-S2, else DVB-S
    f_modulation_system: c_int,
    /// Modulation scheme used
    f_modulation_type: gstmpegts.ModulationType,
    /// Symbol rate (in symbols per second)
    f_symbol_rate: u32,
    /// inner FEC scheme used
    f_fec_inner: gstmpegts.DVBCodeRate,

    extern fn gst_mpegts_satellite_delivery_system_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_satellite_delivery_system_descriptor_get_type;
};

/// ## Generic usage of sections with %GstMpegtsSection
///
/// The `gstmpegts.Section` object is the representation of MPEG-TS Section (SI or
/// PSI).
///
/// Various elements can post those on the bus via `gst.Message` of type
/// `GST_MESSAGE_ELEMENT`. The `gstmpegts.messageParseMpegtsSection` function
/// provides access to the section.
///
/// Applications (or other elements) can create them either by using one of the
/// `gst_mpegts_section_from_*` functions, or by providing the raw SI data via
/// `gstmpegts.Section.new`.
///
/// Elements outputting MPEG-TS streams can also create sections using the
/// various convenience functions and then get the packetized data (to be
/// inserted in MPEG-TS packets) using `gstmpegts.Section.packetize`.
///
/// For more details, refer to the ITU H.222.0 or ISO/IEC 13818-1 specifications
/// and other specifications mentioned in the documentation.
///
/// # Supported base MPEG-TS sections
/// These are the sections for which parsing and packetizing code exists.
///
/// ## Program Association Table (PAT)
/// See:
/// * `gstmpegts.Section.getPat`
/// * `gstmpegts.PatProgram.new`
/// * `gstmpegts.PatProgram`
///
/// ## Conditional Access Table (CAT)
/// See:
/// * `gstmpegts.Section.getCat`
///
/// ## Program Map Table (PMT)
/// See:
/// * `gstmpegts.PMT`
/// * `gstmpegts.Section.getPmt`
/// * `gstmpegts.PMT.new`
/// * `gstmpegts.PMTStream`
///
/// ## Transport Stream Description Table (TSDT)
/// See:
/// * `gstmpegts.Section.getTsdt`
/// # API
pub const Section = extern struct {
    f_parent: gst.MiniObject,
    /// The type of section.
    f_section_type: gstmpegts.SectionType,
    /// The PID on which this section was found or belongs to.
    f_pid: u16,
    /// The table id of this section. See `gstmpegts.SectionTableID` and
    ///      derivates for more information.
    f_table_id: u8,
    /// This meaning differs per section. See the documentation
    ///      of the parsed section type for the meaning of this field
    f_subtable_extension: u16,
    /// Version of the section.
    f_version_number: u8,
    /// Applies to current/next stream or not
    f_current_next_indicator: c_int,
    /// Number of the section (if multiple)
    f_section_number: u8,
    /// Number of the last expected section (if multiple)
    f_last_section_number: u8,
    /// Checksum (if applicable)
    f_crc: u32,
    f_data: ?*u8,
    f_section_length: c_uint,
    f_cached_parsed: ?*anyopaque,
    f_destroy_parsed: ?glib.DestroyNotify,
    f_offset: u64,
    f_short_section: c_int,
    f_packetizer: ?gstmpegts.PacketizeFunc,
    f__gst_reserved: [4]*anyopaque,

    extern fn gst_mpegts_section_from_atsc_mgt(p_mgt: *gstmpegts.AtscMGT) *gstmpegts.Section;
    pub const fromAtscMgt = gst_mpegts_section_from_atsc_mgt;

    extern fn gst_mpegts_section_from_atsc_rrt(p_rrt: *gstmpegts.AtscRRT) *gstmpegts.Section;
    pub const fromAtscRrt = gst_mpegts_section_from_atsc_rrt;

    extern fn gst_mpegts_section_from_atsc_stt(p_stt: *gstmpegts.AtscSTT) *gstmpegts.Section;
    pub const fromAtscStt = gst_mpegts_section_from_atsc_stt;

    /// Ownership of `nit` is taken. The data in `nit` is managed by the `gstmpegts.Section`
    extern fn gst_mpegts_section_from_nit(p_nit: *gstmpegts.NIT) *gstmpegts.Section;
    pub const fromNit = gst_mpegts_section_from_nit;

    /// Creates a PAT `gstmpegts.Section` from the `programs` array of `GstMpegtsPatPrograms`
    extern fn gst_mpegts_section_from_pat(p_programs: *glib.PtrArray, p_ts_id: u16) *gstmpegts.Section;
    pub const fromPat = gst_mpegts_section_from_pat;

    /// Creates a `gstmpegts.Section` from `pmt` that is bound to `pid`
    extern fn gst_mpegts_section_from_pmt(p_pmt: *gstmpegts.PMT, p_pid: u16) *gstmpegts.Section;
    pub const fromPmt = gst_mpegts_section_from_pmt;

    /// Ownership of `sit` is taken. The data in `sit` is managed by the `gstmpegts.Section`
    extern fn gst_mpegts_section_from_scte_sit(p_sit: *gstmpegts.SCTESIT, p_pid: u16) *gstmpegts.Section;
    pub const fromScteSit = gst_mpegts_section_from_scte_sit;

    /// Ownership of `sdt` is taken. The data in `sdt` is managed by the `gstmpegts.Section`
    extern fn gst_mpegts_section_from_sdt(p_sdt: *gstmpegts.SDT) *gstmpegts.Section;
    pub const fromSdt = gst_mpegts_section_from_sdt;

    /// Creates a new `gstmpegts.Section` from the provided `data`.
    ///
    /// Note: Ensuring `data` is big enough to contain the full section is the
    /// responsibility of the caller. If it is not big enough, `NULL` will be
    /// returned.
    ///
    /// Note: it is the responsibility of the caller to ensure `data` does point
    /// to the beginning of the section.
    extern fn gst_mpegts_section_new(p_pid: u16, p_data: [*]u8, p_data_size: usize) *gstmpegts.Section;
    pub const new = gst_mpegts_section_new;

    /// Returns the `gstmpegts.AtscVCT` contained in the `section`
    extern fn gst_mpegts_section_get_atsc_cvct(p_section: *Section) *const gstmpegts.AtscVCT;
    pub const getAtscCvct = gst_mpegts_section_get_atsc_cvct;

    /// Returns the `gstmpegts.AtscEIT` contained in the `section`.
    extern fn gst_mpegts_section_get_atsc_eit(p_section: *Section) *const gstmpegts.AtscEIT;
    pub const getAtscEit = gst_mpegts_section_get_atsc_eit;

    /// Returns the `gstmpegts.AtscETT` contained in the `section`.
    extern fn gst_mpegts_section_get_atsc_ett(p_section: *Section) *const gstmpegts.AtscETT;
    pub const getAtscEtt = gst_mpegts_section_get_atsc_ett;

    /// Returns the `gstmpegts.AtscMGT` contained in the `section`.
    extern fn gst_mpegts_section_get_atsc_mgt(p_section: *Section) *const gstmpegts.AtscMGT;
    pub const getAtscMgt = gst_mpegts_section_get_atsc_mgt;

    /// Returns the `gstmpegts.AtscRRT` contained in the `section`.
    extern fn gst_mpegts_section_get_atsc_rrt(p_section: *Section) *const gstmpegts.AtscRRT;
    pub const getAtscRrt = gst_mpegts_section_get_atsc_rrt;

    /// Returns the `gstmpegts.AtscSTT` contained in the `section`.
    extern fn gst_mpegts_section_get_atsc_stt(p_section: *Section) *const gstmpegts.AtscSTT;
    pub const getAtscStt = gst_mpegts_section_get_atsc_stt;

    /// Returns the `gstmpegts.AtscVCT` contained in the `section`
    extern fn gst_mpegts_section_get_atsc_tvct(p_section: *Section) *const gstmpegts.AtscVCT;
    pub const getAtscTvct = gst_mpegts_section_get_atsc_tvct;

    /// Returns the `gstmpegts.BAT` contained in the `section`.
    extern fn gst_mpegts_section_get_bat(p_section: *Section) *const gstmpegts.BAT;
    pub const getBat = gst_mpegts_section_get_bat;

    /// Parses a Conditional Access Table.
    ///
    /// Returns the array of `gstmpegts.Descriptor` contained in the Conditional
    /// Access Table.
    extern fn gst_mpegts_section_get_cat(p_section: *Section) *glib.PtrArray;
    pub const getCat = gst_mpegts_section_get_cat;

    /// Gets the original unparsed section data.
    extern fn gst_mpegts_section_get_data(p_section: *Section) *glib.Bytes;
    pub const getData = gst_mpegts_section_get_data;

    /// Returns the `gstmpegts.EIT` contained in the `section`.
    extern fn gst_mpegts_section_get_eit(p_section: *Section) *const gstmpegts.EIT;
    pub const getEit = gst_mpegts_section_get_eit;

    /// Returns the `gstmpegts.NIT` contained in the `section`.
    extern fn gst_mpegts_section_get_nit(p_section: *Section) *const gstmpegts.NIT;
    pub const getNit = gst_mpegts_section_get_nit;

    /// Parses a Program Association Table (ITU H.222.0, ISO/IEC 13818-1).
    ///
    /// Returns the array of `gstmpegts.PatProgram` contained in the section.
    ///
    /// Note: The PAT `transport_stream_id` field corresponds to the
    /// "subtable_extension" field of the provided `section`.
    extern fn gst_mpegts_section_get_pat(p_section: *Section) *glib.PtrArray;
    pub const getPat = gst_mpegts_section_get_pat;

    /// Parses the Program Map Table contained in the `section`.
    extern fn gst_mpegts_section_get_pmt(p_section: *Section) *const gstmpegts.PMT;
    pub const getPmt = gst_mpegts_section_get_pmt;

    /// Returns the `gstmpegts.SCTESIT` contained in the `section`.
    extern fn gst_mpegts_section_get_scte_sit(p_section: *Section) *const gstmpegts.SCTESIT;
    pub const getScteSit = gst_mpegts_section_get_scte_sit;

    /// Returns the `gstmpegts.SDT` contained in the `section`.
    extern fn gst_mpegts_section_get_sdt(p_section: *Section) *const gstmpegts.SDT;
    pub const getSdt = gst_mpegts_section_get_sdt;

    /// Returns the `gstmpegts.SIT` contained in the `section`.
    extern fn gst_mpegts_section_get_sit(p_section: *Section) *const gstmpegts.SIT;
    pub const getSit = gst_mpegts_section_get_sit;

    /// Returns the `gst.DateTime` of the TDT
    extern fn gst_mpegts_section_get_tdt(p_section: *Section) *gst.DateTime;
    pub const getTdt = gst_mpegts_section_get_tdt;

    /// Returns the `gstmpegts.TOT` contained in the `section`.
    extern fn gst_mpegts_section_get_tot(p_section: *Section) *const gstmpegts.TOT;
    pub const getTot = gst_mpegts_section_get_tot;

    /// Parses a Transport Stream Description Table.
    ///
    /// Returns the array of `gstmpegts.Descriptor` contained in the section
    extern fn gst_mpegts_section_get_tsdt(p_section: *Section) *glib.PtrArray;
    pub const getTsdt = gst_mpegts_section_get_tsdt;

    /// Packetize (i.e. serialize) the `section`. If the data in `section` has already
    /// been packetized, the data pointer is returned immediately. Otherwise, the
    /// data field is allocated and populated.
    extern fn gst_mpegts_section_packetize(p_section: *Section, p_output_size: *usize) *u8;
    pub const packetize = gst_mpegts_section_packetize;

    /// Creates a custom `gst.Event` with a `GstMpegtsSection` and send it the `element`
    /// `gst.Element`.
    extern fn gst_mpegts_section_send_event(p_section: *Section, p_element: *gst.Element) c_int;
    pub const sendEvent = gst_mpegts_section_send_event;

    extern fn gst_mpegts_section_get_type() usize;
    pub const getGObjectType = gst_mpegts_section_get_type;
};

pub const T2DeliverySystemCell = extern struct {
    /// id of the cell
    f_cell_id: u16,
    /// centre frequencies in Hz
    f_centre_frequencies: ?*glib.Array,
    f_sub_cells: ?*glib.PtrArray,

    extern fn gst_mpegts_t2_delivery_system_cell_get_type() usize;
    pub const getGObjectType = gst_mpegts_t2_delivery_system_cell_get_type;
};

pub const T2DeliverySystemCellExtension = extern struct {
    /// id of the sub cell
    f_cell_id_extension: u8,
    /// centre frequency of the sub cell in Hz
    f_transposer_frequency: u32,

    extern fn gst_mpegts_t2_delivery_system_cell_extension_get_type() usize;
    pub const getGObjectType = gst_mpegts_t2_delivery_system_cell_extension_get_type;
};

/// describe DVB-T2 transmissions according to EN 302 755
pub const T2DeliverySystemDescriptor = extern struct {
    f_plp_id: u8,
    f_t2_system_id: u16,
    f_siso_miso: u8,
    f_bandwidth: u32,
    f_guard_interval: gstmpegts.TerrestrialGuardInterval,
    f_transmission_mode: gstmpegts.TerrestrialTransmissionMode,
    f_other_frequency: c_int,
    f_tfs: c_int,
    f_cells: ?*glib.PtrArray,

    extern fn gst_mpegts_t2_delivery_system_descriptor_free(p_source: *T2DeliverySystemDescriptor) void;
    pub const free = gst_mpegts_t2_delivery_system_descriptor_free;

    extern fn gst_mpegts_t2_delivery_system_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_t2_delivery_system_descriptor_get_type;
};

/// Time Offset Table (EN 300 468)
pub const TOT = extern struct {
    f_utc_time: ?*gst.DateTime,
    /// List of descriptors
    f_descriptors: ?*glib.PtrArray,

    extern fn gst_mpegts_tot_get_type() usize;
    pub const getGObjectType = gst_mpegts_tot_get_type;
};

/// Terrestrial Delivery System Descriptor (EN 300 468 v.1.13.1)
pub const TerrestrialDeliverySystemDescriptor = extern struct {
    /// the frequency in Hz (Hertz)
    f_frequency: u32,
    /// the bandwidth in Hz (Hertz)
    f_bandwidth: u32,
    /// `TRUE` High Priority `FALSE` Low Priority
    f_priority: c_int,
    /// `TRUE` no time slicing `FALSE` time slicing
    f_time_slicing: c_int,
    /// `TRUE` no mpe-fec is used `FALSE` mpe-fec is use
    f_mpe_fec: c_int,
    /// the constellation
    f_constellation: gstmpegts.ModulationType,
    /// the hierarchy
    f_hierarchy: gstmpegts.TerrestrialHierarchy,
    f_code_rate_hp: gstmpegts.DVBCodeRate,
    f_code_rate_lp: gstmpegts.DVBCodeRate,
    f_guard_interval: gstmpegts.TerrestrialGuardInterval,
    f_transmission_mode: gstmpegts.TerrestrialTransmissionMode,
    /// `TRUE` more frequency are use, else not
    f_other_frequency: c_int,

    extern fn gst_mpegts_terrestrial_delivery_system_descriptor_get_type() usize;
    pub const getGObjectType = gst_mpegts_terrestrial_delivery_system_descriptor_get_type;
};

/// These values correspond to the registered descriptor type from
/// the various ATSC specifications.
///
/// Consult the relevant specifications for more details.
pub const ATSCDescriptorType = enum(c_int) {
    stuffing = 128,
    ac3 = 129,
    caption_service = 134,
    content_advisory = 135,
    extended_channel_name = 160,
    service_location = 161,
    time_shifted_service = 162,
    component_name = 163,
    dcc_departing_request = 168,
    dcc_arriving_request = 169,
    redistribution_control = 170,
    genre = 171,
    private_information = 173,
    eac3 = 204,
    enhanced_signaling = 178,
    data_service = 164,
    pid_count = 165,
    download_descriptor = 166,
    multiprotocol_encapsulation = 167,
    module_link = 180,
    crc32 = 181,
    group_link = 184,
    _,
};

/// Type of mpeg-ts streams for ATSC, as defined by the ATSC Code Points
/// Registry. For convenience, some stream types from `gstmpegts.ScteStreamType`
/// are also included.
pub const ATSCStreamType = enum(c_int) {
    dcii_video = 128,
    audio_ac3 = 129,
    subtitling = 130,
    isoch_data = 131,
    sit = 134,
    audio_eac3 = 135,
    audio_dts_hd = 136,
    _,
};

pub const AtscMGTTableType = enum(c_int) {
    eit0 = 256,
    eit127 = 383,
    ett0 = 512,
    ett127 = 639,
    _,
};

pub const CableOuterFECScheme = enum(c_int) {
    undefined = 0,
    none = 1,
    rs_204_188 = 2,
    _,
};

pub const ComponentStreamContent = enum(c_int) {
    mpeg2_video = 1,
    mpeg1_layer2_audio = 2,
    teletext_or_subtitle = 3,
    ac_3 = 4,
    avc = 5,
    aac = 6,
    dts = 7,
    srm_cpcm = 8,
    _,
};

pub const ContentNibbleHi = enum(c_int) {
    movie_drama = 1,
    news_current_affairs = 2,
    show_game_show = 3,
    sports = 4,
    children_youth_program = 5,
    music_ballet_dance = 6,
    arts_culture = 7,
    social_political_economics = 8,
    education_science_factual = 9,
    leisure_hobbies = 10,
    special_characteristics = 11,
    _,
};

pub const DVBCodeRate = enum(c_int) {
    none = 0,
    @"1_2" = 1,
    @"2_3" = 2,
    @"3_4" = 3,
    @"4_5" = 4,
    @"5_6" = 5,
    @"6_7" = 6,
    @"7_8" = 7,
    @"8_9" = 8,
    auto = 9,
    @"3_5" = 10,
    @"9_10" = 11,
    @"2_5" = 12,
    _,
};

/// The type of `gstmpegts.Descriptor`
///
/// These values correspond to the registered descriptor type from
/// the various DVB specifications.
///
/// Consult the relevant specifications for more details.
pub const DVBDescriptorType = enum(c_int) {
    network_name = 64,
    service_list = 65,
    stuffing = 66,
    satellite_delivery_system = 67,
    cable_delivery_system = 68,
    vbi_data = 69,
    vbi_teletext = 70,
    bouquet_name = 71,
    service = 72,
    country_availability = 73,
    linkage = 74,
    nvod_reference = 75,
    time_shifted_service = 76,
    short_event = 77,
    extended_event = 78,
    time_shifted_event = 79,
    component = 80,
    mosaic = 81,
    stream_identifier = 82,
    ca_identifier = 83,
    content = 84,
    parental_rating = 85,
    teletext = 86,
    telephone = 87,
    local_time_offset = 88,
    subtitling = 89,
    terrestrial_delivery_system = 90,
    multilingual_network_name = 91,
    multilingual_bouquet_name = 92,
    multilingual_service_name = 93,
    multilingual_component = 94,
    private_data_specifier = 95,
    service_move = 96,
    short_smoothing_buffer = 97,
    frequency_list = 98,
    partial_transport_stream = 99,
    data_broadcast = 100,
    scrambling = 101,
    data_broadcast_id = 102,
    transport_stream = 103,
    dsng = 104,
    pdc = 105,
    ac3 = 106,
    ancillary_data = 107,
    cell_list = 108,
    cell_frequency_link = 109,
    announcement_support = 110,
    application_signalling = 111,
    adaptation_field_data = 112,
    service_identifier = 113,
    service_availability = 114,
    default_authority = 115,
    related_content = 116,
    tva_id = 117,
    content_identifier = 118,
    timeslice_fec_identifier = 119,
    ecm_repetition_rate = 120,
    s2_satellite_delivery_system = 121,
    enhanced_ac3 = 122,
    dts = 123,
    aac = 124,
    xait_location = 125,
    fta_content_management = 126,
    extension = 127,
    _,
};

/// The type of `gstmpegts.Descriptor`
///
/// These values correspond to the registered extended descriptor
/// type from the various DVB specifications.
///
/// Consult the relevant specifications for more details.
pub const DVBExtendedDescriptorType = enum(c_int) {
    image_icon = 0,
    cpcm_delivery_signalling = 1,
    cp = 2,
    cp_identifier = 3,
    t2_delivery_system = 4,
    sh_delivery_system = 5,
    supplementary_audio = 6,
    network_change_notify = 7,
    message = 8,
    target_region = 9,
    target_region_name = 10,
    service_relocated = 11,
    xait_pid = 12,
    c2_delivery_system = 13,
    dts_hd_audio_stream = 14,
    dts_neutral = 15,
    video_depth_range = 16,
    t2mi = 17,
    uri_linkage = 19,
    ac4 = 21,
    audio_preselection = 25,
    _,
};

pub const DVBLinkageHandOverType = enum(c_int) {
    reserved = 0,
    identical = 1,
    local_variation = 2,
    associated = 3,
    _,
};

/// Linkage Type (EN 300 468 v.1.13.1)
pub const DVBLinkageType = enum(c_int) {
    reserved_00 = 0,
    information = 1,
    epg = 2,
    ca_replacement = 3,
    ts_containing_complete_si = 4,
    service_replacement = 5,
    data_broadcast = 6,
    rcs_map = 7,
    mobile_hand_over = 8,
    system_software_update = 9,
    ts_containing_ssu = 10,
    ip_mac_notification = 11,
    ts_containing_int = 12,
    event = 13,
    extended_event = 14,
    _,
};

pub const DVBScramblingModeType = enum(c_int) {
    reserved = 0,
    csa1 = 1,
    csa2 = 2,
    csa3_standard = 3,
    csa3_minimal_enhanced = 4,
    csa3_full_enhanced = 5,
    cissa = 16,
    atis_0 = 112,
    atis_f = 127,
    _,
};

/// The type of service of a channel.
///
/// As specified in Table 87 of ETSI EN 300 468 v1.13.1
pub const DVBServiceType = enum(c_int) {
    reserved_00 = 0,
    digital_television = 1,
    digital_radio_sound = 2,
    teletext = 3,
    nvod_reference = 4,
    nvod_time_shifted = 5,
    mosaic = 6,
    fm_radio = 7,
    dvb_srm = 8,
    reserved_09 = 9,
    advanced_codec_digital_radio_sound = 10,
    advanced_codec_mosaic = 11,
    data_broadcast = 12,
    reserved_0d_common_interface = 13,
    rcs_map = 14,
    rcs_fls = 15,
    dvb_mhp = 16,
    mpeg2_hd_digital_television = 17,
    advanced_codec_sd_digital_television = 22,
    advanced_codec_sd_nvod_time_shifted = 23,
    advanced_codec_sd_nvod_reference = 24,
    advanced_codec_hd_digital_television = 25,
    advanced_codec_hd_nvod_time_shifted = 26,
    advanced_codec_hd_nvod_reference = 27,
    advanced_codec_stereo_hd_digital_television = 28,
    advanced_codec_stereo_hd_nvod_time_shifted = 29,
    advanced_codec_stereo_hd_nvod_reference = 30,
    reserved_ff = 31,
    _,
};

/// The type of teletext page.
///
/// As specified in Table 100 of ETSI EN 300 468 v1.13.1
pub const DVBTeletextType = enum(c_int) {
    nitial_page = 1,
    ubtitle_page = 2,
    dditional_info_page = 3,
    rogramme_schedule_page = 4,
    earing_impaired_page = 5,
    _,
};

/// The type of `gstmpegts.Descriptor`
///
/// These values correspond to the registered descriptor type from
/// the base MPEG-TS specifications (ITU H.222.0 | ISO/IEC 13818-1).
///
/// Consult the relevant specifications for more details.
pub const DescriptorType = enum(c_int) {
    reserved_00 = 0,
    reserved_01 = 1,
    video_stream = 2,
    audio_stream = 3,
    hierarchy = 4,
    registration = 5,
    data_stream_alignment = 6,
    target_background_grid = 7,
    video_window = 8,
    ca = 9,
    iso_639_language = 10,
    system_clock = 11,
    multiplex_buffer_utilisation = 12,
    copyright = 13,
    maximum_bitrate = 14,
    private_data_indicator = 15,
    smoothing_buffer = 16,
    std = 17,
    ibp = 18,
    dsmcc_carousel_identifier = 19,
    dsmcc_association_tag = 20,
    dsmcc_deferred_association_tag = 21,
    dsmcc_npt_reference = 23,
    dsmcc_npt_endpoint = 24,
    dsmcc_stream_mode = 25,
    dsmcc_stream_event = 26,
    mpeg4_video = 27,
    mpeg4_audio = 28,
    iod = 29,
    sl = 30,
    fmc = 31,
    external_es_id = 32,
    mux_code = 33,
    fmx_buffer_size = 34,
    multiplex_buffer = 35,
    content_labeling = 36,
    metadata_pointer = 37,
    metadata = 38,
    metadata_std = 39,
    avc_video = 40,
    ipmp = 41,
    avc_timing_and_hrd = 42,
    mpeg2_aac_audio = 43,
    flex_mux_timing = 44,
    mpeg4_text = 45,
    mpeg4_audio_extension = 46,
    auxiliary_video_stream = 47,
    svc_extension = 48,
    mvc_extension = 49,
    j2k_video = 50,
    mvc_operation_point = 51,
    mpeg2_stereoscopic_video_format = 52,
    stereoscopic_program_info = 53,
    stereoscopic_video_info = 54,
    _,
};

/// Type of mpeg-ts streams for Blu-ray formats. To be matched with the
/// stream-type of a `gstmpegts.Section`.
pub const HdmvStreamType = enum(c_int) {
    audio_lpcm = 128,
    audio_ac3 = 129,
    audio_dts = 130,
    audio_ac3_true_hd = 131,
    audio_ac3_plus = 132,
    audio_dts_hd = 133,
    audio_dts_hd_master_audio = 134,
    audio_eac3 = 135,
    subpicture_pgs = 144,
    igs = 145,
    subtitle = 146,
    audio_ac3_plus_secondary = 161,
    audio_dts_hd_secondary = 162,
    _,
};

/// These values correspond to the registered descriptor type from
/// the various ISDB specifications.
///
/// Consult the relevant specifications for more details.
pub const ISDBDescriptorType = enum(c_int) {
    hierarchical_transmission = 192,
    digital_copy_control = 193,
    network_identification = 194,
    partial_ts_time = 195,
    audio_component = 196,
    hyperlink = 197,
    target_region = 198,
    data_content = 199,
    video_decode_control = 200,
    download_content = 201,
    ca_emm_ts = 202,
    ca_contract_information = 203,
    ca_service = 204,
    ts_information = 205,
    extended_broadcaster = 206,
    logo_transmission = 207,
    basic_local_event = 208,
    reference = 209,
    node_relation = 210,
    short_node_information = 211,
    stc_reference = 212,
    series = 213,
    event_group = 214,
    si_parameter = 215,
    broadcaster_name = 216,
    component_group = 217,
    si_prime_ts = 218,
    board_information = 219,
    ldt_linkage = 220,
    connected_transmission = 221,
    content_availability = 222,
    service_group = 224,
    _,
};

pub const Iso639AudioType = enum(c_int) {
    undefined = 0,
    clean_effects = 1,
    hearing_impaired = 2,
    visual_impaired_commentary = 3,
    _,
};

/// metadata_descriptor metadata_format valid values. See ISO/IEC 13818-1:2018(E) Table 2-85.
pub const MetadataFormat = enum(c_int) {
    tem = 16,
    bim = 17,
    application_format = 63,
    identifier_field = 255,
    _,
};

/// The type of `gstmpegts.Descriptor`
///
/// These values correspond to miscellaneous descriptor types that are
/// not yet identified from known specifications.
pub const MiscDescriptorType = enum(c_int) {
    mts_desc_dtg_logical_channel = 131,
    _,
};

pub const ModulationType = enum(c_int) {
    qpsk = 0,
    qam_16 = 1,
    qam_32 = 2,
    qam_64 = 3,
    qam_128 = 4,
    qam_256 = 5,
    qam_auto = 6,
    vsb_8 = 7,
    vsb_16 = 8,
    psk_8 = 9,
    apsk_16 = 10,
    apsk_32 = 11,
    dqpsk = 12,
    qam_4_nr_ = 13,
    none = 14,
    _,
};

/// Running status of a service.
///
/// Corresponds to table 6 of ETSI EN 300 468 (v1.13.0)
pub const RunningStatus = enum(c_int) {
    undefined = 0,
    not_running = 1,
    starts_in_few_seconds = 2,
    pausing = 3,
    running = 4,
    off_air = 5,
    _,
};

/// These values correspond to the ones defined by SCTE (amongst other in ANSI/SCTE 57)
pub const SCTEDescriptorType = enum(c_int) {
    stuffing = 128,
    ac3 = 129,
    frame_rate = 130,
    extended_video = 131,
    component_name = 132,
    frequency_spec = 144,
    modulation_params = 145,
    transport_stream_id = 146,
    _,
};

pub const SCTESpliceCommandType = enum(c_int) {
    null = 0,
    schedule = 4,
    insert = 5,
    time = 6,
    bandwidth = 7,
    private = 255,
    _,
};

pub const SCTESpliceDescriptor = enum(c_int) {
    avail = 0,
    dtmf = 1,
    segmentation = 2,
    time = 3,
    audio = 4,
    _,
};

pub const SatellitePolarizationType = enum(c_int) {
    linear_horizontal = 0,
    linear_vertical = 1,
    circular_left = 2,
    circular_right = 3,
    _,
};

pub const SatelliteRolloff = enum(c_int) {
    @"35" = 0,
    @"20" = 1,
    @"25" = 2,
    reserved = 3,
    auto = 4,
    _,
};

/// Type of mpeg-ts streams for SCTE. Most users would want to use the
/// `gstmpegts.ATSCStreamType` instead since it also covers these stream types
pub const ScteStreamType = enum(c_int) {
    subtitling = 130,
    isoch_data = 131,
    sit = 134,
    dst_nrt = 149,
    dsmcc_dcb = 176,
    signaling = 192,
    sync_data = 194,
    async_data = 195,
    _,
};

/// Values for a `gstmpegts.Section` table_id.
///
/// These are the registered ATSC section `table_id` variants. Unless specified
/// otherwise, they are defined in the "ATSC A/65" specification.
///
/// see also: `gstmpegts.SectionTableID` and other variants.
pub const SectionATSCTableID = enum(c_int) {
    master_guide = 199,
    terrestrial_virtual_channel = 200,
    cable_virtual_channel = 201,
    rating_region = 202,
    event_information = 203,
    channel_or_event_extended_text = 204,
    system_time = 205,
    data_event = 206,
    data_service = 207,
    program_identifier = 208,
    network_resource = 209,
    long_term_service = 210,
    directed_channel_change = 211,
    directed_channel_change_section_code = 212,
    aggregate_event_information = 214,
    aggregate_extended_text = 215,
    aggregate_data_event = 217,
    satellite_virtual_channel = 218,
    _,
};

/// Values for a `gstmpegts.Section` table_id.
///
/// These are the registered DVB table_id variants. Unless specified otherwise,
/// they come from the DVB Specification for SI (ETSI EN 300 468).
///
/// see also: `gstmpegts.SectionTableID`
pub const SectionDVBTableID = enum(c_int) {
    network_information_actual_network = 64,
    network_information_other_network = 65,
    service_description_actual_ts = 66,
    service_description_other_ts = 70,
    bouquet_association = 74,
    update_notification = 75,
    downloadable_font_info = 76,
    event_information_actual_ts_present = 78,
    event_information_other_ts_present = 79,
    event_information_actual_ts_schedule_1 = 80,
    event_information_actual_ts_schedule_n = 95,
    event_information_other_ts_schedule_1 = 96,
    event_information_other_ts_schedule_n = 111,
    time_date = 112,
    running_status = 113,
    stuffing = 114,
    time_offset = 115,
    application_information_table = 116,
    container = 117,
    related_content = 118,
    content_identifier = 119,
    mpe_fec = 120,
    resolution_notification = 121,
    mpe_ifec = 122,
    protection_message = 123,
    discontinuity_information = 126,
    selection_information = 127,
    ca_message_ecm_0 = 128,
    ca_message_ecm_1 = 129,
    ca_message_system_private_1 = 130,
    ca_message_system_private_n = 143,
    sct = 160,
    fct = 161,
    tct = 162,
    spt = 163,
    cmt = 164,
    tbtp = 165,
    pcr_packet_payload = 166,
    transmission_mode_support_payload = 170,
    tim = 176,
    ll_fec_parity_data_table = 177,
    _,
};

/// Values for a `gstmpegts.Section` table_id.
///
/// These are the registered SCTE table_id variants.
///
/// see also: `gstmpegts.SectionTableID`
pub const SectionSCTETableID = enum(c_int) {
    eas = 216,
    ebif = 224,
    reserved = 225,
    eiss = 226,
    dii = 227,
    ddb = 228,
    splice = 252,
    _,
};

/// Values for a `gstmpegts.Section` table_id
///
/// These are the registered ITU H.222.0 | ISO/IEC 13818-1 table_id variants.
///
/// see also `gstmpegts.SectionATSCTableID`, `gstmpegts.SectionDVBTableID`, and
/// `gstmpegts.SectionSCTETableID`
pub const SectionTableID = enum(c_int) {
    program_association = 0,
    conditional_access = 1,
    ts_program_map = 2,
    ts_description = 3,
    @"14496_scene_description" = 4,
    @"14496_objet_descriptor" = 5,
    metadata = 6,
    ipmp_control_information = 7,
    @"14496_section" = 8,
    @"23001_11_section" = 9,
    @"23001_10_section" = 10,
    dsm_cc_multiproto_encapsulated_data = 58,
    dsm_cc_u_n_messages = 59,
    dsm_cc_download_data_messages = 60,
    dsm_cc_stream_descriptors = 61,
    dsm_cc_private_data = 62,
    dsm_cc_addressable_sections = 63,
    unset = 255,
    _,
};

/// Types of `gstmpegts.Section` that the library handles. This covers all the
/// MPEG-TS and derivate specification that the library can properly identify and
/// use.
pub const SectionType = enum(c_int) {
    unknown = 0,
    pat = 1,
    pmt = 2,
    cat = 3,
    tsdt = 4,
    eit = 5,
    nit = 6,
    bat = 7,
    sdt = 8,
    tdt = 9,
    tot = 10,
    sit = 11,
    atsc_tvct = 12,
    atsc_cvct = 13,
    atsc_mgt = 14,
    atsc_ett = 15,
    atsc_eit = 16,
    atsc_stt = 17,
    atsc_rrt = 18,
    scte_sit = 19,
    _,
};

/// Type of MPEG-TS stream type.
///
/// These values correspond to the base standard registered types. Depending
/// on the variant of mpeg-ts being used (Bluray, ATSC, DVB, ...), other
/// types might also be used, but will not conflict with these.
///
/// Corresponds to table 2-34 of ITU H.222.0 | ISO/IEC 13818-1
pub const StreamType = enum(c_int) {
    reserved_00 = 0,
    video_mpeg1 = 1,
    video_mpeg2 = 2,
    audio_mpeg1 = 3,
    audio_mpeg2 = 4,
    private_sections = 5,
    private_pes_packets = 6,
    mheg = 7,
    dsm_cc = 8,
    h_222_1 = 9,
    dsmcc_a = 10,
    dsmcc_b = 11,
    dsmcc_c = 12,
    dsmcc_d = 13,
    auxiliary = 14,
    audio_aac_adts = 15,
    video_mpeg4 = 16,
    audio_aac_latm = 17,
    sl_flexmux_pes_packets = 18,
    sl_flexmux_sections = 19,
    synchronized_download = 20,
    metadata_pes_packets = 21,
    metadata_sections = 22,
    metadata_data_carousel = 23,
    metadata_object_carousel = 24,
    metadata_synchronized_download = 25,
    mpeg2_ipmp = 26,
    video_h264 = 27,
    audio_aac_clean = 28,
    mpeg4_timed_text = 29,
    video_rvc = 30,
    video_h264_svc_sub_bitstream = 31,
    video_h264_mvc_sub_bitstream = 32,
    video_jp2k = 33,
    video_mpeg2_stereo_additional_view = 34,
    video_h264_stereo_additional_view = 35,
    video_hevc = 36,
    ipmp_stream = 127,
    user_private_ea = 234,
    _,
};

pub const TerrestrialGuardInterval = enum(c_int) {
    @"1_32" = 0,
    @"1_16" = 1,
    @"1_8" = 2,
    @"1_4" = 3,
    auto = 4,
    @"1_128" = 5,
    @"19_128" = 6,
    @"19_256" = 7,
    pn420 = 8,
    pn595 = 9,
    pn945 = 10,
    _,
};

pub const TerrestrialHierarchy = enum(c_int) {
    none = 0,
    @"1" = 1,
    @"2" = 2,
    @"4" = 3,
    auto = 4,
    _,
};

pub const TerrestrialTransmissionMode = enum(c_int) {
    @"2k" = 0,
    @"8k" = 1,
    auto = 2,
    @"4k" = 3,
    @"1k" = 4,
    @"16k" = 5,
    @"32k" = 6,
    c1 = 7,
    c3780 = 8,
    _,
};

/// Well-known registration ids, expressed as native-endian 32bit integers. These
/// are used in descriptors of type `GST_MTS_DESC_REGISTRATION`. Unless specified
/// otherwise (by use of the "OTHER" prefix), they are all registered by the
/// [SMPTE Registration Authority](https://smpte-ra.org/) or specified in
/// "official" documentation for the given format.
pub const RegistrationId = packed struct(c_uint) {
    _padding0: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_0: RegistrationId = @bitCast(@as(c_uint, 0));
    const flags_ac_3: RegistrationId = @bitCast(@as(c_uint, 1094921523));
    const flags_cuei: RegistrationId = @bitCast(@as(c_uint, 1129661769));
    const flags_drac: RegistrationId = @bitCast(@as(c_uint, 1685217635));
    const flags_dts1: RegistrationId = @bitCast(@as(c_uint, 1146377009));
    const flags_dts2: RegistrationId = @bitCast(@as(c_uint, 1146377010));
    const flags_dts3: RegistrationId = @bitCast(@as(c_uint, 1146377011));
    const flags_bssd: RegistrationId = @bitCast(@as(c_uint, 1112757060));
    const flags_eac3: RegistrationId = @bitCast(@as(c_uint, 1161904947));
    const flags_etv1: RegistrationId = @bitCast(@as(c_uint, 1163154993));
    const flags_ga94: RegistrationId = @bitCast(@as(c_uint, 1195456820));
    const flags_hdmv: RegistrationId = @bitCast(@as(c_uint, 1212435798));
    const flags_klva: RegistrationId = @bitCast(@as(c_uint, 1263294017));
    const flags_opus: RegistrationId = @bitCast(@as(c_uint, 1330664787));
    const flags_tshv: RegistrationId = @bitCast(@as(c_uint, 1414744150));
    const flags_vc_1: RegistrationId = @bitCast(@as(c_uint, 1447243057));
    const flags_ac_4: RegistrationId = @bitCast(@as(c_uint, 1094921524));
    const flags_other_hevc: RegistrationId = @bitCast(@as(c_uint, 1212503619));
};

/// Creates and adds a `gstmpegts.PESMetadataMeta` to a `buffer`.
extern fn gst_buffer_add_mpegts_pes_metadata_meta(p_buffer: *gst.Buffer) *gstmpegts.PESMetadataMeta;
pub const bufferAddMpegtsPesMetadataMeta = gst_buffer_add_mpegts_pes_metadata_meta;

extern fn gst_mpegts_dvb_component_descriptor_free(p_source: *gstmpegts.ComponentDescriptor) void;
pub const dvbComponentDescriptorFree = gst_mpegts_dvb_component_descriptor_free;

/// Creates a new `gst.Event` for a `gstmpegts.Section`.
extern fn gst_event_new_mpegts_section(p_section: *gstmpegts.Section) *gst.Event;
pub const eventNewMpegtsSection = gst_event_new_mpegts_section;

/// Extracts the `gstmpegts.Section` contained in the `event` `gst.Event`
extern fn gst_event_parse_mpegts_section(p_event: *gst.Event) *gstmpegts.Section;
pub const eventParseMpegtsSection = gst_event_parse_mpegts_section;

/// Finds the first descriptor of type `tag` in the array.
///
/// Note: To look for descriptors that can be present more than once in an
/// array of descriptors, iterate the `glib.Array` manually.
extern fn gst_mpegts_find_descriptor(p_descriptors: *glib.PtrArray, p_tag: u8) *const gstmpegts.Descriptor;
pub const findDescriptor = gst_mpegts_find_descriptor;

/// Finds the first descriptor of type `tag` with `tag_extension` in the array.
///
/// Note: To look for descriptors that can be present more than once in an
/// array of descriptors, iterate the `glib.Array` manually.
extern fn gst_mpegts_find_descriptor_with_extension(p_descriptors: *glib.PtrArray, p_tag: u8, p_tag_extension: u8) *const gstmpegts.Descriptor;
pub const findDescriptorWithExtension = gst_mpegts_find_descriptor_with_extension;

/// Initializes the MPEG-TS helper library. Must be called before any
/// usage.
extern fn gst_mpegts_initialize() void;
pub const initialize = gst_mpegts_initialize;

/// Creates a new `gst.Message` for a `GstMpegtsSection`.
extern fn gst_message_new_mpegts_section(p_parent: *gst.Object, p_section: *gstmpegts.Section) *gst.Message;
pub const messageNewMpegtsSection = gst_message_new_mpegts_section;

/// Returns the `gstmpegts.Section` contained in a message.
extern fn gst_message_parse_mpegts_section(p_message: *gst.Message) *gstmpegts.Section;
pub const messageParseMpegtsSection = gst_message_parse_mpegts_section;

/// Parses the descriptors present in `buffer` and returns them as an
/// array.
///
/// Note: The data provided in `buffer` will not be copied.
extern fn gst_mpegts_parse_descriptors(p_buffer: *u8, p_buf_len: usize) *glib.PtrArray;
pub const parseDescriptors = gst_mpegts_parse_descriptors;

/// Allocates a new `glib.PtrArray` for `gstmpegts.PatProgram`. The array can be filled
/// and then converted to a PAT section with `gstmpegts.sectionFromPat`.
extern fn gst_mpegts_pat_new() *glib.PtrArray;
pub const patNew = gst_mpegts_pat_new;

/// Return the `gobject.Type` associated with `gstmpegts.PESMetadataMeta`
extern fn gst_mpegts_pes_metadata_meta_api_get_type() usize;
pub const pesMetadataMetaApiGetType = gst_mpegts_pes_metadata_meta_api_get_type;

/// Allocates and initializes a new INSERT command `gstmpegts.SCTESIT`
/// setup to cancel the specified `event_id`.
extern fn gst_mpegts_scte_cancel_new(p_event_id: u32) *gstmpegts.SCTESIT;
pub const scteCancelNew = gst_mpegts_scte_cancel_new;

/// Allocates and initializes a NULL command `gstmpegts.SCTESIT`.
extern fn gst_mpegts_scte_null_new() *gstmpegts.SCTESIT;
pub const scteNullNew = gst_mpegts_scte_null_new;

/// Allocates and initializes a new "Splice In" INSERT command
/// `gstmpegts.SCTESIT` for the given `event_id` and `splice_time`.
///
/// If the `splice_time` is `G_MAXUINT64` then the event will be
/// immediate as opposed to for the target `splice_time`.
extern fn gst_mpegts_scte_splice_in_new(p_event_id: u32, p_splice_time: gst.ClockTime) *gstmpegts.SCTESIT;
pub const scteSpliceInNew = gst_mpegts_scte_splice_in_new;

/// Allocates and initializes a new "Splice Out" INSERT command
/// `gstmpegts.SCTESIT` for the given `event_id`, `splice_time` and
/// `duration`.
///
/// If the `splice_time` is `G_MAXUINT64` then the event will be
/// immediate as opposed to for the target `splice_time`.
///
/// If the `duration` is 0 it won't be specified in the event.
extern fn gst_mpegts_scte_splice_out_new(p_event_id: u32, p_splice_time: gst.ClockTime, p_duration: gst.ClockTime) *gstmpegts.SCTESIT;
pub const scteSpliceOutNew = gst_mpegts_scte_splice_out_new;

pub const PacketizeFunc = *const fn (p_section: *gstmpegts.Section) callconv(.C) c_int;
