pub const ext = @import("ext.zig");
const gtksource = @This();

const std = @import("std");
const compat = @import("compat");
const gtk = @import("gtk4");
const gsk = @import("gsk4");
const graphene = @import("graphene1");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gdk = @import("gdk4");
const cairo = @import("cairo1");
const pangocairo = @import("pangocairo1");
const pango = @import("pango1");
const harfbuzz = @import("harfbuzz0");
const freetype2 = @import("freetype22");
const gio = @import("gio2");
const gmodule = @import("gmodule2");
const gdkpixbuf = @import("gdkpixbuf2");
/// Subclass of `gtk.TextBuffer`.
///
/// A `GtkSourceBuffer` object is the model for `View` widgets.
/// It extends the `gtk.TextBuffer` class by adding features useful to display
/// and edit source code such as syntax highlighting and bracket matching.
///
/// To create a `GtkSourceBuffer` use `gtksource.Buffer.new` or
/// `gtksource.Buffer.newWithLanguage`. The second form is just a convenience
/// function which allows you to initially set a `Language`. You can also
/// directly create a `View` and get its `Buffer` with
/// `gtk.TextView.getBuffer`.
///
/// The highlighting is enabled by default, but you can disable it with
/// `Buffer.setHighlightSyntax`.
///
/// # Context Classes:
///
/// It is possible to retrieve some information from the syntax highlighting
/// engine. The default context classes that are applied to regions of a
/// `GtkSourceBuffer`:
///
///  - **comment**: the region delimits a comment;
///  - **no-spell-check**: the region should not be spell checked;
///  - **path**: the region delimits a path to a file;
///  - **string**: the region delimits a string.
///
/// Custom language definition files can create their own context classes,
/// since the functions like `Buffer.iterHasContextClass` take
/// a string parameter as the context class.
///
/// `GtkSourceBuffer` provides an API to access the context classes:
/// `Buffer.iterHasContextClass`,
/// `Buffer.getContextClassesAtIter`,
/// `Buffer.iterForwardToContextClassToggle` and
/// `Buffer.iterBackwardToContextClassToggle`.
///
/// And the `gtksource.Buffer.signals.highlight_updated` signal permits to be notified
/// when a context class region changes.
///
/// Each context class has also an associated `gtk.TextTag` with the name
/// `gtksourceview:context-classes:<name>`. For example to
/// retrieve the `gtk.TextTag` for the string context class, one can write:
/// ```c
/// GtkTextTagTable *tag_table;
/// GtkTextTag *tag;
///
/// tag_table = gtk_text_buffer_get_tag_table (buffer);
/// tag = gtk_text_tag_table_lookup (tag_table, "gtksourceview:context-classes:string");
/// ```
///
/// The tag must be used for read-only purposes.
///
/// Accessing a context class via the associated `gtk.TextTag` is less
/// convenient than the `GtkSourceBuffer` API, because:
///
///  - The tag doesn't always exist, you need to listen to the
///    `gtk.TextTagTable.signals.tag_added` and `gtk.TextTagTable.signals.tag_removed` signals.
///  - Instead of the `gtksource.Buffer.signals.highlight_updated` signal, you can listen
///    to the `gtk.TextBuffer.signals.apply_tag` and `gtk.TextBuffer.signals.remove_tag` signals.
///
/// A possible use-case for accessing a context class via the associated
/// `gtk.TextTag` is to read the region but without adding a hard dependency on the
/// GtkSourceView library (for example for a spell-checking library that wants to
/// read the no-spell-check region).
pub const Buffer = extern struct {
    pub const Parent = gtk.TextBuffer;
    pub const Implements = [_]type{};
    pub const Class = gtksource.BufferClass;
    f_parent_instance: gtk.TextBuffer,

    pub const virtual_methods = struct {
        pub const bracket_matched = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_state: gtksource.BracketMatchType) void {
                return gobject.ext.as(Buffer.Class, p_class).f_bracket_matched.?(gobject.ext.as(Buffer, p_buffer), p_iter, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_state: gtksource.BracketMatchType) callconv(.C) void) void {
                gobject.ext.as(Buffer.Class, p_class).f_bracket_matched = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Whether to highlight matching brackets in the buffer.
        pub const highlight_matching_brackets = struct {
            pub const name = "highlight-matching-brackets";

            pub const Type = c_int;
        };

        /// Whether to highlight syntax in the buffer.
        pub const highlight_syntax = struct {
            pub const name = "highlight-syntax";

            pub const Type = c_int;
        };

        /// Whether the buffer has an implicit trailing newline. See
        /// `Buffer.setImplicitTrailingNewline`.
        pub const implicit_trailing_newline = struct {
            pub const name = "implicit-trailing-newline";

            pub const Type = c_int;
        };

        pub const language = struct {
            pub const name = "language";

            pub const Type = ?*gtksource.Language;
        };

        /// The "loading" property denotes that a `GtkSourceFileLoader` is
        /// currently loading the buffer.
        ///
        /// Applications may want to use this setting to avoid doing work
        /// while the buffer is loading such as spellchecking.
        pub const loading = struct {
            pub const name = "loading";

            pub const Type = c_int;
        };

        /// Style scheme. It contains styles for syntax highlighting, optionally
        /// foreground, background, cursor color, current line color, and matching
        /// brackets style.
        pub const style_scheme = struct {
            pub const name = "style-scheme";

            pub const Type = ?*gtksource.StyleScheme;
        };
    };

    pub const signals = struct {
        /// `iter` is set to a valid iterator pointing to the matching bracket
        /// if `state` is `GTK_SOURCE_BRACKET_MATCH_FOUND`. Otherwise `iter` is
        /// meaningless.
        ///
        /// The signal is emitted only when the `state` changes, typically when
        /// the cursor moves.
        ///
        /// A use-case for this signal is to show messages in a `gtk.Statusbar`.
        pub const bracket_matched = struct {
            pub const name = "bracket-matched";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_iter: ?*gtk.TextIter, p_state: gtksource.BracketMatchType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Buffer, p_instance))),
                    gobject.signalLookup("bracket-matched", Buffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "cursor-moved" signal is emitted when then insertion mark has moved.
        pub const cursor_moved = struct {
            pub const name = "cursor-moved";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Buffer, p_instance))),
                    gobject.signalLookup("cursor-moved", Buffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::highlight-updated signal is emitted when the syntax
        /// highlighting and [context classes](./class.Buffer.html`context`-classes) are updated in a
        /// certain region of the `buffer`.
        pub const highlight_updated = struct {
            pub const name = "highlight-updated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_start: *gtk.TextIter, p_end: *gtk.TextIter, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Buffer, p_instance))),
                    gobject.signalLookup("highlight-updated", Buffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::source-mark-updated signal is emitted each time
        /// a mark is added to, moved or removed from the `buffer`.
        pub const source_mark_updated = struct {
            pub const name = "source-mark-updated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_mark: *gtk.TextMark, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Buffer, p_instance))),
                    gobject.signalLookup("source-mark-updated", Buffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new source buffer.
    extern fn gtk_source_buffer_new(p_table: ?*gtk.TextTagTable) *gtksource.Buffer;
    pub const new = gtk_source_buffer_new;

    /// Creates a new source buffer using the highlighting patterns in `language`.
    ///
    /// This is equivalent to creating a new source buffer with
    /// a new tag table and then calling `Buffer.setLanguage`.
    extern fn gtk_source_buffer_new_with_language(p_language: *gtksource.Language) *gtksource.Buffer;
    pub const newWithLanguage = gtk_source_buffer_new_with_language;

    /// Moves `iter` to the position of the previous `Mark` of the given
    /// category.
    ///
    /// Returns `TRUE` if `iter` was moved. If `category` is NULL, the
    /// previous source mark can be of any category.
    extern fn gtk_source_buffer_backward_iter_to_source_mark(p_buffer: *Buffer, p_iter: *gtk.TextIter, p_category: ?[*:0]const u8) c_int;
    pub const backwardIterToSourceMark = gtk_source_buffer_backward_iter_to_source_mark;

    /// Changes the case of the text between the specified iterators.
    ///
    /// Since 5.4, this function will update the position of `start` and
    /// `end` to surround the modified text.
    extern fn gtk_source_buffer_change_case(p_buffer: *Buffer, p_case_type: gtksource.ChangeCaseType, p_start: *gtk.TextIter, p_end: *gtk.TextIter) void;
    pub const changeCase = gtk_source_buffer_change_case;

    /// Creates a source mark in the `buffer` of category `category`.
    ///
    /// A source mark is a `gtk.TextMark` but organized into categories.
    /// Depending on the category a pixbuf can be specified that will be displayed
    /// along the line of the mark.
    ///
    /// Like a `gtk.TextMark`, a `Mark` can be anonymous if the
    /// passed `name` is `NULL`.  Also, the buffer owns the marks so you
    /// shouldn't unreference it.
    ///
    /// Marks always have left gravity and are moved to the beginning of
    /// the line when the user deletes the line they were in.
    ///
    /// Typical uses for a source mark are bookmarks, breakpoints, current
    /// executing instruction indication in a source file, etc..
    extern fn gtk_source_buffer_create_source_mark(p_buffer: *Buffer, p_name: ?[*:0]const u8, p_category: [*:0]const u8, p_where: *const gtk.TextIter) *gtksource.Mark;
    pub const createSourceMark = gtk_source_buffer_create_source_mark;

    /// In short, this is the same function as `gtk.TextBuffer.createTag` but
    /// instead of creating a `gtk.TextTag`, this function creates a `Tag`.
    ///
    /// This function creates a `Tag` and adds it to the tag table for
    /// `buffer`.  Equivalent to calling `gtk.TextTag.new` and then adding the tag to
    /// the buffer’s tag table. The returned tag is owned by the buffer’s tag table,
    /// so the ref count will be equal to one.
    ///
    /// If `tag_name` is `NULL`, the tag is anonymous.
    ///
    /// If `tag_name` is non-`NULL`, a tag called `tag_name` must not already
    /// exist in the tag table for this buffer.
    ///
    /// The `first_property_name` argument and subsequent arguments are a list
    /// of properties to set on the tag, as with `gobject.Object.set`.
    extern fn gtk_source_buffer_create_source_tag(p_buffer: *Buffer, p_tag_name: ?[*:0]const u8, p_first_property_name: ?[*:0]const u8, ...) *gtk.TextTag;
    pub const createSourceTag = gtk_source_buffer_create_source_tag;

    /// Forces buffer to analyze and highlight the given area synchronously.
    ///
    /// **Note**:
    ///
    /// This is a potentially slow operation and should be used only
    /// when you need to make sure that some text not currently
    /// visible is highlighted, for instance before printing.
    extern fn gtk_source_buffer_ensure_highlight(p_buffer: *Buffer, p_start: *const gtk.TextIter, p_end: *const gtk.TextIter) void;
    pub const ensureHighlight = gtk_source_buffer_ensure_highlight;

    /// Moves `iter` to the position of the next `Mark` of the given
    /// `category`.
    ///
    /// Returns `TRUE` if `iter` was moved. If `category` is NULL, the
    /// next source mark can be of any category.
    extern fn gtk_source_buffer_forward_iter_to_source_mark(p_buffer: *Buffer, p_iter: *gtk.TextIter, p_category: ?[*:0]const u8) c_int;
    pub const forwardIterToSourceMark = gtk_source_buffer_forward_iter_to_source_mark;

    /// Get all defined context classes at `iter`.
    ///
    /// See the `Buffer` description for the list of default context classes.
    extern fn gtk_source_buffer_get_context_classes_at_iter(p_buffer: *Buffer, p_iter: *const gtk.TextIter) [*][*:0]u8;
    pub const getContextClassesAtIter = gtk_source_buffer_get_context_classes_at_iter;

    /// Determines whether bracket match highlighting is activated for the
    /// source buffer.
    extern fn gtk_source_buffer_get_highlight_matching_brackets(p_buffer: *Buffer) c_int;
    pub const getHighlightMatchingBrackets = gtk_source_buffer_get_highlight_matching_brackets;

    /// Determines whether syntax highlighting is activated in the source
    /// buffer.
    extern fn gtk_source_buffer_get_highlight_syntax(p_buffer: *Buffer) c_int;
    pub const getHighlightSyntax = gtk_source_buffer_get_highlight_syntax;

    extern fn gtk_source_buffer_get_implicit_trailing_newline(p_buffer: *Buffer) c_int;
    pub const getImplicitTrailingNewline = gtk_source_buffer_get_implicit_trailing_newline;

    /// Returns the `Language` associated with the buffer,
    /// see `Buffer.setLanguage`.
    ///
    /// The returned object should not be unreferenced by the user.
    extern fn gtk_source_buffer_get_language(p_buffer: *Buffer) ?*gtksource.Language;
    pub const getLanguage = gtk_source_buffer_get_language;

    extern fn gtk_source_buffer_get_loading(p_buffer: *Buffer) c_int;
    pub const getLoading = gtk_source_buffer_get_loading;

    /// Returns the list of marks of the given category at `iter`.
    ///
    /// If `category` is `NULL` it returns all marks at `iter`.
    extern fn gtk_source_buffer_get_source_marks_at_iter(p_buffer: *Buffer, p_iter: *gtk.TextIter, p_category: ?[*:0]const u8) *glib.SList;
    pub const getSourceMarksAtIter = gtk_source_buffer_get_source_marks_at_iter;

    /// Returns the list of marks of the given category at `line`.
    ///
    /// If `category` is `NULL`, all marks at `line` are returned.
    extern fn gtk_source_buffer_get_source_marks_at_line(p_buffer: *Buffer, p_line: c_int, p_category: ?[*:0]const u8) *glib.SList;
    pub const getSourceMarksAtLine = gtk_source_buffer_get_source_marks_at_line;

    /// Returns the `StyleScheme` associated with the buffer,
    /// see `Buffer.setStyleScheme`.
    ///
    /// The returned object should not be unreferenced by the user.
    extern fn gtk_source_buffer_get_style_scheme(p_buffer: *Buffer) ?*gtksource.StyleScheme;
    pub const getStyleScheme = gtk_source_buffer_get_style_scheme;

    /// Moves backward to the next toggle (on or off) of the context class.
    ///
    /// If no matching context class toggles are found, returns `FALSE`, otherwise `TRUE`.
    /// Does not return toggles located at `iter`, only toggles after `iter`. Sets
    /// `iter` to the location of the toggle, or to the end of the buffer if no
    /// toggle is found.
    ///
    /// See the `Buffer` description for the list of default context classes.
    extern fn gtk_source_buffer_iter_backward_to_context_class_toggle(p_buffer: *Buffer, p_iter: *gtk.TextIter, p_context_class: [*:0]const u8) c_int;
    pub const iterBackwardToContextClassToggle = gtk_source_buffer_iter_backward_to_context_class_toggle;

    /// Moves forward to the next toggle (on or off) of the context class.
    ///
    /// If no matching context class toggles are found, returns `FALSE`, otherwise `TRUE`.
    /// Does not return toggles located at `iter`, only toggles after `iter`. Sets
    /// `iter` to the location of the toggle, or to the end of the buffer if no
    /// toggle is found.
    ///
    /// See the `Buffer` description for the list of default context classes.
    extern fn gtk_source_buffer_iter_forward_to_context_class_toggle(p_buffer: *Buffer, p_iter: *gtk.TextIter, p_context_class: [*:0]const u8) c_int;
    pub const iterForwardToContextClassToggle = gtk_source_buffer_iter_forward_to_context_class_toggle;

    /// Check if the class `context_class` is set on `iter`.
    ///
    /// See the `Buffer` description for the list of default context classes.
    extern fn gtk_source_buffer_iter_has_context_class(p_buffer: *Buffer, p_iter: *const gtk.TextIter, p_context_class: [*:0]const u8) c_int;
    pub const iterHasContextClass = gtk_source_buffer_iter_has_context_class;

    /// Joins the lines of text between the specified iterators.
    extern fn gtk_source_buffer_join_lines(p_buffer: *Buffer, p_start: *gtk.TextIter, p_end: *gtk.TextIter) void;
    pub const joinLines = gtk_source_buffer_join_lines;

    /// Remove all marks of `category` between `start` and `end` from the buffer.
    ///
    /// If `category` is NULL, all marks in the range will be removed.
    extern fn gtk_source_buffer_remove_source_marks(p_buffer: *Buffer, p_start: *const gtk.TextIter, p_end: *const gtk.TextIter, p_category: ?[*:0]const u8) void;
    pub const removeSourceMarks = gtk_source_buffer_remove_source_marks;

    /// Controls the bracket match highlighting function in the buffer.
    ///
    /// If activated, when you position your cursor over a bracket character
    /// (a parenthesis, a square bracket, etc.) the matching opening or
    /// closing bracket character will be highlighted.
    extern fn gtk_source_buffer_set_highlight_matching_brackets(p_buffer: *Buffer, p_highlight: c_int) void;
    pub const setHighlightMatchingBrackets = gtk_source_buffer_set_highlight_matching_brackets;

    /// Controls whether syntax is highlighted in the buffer.
    ///
    /// If `highlight` is `TRUE`, the text will be highlighted according to the syntax
    /// patterns specified in the `Language` set with `Buffer.setLanguage`.
    ///
    /// If `highlight` is `FALSE`, syntax highlighting is disabled and all the
    /// `gtk.TextTag` objects that have been added by the syntax highlighting engine
    /// are removed from the buffer.
    extern fn gtk_source_buffer_set_highlight_syntax(p_buffer: *Buffer, p_highlight: c_int) void;
    pub const setHighlightSyntax = gtk_source_buffer_set_highlight_syntax;

    /// Sets whether the `buffer` has an implicit trailing newline.
    ///
    /// If an explicit trailing newline is present in a `gtk.TextBuffer`, `gtk.TextView`
    /// shows it as an empty line. This is generally not what the user expects.
    ///
    /// If `implicit_trailing_newline` is `TRUE` (the default value):
    ///  - when a `FileLoader` loads the content of a file into the `buffer`,
    ///    the trailing newline (if present in the file) is not inserted into the
    ///    `buffer`.
    ///  - when a `FileSaver` saves the content of the `buffer` into a file, a
    ///    trailing newline is added to the file.
    ///
    /// On the other hand, if `implicit_trailing_newline` is `FALSE`, the file's
    /// content is not modified when loaded into the `buffer`, and the `buffer`'s
    /// content is not modified when saved into a file.
    extern fn gtk_source_buffer_set_implicit_trailing_newline(p_buffer: *Buffer, p_implicit_trailing_newline: c_int) void;
    pub const setImplicitTrailingNewline = gtk_source_buffer_set_implicit_trailing_newline;

    /// Associates a `Language` with the buffer.
    ///
    /// Note that a `Language` affects not only the syntax highlighting, but
    /// also the [context classes](./class.Buffer.html`context`-classes). If you want to disable just the
    /// syntax highlighting, see `Buffer.setHighlightSyntax`.
    ///
    /// The buffer holds a reference to `language`.
    extern fn gtk_source_buffer_set_language(p_buffer: *Buffer, p_language: ?*gtksource.Language) void;
    pub const setLanguage = gtk_source_buffer_set_language;

    /// Sets a `StyleScheme` to be used by the buffer and the view.
    ///
    /// Note that a `StyleScheme` affects not only the syntax highlighting,
    /// but also other `View` features such as highlighting the current line,
    /// matching brackets, the line numbers, etc.
    ///
    /// Instead of setting a `NULL` `scheme`, it is better to disable syntax
    /// highlighting with `Buffer.setHighlightSyntax`, and setting the
    /// `StyleScheme` with the "classic" or "tango" ID, because those two
    /// style schemes follow more closely the GTK theme (for example for the
    /// background color).
    ///
    /// The buffer holds a reference to `scheme`.
    extern fn gtk_source_buffer_set_style_scheme(p_buffer: *Buffer, p_scheme: ?*gtksource.StyleScheme) void;
    pub const setStyleScheme = gtk_source_buffer_set_style_scheme;

    /// Sort the lines of text between the specified iterators.
    extern fn gtk_source_buffer_sort_lines(p_buffer: *Buffer, p_start: *gtk.TextIter, p_end: *gtk.TextIter, p_flags: gtksource.SortFlags, p_column: c_int) void;
    pub const sortLines = gtk_source_buffer_sort_lines;

    extern fn gtk_source_buffer_get_type() usize;
    pub const getGObjectType = gtk_source_buffer_get_type;

    extern fn g_object_ref(p_self: *gtksource.Buffer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Buffer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Buffer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Main Completion Object.
///
/// The completion system helps the user when they writes some text,
/// such as words, command names, functions, and suchlike. Proposals can
/// be shown, to complete the text the user is writing. Each proposal can
/// contain an additional piece of information (for example
/// documentation), that is displayed when the "Details" button is
/// clicked.
///
/// Proposals are created via a `CompletionProvider`. There can
/// be for example a provider to complete words (see `CompletionWords`),
/// another provider for the completion of
/// function names, etc. To add a provider, call
/// `Completion.addProvider`.
///
/// The `CompletionProposal` interface represents a proposal.
///
/// If a proposal contains extra information (see
/// `GTK_SOURCE_COMPLETION_COLUMN_DETAILS`), it will be
/// displayed in a supplemental details window, which appears when
/// the "Details" button is clicked.
///
/// Each `View` object is associated with a `Completion`
/// instance. This instance can be obtained with
/// `View.getCompletion`. The `View` class contains also the
/// `View.signals.show_completion` signal.
///
/// A same `CompletionProvider` object can be used for several
/// `GtkSourceCompletion`'s.
pub const Completion = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.CompletionClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gtk.TextBuffer` for the `gtksource.Completion.properties.view`.
        /// This is a convenience property for providers.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtk.TextView;
        };

        /// The number of rows to display to the user before scrolling.
        pub const page_size = struct {
            pub const name = "page-size";

            pub const Type = c_uint;
        };

        /// Determines whether the visibility of the info window should be saved when the
        /// completion is hidden, and restored when the completion is shown again.
        pub const remember_info_visibility = struct {
            pub const name = "remember-info-visibility";

            pub const Type = c_int;
        };

        /// Determines whether the first proposal should be selected when the completion
        /// is first shown.
        pub const select_on_show = struct {
            pub const name = "select-on-show";

            pub const Type = c_int;
        };

        /// The "show-icons" property denotes if icons should be displayed within
        /// the list of completions presented to the user.
        pub const show_icons = struct {
            pub const name = "show-icons";

            pub const Type = c_int;
        };

        /// The "view" property is the `gtk.TextView` for which this `gtksource.Completion`
        /// is providing completion features.
        pub const view = struct {
            pub const name = "view";

            pub const Type = ?*gtksource.View;
        };
    };

    pub const signals = struct {
        /// The "hide" signal is emitted when the completion window should
        /// be hidden.
        pub const hide = struct {
            pub const name = "hide";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Completion, p_instance))),
                    gobject.signalLookup("hide", Completion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "provided-added" signal is emitted when a new provider is
        /// added to the completion.
        pub const provider_added = struct {
            pub const name = "provider-added";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_provider: *gtksource.CompletionProvider, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Completion, p_instance))),
                    gobject.signalLookup("provider-added", Completion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "provided-removed" signal is emitted when a provider has
        /// been removed from the completion.
        pub const provider_removed = struct {
            pub const name = "provider-removed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_provider: *gtksource.CompletionProvider, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Completion, p_instance))),
                    gobject.signalLookup("provider-removed", Completion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "show" signal is emitted when the completion window should
        /// be shown.
        pub const show = struct {
            pub const name = "show";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Completion, p_instance))),
                    gobject.signalLookup("show", Completion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// This will add `<b>` tags around matched characters in `haystack`
    /// based on `casefold_query`.
    extern fn gtk_source_completion_fuzzy_highlight(p_haystack: [*:0]const u8, p_casefold_query: [*:0]const u8) ?*pango.AttrList;
    pub const fuzzyHighlight = gtk_source_completion_fuzzy_highlight;

    /// This helper function can do a fuzzy match for you giving a haystack and
    /// casefolded needle.
    ///
    /// Casefold your needle using `glib.utf8Casefold` before
    /// running the query.
    ///
    /// Score will be set with the score of the match upon success. Otherwise,
    /// it will be set to zero.
    extern fn gtk_source_completion_fuzzy_match(p_haystack: ?[*:0]const u8, p_casefold_needle: [*:0]const u8, p_priority: ?*c_uint) c_int;
    pub const fuzzyMatch = gtk_source_completion_fuzzy_match;

    /// Adds a `CompletionProvider` to the list of providers to be queried
    /// for completion results.
    extern fn gtk_source_completion_add_provider(p_self: *Completion, p_provider: *gtksource.CompletionProvider) void;
    pub const addProvider = gtk_source_completion_add_provider;

    extern fn gtk_source_completion_block_interactive(p_self: *Completion) void;
    pub const blockInteractive = gtk_source_completion_block_interactive;

    /// Gets the connected `View`'s `Buffer`
    extern fn gtk_source_completion_get_buffer(p_self: *Completion) *gtksource.Buffer;
    pub const getBuffer = gtk_source_completion_get_buffer;

    extern fn gtk_source_completion_get_page_size(p_self: *Completion) c_uint;
    pub const getPageSize = gtk_source_completion_get_page_size;

    /// Gets the `View` that owns the `Completion`.
    extern fn gtk_source_completion_get_view(p_self: *Completion) *gtksource.View;
    pub const getView = gtk_source_completion_get_view;

    /// Emits the "hide" signal.
    ///
    /// When the "hide" signal is emitted, the completion window will be
    /// dismissed.
    extern fn gtk_source_completion_hide(p_self: *Completion) void;
    pub const hide = gtk_source_completion_hide;

    /// Removes a `CompletionProvider` previously added with
    /// `Completion.addProvider`.
    extern fn gtk_source_completion_remove_provider(p_self: *Completion, p_provider: *gtksource.CompletionProvider) void;
    pub const removeProvider = gtk_source_completion_remove_provider;

    extern fn gtk_source_completion_set_page_size(p_self: *Completion, p_page_size: c_uint) void;
    pub const setPageSize = gtk_source_completion_set_page_size;

    /// Emits the "show" signal.
    ///
    /// When the "show" signal is emitted, the completion window will be
    /// displayed if there are any results available.
    extern fn gtk_source_completion_show(p_self: *Completion) void;
    pub const show = gtk_source_completion_show;

    extern fn gtk_source_completion_unblock_interactive(p_self: *Completion) void;
    pub const unblockInteractive = gtk_source_completion_unblock_interactive;

    extern fn gtk_source_completion_get_type() usize;
    pub const getGObjectType = gtk_source_completion_get_type;

    extern fn g_object_ref(p_self: *gtksource.Completion) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Completion) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Completion, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Widget for single cell of completion proposal.
///
/// The `GtkSourceCompletionCell` widget provides a container to display various
/// types of information with the completion display.
///
/// Each proposal may consist of multiple cells depending on the complexity of
/// the proposal. For example, programming language proposals may contain a cell
/// for the "left-hand-side" of an operation along with the "typed-text" for a
/// function name and "parameters". They may also optionally set an icon to
/// signify the kind of result.
///
/// A `CompletionProvider` should implement the
/// `CompletionProvider.virtual_methods.display` virtual function to control
/// how to convert data from their `CompletionProposal` to content for
/// the `GtkSourceCompletionCell`.
pub const CompletionCell = opaque {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.CompletionCellClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const column = struct {
            pub const name = "column";

            pub const Type = gtksource.CompletionColumn;
        };

        pub const markup = struct {
            pub const name = "markup";

            pub const Type = ?[*:0]u8;
        };

        pub const paintable = struct {
            pub const name = "paintable";

            pub const Type = ?*gdk.Paintable;
        };

        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };

        pub const widget = struct {
            pub const name = "widget";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_completion_cell_get_column(p_self: *CompletionCell) gtksource.CompletionColumn;
    pub const getColumn = gtk_source_completion_cell_get_column;

    /// Gets the child `gtk.Widget`, if any.
    extern fn gtk_source_completion_cell_get_widget(p_self: *CompletionCell) ?*gtk.Widget;
    pub const getWidget = gtk_source_completion_cell_get_widget;

    extern fn gtk_source_completion_cell_set_gicon(p_self: *CompletionCell, p_gicon: *gio.Icon) void;
    pub const setGicon = gtk_source_completion_cell_set_gicon;

    extern fn gtk_source_completion_cell_set_icon_name(p_self: *CompletionCell, p_icon_name: [*:0]const u8) void;
    pub const setIconName = gtk_source_completion_cell_set_icon_name;

    extern fn gtk_source_completion_cell_set_markup(p_self: *CompletionCell, p_markup: [*:0]const u8) void;
    pub const setMarkup = gtk_source_completion_cell_set_markup;

    extern fn gtk_source_completion_cell_set_paintable(p_self: *CompletionCell, p_paintable: *gdk.Paintable) void;
    pub const setPaintable = gtk_source_completion_cell_set_paintable;

    /// Sets the text for the column cell. Use `NULL` to unset.
    extern fn gtk_source_completion_cell_set_text(p_self: *CompletionCell, p_text: ?[*:0]const u8) void;
    pub const setText = gtk_source_completion_cell_set_text;

    extern fn gtk_source_completion_cell_set_text_with_attributes(p_self: *CompletionCell, p_text: [*:0]const u8, p_attrs: *pango.AttrList) void;
    pub const setTextWithAttributes = gtk_source_completion_cell_set_text_with_attributes;

    extern fn gtk_source_completion_cell_set_widget(p_self: *CompletionCell, p_child: *gtk.Widget) void;
    pub const setWidget = gtk_source_completion_cell_set_widget;

    extern fn gtk_source_completion_cell_get_type() usize;
    pub const getGObjectType = gtk_source_completion_cell_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionCell) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionCell) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionCell, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The context of a completion.
///
/// `GtkSourceCompletionContext` contains information about an attept to display
/// completion proposals to the user based on typed text in the `View`.
///
/// When typing, `Completion` may use registered
/// `CompletionProvider` to determine if there may be results which
/// could be displayed. If so, a `GtkSourceCompletionContext` is created with
/// information that is provided to the `CompletionProvider` to populate
/// results which might be useful to the user.
///
/// `CompletionProvider` are expected to provide `gio.ListModel` with
/// `CompletionProposal` which may be joined together in a list of
/// results for the user. They are also responsible for how the contents are
/// displayed using `CompletionCell` which allows for some level of
/// customization.
pub const CompletionContext = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.ListModel};
    pub const Class = gtksource.CompletionContextClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The "busy" property is `TRUE` while the completion context is
        /// populating completion proposals.
        pub const busy = struct {
            pub const name = "busy";

            pub const Type = c_int;
        };

        /// The "completion" is the `gtksource.Completion` that was used to create the context.
        pub const completion = struct {
            pub const name = "completion";

            pub const Type = ?*gtksource.Completion;
        };

        /// The "empty" property is `TRUE` when there are no results.
        ///
        /// It will be notified when the first result is added or the last
        /// result is removed.
        pub const empty = struct {
            pub const name = "empty";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when a provider changes a model.
        ///
        /// This signal is primarily useful for `gtksource.CompletionProvider`'s
        /// that want to track other providers in context. For example, it can
        /// be used to create a "top results" provider.
        pub const provider_model_changed = struct {
            pub const name = "provider-model-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_provider: *gtksource.CompletionProvider, p_model: ?*gio.ListModel, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CompletionContext, p_instance))),
                    gobject.signalLookup("provider-model-changed", CompletionContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Gets the mode for which the context was activated.
    extern fn gtk_source_completion_context_get_activation(p_self: *CompletionContext) gtksource.CompletionActivation;
    pub const getActivation = gtk_source_completion_context_get_activation;

    /// Gets the bounds for the completion, which is the beginning of the
    /// current word (taking break characters into account) to the current
    /// insertion cursor.
    ///
    /// If `begin` is non-`NULL`, it will be set to the start position of the
    /// current word being completed.
    ///
    /// If `end` is non-`NULL`, it will be set to the insertion cursor for the
    /// current word being completed.
    extern fn gtk_source_completion_context_get_bounds(p_self: *CompletionContext, p_begin: ?*gtk.TextIter, p_end: ?*gtk.TextIter) c_int;
    pub const getBounds = gtk_source_completion_context_get_bounds;

    /// Gets the underlying buffer used by the context.
    ///
    /// This is a convenience function to get the buffer via the `gtksource.Completion`
    /// property.
    extern fn gtk_source_completion_context_get_buffer(p_self: *CompletionContext) ?*gtksource.Buffer;
    pub const getBuffer = gtk_source_completion_context_get_buffer;

    /// Gets the "busy" property. This is set to `TRUE` while the completion
    /// context is actively fetching proposals from registered
    /// `gtksource.CompletionProvider`'s.
    extern fn gtk_source_completion_context_get_busy(p_self: *CompletionContext) c_int;
    pub const getBusy = gtk_source_completion_context_get_busy;

    /// Gets the `gtksource.Completion` that created the context.
    extern fn gtk_source_completion_context_get_completion(p_self: *CompletionContext) ?*gtksource.Completion;
    pub const getCompletion = gtk_source_completion_context_get_completion;

    /// Checks if any proposals have been provided to the context.
    ///
    /// Out of convenience, this function will return `TRUE` if `self` is `NULL`.
    extern fn gtk_source_completion_context_get_empty(p_self: *CompletionContext) c_int;
    pub const getEmpty = gtk_source_completion_context_get_empty;

    /// Gets the language of the underlying buffer, if any.
    extern fn gtk_source_completion_context_get_language(p_self: *CompletionContext) ?*gtksource.Language;
    pub const getLanguage = gtk_source_completion_context_get_language;

    /// Gets the `gio.ListModel` associated with the provider.
    ///
    /// You can connect to `gtksource.CompletionContext.signals.model`-changed to receive
    /// notifications about when the model has been replaced by a new model.
    extern fn gtk_source_completion_context_get_proposals_for_provider(p_self: *CompletionContext, p_provider: *gtksource.CompletionProvider) ?*gio.ListModel;
    pub const getProposalsForProvider = gtk_source_completion_context_get_proposals_for_provider;

    /// Gets the text view for the context.
    extern fn gtk_source_completion_context_get_view(p_self: *CompletionContext) ?*gtksource.View;
    pub const getView = gtk_source_completion_context_get_view;

    /// Gets the word that is being completed up to the position of the insert mark.
    extern fn gtk_source_completion_context_get_word(p_self: *CompletionContext) [*:0]u8;
    pub const getWord = gtk_source_completion_context_get_word;

    /// Gets the providers that are associated with the context.
    extern fn gtk_source_completion_context_list_providers(p_self: *CompletionContext) *gio.ListModel;
    pub const listProviders = gtk_source_completion_context_list_providers;

    /// This function allows providers to update their results for a context
    /// outside of a call to `CompletionProvider.populateAsync`.
    ///
    /// This can be used to immediately return results for a provider while it does
    /// additional asynchronous work. Doing so will allow the completions to
    /// update while the operation is in progress.
    extern fn gtk_source_completion_context_set_proposals_for_provider(p_self: *CompletionContext, p_provider: *gtksource.CompletionProvider, p_results: ?*gio.ListModel) void;
    pub const setProposalsForProvider = gtk_source_completion_context_set_proposals_for_provider;

    extern fn gtk_source_completion_context_get_type() usize;
    pub const getGObjectType = gtk_source_completion_context_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `CompletionProvider` for the completion of snippets.
///
/// The `GtkSourceCompletionSnippets` is an example of an implementation of
/// the `CompletionProvider` interface. The proposals are snippets
/// registered with the `SnippetManager`.
pub const CompletionSnippets = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtksource.CompletionProvider};
    pub const Class = gtksource.CompletionSnippetsClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const priority = struct {
            pub const name = "priority";

            pub const Type = c_int;
        };

        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_completion_snippets_new() *gtksource.CompletionSnippets;
    pub const new = gtk_source_completion_snippets_new;

    extern fn gtk_source_completion_snippets_get_type() usize;
    pub const getGObjectType = gtk_source_completion_snippets_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionSnippets) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionSnippets) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionSnippets, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `CompletionProvider` for the completion of words.
///
/// The `GtkSourceCompletionWords` is an example of an implementation of
/// the `CompletionProvider` interface. The proposals are words
/// appearing in the registered `gtk.TextBuffer`s.
pub const CompletionWords = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtksource.CompletionProvider};
    pub const Class = gtksource.CompletionWordsClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const minimum_word_size = struct {
            pub const name = "minimum-word-size";

            pub const Type = c_uint;
        };

        pub const priority = struct {
            pub const name = "priority";

            pub const Type = c_int;
        };

        pub const proposals_batch_size = struct {
            pub const name = "proposals-batch-size";

            pub const Type = c_uint;
        };

        pub const scan_batch_size = struct {
            pub const name = "scan-batch-size";

            pub const Type = c_uint;
        };

        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_completion_words_new(p_title: ?[*:0]const u8) *gtksource.CompletionWords;
    pub const new = gtk_source_completion_words_new;

    /// Registers `buffer` in the `words` provider.
    extern fn gtk_source_completion_words_register(p_words: *CompletionWords, p_buffer: *gtk.TextBuffer) void;
    pub const register = gtk_source_completion_words_register;

    /// Unregisters `buffer` from the `words` provider.
    extern fn gtk_source_completion_words_unregister(p_words: *CompletionWords, p_buffer: *gtk.TextBuffer) void;
    pub const unregister = gtk_source_completion_words_unregister;

    extern fn gtk_source_completion_words_get_type() usize;
    pub const getGObjectType = gtk_source_completion_words_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionWords) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionWords) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionWords, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// On-disk representation of a `Buffer`.
///
/// A `GtkSourceFile` object is the on-disk representation of a `Buffer`.
/// With a `GtkSourceFile`, you can create and configure a `FileLoader`
/// and `FileSaver` which take by default the values of the
/// `GtkSourceFile` properties (except for the file loader which auto-detect some
/// properties). On a successful load or save operation, the `GtkSourceFile`
/// properties are updated. If an operation fails, the `GtkSourceFile` properties
/// have still the previous valid values.
pub const File = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.FileClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The compression type.
        pub const compression_type = struct {
            pub const name = "compression-type";

            pub const Type = gtksource.CompressionType;
        };

        /// The character encoding, initially `NULL`. After a successful file
        /// loading or saving operation, the encoding is non-`NULL`.
        pub const encoding = struct {
            pub const name = "encoding";

            pub const Type = ?*gtksource.Encoding;
        };

        /// The location.
        pub const location = struct {
            pub const name = "location";

            pub const Type = ?*gio.File;
        };

        /// The line ending type.
        pub const newline_type = struct {
            pub const name = "newline-type";

            pub const Type = gtksource.NewlineType;
        };

        /// Whether the file is read-only or not. The value of this property is
        /// not updated automatically (there is no file monitors).
        pub const read_only = struct {
            pub const name = "read-only";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_file_new() *gtksource.File;
    pub const new = gtk_source_file_new;

    /// Checks synchronously the file on disk, to know whether the file is externally
    /// modified, or has been deleted, and whether the file is read-only.
    ///
    /// `gtksource.File` doesn't create a `gio.FileMonitor` to track those properties, so
    /// this function needs to be called instead. Creating lots of `gio.FileMonitor`'s
    /// would take lots of resources.
    ///
    /// Since this function is synchronous, it is advised to call it only on local
    /// files. See `File.isLocal`.
    extern fn gtk_source_file_check_file_on_disk(p_file: *File) void;
    pub const checkFileOnDisk = gtk_source_file_check_file_on_disk;

    extern fn gtk_source_file_get_compression_type(p_file: *File) gtksource.CompressionType;
    pub const getCompressionType = gtk_source_file_get_compression_type;

    /// The encoding is initially `NULL`. After a successful file loading or saving
    /// operation, the encoding is non-`NULL`.
    extern fn gtk_source_file_get_encoding(p_file: *File) *const gtksource.Encoding;
    pub const getEncoding = gtk_source_file_get_encoding;

    extern fn gtk_source_file_get_location(p_file: *File) *gio.File;
    pub const getLocation = gtk_source_file_get_location;

    extern fn gtk_source_file_get_newline_type(p_file: *File) gtksource.NewlineType;
    pub const getNewlineType = gtk_source_file_get_newline_type;

    /// Returns whether the file has been deleted. If the
    /// `File.properties.location` is `NULL`, returns `FALSE`.
    ///
    /// To have an up-to-date value, you must first call
    /// `File.checkFileOnDisk`.
    extern fn gtk_source_file_is_deleted(p_file: *File) c_int;
    pub const isDeleted = gtk_source_file_is_deleted;

    /// Returns whether the file is externally modified. If the
    /// `File.properties.location` is `NULL`, returns `FALSE`.
    ///
    /// To have an up-to-date value, you must first call
    /// `File.checkFileOnDisk`.
    extern fn gtk_source_file_is_externally_modified(p_file: *File) c_int;
    pub const isExternallyModified = gtk_source_file_is_externally_modified;

    /// Returns whether the file is local. If the `File.properties.location` is `NULL`,
    /// returns `FALSE`.
    extern fn gtk_source_file_is_local(p_file: *File) c_int;
    pub const isLocal = gtk_source_file_is_local;

    /// Returns whether the file is read-only. If the
    /// `File.properties.location` is `NULL`, returns `FALSE`.
    ///
    /// To have an up-to-date value, you must first call
    /// `File.checkFileOnDisk`.
    extern fn gtk_source_file_is_readonly(p_file: *File) c_int;
    pub const isReadonly = gtk_source_file_is_readonly;

    /// Sets the location.
    extern fn gtk_source_file_set_location(p_file: *File, p_location: ?*gio.File) void;
    pub const setLocation = gtk_source_file_set_location;

    /// Sets a `MountOperationFactory` function that will be called when a
    /// `gio.MountOperation` must be created.
    ///
    /// This is useful for creating a `gtk.MountOperation` with the parent `gtk.Window`.
    ///
    /// If a mount operation factory isn't set, `gio.MountOperation.new` will be
    /// called.
    extern fn gtk_source_file_set_mount_operation_factory(p_file: *File, p_callback: gtksource.MountOperationFactory, p_user_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) void;
    pub const setMountOperationFactory = gtk_source_file_set_mount_operation_factory;

    extern fn gtk_source_file_get_type() usize;
    pub const getGObjectType = gtk_source_file_get_type;

    extern fn g_object_ref(p_self: *gtksource.File) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.File) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *File, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Load a file into a GtkSourceBuffer.
///
/// A `GtkSourceFileLoader` object permits to load the contents of a `gio.File` or a
/// `gio.InputStream` into a `Buffer`.
///
/// A file loader should be used only for one load operation, including errors
/// handling. If an error occurs, you can reconfigure the loader and relaunch the
/// operation with `FileLoader.loadAsync`.
///
/// Running a `GtkSourceFileLoader` is an undoable action for the
/// `Buffer`.
///
/// After a file loading, the buffer is reset to the contents provided by the
/// `gio.File` or `gio.InputStream`, so the buffer is set as “unmodified”, that is,
/// `gtk.TextBuffer.setModified` is called with `FALSE`. If the contents isn't
/// saved somewhere (for example if you load from stdin), then you should
/// probably call `gtk.TextBuffer.setModified` with `TRUE` after calling
/// `FileLoader.loadFinish`.
pub const FileLoader = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.FileLoaderClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gtksource.Buffer` to load the contents into. The
        /// `gtksource.FileLoader` object has a weak reference to the buffer.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtksource.Buffer;
        };

        /// The `gtksource.File`. The `gtksource.FileLoader` object has a weak
        /// reference to the file.
        pub const file = struct {
            pub const name = "file";

            pub const Type = ?*gtksource.File;
        };

        /// The `gio.InputStream` to load. Useful for reading stdin. If this property
        /// is set, the `gtksource.FileLoader.properties.location` property is ignored.
        pub const input_stream = struct {
            pub const name = "input-stream";

            pub const Type = ?*gio.InputStream;
        };

        /// The `gio.File` to load. If the `gtksource.FileLoader.properties.input`-stream is
        /// `NULL`, by default the location is taken from the `gtksource.File` at
        /// construction time.
        pub const location = struct {
            pub const name = "location";

            pub const Type = ?*gio.File;
        };
    };

    pub const signals = struct {};

    /// Creates a new `GtkSourceFileLoader` object. The contents is read from the
    /// `File`'s location.
    ///
    /// If not already done, call `File.setLocation` before calling this constructor.
    /// The previous location is anyway not needed, because as soon as the file loading begins,
    /// the `buffer` is emptied.
    extern fn gtk_source_file_loader_new(p_buffer: *gtksource.Buffer, p_file: *gtksource.File) *gtksource.FileLoader;
    pub const new = gtk_source_file_loader_new;

    /// Creates a new `gtksource.FileLoader` object. The contents is read from `stream`.
    extern fn gtk_source_file_loader_new_from_stream(p_buffer: *gtksource.Buffer, p_file: *gtksource.File, p_stream: *gio.InputStream) *gtksource.FileLoader;
    pub const newFromStream = gtk_source_file_loader_new_from_stream;

    extern fn gtk_source_file_loader_get_buffer(p_loader: *FileLoader) *gtksource.Buffer;
    pub const getBuffer = gtk_source_file_loader_get_buffer;

    extern fn gtk_source_file_loader_get_compression_type(p_loader: *FileLoader) gtksource.CompressionType;
    pub const getCompressionType = gtk_source_file_loader_get_compression_type;

    extern fn gtk_source_file_loader_get_encoding(p_loader: *FileLoader) *const gtksource.Encoding;
    pub const getEncoding = gtk_source_file_loader_get_encoding;

    extern fn gtk_source_file_loader_get_file(p_loader: *FileLoader) *gtksource.File;
    pub const getFile = gtk_source_file_loader_get_file;

    extern fn gtk_source_file_loader_get_input_stream(p_loader: *FileLoader) ?*gio.InputStream;
    pub const getInputStream = gtk_source_file_loader_get_input_stream;

    extern fn gtk_source_file_loader_get_location(p_loader: *FileLoader) ?*gio.File;
    pub const getLocation = gtk_source_file_loader_get_location;

    extern fn gtk_source_file_loader_get_newline_type(p_loader: *FileLoader) gtksource.NewlineType;
    pub const getNewlineType = gtk_source_file_loader_get_newline_type;

    /// Loads asynchronously the file or input stream contents into the `Buffer`.
    ///
    /// See the `gio.AsyncResult` documentation to know how to use this
    /// function.
    extern fn gtk_source_file_loader_load_async(p_loader: *FileLoader, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_progress_callback: ?gio.FileProgressCallback, p_progress_callback_data: ?*anyopaque, p_progress_callback_notify: ?glib.DestroyNotify, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const loadAsync = gtk_source_file_loader_load_async;

    /// Finishes a file loading started with `FileLoader.loadAsync`.
    ///
    /// If the contents has been loaded, the following `File` properties will
    /// be updated: the location, the encoding, the newline type and the compression
    /// type.
    extern fn gtk_source_file_loader_load_finish(p_loader: *FileLoader, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const loadFinish = gtk_source_file_loader_load_finish;

    /// Sets the candidate encodings for the file loading.
    ///
    /// The encodings are tried in the same order as the list.
    ///
    /// For convenience, `candidate_encodings` can contain duplicates. Only the first
    /// occurrence of a duplicated encoding is kept in the list.
    ///
    /// By default the candidate encodings are (in that order in the list):
    ///
    /// 1. If set, the `File`'s encoding as returned by `File.getEncoding`.
    /// 2. The default candidates as returned by `Encoding.getDefaultCandidates`.
    extern fn gtk_source_file_loader_set_candidate_encodings(p_loader: *FileLoader, p_candidate_encodings: *glib.SList) void;
    pub const setCandidateEncodings = gtk_source_file_loader_set_candidate_encodings;

    extern fn gtk_source_file_loader_get_type() usize;
    pub const getGObjectType = gtk_source_file_loader_get_type;

    extern fn g_object_ref(p_self: *gtksource.FileLoader) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.FileLoader) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileLoader, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Save a `Buffer` into a file.
///
/// A `GtkSourceFileSaver` object permits to save a `Buffer` into a
/// `gio.File`.
///
/// A file saver should be used only for one save operation, including errors
/// handling. If an error occurs, you can reconfigure the saver and relaunch the
/// operation with `FileSaver.saveAsync`.
pub const FileSaver = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.FileSaverClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gtksource.Buffer` to save. The `gtksource.FileSaver` object has a
        /// weak reference to the buffer.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtksource.Buffer;
        };

        /// The compression type.
        pub const compression_type = struct {
            pub const name = "compression-type";

            pub const Type = gtksource.CompressionType;
        };

        /// The file's encoding.
        pub const encoding = struct {
            pub const name = "encoding";

            pub const Type = ?*gtksource.Encoding;
        };

        /// The `gtksource.File`. The `gtksource.FileSaver` object has a weak
        /// reference to the file.
        pub const file = struct {
            pub const name = "file";

            pub const Type = ?*gtksource.File;
        };

        /// File saving flags.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gtksource.FileSaverFlags;
        };

        /// The `gio.File` where to save the buffer. By default the location is taken
        /// from the `gtksource.File` at construction time.
        pub const location = struct {
            pub const name = "location";

            pub const Type = ?*gio.File;
        };

        /// The newline type.
        pub const newline_type = struct {
            pub const name = "newline-type";

            pub const Type = gtksource.NewlineType;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtksource.FileSaver` object. The `buffer` will be saved to the
    /// `File`'s location.
    ///
    /// This constructor is suitable for a simple "save" operation, when the `file`
    /// already contains a non-`NULL` `File.properties.location`.
    extern fn gtk_source_file_saver_new(p_buffer: *gtksource.Buffer, p_file: *gtksource.File) *gtksource.FileSaver;
    pub const new = gtk_source_file_saver_new;

    /// Creates a new `gtksource.FileSaver` object with a target location.
    ///
    /// When the file saving is finished successfully, `target_location` is set to the `file`'s
    /// `File.properties.location` property. If an error occurs, the previous valid
    /// location is still available in `File`.
    ///
    /// This constructor is suitable for a "save as" operation, or for saving a new
    /// buffer for the first time.
    extern fn gtk_source_file_saver_new_with_target(p_buffer: *gtksource.Buffer, p_file: *gtksource.File, p_target_location: *gio.File) *gtksource.FileSaver;
    pub const newWithTarget = gtk_source_file_saver_new_with_target;

    extern fn gtk_source_file_saver_get_buffer(p_saver: *FileSaver) *gtksource.Buffer;
    pub const getBuffer = gtk_source_file_saver_get_buffer;

    extern fn gtk_source_file_saver_get_compression_type(p_saver: *FileSaver) gtksource.CompressionType;
    pub const getCompressionType = gtk_source_file_saver_get_compression_type;

    extern fn gtk_source_file_saver_get_encoding(p_saver: *FileSaver) *const gtksource.Encoding;
    pub const getEncoding = gtk_source_file_saver_get_encoding;

    extern fn gtk_source_file_saver_get_file(p_saver: *FileSaver) *gtksource.File;
    pub const getFile = gtk_source_file_saver_get_file;

    extern fn gtk_source_file_saver_get_flags(p_saver: *FileSaver) gtksource.FileSaverFlags;
    pub const getFlags = gtk_source_file_saver_get_flags;

    extern fn gtk_source_file_saver_get_location(p_saver: *FileSaver) *gio.File;
    pub const getLocation = gtk_source_file_saver_get_location;

    extern fn gtk_source_file_saver_get_newline_type(p_saver: *FileSaver) gtksource.NewlineType;
    pub const getNewlineType = gtk_source_file_saver_get_newline_type;

    /// Saves asynchronously the buffer into the file.
    ///
    /// See the `gio.AsyncResult` documentation to know how to use this function.
    extern fn gtk_source_file_saver_save_async(p_saver: *FileSaver, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_progress_callback: ?gio.FileProgressCallback, p_progress_callback_data: ?*anyopaque, p_progress_callback_notify: ?glib.DestroyNotify, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const saveAsync = gtk_source_file_saver_save_async;

    /// Finishes a file saving started with `FileSaver.saveAsync`.
    ///
    /// If the file has been saved successfully, the following `File`
    /// properties will be updated: the location, the encoding, the newline type and
    /// the compression type.
    ///
    /// Since the 3.20 version, `gtk.TextBuffer.setModified` is called with `FALSE`
    /// if the file has been saved successfully.
    extern fn gtk_source_file_saver_save_finish(p_saver: *FileSaver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const saveFinish = gtk_source_file_saver_save_finish;

    /// Sets the compression type. By default the compression type is taken from the
    /// `gtksource.File`.
    extern fn gtk_source_file_saver_set_compression_type(p_saver: *FileSaver, p_compression_type: gtksource.CompressionType) void;
    pub const setCompressionType = gtk_source_file_saver_set_compression_type;

    /// Sets the encoding. If `encoding` is `NULL`, the UTF-8 encoding will be set.
    ///
    /// By default the encoding is taken from the `gtksource.File`.
    extern fn gtk_source_file_saver_set_encoding(p_saver: *FileSaver, p_encoding: ?*const gtksource.Encoding) void;
    pub const setEncoding = gtk_source_file_saver_set_encoding;

    extern fn gtk_source_file_saver_set_flags(p_saver: *FileSaver, p_flags: gtksource.FileSaverFlags) void;
    pub const setFlags = gtk_source_file_saver_set_flags;

    /// Sets the newline type. By default the newline type is taken from the
    /// `gtksource.File`.
    extern fn gtk_source_file_saver_set_newline_type(p_saver: *FileSaver, p_newline_type: gtksource.NewlineType) void;
    pub const setNewlineType = gtk_source_file_saver_set_newline_type;

    extern fn gtk_source_file_saver_get_type() usize;
    pub const getGObjectType = gtk_source_file_saver_get_type;

    extern fn g_object_ref(p_self: *gtksource.FileSaver) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.FileSaver) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileSaver, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Gutter object for `View`.
///
/// The `GtkSourceGutter` object represents the left or right gutter of the text
/// view. It is used by `View` to draw the line numbers and
/// `Mark`s that might be present on a line. By packing
/// additional `GutterRenderer` objects in the gutter, you can extend the
/// gutter with your own custom drawings.
///
/// To get a `GtkSourceGutter`, use the `View.getGutter` function.
///
/// The gutter works very much the same way as cells rendered in a `gtk.TreeView`.
/// The concept is similar, with the exception that the gutter does not have an
/// underlying `gtk.TreeModel`. The builtin line number renderer is at position
/// `GTK_SOURCE_VIEW_GUTTER_POSITION_LINES` (-30) and the marks renderer is at
/// `GTK_SOURCE_VIEW_GUTTER_POSITION_MARKS` (-20). The gutter sorts the renderers
/// in ascending order, from left to right. So the marks are displayed on the
/// right of the line numbers.
pub const Gutter = opaque {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.GutterClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gtksource.View` of the gutter.
        pub const view = struct {
            pub const name = "view";

            pub const Type = ?*gtksource.View;
        };

        /// The text window type on which the window is placed.
        pub const window_type = struct {
            pub const name = "window-type";

            pub const Type = gtk.TextWindowType;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_gutter_get_view(p_gutter: *Gutter) *gtksource.View;
    pub const getView = gtk_source_gutter_get_view;

    /// Insert `renderer` into the gutter. If `renderer` is yet unowned then gutter
    /// claims its ownership. Otherwise just increases renderer's reference count.
    /// `renderer` cannot be already inserted to another gutter.
    extern fn gtk_source_gutter_insert(p_gutter: *Gutter, p_renderer: *gtksource.GutterRenderer, p_position: c_int) c_int;
    pub const insert = gtk_source_gutter_insert;

    extern fn gtk_source_gutter_remove(p_gutter: *Gutter, p_renderer: *gtksource.GutterRenderer) void;
    pub const remove = gtk_source_gutter_remove;

    /// Reorders `renderer` in `gutter` to new `position`.
    extern fn gtk_source_gutter_reorder(p_gutter: *Gutter, p_renderer: *gtksource.GutterRenderer, p_position: c_int) void;
    pub const reorder = gtk_source_gutter_reorder;

    extern fn gtk_source_gutter_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_get_type;

    extern fn g_object_ref(p_self: *gtksource.Gutter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Gutter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Gutter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Collected information about visible lines.
///
/// The `GtkSourceGutterLines` object is used to collect information about
/// visible lines.
///
/// Use this from your `GutterRenderer.signals.query_data` to collect the
/// necessary information on visible lines. Doing so reduces the number of
/// passes through the text btree allowing GtkSourceView to reach more
/// frames-per-second while performing kinetic scrolling.
pub const GutterLines = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.GutterLinesClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Adds the class `name` to `line`.
    ///
    /// `name` will be converted to a `glib.Quark` as part of this process. A
    /// faster version of this function is available via
    /// `GutterLines.addQclass` for situations where the `glib.Quark` is
    /// known ahead of time.
    extern fn gtk_source_gutter_lines_add_class(p_lines: *GutterLines, p_line: c_uint, p_name: [*:0]const u8) void;
    pub const addClass = gtk_source_gutter_lines_add_class;

    /// Adds the class denoted by `qname` to `line`.
    ///
    /// You may check if a line has `qname` by calling
    /// `GutterLines.hasQclass`.
    ///
    /// You can remove `qname` by calling
    /// `GutterLines.removeQclass`.
    extern fn gtk_source_gutter_lines_add_qclass(p_lines: *GutterLines, p_line: c_uint, p_qname: glib.Quark) void;
    pub const addQclass = gtk_source_gutter_lines_add_qclass;

    /// Gets the `gtk.TextBuffer` that the `GtkSourceGutterLines` represents.
    extern fn gtk_source_gutter_lines_get_buffer(p_lines: *GutterLines) *gtk.TextBuffer;
    pub const getBuffer = gtk_source_gutter_lines_get_buffer;

    /// Gets the line number (starting from 0) for the first line that is
    /// user visible.
    extern fn gtk_source_gutter_lines_get_first(p_lines: *GutterLines) c_uint;
    pub const getFirst = gtk_source_gutter_lines_get_first;

    /// Gets a `gtk.TextIter` for the current buffer at `line`
    extern fn gtk_source_gutter_lines_get_iter_at_line(p_lines: *GutterLines, p_iter: *gtk.TextIter, p_line: c_uint) void;
    pub const getIterAtLine = gtk_source_gutter_lines_get_iter_at_line;

    /// Gets the line number (starting from 0) for the last line that is
    /// user visible.
    extern fn gtk_source_gutter_lines_get_last(p_lines: *GutterLines) c_uint;
    pub const getLast = gtk_source_gutter_lines_get_last;

    /// Gets the Y range for a line based on `mode`.
    ///
    /// The value for `y` is relative to the renderers widget coordinates.
    extern fn gtk_source_gutter_lines_get_line_yrange(p_lines: *GutterLines, p_line: c_uint, p_mode: gtksource.GutterRendererAlignmentMode, p_y: *c_int, p_height: *c_int) void;
    pub const getLineYrange = gtk_source_gutter_lines_get_line_yrange;

    /// Gets the `gtk.TextView` that the `GtkSourceGutterLines` represents.
    extern fn gtk_source_gutter_lines_get_view(p_lines: *GutterLines) *gtk.TextView;
    pub const getView = gtk_source_gutter_lines_get_view;

    /// Checks to see if the line has any GQuark classes set. This can be
    /// used to help renderer implementations avoid work if nothing has
    /// been set on the class.
    extern fn gtk_source_gutter_lines_has_any_class(p_lines: *GutterLines, p_line: c_uint) c_int;
    pub const hasAnyClass = gtk_source_gutter_lines_has_any_class;

    /// Checks to see if `GutterLines.addClass` was called with
    /// the `name` for `line`.
    ///
    /// A faster version of this function is provided via
    /// `GutterLines.hasQclass` for situations where the quark
    /// is known ahead of time.
    extern fn gtk_source_gutter_lines_has_class(p_lines: *GutterLines, p_line: c_uint, p_name: [*:0]const u8) c_int;
    pub const hasClass = gtk_source_gutter_lines_has_class;

    /// Checks to see if `GutterLines.addQclass` was called with
    /// the quark denoted by `qname` for `line`.
    extern fn gtk_source_gutter_lines_has_qclass(p_lines: *GutterLines, p_line: c_uint, p_qname: glib.Quark) c_int;
    pub const hasQclass = gtk_source_gutter_lines_has_qclass;

    /// Checks to see if `line` contains the insertion cursor.
    extern fn gtk_source_gutter_lines_is_cursor(p_lines: *GutterLines, p_line: c_uint) c_int;
    pub const isCursor = gtk_source_gutter_lines_is_cursor;

    /// Checks to see if `line` is marked as prelit. Generally, this means
    /// the mouse pointer is over the line within the gutter.
    extern fn gtk_source_gutter_lines_is_prelit(p_lines: *GutterLines, p_line: c_uint) c_int;
    pub const isPrelit = gtk_source_gutter_lines_is_prelit;

    /// Checks to see if the view had a selection and if that selection overlaps
    /// `line` in some way.
    extern fn gtk_source_gutter_lines_is_selected(p_lines: *GutterLines, p_line: c_uint) c_int;
    pub const isSelected = gtk_source_gutter_lines_is_selected;

    /// Removes the class matching `name` from `line`.
    ///
    /// A faster version of this function is available via
    /// `GutterLines.removeQclass` for situations where the
    /// `glib.Quark` is known ahead of time.
    extern fn gtk_source_gutter_lines_remove_class(p_lines: *GutterLines, p_line: c_uint, p_name: [*:0]const u8) void;
    pub const removeClass = gtk_source_gutter_lines_remove_class;

    /// Reverses a call to `GutterLines.addQclass` by removing
    /// the `glib.Quark` matching `qname`.
    extern fn gtk_source_gutter_lines_remove_qclass(p_lines: *GutterLines, p_line: c_uint, p_qname: glib.Quark) void;
    pub const removeQclass = gtk_source_gutter_lines_remove_qclass;

    extern fn gtk_source_gutter_lines_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_lines_get_type;

    extern fn g_object_ref(p_self: *gtksource.GutterLines) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.GutterLines) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GutterLines, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Gutter cell renderer.
///
/// A `GtkSourceGutterRenderer` represents a column in a `Gutter`. The
/// column contains one cell for each visible line of the `gtk.TextBuffer`. Due to
/// text wrapping, a cell can thus span multiple lines of the `gtk.TextView`. In
/// this case, `GutterRendererAlignmentMode` controls the alignment of
/// the cell.
///
/// The gutter renderer is a `gtk.Widget` and is measured using the normal widget
/// measurement facilities. The width of the gutter will be determined by the
/// measurements of the gutter renderers.
///
/// The width of a gutter renderer generally takes into account the entire text
/// buffer. For instance, to display the line numbers, if the buffer contains 100
/// lines, the gutter renderer will always set its width such as three digits can
/// be printed, even if only the first 20 lines are shown. Another strategy is to
/// take into account only the visible lines.  In this case, only two digits are
/// necessary to display the line numbers of the first 20 lines. To take another
/// example, the gutter renderer for `Mark`s doesn't need to take
/// into account the text buffer to announce its width. It only depends on the
/// icons size displayed in the gutter column.
///
/// When the available size to render a cell is greater than the required size to
/// render the cell contents, the cell contents can be aligned horizontally and
/// vertically with `GutterRenderer.setAlignmentMode`.
///
/// The cells rendering occurs using `gtk.Widget.virtual_methods.snapshot`. Implementations
/// should use ``gtk_source_gutter_renderer_get_lines`` to retrieve information
/// about the lines to be rendered. To help with aligning content which takes
/// into account the padding and alignment of a cell, implementations may call
/// `GutterRenderer.alignCell` for a given line number with the
/// width and height measurement of the content they width to render.
pub const GutterRenderer = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.GutterRendererClass;
    f_parent_instance: gtk.Widget,

    pub const virtual_methods = struct {
        /// Emits the `GutterRenderer.signals.activate` signal of the renderer. This is
        /// called from `Gutter` and should never have to be called manually.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_activate.?(gobject.ext.as(GutterRenderer, p_renderer), p_iter, p_area, p_button, p_state, p_n_presses);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        pub const begin = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lines: *gtksource.GutterLines) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_begin.?(gobject.ext.as(GutterRenderer, p_renderer), p_lines);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lines: *gtksource.GutterLines) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_begin = @ptrCast(p_implementation);
            }
        };

        /// This is called when the text buffer changes for `renderer`.
        pub const change_buffer = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_old_buffer: ?*gtksource.Buffer) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_change_buffer.?(gobject.ext.as(GutterRenderer, p_renderer), p_old_buffer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_old_buffer: ?*gtksource.Buffer) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_change_buffer = @ptrCast(p_implementation);
            }
        };

        /// This is called when the text view changes for `renderer`.
        pub const change_view = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_old_view: ?*gtksource.View) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_change_view.?(gobject.ext.as(GutterRenderer, p_renderer), p_old_view);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_old_view: ?*gtksource.View) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_change_view = @ptrCast(p_implementation);
            }
        };

        pub const end = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_end.?(gobject.ext.as(GutterRenderer, p_renderer));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_end = @ptrCast(p_implementation);
            }
        };

        /// Get whether the renderer is activatable at the location provided. This is
        /// called from `Gutter` to determine whether a renderer is activatable
        /// using the mouse pointer.
        pub const query_activatable = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle) c_int {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_query_activatable.?(gobject.ext.as(GutterRenderer, p_renderer), p_iter, p_area);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle) callconv(.C) c_int) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_query_activatable = @ptrCast(p_implementation);
            }
        };

        pub const query_data = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lines: *gtksource.GutterLines, p_line: c_uint) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_query_data.?(gobject.ext.as(GutterRenderer, p_renderer), p_lines, p_line);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_lines: *gtksource.GutterLines, p_line: c_uint) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_query_data = @ptrCast(p_implementation);
            }
        };

        pub const snapshot_line = struct {
            pub fn call(p_class: anytype, p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_snapshot: *gtk.Snapshot, p_lines: *gtksource.GutterLines, p_line: c_uint) void {
                return gobject.ext.as(GutterRenderer.Class, p_class).f_snapshot_line.?(gobject.ext.as(GutterRenderer, p_renderer), p_snapshot, p_lines, p_line);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_renderer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_snapshot: *gtk.Snapshot, p_lines: *gtksource.GutterLines, p_line: c_uint) callconv(.C) void) void {
                gobject.ext.as(GutterRenderer.Class, p_class).f_snapshot_line = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The alignment mode of the renderer.
        ///
        /// This can be used to indicate that in the case a cell spans multiple lines (due to text wrapping)
        /// the alignment should work on either the full cell, the first line or the last line.
        pub const alignment_mode = struct {
            pub const name = "alignment-mode";

            pub const Type = gtksource.GutterRendererAlignmentMode;
        };

        /// Contains information about the lines to be rendered.
        ///
        /// It should be used by `gtksource.GutterRenderer` implementations from `gtk.Widget.virtual_methods.snapshot`.
        pub const lines = struct {
            pub const name = "lines";

            pub const Type = ?*gtksource.GutterLines;
        };

        /// The view on which the renderer is placed.
        pub const view = struct {
            pub const name = "view";

            pub const Type = ?*gtk.TextView;
        };

        /// The horizontal alignment of the renderer.
        ///
        /// Set to 0 for a left alignment. 1 for a right alignment. And 0.5 for centering the cells.
        /// A value lower than 0 doesn't modify the alignment.
        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        /// The left and right padding of the renderer.
        pub const xpad = struct {
            pub const name = "xpad";

            pub const Type = c_int;
        };

        /// The vertical alignment of the renderer.
        ///
        /// Set to 0 for a top alignment. 1 for a bottom alignment. And 0.5 for centering the cells.
        /// A value lower than 0 doesn't modify the alignment.
        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };

        /// The top and bottom padding of the renderer.
        pub const ypad = struct {
            pub const name = "ypad";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The signal is emitted when the renderer is activated.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_iter: *gtk.TextIter, p_area: *gdk.Rectangle, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GutterRenderer, p_instance))),
                    gobject.signalLookup("activate", GutterRenderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is emitted when the renderer can possibly be activated.
        pub const query_activatable = struct {
            pub const name = "query-activatable";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_iter: *gtk.TextIter, p_area: *gdk.Rectangle, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GutterRenderer, p_instance))),
                    gobject.signalLookup("query-activatable", GutterRenderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const query_data = struct {
            pub const name = "query-data";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: *gobject.Object, p_p0: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GutterRenderer, p_instance))),
                    gobject.signalLookup("query-data", GutterRenderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Emits the `GutterRenderer.signals.activate` signal of the renderer. This is
    /// called from `Gutter` and should never have to be called manually.
    extern fn gtk_source_gutter_renderer_activate(p_renderer: *GutterRenderer, p_iter: *const gtk.TextIter, p_area: *const gdk.Rectangle, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) void;
    pub const activate = gtk_source_gutter_renderer_activate;

    /// Locates where to render content that is `width` x `height` based on
    /// the renderers alignment and padding.
    ///
    /// The location will be placed into `x` and `y` and is relative to the
    /// renderer's coordinates.
    ///
    /// It is encouraged that renderers use this function when snappshotting
    /// to ensure consistent placement of their contents.
    extern fn gtk_source_gutter_renderer_align_cell(p_renderer: *GutterRenderer, p_line: c_uint, p_width: f32, p_height: f32, p_x: *f32, p_y: *f32) void;
    pub const alignCell = gtk_source_gutter_renderer_align_cell;

    /// Get the alignment mode.
    ///
    /// The alignment mode describes the manner in which the
    /// renderer is aligned (see `GutterRenderer.properties.xalign` and
    /// `GutterRenderer.properties.yalign`).
    extern fn gtk_source_gutter_renderer_get_alignment_mode(p_renderer: *GutterRenderer) gtksource.GutterRendererAlignmentMode;
    pub const getAlignmentMode = gtk_source_gutter_renderer_get_alignment_mode;

    /// Gets the `Buffer` for which the gutter renderer is drawing.
    extern fn gtk_source_gutter_renderer_get_buffer(p_renderer: *GutterRenderer) ?*gtksource.Buffer;
    pub const getBuffer = gtk_source_gutter_renderer_get_buffer;

    /// Get the view associated to the gutter renderer
    extern fn gtk_source_gutter_renderer_get_view(p_renderer: *GutterRenderer) *gtksource.View;
    pub const getView = gtk_source_gutter_renderer_get_view;

    /// Gets the `xalign` property.
    ///
    /// This may be used to adjust where within the cell rectangle the renderer will draw.
    extern fn gtk_source_gutter_renderer_get_xalign(p_renderer: *GutterRenderer) f32;
    pub const getXalign = gtk_source_gutter_renderer_get_xalign;

    /// Gets the `xpad` property.
    ///
    /// This may be used to adjust the cell rectangle that the renderer will use to draw.
    extern fn gtk_source_gutter_renderer_get_xpad(p_renderer: *GutterRenderer) c_int;
    pub const getXpad = gtk_source_gutter_renderer_get_xpad;

    /// Gets the `yalign` property.
    ///
    /// This may be used to adjust where within the cell rectangle the renderer will draw.
    extern fn gtk_source_gutter_renderer_get_yalign(p_renderer: *GutterRenderer) f32;
    pub const getYalign = gtk_source_gutter_renderer_get_yalign;

    /// Gets the `ypad` property.
    ///
    /// This may be used to adjust the cell rectangle that the renderer will use to draw.
    extern fn gtk_source_gutter_renderer_get_ypad(p_renderer: *GutterRenderer) c_int;
    pub const getYpad = gtk_source_gutter_renderer_get_ypad;

    /// Get whether the renderer is activatable at the location provided. This is
    /// called from `Gutter` to determine whether a renderer is activatable
    /// using the mouse pointer.
    extern fn gtk_source_gutter_renderer_query_activatable(p_renderer: *GutterRenderer, p_iter: *const gtk.TextIter, p_area: *const gdk.Rectangle) c_int;
    pub const queryActivatable = gtk_source_gutter_renderer_query_activatable;

    /// Set the alignment mode. The alignment mode describes the manner in which the
    /// renderer is aligned (see `GutterRenderer.properties.xalign` and
    /// `GutterRenderer.properties.yalign`).
    extern fn gtk_source_gutter_renderer_set_alignment_mode(p_renderer: *GutterRenderer, p_mode: gtksource.GutterRendererAlignmentMode) void;
    pub const setAlignmentMode = gtk_source_gutter_renderer_set_alignment_mode;

    /// Adjusts the `xalign` property.
    ///
    /// This may be used to adjust where within the cell rectangle the renderer will draw.
    extern fn gtk_source_gutter_renderer_set_xalign(p_renderer: *GutterRenderer, p_xalign: f32) void;
    pub const setXalign = gtk_source_gutter_renderer_set_xalign;

    /// Adjusts the `xpad` property.
    ///
    /// This may be used to adjust the cell rectangle that the renderer will use to draw.
    extern fn gtk_source_gutter_renderer_set_xpad(p_renderer: *GutterRenderer, p_xpad: c_int) void;
    pub const setXpad = gtk_source_gutter_renderer_set_xpad;

    /// Adjusts the `yalign` property.
    ///
    /// This may be used to adjust where within the cell rectangle the renderer will draw.
    extern fn gtk_source_gutter_renderer_set_yalign(p_renderer: *GutterRenderer, p_yalign: f32) void;
    pub const setYalign = gtk_source_gutter_renderer_set_yalign;

    /// Adjusts the `ypad` property.
    ///
    /// This may be used to adjust the cell rectangle that the renderer will use to draw.
    extern fn gtk_source_gutter_renderer_set_ypad(p_renderer: *GutterRenderer, p_ypad: c_int) void;
    pub const setYpad = gtk_source_gutter_renderer_set_ypad;

    extern fn gtk_source_gutter_renderer_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_renderer_get_type;

    extern fn g_object_ref(p_self: *gtksource.GutterRenderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.GutterRenderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GutterRenderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Renders a pixbuf in the gutter.
///
/// A `GtkSourceGutterRendererPixbuf` can be used to render an image in a cell of
/// `Gutter`.
pub const GutterRendererPixbuf = extern struct {
    pub const Parent = gtksource.GutterRenderer;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.GutterRendererPixbufClass;
    f_parent_instance: gtksource.GutterRenderer,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const gicon = struct {
            pub const name = "gicon";

            pub const Type = ?*gio.Icon;
        };

        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        pub const paintable = struct {
            pub const name = "paintable";

            pub const Type = ?*gdk.Paintable;
        };

        pub const pixbuf = struct {
            pub const name = "pixbuf";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };
    };

    pub const signals = struct {};

    /// Create a new `gtksource.GutterRendererPixbuf`.
    extern fn gtk_source_gutter_renderer_pixbuf_new() *gtksource.GutterRendererPixbuf;
    pub const new = gtk_source_gutter_renderer_pixbuf_new;

    /// Get the gicon of the renderer
    extern fn gtk_source_gutter_renderer_pixbuf_get_gicon(p_renderer: *GutterRendererPixbuf) *gio.Icon;
    pub const getGicon = gtk_source_gutter_renderer_pixbuf_get_gicon;

    extern fn gtk_source_gutter_renderer_pixbuf_get_icon_name(p_renderer: *GutterRendererPixbuf) [*:0]const u8;
    pub const getIconName = gtk_source_gutter_renderer_pixbuf_get_icon_name;

    /// Gets a `gdk.Paintable` that was set with
    /// `GutterRendererPixbuf.setPaintable`
    extern fn gtk_source_gutter_renderer_pixbuf_get_paintable(p_renderer: *GutterRendererPixbuf) ?*gdk.Paintable;
    pub const getPaintable = gtk_source_gutter_renderer_pixbuf_get_paintable;

    /// Get the pixbuf of the renderer.
    extern fn gtk_source_gutter_renderer_pixbuf_get_pixbuf(p_renderer: *GutterRendererPixbuf) *gdkpixbuf.Pixbuf;
    pub const getPixbuf = gtk_source_gutter_renderer_pixbuf_get_pixbuf;

    /// Allows overlaying a paintable on top of any other image that
    /// has been set for the pixbuf. This will be applied when the
    /// widget is next snapshot.
    extern fn gtk_source_gutter_renderer_pixbuf_overlay_paintable(p_renderer: *GutterRendererPixbuf, p_paintable: *gdk.Paintable) void;
    pub const overlayPaintable = gtk_source_gutter_renderer_pixbuf_overlay_paintable;

    extern fn gtk_source_gutter_renderer_pixbuf_set_gicon(p_renderer: *GutterRendererPixbuf, p_icon: ?*gio.Icon) void;
    pub const setGicon = gtk_source_gutter_renderer_pixbuf_set_gicon;

    extern fn gtk_source_gutter_renderer_pixbuf_set_icon_name(p_renderer: *GutterRendererPixbuf, p_icon_name: ?[*:0]const u8) void;
    pub const setIconName = gtk_source_gutter_renderer_pixbuf_set_icon_name;

    extern fn gtk_source_gutter_renderer_pixbuf_set_paintable(p_renderer: *GutterRendererPixbuf, p_paintable: ?*gdk.Paintable) void;
    pub const setPaintable = gtk_source_gutter_renderer_pixbuf_set_paintable;

    extern fn gtk_source_gutter_renderer_pixbuf_set_pixbuf(p_renderer: *GutterRendererPixbuf, p_pixbuf: ?*gdkpixbuf.Pixbuf) void;
    pub const setPixbuf = gtk_source_gutter_renderer_pixbuf_set_pixbuf;

    extern fn gtk_source_gutter_renderer_pixbuf_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_renderer_pixbuf_get_type;

    extern fn g_object_ref(p_self: *gtksource.GutterRendererPixbuf) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.GutterRendererPixbuf) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GutterRendererPixbuf, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Renders text in the gutter.
///
/// A `GtkSourceGutterRendererText` can be used to render text in a cell of
/// `Gutter`.
pub const GutterRendererText = extern struct {
    pub const Parent = gtksource.GutterRenderer;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.GutterRendererTextClass;
    f_parent_instance: gtksource.GutterRenderer,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const markup = struct {
            pub const name = "markup";

            pub const Type = ?[*:0]u8;
        };

        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a new `gtksource.GutterRendererText`.
    extern fn gtk_source_gutter_renderer_text_new() *gtksource.GutterRendererText;
    pub const new = gtk_source_gutter_renderer_text_new;

    /// Measures the text provided using the pango layout used by the
    /// `gtksource.GutterRendererText`.
    extern fn gtk_source_gutter_renderer_text_measure(p_renderer: *GutterRendererText, p_text: [*:0]const u8, p_width: ?*c_int, p_height: ?*c_int) void;
    pub const measure = gtk_source_gutter_renderer_text_measure;

    /// Measures the pango markup provided using the pango layout used by the
    /// `gtksource.GutterRendererText`.
    extern fn gtk_source_gutter_renderer_text_measure_markup(p_renderer: *GutterRendererText, p_markup: [*:0]const u8, p_width: ?*c_int, p_height: ?*c_int) void;
    pub const measureMarkup = gtk_source_gutter_renderer_text_measure_markup;

    extern fn gtk_source_gutter_renderer_text_set_markup(p_renderer: *GutterRendererText, p_markup: [*:0]const u8, p_length: c_int) void;
    pub const setMarkup = gtk_source_gutter_renderer_text_set_markup;

    extern fn gtk_source_gutter_renderer_text_set_text(p_renderer: *GutterRendererText, p_text: [*:0]const u8, p_length: c_int) void;
    pub const setText = gtk_source_gutter_renderer_text_set_text;

    extern fn gtk_source_gutter_renderer_text_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_renderer_text_get_type;

    extern fn g_object_ref(p_self: *gtksource.GutterRendererText) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.GutterRendererText) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GutterRendererText, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interactive tooltips.
///
/// `GtkSourceHover` allows a `View` to provide contextual information.
/// When enabled, if the user hovers over a word in the text editor, a series
/// of registered `HoverProvider` can populate a `HoverDisplay`
/// with useful information.
///
/// To enable call `View.getHover` and add `HoverProvider`
/// using `Hover.addProvider`. To disable, remove all registered
/// providers with `Hover.removeProvider`.
///
/// You can change how long to wait to display the interactive tooltip by
/// setting the `Hover.properties.hover_delay` property in milliseconds.
pub const Hover = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.HoverClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Contains the number of milliseconds to delay before showing the hover assistant.
        pub const hover_delay = struct {
            pub const name = "hover-delay";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_hover_add_provider(p_self: *Hover, p_provider: *gtksource.HoverProvider) void;
    pub const addProvider = gtk_source_hover_add_provider;

    extern fn gtk_source_hover_remove_provider(p_self: *Hover, p_provider: *gtksource.HoverProvider) void;
    pub const removeProvider = gtk_source_hover_remove_provider;

    extern fn gtk_source_hover_get_type() usize;
    pub const getGObjectType = gtk_source_hover_get_type;

    extern fn g_object_ref(p_self: *gtksource.Hover) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Hover) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Hover, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Context for populating `HoverDisplay` contents.
///
/// `GtkSourceHoverContext` contains information about the request to populate
/// contents for a `HoverDisplay`.
///
/// It can be used to retrieve the `View`, `Buffer`, and
/// `gtk.TextIter` for the regions of text which are being displayed.
///
/// Use `HoverContext.getBounds` to get the word that was
/// requested. `HoverContext.getIter` will get you the location
/// of the pointer when the request was made.
pub const HoverContext = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.HoverContextClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the current word bounds of the hover.
    ///
    /// If `begin` is non-`NULL`, it will be set to the start position of the
    /// current word being hovered.
    ///
    /// If `end` is non-`NULL`, it will be set to the end position for the
    /// current word being hovered.
    extern fn gtk_source_hover_context_get_bounds(p_self: *HoverContext, p_begin: ?*gtk.TextIter, p_end: ?*gtk.TextIter) c_int;
    pub const getBounds = gtk_source_hover_context_get_bounds;

    /// A convenience function to get the buffer.
    extern fn gtk_source_hover_context_get_buffer(p_self: *HoverContext) *gtksource.Buffer;
    pub const getBuffer = gtk_source_hover_context_get_buffer;

    extern fn gtk_source_hover_context_get_iter(p_self: *HoverContext, p_iter: *gtk.TextIter) c_int;
    pub const getIter = gtk_source_hover_context_get_iter;

    extern fn gtk_source_hover_context_get_view(p_self: *HoverContext) *gtksource.View;
    pub const getView = gtk_source_hover_context_get_view;

    extern fn gtk_source_hover_context_get_type() usize;
    pub const getGObjectType = gtk_source_hover_context_get_type;

    extern fn g_object_ref(p_self: *gtksource.HoverContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.HoverContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HoverContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Display for interactive tooltips.
///
/// `GtkSourceHoverDisplay` is a `gtk.Widget` that may be populated with widgets
/// to be displayed to the user in interactive tooltips. The children widgets
/// are packed vertically using a `gtk.Box`.
///
/// Implement the `HoverProvider` interface to be notified of when
/// to populate a `GtkSourceHoverDisplay` on behalf of the user.
pub const HoverDisplay = opaque {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.HoverDisplayClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_source_hover_display_append(p_self: *HoverDisplay, p_child: *gtk.Widget) void;
    pub const append = gtk_source_hover_display_append;

    extern fn gtk_source_hover_display_insert_after(p_self: *HoverDisplay, p_child: *gtk.Widget, p_sibling: *gtk.Widget) void;
    pub const insertAfter = gtk_source_hover_display_insert_after;

    extern fn gtk_source_hover_display_prepend(p_self: *HoverDisplay, p_child: *gtk.Widget) void;
    pub const prepend = gtk_source_hover_display_prepend;

    extern fn gtk_source_hover_display_remove(p_self: *HoverDisplay, p_child: *gtk.Widget) void;
    pub const remove = gtk_source_hover_display_remove;

    extern fn gtk_source_hover_display_get_type() usize;
    pub const getGObjectType = gtk_source_hover_display_get_type;

    extern fn g_object_ref(p_self: *gtksource.HoverDisplay) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.HoverDisplay) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HoverDisplay, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Represents a syntax highlighted language.
///
/// A `GtkSourceLanguage` represents a programming or markup language, affecting
/// syntax highlighting and [context classes](./class.Buffer.html`context`-classes).
///
/// Use `LanguageManager` to obtain a `GtkSourceLanguage` instance, and
/// `Buffer.setLanguage` to apply it to a `Buffer`.
pub const Language = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.LanguageClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const hidden = struct {
            pub const name = "hidden";

            pub const Type = c_int;
        };

        pub const id = struct {
            pub const name = "id";

            pub const Type = ?[*:0]u8;
        };

        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        pub const section = struct {
            pub const name = "section";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Returns the globs associated to this language.
    ///
    /// This is just an utility wrapper around `Language.getMetadata` to
    /// retrieve the "globs" metadata property and split it into an array.
    extern fn gtk_source_language_get_globs(p_language: *Language) ?[*][*:0]u8;
    pub const getGlobs = gtk_source_language_get_globs;

    /// Returns whether the language should be hidden from the user.
    extern fn gtk_source_language_get_hidden(p_language: *Language) c_int;
    pub const getHidden = gtk_source_language_get_hidden;

    /// Returns the ID of the language.
    ///
    /// The ID is not locale-dependent.The returned string is owned by `language`
    /// and should not be freed or modified.
    extern fn gtk_source_language_get_id(p_language: *Language) [*:0]const u8;
    pub const getId = gtk_source_language_get_id;

    extern fn gtk_source_language_get_metadata(p_language: *Language, p_name: [*:0]const u8) ?[*:0]const u8;
    pub const getMetadata = gtk_source_language_get_metadata;

    /// Returns the mime types associated to this language.
    ///
    /// This is just an utility wrapper around `Language.getMetadata` to
    /// retrieve the "mimetypes" metadata property and split it into an
    /// array.
    extern fn gtk_source_language_get_mime_types(p_language: *Language) ?[*][*:0]u8;
    pub const getMimeTypes = gtk_source_language_get_mime_types;

    /// Returns the localized name of the language.
    ///
    /// The returned string is owned by `language` and should not be freed
    /// or modified.
    extern fn gtk_source_language_get_name(p_language: *Language) [*:0]const u8;
    pub const getName = gtk_source_language_get_name;

    /// Returns the localized section of the language.
    ///
    /// Each language belong to a section (ex. HTML belongs to the
    /// Markup section).
    /// The returned string is owned by `language` and should not be freed
    /// or modified.
    extern fn gtk_source_language_get_section(p_language: *Language) [*:0]const u8;
    pub const getSection = gtk_source_language_get_section;

    /// Returns the ID of the style to use if the specified `style_id`
    /// is not present in the current style scheme.
    extern fn gtk_source_language_get_style_fallback(p_language: *Language, p_style_id: [*:0]const u8) ?[*:0]const u8;
    pub const getStyleFallback = gtk_source_language_get_style_fallback;

    /// Returns the ids of the styles defined by this `language`.
    extern fn gtk_source_language_get_style_ids(p_language: *Language) ?[*][*:0]u8;
    pub const getStyleIds = gtk_source_language_get_style_ids;

    /// Returns the name of the style with ID `style_id` defined by this `language`.
    extern fn gtk_source_language_get_style_name(p_language: *Language, p_style_id: [*:0]const u8) ?[*:0]const u8;
    pub const getStyleName = gtk_source_language_get_style_name;

    extern fn gtk_source_language_get_type() usize;
    pub const getGObjectType = gtk_source_language_get_type;

    extern fn g_object_ref(p_self: *gtksource.Language) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Language) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Language, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Provides access to `Language`s.
///
/// `GtkSourceLanguageManager` is an object which processes language description
/// files and creates and stores `Language` objects, and provides API to
/// access them.
///
/// Use `LanguageManager.getDefault` to retrieve the default
/// instance of `GtkSourceLanguageManager`, and
/// `LanguageManager.guessLanguage` to get a `Language` for
/// given file name and content type.
pub const LanguageManager = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.LanguageManagerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const language_ids = struct {
            pub const name = "language-ids";

            pub const Type = ?[*][*:0]u8;
        };

        pub const search_path = struct {
            pub const name = "search-path";

            pub const Type = ?[*][*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Returns the default `gtksource.LanguageManager` instance.
    extern fn gtk_source_language_manager_get_default() *gtksource.LanguageManager;
    pub const getDefault = gtk_source_language_manager_get_default;

    /// Creates a new language manager.
    ///
    /// If you do not need more than one language manager or a private language manager
    /// instance then use `LanguageManager.getDefault` instead.
    extern fn gtk_source_language_manager_new() *gtksource.LanguageManager;
    pub const new = gtk_source_language_manager_new;

    /// Appends `path` to the list of directories where the `manager` looks for
    /// language files.
    ///
    /// See `LanguageManager.setSearchPath` for details.
    extern fn gtk_source_language_manager_append_search_path(p_lm: *LanguageManager, p_path: [*:0]const u8) void;
    pub const appendSearchPath = gtk_source_language_manager_append_search_path;

    /// Gets the `Language` identified by the given `id` in the language
    /// manager.
    extern fn gtk_source_language_manager_get_language(p_lm: *LanguageManager, p_id: [*:0]const u8) ?*gtksource.Language;
    pub const getLanguage = gtk_source_language_manager_get_language;

    /// Returns the ids of the available languages.
    extern fn gtk_source_language_manager_get_language_ids(p_lm: *LanguageManager) ?[*]const [*:0]const u8;
    pub const getLanguageIds = gtk_source_language_manager_get_language_ids;

    /// Gets the list directories where `lm` looks for language files.
    extern fn gtk_source_language_manager_get_search_path(p_lm: *LanguageManager) [*]const [*:0]const u8;
    pub const getSearchPath = gtk_source_language_manager_get_search_path;

    /// Picks a `Language` for given file name and content type,
    /// according to the information in lang files.
    ///
    /// Either `filename` or `content_type` may be `NULL`. This function can be used as follows:
    ///
    /// ```c
    /// GtkSourceLanguage *lang;
    /// GtkSourceLanguageManager *manager;
    /// lm = gtk_source_language_manager_get_default ();
    /// lang = gtk_source_language_manager_guess_language (manager, filename, NULL);
    /// gtk_source_buffer_set_language (buffer, lang);
    /// ```
    ///
    /// or
    ///
    /// ```c
    /// GtkSourceLanguage *lang = NULL;
    /// GtkSourceLanguageManager *manager;
    /// gboolean result_uncertain;
    /// gchar *content_type;
    ///
    /// content_type = g_content_type_guess (filename, NULL, 0, &result_uncertain);
    /// if (result_uncertain)
    ///   {
    ///     g_free (content_type);
    ///     content_type = NULL;
    ///   }
    ///
    /// manager = gtk_source_language_manager_get_default ();
    /// lang = gtk_source_language_manager_guess_language (manager, filename, content_type);
    /// gtk_source_buffer_set_language (buffer, lang);
    ///
    /// g_free (content_type);
    /// ```
    ///
    /// etc. Use `Language.getMimeTypes` and `Language.getGlobs`
    /// if you need full control over file -> language mapping.
    extern fn gtk_source_language_manager_guess_language(p_lm: *LanguageManager, p_filename: ?[*:0]const u8, p_content_type: ?[*:0]const u8) ?*gtksource.Language;
    pub const guessLanguage = gtk_source_language_manager_guess_language;

    /// Prepends `path` to the list of directories where the `manager` looks
    /// for language files.
    ///
    /// See `LanguageManager.setSearchPath` for details.
    extern fn gtk_source_language_manager_prepend_search_path(p_lm: *LanguageManager, p_path: [*:0]const u8) void;
    pub const prependSearchPath = gtk_source_language_manager_prepend_search_path;

    /// Sets the list of directories where the `lm` looks for
    /// language files.
    ///
    /// If `dirs` is `NULL`, the search path is reset to default.
    ///
    /// At the moment this function can be called only before the
    /// language files are loaded for the first time. In practice
    /// to set a custom search path for a `GtkSourceLanguageManager`,
    /// you have to call this function right after creating it.
    ///
    /// Since GtkSourceView 5.4 this function will allow you to provide
    /// paths in the form of "resource:///" URIs to embedded `GResource`s.
    /// They must contain the path of a directory within the `GResource`.
    extern fn gtk_source_language_manager_set_search_path(p_lm: *LanguageManager, p_dirs: ?[*]const [*:0]const u8) void;
    pub const setSearchPath = gtk_source_language_manager_set_search_path;

    extern fn gtk_source_language_manager_get_type() usize;
    pub const getGObjectType = gtk_source_language_manager_get_type;

    extern fn g_object_ref(p_self: *gtksource.LanguageManager) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.LanguageManager) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *LanguageManager, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Widget that displays a map for a specific `View`.
///
/// `GtkSourceMap` is a widget that maps the content of a `View` into
/// a smaller view so the user can have a quick overview of the whole document.
///
/// This works by connecting a `View` to to the `GtkSourceMap` using
/// the `Map.properties.view` property or `Map.setView`.
///
/// `GtkSourceMap` is a `View` object. This means that you can add a
/// `GutterRenderer` to a gutter in the same way you would for a
/// `View`. One example might be a `GutterRenderer` that shows
/// which lines have changed in the document.
///
/// Additionally, it is desirable to match the font of the `GtkSourceMap` and
/// the `View` used for editing. Therefore, `Map.properties.font_desc`
/// should be used to set the target font. You will need to adjust this to the
/// desired font size for the map. A 1pt font generally seems to be an
/// appropriate font size. "Monospace 1" is the default. See
/// `pango.FontDescription.setSize` for how to alter the size of an existing
/// `pango.FontDescription`.
///
/// When FontConfig is available, `GtkSourceMap` will try to use a bundled
/// "block" font to make the map more legible.
pub const Map = extern struct {
    pub const Parent = gtksource.View;
    pub const Implements = [_]type{ gtk.Accessible, gtk.AccessibleText, gtk.Buildable, gtk.ConstraintTarget, gtk.Scrollable };
    pub const Class = gtksource.MapClass;
    f_parent_instance: gtksource.View,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const font_desc = struct {
            pub const name = "font-desc";

            pub const Type = ?*pango.FontDescription;
        };

        pub const view = struct {
            pub const name = "view";

            pub const Type = ?*gtksource.View;
        };
    };

    pub const signals = struct {};

    /// Creates a new `GtkSourceMap`.
    extern fn gtk_source_map_new() *gtksource.Map;
    pub const new = gtk_source_map_new;

    /// Gets the `Map.properties.view` property, which is the view this widget is mapping.
    extern fn gtk_source_map_get_view(p_map: *Map) ?*gtksource.View;
    pub const getView = gtk_source_map_get_view;

    /// Sets the view that `map` will be doing the mapping to.
    extern fn gtk_source_map_set_view(p_map: *Map, p_view: *gtksource.View) void;
    pub const setView = gtk_source_map_set_view;

    extern fn gtk_source_map_get_type() usize;
    pub const getGObjectType = gtk_source_map_get_type;

    extern fn g_object_ref(p_self: *gtksource.Map) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Map) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Map, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Mark object for `Buffer`.
///
/// A `GtkSourceMark` marks a position in the text where you want to display
/// additional info. It is based on `gtk.TextMark` and thus is still valid after
/// the text has changed though its position may change.
///
/// `GtkSourceMark`s are organized in categories which you have to set
/// when you create the mark. Each category can have a priority, a pixbuf and
/// other associated attributes. See `View.setMarkAttributes`.
/// The pixbuf will be displayed in the margin at the line where the mark
/// residents if the `View.properties.show_line_marks` property is set to `TRUE`. If
/// there are multiple marks in the same line, the pixbufs will be drawn on top
/// of each other. The mark with the highest priority will be drawn on top.
pub const Mark = extern struct {
    pub const Parent = gtk.TextMark;
    pub const Implements = [_]type{};
    pub const Class = gtksource.MarkClass;
    f_parent_instance: gtk.TextMark,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The category of the `GtkSourceMark`, classifies the mark and controls
        /// which pixbuf is used and with which priority it is drawn.
        pub const category = struct {
            pub const name = "category";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a text mark.
    ///
    /// Add it to a buffer using `gtk.TextBuffer.addMark`.
    /// If name is NULL, the mark is anonymous; otherwise, the mark can be retrieved
    /// by name using `gtk.TextBuffer.getMark`.
    /// Normally marks are created using the utility function
    /// `Buffer.createSourceMark`.
    extern fn gtk_source_mark_new(p_name: ?[*:0]const u8, p_category: [*:0]const u8) *gtksource.Mark;
    pub const new = gtk_source_mark_new;

    /// Returns the mark category.
    extern fn gtk_source_mark_get_category(p_mark: *Mark) [*:0]const u8;
    pub const getCategory = gtk_source_mark_get_category;

    /// Returns the next `GtkSourceMark` in the buffer or `NULL` if the mark
    /// was not added to a buffer.
    ///
    ///  If there is no next mark, `NULL` will be returned.
    ///
    /// If `category` is `NULL`, looks for marks of any category.
    extern fn gtk_source_mark_next(p_mark: *Mark, p_category: ?[*:0]const u8) ?*gtksource.Mark;
    pub const next = gtk_source_mark_next;

    /// Returns the previous `GtkSourceMark` in the buffer or `NULL` if the mark
    /// was not added to a buffer.
    ///
    /// If there is no previous mark, `NULL` is returned.
    ///
    /// If `category` is `NULL`, looks for marks of any category
    extern fn gtk_source_mark_prev(p_mark: *Mark, p_category: ?[*:0]const u8) ?*gtksource.Mark;
    pub const prev = gtk_source_mark_prev;

    extern fn gtk_source_mark_get_type() usize;
    pub const getGObjectType = gtk_source_mark_get_type;

    extern fn g_object_ref(p_self: *gtksource.Mark) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Mark) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Mark, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The source mark attributes object.
///
/// `GtkSourceMarkAttributes` is an object specifying attributes used by
/// a `View` to visually show lines marked with `Mark`s
/// of a specific category. It allows you to define a background color of a line,
/// an icon shown in gutter and tooltips.
///
/// The background color is used as a background of a line where a mark is placed
/// and it can be set with `MarkAttributes.setBackground`. To check
/// if any custom background color was defined and what color it is, use
/// `MarkAttributes.getBackground`.
///
/// An icon is a graphic element which is shown in the gutter of a view. An
/// example use is showing a red filled circle in a debugger to show that a
/// breakpoint was set in certain line. To get an icon that will be placed in
/// a gutter, first a base for it must be specified and then
/// `MarkAttributes.renderIcon` must be called.
/// There are several ways to specify a base for an icon:
///
/// - `MarkAttributes.setIconName`
/// - `MarkAttributes.setGicon`
/// - `MarkAttributes.setPixbuf`
///
/// Using any of the above functions overrides the one used earlier. But note
/// that a getter counterpart of earlier used function can still return some
/// value, but it is just not used when rendering the proper icon.
///
/// To provide meaningful tooltips for a given mark of a category, you should
/// connect to `MarkAttributes.signals.query_tooltip_text` or
/// `MarkAttributes.signals.query_tooltip_markup` where the latter
/// takes precedence.
pub const MarkAttributes = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.MarkAttributesClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// A color used for background of a line.
        pub const background = struct {
            pub const name = "background";

            pub const Type = ?*gdk.RGBA;
        };

        /// A `gio.Icon` that may be a base of a rendered icon.
        pub const gicon = struct {
            pub const name = "gicon";

            pub const Type = ?*gio.Icon;
        };

        /// An icon name that may be a base of a rendered icon.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// A `gdkpixbuf.Pixbuf` that may be a base of a rendered icon.
        pub const pixbuf = struct {
            pub const name = "pixbuf";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };
    };

    pub const signals = struct {
        /// The code should connect to this signal to provide a tooltip for given
        /// `mark`. The tooltip can contain a markup.
        pub const query_tooltip_markup = struct {
            pub const name = "query-tooltip-markup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_mark: *gtksource.Mark, P_Data) callconv(.C) [*:0]u8, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MarkAttributes, p_instance))),
                    gobject.signalLookup("query-tooltip-markup", MarkAttributes.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The code should connect to this signal to provide a tooltip for given
        /// `mark`. The tooltip should be just a plain text.
        pub const query_tooltip_text = struct {
            pub const name = "query-tooltip-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_mark: *gtksource.Mark, P_Data) callconv(.C) [*:0]u8, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MarkAttributes, p_instance))),
                    gobject.signalLookup("query-tooltip-text", MarkAttributes.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new source mark attributes.
    extern fn gtk_source_mark_attributes_new() *gtksource.MarkAttributes;
    pub const new = gtk_source_mark_attributes_new;

    /// Stores background color in `background`.
    extern fn gtk_source_mark_attributes_get_background(p_attributes: *MarkAttributes, p_background: *gdk.RGBA) c_int;
    pub const getBackground = gtk_source_mark_attributes_get_background;

    /// Gets a `gio.Icon` to be used as a base for rendered icon.
    ///
    /// Note that the icon can be `NULL` if it wasn't set earlier.
    extern fn gtk_source_mark_attributes_get_gicon(p_attributes: *MarkAttributes) *gio.Icon;
    pub const getGicon = gtk_source_mark_attributes_get_gicon;

    /// Gets a name of an icon to be used as a base for rendered icon.
    ///
    /// Note that the icon name can be `NULL` if it wasn't set earlier.
    extern fn gtk_source_mark_attributes_get_icon_name(p_attributes: *MarkAttributes) [*:0]const u8;
    pub const getIconName = gtk_source_mark_attributes_get_icon_name;

    /// Gets a `gdkpixbuf.Pixbuf` to be used as a base for rendered icon.
    ///
    /// Note that the pixbuf can be `NULL` if it wasn't set earlier.
    extern fn gtk_source_mark_attributes_get_pixbuf(p_attributes: *MarkAttributes) *const gdkpixbuf.Pixbuf;
    pub const getPixbuf = gtk_source_mark_attributes_get_pixbuf;

    /// Queries for a tooltip by emitting a `MarkAttributes.signals.query_tooltip_markup` signal.
    ///
    /// The tooltip may contain a markup.
    extern fn gtk_source_mark_attributes_get_tooltip_markup(p_attributes: *MarkAttributes, p_mark: *gtksource.Mark) [*:0]u8;
    pub const getTooltipMarkup = gtk_source_mark_attributes_get_tooltip_markup;

    /// Queries for a tooltip by emitting a `MarkAttributes.signals.query_tooltip_text` signal.
    ///
    /// The tooltip is a plain text.
    extern fn gtk_source_mark_attributes_get_tooltip_text(p_attributes: *MarkAttributes, p_mark: *gtksource.Mark) [*:0]u8;
    pub const getTooltipText = gtk_source_mark_attributes_get_tooltip_text;

    /// Renders an icon of given size.
    ///
    /// The base of the icon is set by the last call to one of:
    ///
    /// - `MarkAttributes.setPixbuf`
    /// - `MarkAttributes.setGicon`
    /// - `MarkAttributes.setIconName`
    ///
    /// `size` cannot be lower than 1.
    extern fn gtk_source_mark_attributes_render_icon(p_attributes: *MarkAttributes, p_widget: *gtk.Widget, p_size: c_int) *gdk.Paintable;
    pub const renderIcon = gtk_source_mark_attributes_render_icon;

    /// Sets background color to the one given in `background`.
    extern fn gtk_source_mark_attributes_set_background(p_attributes: *MarkAttributes, p_background: *const gdk.RGBA) void;
    pub const setBackground = gtk_source_mark_attributes_set_background;

    /// Sets an icon to be used as a base for rendered icon.
    extern fn gtk_source_mark_attributes_set_gicon(p_attributes: *MarkAttributes, p_gicon: *gio.Icon) void;
    pub const setGicon = gtk_source_mark_attributes_set_gicon;

    /// Sets a name of an icon to be used as a base for rendered icon.
    extern fn gtk_source_mark_attributes_set_icon_name(p_attributes: *MarkAttributes, p_icon_name: [*:0]const u8) void;
    pub const setIconName = gtk_source_mark_attributes_set_icon_name;

    /// Sets a pixbuf to be used as a base for rendered icon.
    extern fn gtk_source_mark_attributes_set_pixbuf(p_attributes: *MarkAttributes, p_pixbuf: *const gdkpixbuf.Pixbuf) void;
    pub const setPixbuf = gtk_source_mark_attributes_set_pixbuf;

    extern fn gtk_source_mark_attributes_get_type() usize;
    pub const getGObjectType = gtk_source_mark_attributes_get_type;

    extern fn g_object_ref(p_self: *gtksource.MarkAttributes) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.MarkAttributes) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MarkAttributes, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Compose a `Buffer` for printing.
///
/// The `GtkSourcePrintCompositor` object is used to compose a `Buffer`
/// for printing. You can set various configuration options to customize the
/// printed output. `GtkSourcePrintCompositor` is designed to be used with the
/// high-level printing API of gtk+, i.e. `gtk.PrintOperation`.
///
/// The margins specified in this object are the layout margins: they define the
/// blank space bordering the printed area of the pages. They must not be
/// confused with the "print margins", i.e. the parts of the page that the
/// printer cannot print on, defined in the `gtk.PageSetup` objects. If the
/// specified layout margins are smaller than the "print margins", the latter
/// ones are used as a fallback by the `GtkSourcePrintCompositor` object, so that
/// the printed area is not clipped.
pub const PrintCompositor = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.PrintCompositorClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Name of the font used for the text body.
        ///
        /// Accepted values are strings representing a font description Pango can understand.
        /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
        /// for a description of the format of the string representation.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const body_font_name = struct {
            pub const name = "body-font-name";

            pub const Type = ?[*:0]u8;
        };

        /// The `Buffer` object to print.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtksource.Buffer;
        };

        /// Name of the font used to print page footer.
        /// If this property is unspecified, the text body font is used.
        ///
        /// Accepted values are strings representing a font description Pango can understand.
        /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
        /// for a description of the format of the string representation.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const footer_font_name = struct {
            pub const name = "footer-font-name";

            pub const Type = ?[*:0]u8;
        };

        /// Name of the font used to print page header.
        /// If this property is unspecified, the text body font is used.
        ///
        /// Accepted values are strings representing a font description Pango can understand.
        /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
        /// for a description of the format of the string representation.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const header_font_name = struct {
            pub const name = "header-font-name";

            pub const Type = ?[*:0]u8;
        };

        /// Whether to print the document with highlighted syntax.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const highlight_syntax = struct {
            pub const name = "highlight-syntax";

            pub const Type = c_int;
        };

        /// Name of the font used to print line numbers on the left margin.
        /// If this property is unspecified, the text body font is used.
        ///
        /// Accepted values are strings representing a font description Pango can understand.
        /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
        /// for a description of the format of the string representation.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const line_numbers_font_name = struct {
            pub const name = "line-numbers-font-name";

            pub const Type = ?[*:0]u8;
        };

        /// The number of pages in the document or <code>-1</code> if the
        /// document has not been completely paginated.
        pub const n_pages = struct {
            pub const name = "n-pages";

            pub const Type = c_int;
        };

        /// Whether to print a footer in each page.
        ///
        /// Note that by default the footer format is unspecified, and if it is
        /// unspecified the footer will not be printed, regardless of the value of
        /// this property.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const print_footer = struct {
            pub const name = "print-footer";

            pub const Type = c_int;
        };

        /// Whether to print a header in each page.
        ///
        /// Note that by default the header format is unspecified, and if it is
        /// unspecified the header will not be printed, regardless of the value of
        /// this property.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const print_header = struct {
            pub const name = "print-header";

            pub const Type = c_int;
        };

        /// Interval of printed line numbers.
        ///
        /// If this property is set to 0 no numbers will be printed.
        /// If greater than 0, a number will be printed every "print-line-numbers"
        /// lines (i.e. 1 will print all line numbers).
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const print_line_numbers = struct {
            pub const name = "print-line-numbers";

            pub const Type = c_uint;
        };

        /// Width of a tab character expressed in spaces.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const tab_width = struct {
            pub const name = "tab-width";

            pub const Type = c_uint;
        };

        /// Whether to wrap lines never, at word boundaries, or at character boundaries.
        ///
        /// The value of this property cannot be changed anymore after the first
        /// call to the `PrintCompositor.paginate` function.
        pub const wrap_mode = struct {
            pub const name = "wrap-mode";

            pub const Type = gtk.WrapMode;
        };
    };

    pub const signals = struct {};

    /// Creates a new print compositor that can be used to print `buffer`.
    extern fn gtk_source_print_compositor_new(p_buffer: *gtksource.Buffer) *gtksource.PrintCompositor;
    pub const new = gtk_source_print_compositor_new;

    /// Creates a new print compositor that can be used to print the buffer
    /// associated with `view`.
    ///
    /// This constructor sets some configuration properties to make the
    /// printed output match `view` as much as possible.  The properties set are
    /// `PrintCompositor.properties.tab_width`, `PrintCompositor.properties.highlight_syntax`,
    /// `PrintCompositor.properties.wrap_mode`, `PrintCompositor.properties.body_font_name` and
    /// `PrintCompositor.properties.print_line_numbers`.
    extern fn gtk_source_print_compositor_new_from_view(p_view: *gtksource.View) *gtksource.PrintCompositor;
    pub const newFromView = gtk_source_print_compositor_new_from_view;

    /// Draw page `page_nr` for printing on the the Cairo context encapsuled in `context`.
    ///
    /// This method has been designed to be called in the handler of the `gtk.PrintOperation.signals.draw_page` signal
    /// as shown in the following example:
    ///
    /// ```c
    /// // Signal handler for the GtkPrintOperation::draw_page signal
    ///
    /// static void
    /// draw_page (GtkPrintOperation *operation,
    ///            GtkPrintContext   *context,
    ///            gint               page_nr,
    ///            gpointer           user_data)
    /// {
    ///     GtkSourcePrintCompositor *compositor;
    ///
    ///     compositor = GTK_SOURCE_PRINT_COMPOSITOR (user_data);
    ///
    ///     gtk_source_print_compositor_draw_page (compositor,
    ///                                            context,
    ///                                            page_nr);
    /// }
    /// ```
    extern fn gtk_source_print_compositor_draw_page(p_compositor: *PrintCompositor, p_context: *gtk.PrintContext, p_page_nr: c_int) void;
    pub const drawPage = gtk_source_print_compositor_draw_page;

    /// Returns the name of the font used to print the text body.
    ///
    /// The returned string must be freed with `glib.free`.
    extern fn gtk_source_print_compositor_get_body_font_name(p_compositor: *PrintCompositor) [*:0]u8;
    pub const getBodyFontName = gtk_source_print_compositor_get_body_font_name;

    /// Gets the bottom margin in units of `unit`.
    extern fn gtk_source_print_compositor_get_bottom_margin(p_compositor: *PrintCompositor, p_unit: gtk.Unit) f64;
    pub const getBottomMargin = gtk_source_print_compositor_get_bottom_margin;

    /// Gets the `Buffer` associated with the compositor.
    ///
    /// The returned object reference is owned by the compositor object and
    /// should not be unreferenced.
    extern fn gtk_source_print_compositor_get_buffer(p_compositor: *PrintCompositor) *gtksource.Buffer;
    pub const getBuffer = gtk_source_print_compositor_get_buffer;

    /// Returns the name of the font used to print the page footer.
    ///
    /// The returned string must be freed with `glib.free`.
    extern fn gtk_source_print_compositor_get_footer_font_name(p_compositor: *PrintCompositor) [*:0]u8;
    pub const getFooterFontName = gtk_source_print_compositor_get_footer_font_name;

    /// Returns the name of the font used to print the page header.
    ///
    /// The returned string must be freed with `glib.free`.
    extern fn gtk_source_print_compositor_get_header_font_name(p_compositor: *PrintCompositor) [*:0]u8;
    pub const getHeaderFontName = gtk_source_print_compositor_get_header_font_name;

    /// Determines whether the printed text will be highlighted according to the
    /// buffer rules.
    ///
    /// Note that highlighting will happen only if the buffer to print has highlighting activated.
    extern fn gtk_source_print_compositor_get_highlight_syntax(p_compositor: *PrintCompositor) c_int;
    pub const getHighlightSyntax = gtk_source_print_compositor_get_highlight_syntax;

    /// Gets the left margin in units of `unit`.
    extern fn gtk_source_print_compositor_get_left_margin(p_compositor: *PrintCompositor, p_unit: gtk.Unit) f64;
    pub const getLeftMargin = gtk_source_print_compositor_get_left_margin;

    /// Returns the name of the font used to print line numbers on the left margin.
    ///
    /// The returned string must be freed with `glib.free`.
    extern fn gtk_source_print_compositor_get_line_numbers_font_name(p_compositor: *PrintCompositor) [*:0]u8;
    pub const getLineNumbersFontName = gtk_source_print_compositor_get_line_numbers_font_name;

    /// Returns the number of pages in the document or <code>-1</code> if the
    /// document has not been completely paginated.
    extern fn gtk_source_print_compositor_get_n_pages(p_compositor: *PrintCompositor) c_int;
    pub const getNPages = gtk_source_print_compositor_get_n_pages;

    /// Returns the current fraction of the document pagination that has been completed.
    extern fn gtk_source_print_compositor_get_pagination_progress(p_compositor: *PrintCompositor) f64;
    pub const getPaginationProgress = gtk_source_print_compositor_get_pagination_progress;

    /// Determines if a footer is set to be printed for each page.
    ///
    /// A footer will be printed if this function returns `TRUE`
    /// **and** some format strings have been specified
    /// with `PrintCompositor.setFooterFormat`.
    extern fn gtk_source_print_compositor_get_print_footer(p_compositor: *PrintCompositor) c_int;
    pub const getPrintFooter = gtk_source_print_compositor_get_print_footer;

    /// Determines if a header is set to be printed for each page.
    ///
    /// A header will be printed if this function returns `TRUE`
    /// **and** some format strings have been specified
    /// with `PrintCompositor.setHeaderFormat`.
    extern fn gtk_source_print_compositor_get_print_header(p_compositor: *PrintCompositor) c_int;
    pub const getPrintHeader = gtk_source_print_compositor_get_print_header;

    /// Returns the interval used for line number printing.
    ///
    /// If the value is 0, no line numbers will be printed. The default value is
    /// 1 (i.e. numbers printed in all lines).
    extern fn gtk_source_print_compositor_get_print_line_numbers(p_compositor: *PrintCompositor) c_uint;
    pub const getPrintLineNumbers = gtk_source_print_compositor_get_print_line_numbers;

    /// Gets the right margin in units of `unit`.
    extern fn gtk_source_print_compositor_get_right_margin(p_compositor: *PrintCompositor, p_unit: gtk.Unit) f64;
    pub const getRightMargin = gtk_source_print_compositor_get_right_margin;

    /// Returns the width of tabulation in characters for printed text.
    extern fn gtk_source_print_compositor_get_tab_width(p_compositor: *PrintCompositor) c_uint;
    pub const getTabWidth = gtk_source_print_compositor_get_tab_width;

    /// Gets the top margin in units of `unit`.
    extern fn gtk_source_print_compositor_get_top_margin(p_compositor: *PrintCompositor, p_unit: gtk.Unit) f64;
    pub const getTopMargin = gtk_source_print_compositor_get_top_margin;

    /// Gets the line wrapping mode for the printed text.
    extern fn gtk_source_print_compositor_get_wrap_mode(p_compositor: *PrintCompositor) gtk.WrapMode;
    pub const getWrapMode = gtk_source_print_compositor_get_wrap_mode;

    /// Specifies a tag whose style should be ignored when compositing the
    /// document to the printable page.
    extern fn gtk_source_print_compositor_ignore_tag(p_compositor: *PrintCompositor, p_tag: *gtk.TextTag) void;
    pub const ignoreTag = gtk_source_print_compositor_ignore_tag;

    /// Paginate the document associated with the `compositor`.
    ///
    /// In order to support non-blocking pagination, document is paginated in small chunks.
    /// Each time `PrintCompositor.paginate` is invoked, a chunk of the document
    /// is paginated. To paginate the entire document, `PrintCompositor.paginate`
    /// must be invoked multiple times.
    /// It returns `TRUE` if the document has been completely paginated, otherwise it returns `FALSE`.
    ///
    /// This method has been designed to be invoked in the handler of the `gtk.PrintOperation.signals.paginate` signal,
    /// as shown in the following example:
    ///
    /// ```c
    /// // Signal handler for the GtkPrintOperation::paginate signal
    ///
    /// static gboolean
    /// paginate (GtkPrintOperation *operation,
    ///           GtkPrintContext   *context,
    ///           gpointer           user_data)
    /// {
    ///     GtkSourcePrintCompositor *compositor;
    ///
    ///     compositor = GTK_SOURCE_PRINT_COMPOSITOR (user_data);
    ///
    ///     if (gtk_source_print_compositor_paginate (compositor, context))
    ///     {
    ///         gint n_pages;
    ///
    ///         n_pages = gtk_source_print_compositor_get_n_pages (compositor);
    ///         gtk_print_operation_set_n_pages (operation, n_pages);
    ///
    ///         return TRUE;
    ///     }
    ///
    ///     return FALSE;
    /// }
    /// ```
    ///
    /// If you don't need to do pagination in chunks, you can simply do it all in the
    /// `gtk.PrintOperation.signals.begin_print` handler, and set the number of pages from there, like
    /// in the following example:
    ///
    /// ```c
    /// // Signal handler for the GtkPrintOperation::begin-print signal
    ///
    /// static void
    /// begin_print (GtkPrintOperation *operation,
    ///              GtkPrintContext   *context,
    ///              gpointer           user_data)
    /// {
    ///     GtkSourcePrintCompositor *compositor;
    ///     gint n_pages;
    ///
    ///     compositor = GTK_SOURCE_PRINT_COMPOSITOR (user_data);
    ///
    ///     while (!gtk_source_print_compositor_paginate (compositor, context));
    ///
    ///     n_pages = gtk_source_print_compositor_get_n_pages (compositor);
    ///     gtk_print_operation_set_n_pages (operation, n_pages);
    /// }
    /// ```
    extern fn gtk_source_print_compositor_paginate(p_compositor: *PrintCompositor, p_context: *gtk.PrintContext) c_int;
    pub const paginate = gtk_source_print_compositor_paginate;

    /// Sets the default font for the printed text.
    ///
    /// `font_name` should be a
    /// string representation of a font description Pango can understand.
    /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
    /// for a description of the format of the string representation.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_body_font_name(p_compositor: *PrintCompositor, p_font_name: [*:0]const u8) void;
    pub const setBodyFontName = gtk_source_print_compositor_set_body_font_name;

    /// Sets the bottom margin used by `compositor`.
    extern fn gtk_source_print_compositor_set_bottom_margin(p_compositor: *PrintCompositor, p_margin: f64, p_unit: gtk.Unit) void;
    pub const setBottomMargin = gtk_source_print_compositor_set_bottom_margin;

    /// Sets the font for printing the page footer.
    ///
    /// If `NULL` is supplied, the default font (i.e. the one being used for the
    /// text) will be used instead.
    ///
    /// `font_name` should be a
    /// string representation of a font description Pango can understand.
    /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
    /// for a description of the format of the string representation.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_footer_font_name(p_compositor: *PrintCompositor, p_font_name: ?[*:0]const u8) void;
    pub const setFooterFontName = gtk_source_print_compositor_set_footer_font_name;

    /// See `PrintCompositor.setHeaderFormat` for more information
    /// about the parameters.
    extern fn gtk_source_print_compositor_set_footer_format(p_compositor: *PrintCompositor, p_separator: c_int, p_left: ?[*:0]const u8, p_center: ?[*:0]const u8, p_right: ?[*:0]const u8) void;
    pub const setFooterFormat = gtk_source_print_compositor_set_footer_format;

    /// Sets the font for printing the page header.
    ///
    /// If `NULL` is supplied, the default font (i.e. the one being used for the
    /// text) will be used instead.
    ///
    /// `font_name` should be a
    /// string representation of a font description Pango can understand.
    /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
    /// for a description of the format of the string representation.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_header_font_name(p_compositor: *PrintCompositor, p_font_name: ?[*:0]const u8) void;
    pub const setHeaderFontName = gtk_source_print_compositor_set_header_font_name;

    /// Sets strftime like header format strings, to be printed on the
    /// left, center and right of the top of each page.
    ///
    /// The strings may include strftime(3) codes which will be expanded at print time.
    /// A subset of `strftime` codes are accepted, see `glib.DateTime.format`
    /// for more details on the accepted format specifiers.
    /// Additionally the following format specifiers are accepted:
    ///
    /// - `N`: the page number
    /// - `Q`: the page count.
    ///
    /// `separator` specifies if a solid line should be drawn to separate
    /// the header from the document text.
    ///
    /// If `NULL` is given for any of the three arguments, that particular
    /// string will not be printed.
    ///
    /// For the header to be printed, in
    /// addition to specifying format strings, you need to enable header
    /// printing with `PrintCompositor.setPrintHeader`.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_header_format(p_compositor: *PrintCompositor, p_separator: c_int, p_left: ?[*:0]const u8, p_center: ?[*:0]const u8, p_right: ?[*:0]const u8) void;
    pub const setHeaderFormat = gtk_source_print_compositor_set_header_format;

    /// Sets whether the printed text will be highlighted according to the
    /// buffer rules.  Both color and font style are applied.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_highlight_syntax(p_compositor: *PrintCompositor, p_highlight: c_int) void;
    pub const setHighlightSyntax = gtk_source_print_compositor_set_highlight_syntax;

    /// Sets the left margin used by `compositor`.
    extern fn gtk_source_print_compositor_set_left_margin(p_compositor: *PrintCompositor, p_margin: f64, p_unit: gtk.Unit) void;
    pub const setLeftMargin = gtk_source_print_compositor_set_left_margin;

    /// Sets the font for printing line numbers on the left margin.
    ///
    /// If `NULL` is supplied, the default font (i.e. the one being used for the
    /// text) will be used instead.
    ///
    /// `font_name` should be a
    /// string representation of a font description Pango can understand.
    /// (e.g. &quot;Monospace 10&quot;). See `pango.FontDescription.fromString`
    /// for a description of the format of the string representation.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_line_numbers_font_name(p_compositor: *PrintCompositor, p_font_name: ?[*:0]const u8) void;
    pub const setLineNumbersFontName = gtk_source_print_compositor_set_line_numbers_font_name;

    /// Sets whether you want to print a footer in each page.
    ///
    /// The footer consists of three pieces of text and an optional line
    /// separator, configurable with
    /// `PrintCompositor.setFooterFormat`.
    ///
    /// Note that by default the footer format is unspecified, and if it's
    /// empty it will not be printed, regardless of this setting.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_print_footer(p_compositor: *PrintCompositor, p_print: c_int) void;
    pub const setPrintFooter = gtk_source_print_compositor_set_print_footer;

    /// Sets whether you want to print a header in each page.
    ///
    /// The header consists of three pieces of text and an optional line
    /// separator, configurable with `PrintCompositor.setHeaderFormat`.
    ///
    /// Note that by default the header format is unspecified, and if it's
    /// empty it will not be printed, regardless of this setting.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_print_header(p_compositor: *PrintCompositor, p_print: c_int) void;
    pub const setPrintHeader = gtk_source_print_compositor_set_print_header;

    /// Sets the interval for printed line numbers.
    ///
    /// If `interval` is 0 no numbers will be printed. If greater than 0, a number will be
    /// printed every `interval` lines (i.e. 1 will print all line numbers).
    ///
    /// Maximum accepted value for `interval` is 100.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_print_line_numbers(p_compositor: *PrintCompositor, p_interval: c_uint) void;
    pub const setPrintLineNumbers = gtk_source_print_compositor_set_print_line_numbers;

    /// Sets the right margin used by `compositor`.
    extern fn gtk_source_print_compositor_set_right_margin(p_compositor: *PrintCompositor, p_margin: f64, p_unit: gtk.Unit) void;
    pub const setRightMargin = gtk_source_print_compositor_set_right_margin;

    /// Sets the width of tabulation in characters for printed text.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_tab_width(p_compositor: *PrintCompositor, p_width: c_uint) void;
    pub const setTabWidth = gtk_source_print_compositor_set_tab_width;

    /// Sets the top margin used by `compositor`.
    extern fn gtk_source_print_compositor_set_top_margin(p_compositor: *PrintCompositor, p_margin: f64, p_unit: gtk.Unit) void;
    pub const setTopMargin = gtk_source_print_compositor_set_top_margin;

    /// Sets the line wrapping mode for the printed text.
    ///
    /// This function cannot be called anymore after the first call to the
    /// `PrintCompositor.paginate` function.
    extern fn gtk_source_print_compositor_set_wrap_mode(p_compositor: *PrintCompositor, p_wrap_mode: gtk.WrapMode) void;
    pub const setWrapMode = gtk_source_print_compositor_set_wrap_mode;

    extern fn gtk_source_print_compositor_get_type() usize;
    pub const getGObjectType = gtk_source_print_compositor_get_type;

    extern fn g_object_ref(p_self: *gtksource.PrintCompositor) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.PrintCompositor) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PrintCompositor, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Region utility.
///
/// A `GtkSourceRegion` permits to store a group of subregions of a
/// `gtk.TextBuffer`. `GtkSourceRegion` stores the subregions with pairs of
/// `gtk.TextMark`'s, so the region is still valid after insertions and deletions
/// in the `gtk.TextBuffer`.
///
/// The `gtk.TextMark` for the start of a subregion has a left gravity, while the
/// `gtk.TextMark` for the end of a subregion has a right gravity.
///
/// The typical use-case of `GtkSourceRegion` is to scan a `gtk.TextBuffer` chunk by
/// chunk, not the whole buffer at once to not block the user interface. The
/// `GtkSourceRegion` represents in that case the remaining region to scan. You
/// can listen to the `gtk.TextBuffer.signals.insert_text` and
/// `gtk.TextBuffer.signals.delete_range` signals to update the `GtkSourceRegion`
/// accordingly.
///
/// To iterate through the subregions, you need to use a `RegionIter`,
/// for example:
/// ```c
/// GtkSourceRegion *region;
/// GtkSourceRegionIter region_iter;
///
/// gtk_source_region_get_start_region_iter (region, &region_iter);
///
/// while (!gtk_source_region_iter_is_end (&region_iter))
/// {
///         GtkTextIter subregion_start;
///         GtkTextIter subregion_end;
///
///         if (!gtk_source_region_iter_get_subregion (&region_iter,
///                                                    &subregion_start,
///                                                    &subregion_end))
///         {
///                 break;
///         }
///
///         // Do something useful with the subregion.
///
///         gtk_source_region_iter_next (&region_iter);
/// }
/// ```
pub const Region = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.RegionClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gtk.TextBuffer`. The `gtksource.Region` has a weak reference to the
        /// buffer.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtk.TextBuffer;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_region_new(p_buffer: *gtk.TextBuffer) *gtksource.Region;
    pub const new = gtk_source_region_new;

    /// Adds `region_to_add` to `region`.
    ///
    /// `region_to_add` is not modified.
    extern fn gtk_source_region_add_region(p_region: *Region, p_region_to_add: ?*gtksource.Region) void;
    pub const addRegion = gtk_source_region_add_region;

    /// Adds the subregion delimited by `_start` and `_end` to `region`.
    extern fn gtk_source_region_add_subregion(p_region: *Region, p__start: *const gtk.TextIter, p__end: *const gtk.TextIter) void;
    pub const addSubregion = gtk_source_region_add_subregion;

    /// Gets the `start` and `end` bounds of the `region`.
    extern fn gtk_source_region_get_bounds(p_region: *Region, p_start: ?*gtk.TextIter, p_end: ?*gtk.TextIter) c_int;
    pub const getBounds = gtk_source_region_get_bounds;

    extern fn gtk_source_region_get_buffer(p_region: *Region) ?*gtk.TextBuffer;
    pub const getBuffer = gtk_source_region_get_buffer;

    /// Initializes a `RegionIter` to the first subregion of `region`.
    ///
    /// If `region` is empty, `iter` will be initialized to the end iterator.
    extern fn gtk_source_region_get_start_region_iter(p_region: *Region, p_iter: *gtksource.RegionIter) void;
    pub const getStartRegionIter = gtk_source_region_get_start_region_iter;

    /// Returns the intersection between `region1` and `region2`.
    ///
    /// `region1` and `region2` are not modified.
    extern fn gtk_source_region_intersect_region(p_region1: ?*Region, p_region2: ?*gtksource.Region) ?*gtksource.Region;
    pub const intersectRegion = gtk_source_region_intersect_region;

    /// Returns the intersection between `region` and the subregion delimited by
    /// `_start` and `_end`.
    ///
    /// `region` is not modified.
    extern fn gtk_source_region_intersect_subregion(p_region: *Region, p__start: *const gtk.TextIter, p__end: *const gtk.TextIter) ?*gtksource.Region;
    pub const intersectSubregion = gtk_source_region_intersect_subregion;

    /// Returns whether the `region` is empty.
    ///
    /// A `NULL` `region` is considered empty.
    extern fn gtk_source_region_is_empty(p_region: ?*Region) c_int;
    pub const isEmpty = gtk_source_region_is_empty;

    /// Subtracts `region_to_subtract` from `region`.
    ///
    /// `region_to_subtract` is not modified.
    extern fn gtk_source_region_subtract_region(p_region: *Region, p_region_to_subtract: ?*gtksource.Region) void;
    pub const subtractRegion = gtk_source_region_subtract_region;

    /// Subtracts the subregion delimited by `_start` and `_end` from `region`.
    extern fn gtk_source_region_subtract_subregion(p_region: *Region, p__start: *const gtk.TextIter, p__end: *const gtk.TextIter) void;
    pub const subtractSubregion = gtk_source_region_subtract_subregion;

    /// Gets a string represention of `region`, for debugging purposes.
    ///
    /// The returned string contains the character offsets of the subregions. It
    /// doesn't include a newline character at the end of the string.
    extern fn gtk_source_region_to_string(p_region: *Region) ?[*:0]u8;
    pub const toString = gtk_source_region_to_string;

    extern fn gtk_source_region_get_type() usize;
    pub const getGObjectType = gtk_source_region_get_type;

    extern fn g_object_ref(p_self: *gtksource.Region) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Region) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Region, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Search context.
///
/// A `GtkSourceSearchContext` is used for the search and replace in a
/// `Buffer`. The search settings are represented by a
/// `SearchSettings` object. There can be a many-to-many relationship
/// between buffers and search settings, with the search contexts in-between: a
/// search settings object can be shared between several search contexts; and a
/// buffer can contain several search contexts at the same time.
///
/// The total number of search occurrences can be retrieved with
/// `SearchContext.getOccurrencesCount`. To know the position of a
/// certain match, use `SearchContext.getOccurrencePosition`.
///
/// The buffer is scanned asynchronously, so it doesn't block the user interface.
/// For each search, the buffer is scanned at most once. After that, navigating
/// through the occurrences doesn't require to re-scan the buffer entirely.
///
/// To search forward, use `SearchContext.forward` or
/// `SearchContext.forwardAsync` for the asynchronous version.
/// The backward search is done similarly. To replace a search match, or all
/// matches, use `SearchContext.replace` and
/// `SearchContext.replaceAll`.
///
/// The search occurrences are highlighted by default. To disable it, use
/// `SearchContext.setHighlight`. You can enable the search
/// highlighting for several `GtkSourceSearchContext`s attached to the
/// same buffer. Moreover, each of those `GtkSourceSearchContext`s can
/// have a different text style associated. Use
/// `SearchContext.setMatchStyle` to specify the `Style`
/// to apply on search matches.
///
/// Note that the `SearchContext.properties.highlight` and
/// `SearchContext.properties.match_style` properties are in the
/// `GtkSourceSearchContext` class, not `SearchSettings`. Appearance
/// settings should be tied to one, and only one buffer, as different buffers can
/// have different style scheme associated (a `SearchSettings` object
/// can be bound indirectly to several buffers).
///
/// The concept of "current match" doesn't exist yet. A way to highlight
/// differently the current match is to select it.
///
/// A search occurrence's position doesn't depend on the cursor position or other
/// parameters. Take for instance the buffer "aaaa" with the search text "aa".
/// The two occurrences are at positions [0:2] and [2:4]. If you begin to search
/// at position 1, you will get the occurrence [2:4], not [1:3]. This is a
/// prerequisite for regular expression searches. The pattern ".*" matches the
/// entire line. If the cursor is at the middle of the line, you don't want the
/// rest of the line as the occurrence, you want an entire line. (As a side note,
/// regular expression searches can also match multiple lines.)
///
/// In the GtkSourceView source code, there is an example of how to use the
/// search and replace API: see the tests/test-search.c file. It is a mini
/// application for the search and replace, with a basic user interface.
pub const SearchContext = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SearchContextClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `Buffer` associated to the search context.
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtksource.Buffer;
        };

        /// Highlight the search occurrences.
        pub const highlight = struct {
            pub const name = "highlight";

            pub const Type = c_int;
        };

        /// A `Style`, or `NULL` for theme's scheme default style.
        pub const match_style = struct {
            pub const name = "match-style";

            pub const Type = ?*gtksource.Style;
        };

        /// The total number of search occurrences. If the search is disabled,
        /// the value is 0. If the buffer is not already fully scanned, the value
        /// is -1.
        pub const occurrences_count = struct {
            pub const name = "occurrences-count";

            pub const Type = c_int;
        };

        /// If the regex search pattern doesn't follow all the rules, this
        /// `glib.Error` property will be set. If the pattern is valid, the value is
        /// `NULL`.
        ///
        /// Free with `glib.Error.free`.
        pub const regex_error = struct {
            pub const name = "regex-error";

            pub const Type = ?*glib.Error;
        };

        /// The `SearchSettings` associated to the search context.
        ///
        /// This property is construct-only since version 4.0.
        pub const settings = struct {
            pub const name = "settings";

            pub const Type = ?*gtksource.SearchSettings;
        };
    };

    pub const signals = struct {};

    /// Creates a new search context, associated with `buffer`, and customized with
    /// `settings`.
    ///
    /// If `settings` is `NULL`, a new `SearchSettings` object will
    /// be created, that you can retrieve with `SearchContext.getSettings`.
    extern fn gtk_source_search_context_new(p_buffer: *gtksource.Buffer, p_settings: ?*gtksource.SearchSettings) *gtksource.SearchContext;
    pub const new = gtk_source_search_context_new;

    /// Synchronous backward search.
    ///
    /// It is recommended to use the asynchronous functions instead, to not block the user interface.
    /// However, if you are sure that the `buffer` is small, this function is more convenient to use.
    ///
    /// If the `SearchSettings.properties.wrap_around` property is `FALSE`, this function
    /// doesn't try to wrap around.
    ///
    /// The `has_wrapped_around` out parameter is set independently of whether a match
    /// is found. So if this function returns `FALSE`, `has_wrapped_around` will have
    /// the same value as the `SearchSettings.properties.wrap_around` property.
    extern fn gtk_source_search_context_backward(p_search: *SearchContext, p_iter: *const gtk.TextIter, p_match_start: ?*gtk.TextIter, p_match_end: ?*gtk.TextIter, p_has_wrapped_around: ?*c_int) c_int;
    pub const backward = gtk_source_search_context_backward;

    /// The asynchronous version of `SearchContext.backward`.
    ///
    /// See the `gio.AsyncResult` documentation to know how to use this function.
    ///
    /// If the operation is cancelled, the `callback` will only be called if
    /// `cancellable` was not `NULL`. The method takes
    /// ownership of `cancellable`, so you can unref it after calling this function.
    extern fn gtk_source_search_context_backward_async(p_search: *SearchContext, p_iter: *const gtk.TextIter, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const backwardAsync = gtk_source_search_context_backward_async;

    /// Finishes a backward search started with
    /// `SearchContext.backwardAsync`.
    ///
    /// See the documentation of `SearchContext.backward` for more
    /// details.
    extern fn gtk_source_search_context_backward_finish(p_search: *SearchContext, p_result: *gio.AsyncResult, p_match_start: ?*gtk.TextIter, p_match_end: ?*gtk.TextIter, p_has_wrapped_around: ?*c_int, p_error: ?*?*glib.Error) c_int;
    pub const backwardFinish = gtk_source_search_context_backward_finish;

    /// Synchronous forward search.
    ///
    /// It is recommended to use the asynchronous functions instead, to not block the user interface.
    /// However, if you are sure that the `buffer` is small, this function is more convenient to use.
    ///
    /// If the `SearchSettings.properties.wrap_around` property is `FALSE`, this function
    /// doesn't try to wrap around.
    ///
    /// The `has_wrapped_around` out parameter is set independently of whether a match
    /// is found. So if this function returns `FALSE`, `has_wrapped_around` will have
    /// the same value as the  `SearchSettings.properties.wrap_around` property.
    extern fn gtk_source_search_context_forward(p_search: *SearchContext, p_iter: *const gtk.TextIter, p_match_start: ?*gtk.TextIter, p_match_end: ?*gtk.TextIter, p_has_wrapped_around: ?*c_int) c_int;
    pub const forward = gtk_source_search_context_forward;

    /// The asynchronous version of `SearchContext.forward`.
    ///
    /// See the `gio.AsyncResult` documentation to know how to use this function.
    ///
    /// If the operation is cancelled, the `callback` will only be called if
    /// `cancellable` was not `NULL`. The method takes
    /// ownership of `cancellable`, so you can unref it after calling this function.
    extern fn gtk_source_search_context_forward_async(p_search: *SearchContext, p_iter: *const gtk.TextIter, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const forwardAsync = gtk_source_search_context_forward_async;

    /// Finishes a forward search started with `SearchContext.forwardAsync`.
    ///
    /// See the documentation of `SearchContext.forward` for more
    /// details.
    extern fn gtk_source_search_context_forward_finish(p_search: *SearchContext, p_result: *gio.AsyncResult, p_match_start: ?*gtk.TextIter, p_match_end: ?*gtk.TextIter, p_has_wrapped_around: ?*c_int, p_error: ?*?*glib.Error) c_int;
    pub const forwardFinish = gtk_source_search_context_forward_finish;

    extern fn gtk_source_search_context_get_buffer(p_search: *SearchContext) *gtksource.Buffer;
    pub const getBuffer = gtk_source_search_context_get_buffer;

    extern fn gtk_source_search_context_get_highlight(p_search: *SearchContext) c_int;
    pub const getHighlight = gtk_source_search_context_get_highlight;

    extern fn gtk_source_search_context_get_match_style(p_search: *SearchContext) *gtksource.Style;
    pub const getMatchStyle = gtk_source_search_context_get_match_style;

    /// Gets the position of a search occurrence.
    ///
    /// If the buffer is not already fully scanned, the position may be unknown,
    /// and -1 is returned. If 0 is returned, it means that this part of the buffer
    /// has already been scanned, and that `match_start` and `match_end` don't delimit an occurrence.
    extern fn gtk_source_search_context_get_occurrence_position(p_search: *SearchContext, p_match_start: *const gtk.TextIter, p_match_end: *const gtk.TextIter) c_int;
    pub const getOccurrencePosition = gtk_source_search_context_get_occurrence_position;

    /// Gets the total number of search occurrences.
    ///
    /// If the buffer is not already fully scanned, the total number of occurrences is
    /// unknown, and -1 is returned.
    extern fn gtk_source_search_context_get_occurrences_count(p_search: *SearchContext) c_int;
    pub const getOccurrencesCount = gtk_source_search_context_get_occurrences_count;

    /// Regular expression patterns must follow certain rules. If
    /// `SearchSettings.properties.search_text` breaks a rule, the error can be
    /// retrieved with this function.
    ///
    /// The error domain is `glib.RegexError`.
    ///
    /// Free the return value with `glib.Error.free`.
    extern fn gtk_source_search_context_get_regex_error(p_search: *SearchContext) ?*glib.Error;
    pub const getRegexError = gtk_source_search_context_get_regex_error;

    extern fn gtk_source_search_context_get_settings(p_search: *SearchContext) *gtksource.SearchSettings;
    pub const getSettings = gtk_source_search_context_get_settings;

    /// Replaces a search match by another text. If `match_start` and `match_end`
    /// doesn't correspond to a search match, `FALSE` is returned.
    ///
    /// `match_start` and `match_end` iters are revalidated to point to the replacement
    /// text boundaries.
    ///
    /// For a regular expression replacement, you can check if `replace` is valid by
    /// calling `glib.Regex.checkReplacement`. The `replace` text can contain
    /// backreferences.
    extern fn gtk_source_search_context_replace(p_search: *SearchContext, p_match_start: *gtk.TextIter, p_match_end: *gtk.TextIter, p_replace: [*:0]const u8, p_replace_length: c_int, p_error: ?*?*glib.Error) c_int;
    pub const replace = gtk_source_search_context_replace;

    /// Replaces all search matches by another text.
    ///
    /// It is a synchronous function, so it can block the user interface.
    ///
    /// For a regular expression replacement, you can check if `replace` is valid by
    /// calling `glib.Regex.checkReplacement`. The `replace` text can contain
    /// backreferences.
    extern fn gtk_source_search_context_replace_all(p_search: *SearchContext, p_replace: [*:0]const u8, p_replace_length: c_int, p_error: ?*?*glib.Error) c_uint;
    pub const replaceAll = gtk_source_search_context_replace_all;

    /// Enables or disables the search occurrences highlighting.
    extern fn gtk_source_search_context_set_highlight(p_search: *SearchContext, p_highlight: c_int) void;
    pub const setHighlight = gtk_source_search_context_set_highlight;

    /// Set the style to apply on search matches.
    ///
    /// If `match_style` is `NULL`, default theme's scheme 'match-style' will be used.
    /// To enable or disable the search highlighting, use `SearchContext.setHighlight`.
    extern fn gtk_source_search_context_set_match_style(p_search: *SearchContext, p_match_style: ?*gtksource.Style) void;
    pub const setMatchStyle = gtk_source_search_context_set_match_style;

    extern fn gtk_source_search_context_get_type() usize;
    pub const getGObjectType = gtk_source_search_context_get_type;

    extern fn g_object_ref(p_self: *gtksource.SearchContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SearchContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SearchContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Search settings.
///
/// A `GtkSourceSearchSettings` object represents the settings of a search. The
/// search settings can be associated with one or several
/// `SearchContext`s.
pub const SearchSettings = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SearchSettingsClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// If `TRUE`, a search match must start and end a word. The match can
        /// span multiple words.
        pub const at_word_boundaries = struct {
            pub const name = "at-word-boundaries";

            pub const Type = c_int;
        };

        /// Whether the search is case sensitive.
        pub const case_sensitive = struct {
            pub const name = "case-sensitive";

            pub const Type = c_int;
        };

        /// Search by regular expressions with
        /// `SearchSettings.properties.search_text` as the pattern.
        pub const regex_enabled = struct {
            pub const name = "regex-enabled";

            pub const Type = c_int;
        };

        /// A search string, or `NULL` if the search is disabled.
        ///
        /// If the regular expression search is enabled, `SearchSettings.properties.search_text` is
        /// the pattern.
        pub const search_text = struct {
            pub const name = "search-text";

            pub const Type = ?[*:0]u8;
        };

        /// Exclude invisible text from the search.
        /// A search match may have invisible text interspersed.
        pub const visible_only = struct {
            pub const name = "visible-only";

            pub const Type = c_int;
        };

        /// For a forward search, continue at the beginning of the buffer if no
        /// search occurrence is found. For a backward search, continue at the
        /// end of the buffer.
        pub const wrap_around = struct {
            pub const name = "wrap-around";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new search settings object.
    extern fn gtk_source_search_settings_new() *gtksource.SearchSettings;
    pub const new = gtk_source_search_settings_new;

    extern fn gtk_source_search_settings_get_at_word_boundaries(p_settings: *SearchSettings) c_int;
    pub const getAtWordBoundaries = gtk_source_search_settings_get_at_word_boundaries;

    extern fn gtk_source_search_settings_get_case_sensitive(p_settings: *SearchSettings) c_int;
    pub const getCaseSensitive = gtk_source_search_settings_get_case_sensitive;

    extern fn gtk_source_search_settings_get_regex_enabled(p_settings: *SearchSettings) c_int;
    pub const getRegexEnabled = gtk_source_search_settings_get_regex_enabled;

    /// Gets the text to search.
    ///
    /// The return value must not be freed.
    ///
    /// You may be interested to call `utilsEscapeSearchText` after
    /// this function.
    extern fn gtk_source_search_settings_get_search_text(p_settings: *SearchSettings) ?[*:0]const u8;
    pub const getSearchText = gtk_source_search_settings_get_search_text;

    extern fn gtk_source_search_settings_get_visible_only(p_settings: *SearchSettings) c_int;
    pub const getVisibleOnly = gtk_source_search_settings_get_visible_only;

    extern fn gtk_source_search_settings_get_wrap_around(p_settings: *SearchSettings) c_int;
    pub const getWrapAround = gtk_source_search_settings_get_wrap_around;

    /// Change whether the search is done at word boundaries.
    ///
    /// If `at_word_boundaries` is `TRUE`, a search match must start and end a word.
    /// The match can span multiple words. See also `gtk.TextIter.startsWord` and
    /// `gtk.TextIter.endsWord`.
    extern fn gtk_source_search_settings_set_at_word_boundaries(p_settings: *SearchSettings, p_at_word_boundaries: c_int) void;
    pub const setAtWordBoundaries = gtk_source_search_settings_set_at_word_boundaries;

    /// Enables or disables the case sensitivity for the search.
    extern fn gtk_source_search_settings_set_case_sensitive(p_settings: *SearchSettings, p_case_sensitive: c_int) void;
    pub const setCaseSensitive = gtk_source_search_settings_set_case_sensitive;

    /// Enables or disables whether to search by regular expressions.
    ///
    /// If enabled, the `SearchSettings.properties.search_text` property contains the
    /// pattern of the regular expression.
    ///
    /// `SearchContext` uses `glib.Regex` when regex search is enabled. See the
    /// [Regular expression syntax](https://developer.gnome.org/glib/stable/glib-regex-syntax.html)
    /// page in the GLib reference manual.
    extern fn gtk_source_search_settings_set_regex_enabled(p_settings: *SearchSettings, p_regex_enabled: c_int) void;
    pub const setRegexEnabled = gtk_source_search_settings_set_regex_enabled;

    /// Sets the text to search.
    ///
    /// If `search_text` is `NULL` or is empty, the search will be disabled. A copy of `search_text`
    /// will be made, so you can safely free `search_text` after a call to this function.
    ///
    /// You may be interested to call `utilsUnescapeSearchText` before
    /// this function.
    extern fn gtk_source_search_settings_set_search_text(p_settings: *SearchSettings, p_search_text: ?[*:0]const u8) void;
    pub const setSearchText = gtk_source_search_settings_set_search_text;

    /// Enables or disables whether to exclude invisible text from the search.
    ///
    /// If enabled, only visible text will be searched.
    /// A search match may have invisible text interspersed.
    extern fn gtk_source_search_settings_set_visible_only(p_settings: *SearchSettings, p_visible_only: c_int) void;
    pub const setVisibleOnly = gtk_source_search_settings_set_visible_only;

    /// Enables or disables the wrap around search.
    ///
    /// If `wrap_around` is `TRUE`, the forward search continues at the beginning of the buffer
    /// if no search occurrences are found. Similarly, the backward search continues to search at
    /// the end of the buffer.
    extern fn gtk_source_search_settings_set_wrap_around(p_settings: *SearchSettings, p_wrap_around: c_int) void;
    pub const setWrapAround = gtk_source_search_settings_set_wrap_around;

    extern fn gtk_source_search_settings_get_type() usize;
    pub const getGObjectType = gtk_source_search_settings_get_type;

    extern fn g_object_ref(p_self: *gtksource.SearchSettings) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SearchSettings) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SearchSettings, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Quick insertion code snippets.
///
/// The `GtkSourceSnippet` represents a series of chunks that can quickly be
/// inserted into the `View`.
///
/// Snippets are defined in XML files which are loaded by the
/// `SnippetManager`. Alternatively, applications can create snippets
/// on demand and insert them into the `View` using
/// `View.pushSnippet`.
///
/// Snippet chunks can reference other snippet chunks as well as post-process
/// the values from other chunks such as capitalization.
pub const Snippet = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SnippetClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtk.TextBuffer;
        };

        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        pub const focus_position = struct {
            pub const name = "focus-position";

            pub const Type = c_int;
        };

        pub const language_id = struct {
            pub const name = "language-id";

            pub const Type = ?[*:0]u8;
        };

        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        pub const trigger = struct {
            pub const name = "trigger";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtksource.Snippet`
    extern fn gtk_source_snippet_new(p_trigger: ?[*:0]const u8, p_language_id: ?[*:0]const u8) *gtksource.Snippet;
    pub const new = gtk_source_snippet_new;

    /// Parses the snippet formatted `text` into a series of chunks and adds them
    /// to a new `gtksource.Snippet`.
    extern fn gtk_source_snippet_new_parsed(p_text: [*:0]const u8, p_error: ?*?*glib.Error) ?*gtksource.Snippet;
    pub const newParsed = gtk_source_snippet_new_parsed;

    /// Appends `chunk` to the `snippet`.
    ///
    /// This may only be called before the snippet has been expanded.
    extern fn gtk_source_snippet_add_chunk(p_snippet: *Snippet, p_chunk: *gtksource.SnippetChunk) void;
    pub const addChunk = gtk_source_snippet_add_chunk;

    /// Does a deep copy of the snippet.
    extern fn gtk_source_snippet_copy(p_snippet: *Snippet) *gtksource.Snippet;
    pub const copy = gtk_source_snippet_copy;

    /// Gets the context used for expanding the snippet.
    extern fn gtk_source_snippet_get_context(p_snippet: *Snippet) ?*gtksource.SnippetContext;
    pub const getContext = gtk_source_snippet_get_context;

    /// Gets the description for the snippet.
    extern fn gtk_source_snippet_get_description(p_snippet: *Snippet) [*:0]const u8;
    pub const getDescription = gtk_source_snippet_get_description;

    /// Gets the current focus for the snippet.
    ///
    /// This is changed as the user tabs through focus locations.
    extern fn gtk_source_snippet_get_focus_position(p_snippet: *Snippet) c_int;
    pub const getFocusPosition = gtk_source_snippet_get_focus_position;

    /// Gets the language-id used for the source snippet.
    ///
    /// The language identifier should be one that matches a
    /// source language `Language.properties.id` property.
    extern fn gtk_source_snippet_get_language_id(p_snippet: *Snippet) [*:0]const u8;
    pub const getLanguageId = gtk_source_snippet_get_language_id;

    /// Gets the number of chunks in the snippet.
    ///
    /// Note that not all chunks are editable.
    extern fn gtk_source_snippet_get_n_chunks(p_snippet: *Snippet) c_uint;
    pub const getNChunks = gtk_source_snippet_get_n_chunks;

    /// Gets the name for the snippet.
    extern fn gtk_source_snippet_get_name(p_snippet: *Snippet) [*:0]const u8;
    pub const getName = gtk_source_snippet_get_name;

    /// Gets the chunk at `nth`.
    extern fn gtk_source_snippet_get_nth_chunk(p_snippet: *Snippet, p_nth: c_uint) *gtksource.SnippetChunk;
    pub const getNthChunk = gtk_source_snippet_get_nth_chunk;

    /// Gets the trigger for the source snippet.
    ///
    /// A trigger is a word that can be expanded into the full snippet when
    /// the user presses Tab.
    extern fn gtk_source_snippet_get_trigger(p_snippet: *Snippet) ?[*:0]const u8;
    pub const getTrigger = gtk_source_snippet_get_trigger;

    /// Sets the description for the snippet.
    extern fn gtk_source_snippet_set_description(p_snippet: *Snippet, p_description: [*:0]const u8) void;
    pub const setDescription = gtk_source_snippet_set_description;

    /// Sets the language identifier for the snippet.
    ///
    /// This should match the `Language.properties.id` identifier.
    extern fn gtk_source_snippet_set_language_id(p_snippet: *Snippet, p_language_id: [*:0]const u8) void;
    pub const setLanguageId = gtk_source_snippet_set_language_id;

    /// Sets the name for the snippet.
    extern fn gtk_source_snippet_set_name(p_snippet: *Snippet, p_name: [*:0]const u8) void;
    pub const setName = gtk_source_snippet_set_name;

    /// Sets the trigger for the snippet.
    extern fn gtk_source_snippet_set_trigger(p_snippet: *Snippet, p_trigger: [*:0]const u8) void;
    pub const setTrigger = gtk_source_snippet_set_trigger;

    extern fn gtk_source_snippet_get_type() usize;
    pub const getGObjectType = gtk_source_snippet_get_type;

    extern fn g_object_ref(p_self: *gtksource.Snippet) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Snippet) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Snippet, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A chunk of text within the source snippet.
///
/// The `GtkSourceSnippetChunk` represents a single chunk of text that
/// may or may not be an edit point within the snippet. Chunks that are
/// an edit point (also called a tab stop) have the
/// `SnippetChunk.properties.focus_position` property set.
pub const SnippetChunk = opaque {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SnippetChunkClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const context = struct {
            pub const name = "context";

            pub const Type = ?*gtksource.SnippetContext;
        };

        pub const focus_position = struct {
            pub const name = "focus-position";

            pub const Type = c_int;
        };

        pub const spec = struct {
            pub const name = "spec";

            pub const Type = ?[*:0]u8;
        };

        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };

        pub const text_set = struct {
            pub const name = "text-set";

            pub const Type = c_int;
        };

        pub const tooltip_text = struct {
            pub const name = "tooltip-text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a new `GtkSourceSnippetChunk` that can be added to
    /// a `Snippet`.
    extern fn gtk_source_snippet_chunk_new() *gtksource.SnippetChunk;
    pub const new = gtk_source_snippet_chunk_new;

    /// Copies the source snippet.
    extern fn gtk_source_snippet_chunk_copy(p_chunk: *SnippetChunk) *gtksource.SnippetChunk;
    pub const copy = gtk_source_snippet_chunk_copy;

    /// Gets the context for the snippet insertion.
    extern fn gtk_source_snippet_chunk_get_context(p_chunk: *SnippetChunk) *gtksource.SnippetContext;
    pub const getContext = gtk_source_snippet_chunk_get_context;

    /// Gets the `SnippetChunk.properties.focus_position`.
    ///
    /// The focus-position is used to determine how many tabs it takes for the
    /// snippet to advanced to this chunk.
    ///
    /// A focus-position of zero will be the last focus position of the snippet
    /// and snippet editing ends when it has been reached.
    ///
    /// A focus-position of -1 means the chunk cannot be focused by the user.
    extern fn gtk_source_snippet_chunk_get_focus_position(p_chunk: *SnippetChunk) c_int;
    pub const getFocusPosition = gtk_source_snippet_chunk_get_focus_position;

    /// Gets the specification for the chunk.
    ///
    /// The specification is evaluated for variables when other chunks are edited
    /// within the snippet context. If the user has changed the text, the
    /// `SnippetChunk.properties.text` and `SnippetChunk.properties.text_set` properties
    /// are updated.
    extern fn gtk_source_snippet_chunk_get_spec(p_chunk: *SnippetChunk) ?[*:0]const u8;
    pub const getSpec = gtk_source_snippet_chunk_get_spec;

    /// Gets the `SnippetChunk.properties.text` property.
    ///
    /// The text property is updated when the user edits the text of the chunk.
    /// If it has not been edited, the `SnippetChunk.properties.spec` property is
    /// returned.
    extern fn gtk_source_snippet_chunk_get_text(p_chunk: *SnippetChunk) [*:0]const u8;
    pub const getText = gtk_source_snippet_chunk_get_text;

    /// Gets the `SnippetChunk.properties.text_set` property.
    ///
    /// This is typically set when the user has edited a snippet chunk.
    extern fn gtk_source_snippet_chunk_get_text_set(p_chunk: *SnippetChunk) c_int;
    pub const getTextSet = gtk_source_snippet_chunk_get_text_set;

    extern fn gtk_source_snippet_chunk_get_tooltip_text(p_chunk: *SnippetChunk) [*:0]const u8;
    pub const getTooltipText = gtk_source_snippet_chunk_get_tooltip_text;

    extern fn gtk_source_snippet_chunk_set_context(p_chunk: *SnippetChunk, p_context: *gtksource.SnippetContext) void;
    pub const setContext = gtk_source_snippet_chunk_set_context;

    /// Sets the `SnippetChunk.properties.focus_position` property.
    ///
    /// The focus-position is used to determine how many tabs it takes for the
    /// snippet to advanced to this chunk.
    ///
    /// A focus-position of zero will be the last focus position of the snippet
    /// and snippet editing ends when it has been reached.
    ///
    /// A focus-position of -1 means the chunk cannot be focused by the user.
    extern fn gtk_source_snippet_chunk_set_focus_position(p_chunk: *SnippetChunk, p_focus_position: c_int) void;
    pub const setFocusPosition = gtk_source_snippet_chunk_set_focus_position;

    /// Sets the specification for the chunk.
    ///
    /// The specification is evaluated for variables when other chunks are edited
    /// within the snippet context. If the user has changed the text, the
    /// `SnippetChunk.properties.@"text and"` `SnippetChunk.properties.text_set` properties
    /// are updated.
    extern fn gtk_source_snippet_chunk_set_spec(p_chunk: *SnippetChunk, p_spec: [*:0]const u8) void;
    pub const setSpec = gtk_source_snippet_chunk_set_spec;

    /// Sets the text for the snippet chunk.
    ///
    /// This is usually used by the snippet engine to update the text, but may
    /// be useful when creating custom snippets to avoid expansion of any
    /// specification.
    extern fn gtk_source_snippet_chunk_set_text(p_chunk: *SnippetChunk, p_text: [*:0]const u8) void;
    pub const setText = gtk_source_snippet_chunk_set_text;

    /// Sets the `SnippetChunk.properties.text_set` property.
    ///
    /// This is typically set when the user has edited a snippet chunk by the
    /// snippet engine.
    extern fn gtk_source_snippet_chunk_set_text_set(p_chunk: *SnippetChunk, p_text_set: c_int) void;
    pub const setTextSet = gtk_source_snippet_chunk_set_text_set;

    extern fn gtk_source_snippet_chunk_set_tooltip_text(p_chunk: *SnippetChunk, p_tooltip_text: [*:0]const u8) void;
    pub const setTooltipText = gtk_source_snippet_chunk_set_tooltip_text;

    extern fn gtk_source_snippet_chunk_get_type() usize;
    pub const getGObjectType = gtk_source_snippet_chunk_get_type;

    extern fn g_object_ref(p_self: *gtksource.SnippetChunk) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SnippetChunk) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SnippetChunk, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Context for expanding `SnippetChunk`.
///
/// This class is currently used primary as a hashtable. However, the longer
/// term goal is to have it hold onto a `GjsContext` as well as other languages
/// so that `SnippetChunk` can expand themselves by executing
/// script within the context.
///
/// The `Snippet` will build the context and then expand each of the
/// chunks during the insertion/edit phase.
pub const SnippetContext = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SnippetContextClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// The signal is emitted when a change has been
        /// discovered in one of the chunks of the snippet which has
        /// caused a variable or other dynamic data within the context
        /// to have changed.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SnippetContext, p_instance))),
                    gobject.signalLookup("changed", SnippetContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtksource.SnippetContext`.
    ///
    /// Generally, this isn't needed unless you are controlling the
    /// expansion of snippets manually.
    extern fn gtk_source_snippet_context_new() *gtksource.SnippetContext;
    pub const new = gtk_source_snippet_context_new;

    /// Removes all variables from the context.
    extern fn gtk_source_snippet_context_clear_variables(p_self: *SnippetContext) void;
    pub const clearVariables = gtk_source_snippet_context_clear_variables;

    extern fn gtk_source_snippet_context_expand(p_self: *SnippetContext, p_input: [*:0]const u8) [*:0]u8;
    pub const expand = gtk_source_snippet_context_expand;

    /// Gets the current value for a variable named `key`.
    extern fn gtk_source_snippet_context_get_variable(p_self: *SnippetContext, p_key: [*:0]const u8) ?[*:0]const u8;
    pub const getVariable = gtk_source_snippet_context_get_variable;

    /// Sets a constatnt within the context.
    ///
    /// This is similar to a variable set with `SnippetContext.setVariable`
    /// but is expected to not change during use of the snippet.
    ///
    /// Examples would be the date or users name.
    extern fn gtk_source_snippet_context_set_constant(p_self: *SnippetContext, p_key: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setConstant = gtk_source_snippet_context_set_constant;

    extern fn gtk_source_snippet_context_set_line_prefix(p_self: *SnippetContext, p_line_prefix: [*:0]const u8) void;
    pub const setLinePrefix = gtk_source_snippet_context_set_line_prefix;

    extern fn gtk_source_snippet_context_set_tab_width(p_self: *SnippetContext, p_tab_width: c_int) void;
    pub const setTabWidth = gtk_source_snippet_context_set_tab_width;

    extern fn gtk_source_snippet_context_set_use_spaces(p_self: *SnippetContext, p_use_spaces: c_int) void;
    pub const setUseSpaces = gtk_source_snippet_context_set_use_spaces;

    /// Sets a variable within the context.
    ///
    /// This variable may be overridden by future updates to the
    /// context.
    extern fn gtk_source_snippet_context_set_variable(p_self: *SnippetContext, p_key: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setVariable = gtk_source_snippet_context_set_variable;

    extern fn gtk_source_snippet_context_get_type() usize;
    pub const getGObjectType = gtk_source_snippet_context_get_type;

    extern fn g_object_ref(p_self: *gtksource.SnippetContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SnippetContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SnippetContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Provides access to `Snippet`.
///
/// `GtkSourceSnippetManager` is an object which processes snippet description
/// files and creates `Snippet` objects.
///
/// Use `SnippetManager.getDefault` to retrieve the default
/// instance of `GtkSourceSnippetManager`.
///
/// Use `SnippetManager.getSnippet` to retrieve snippets for
/// a given snippets.
pub const SnippetManager = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SnippetManagerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Contains a list of directories to search for files containing snippets (*.snippets).
        pub const search_path = struct {
            pub const name = "search-path";

            pub const Type = ?[*][*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Returns the default `gtksource.SnippetManager` instance.
    extern fn gtk_source_snippet_manager_get_default() *gtksource.SnippetManager;
    pub const getDefault = gtk_source_snippet_manager_get_default;

    /// Gets the list directories where `self` looks for snippet files.
    extern fn gtk_source_snippet_manager_get_search_path(p_self: *SnippetManager) [*]const [*:0]const u8;
    pub const getSearchPath = gtk_source_snippet_manager_get_search_path;

    /// Queries the known snippets for the first matching `group`, `language_id`,
    /// and/or `trigger`.
    ///
    /// If `group` or `language_id` are `NULL`, they will be ignored.
    extern fn gtk_source_snippet_manager_get_snippet(p_self: *SnippetManager, p_group: ?[*:0]const u8, p_language_id: ?[*:0]const u8, p_trigger: [*:0]const u8) ?*gtksource.Snippet;
    pub const getSnippet = gtk_source_snippet_manager_get_snippet;

    /// Gets a `gio.ListModel` of all snippets.
    ///
    /// This can be used to get an unfiltered list of all of the snippets
    /// known to the snippet manager.
    extern fn gtk_source_snippet_manager_list_all(p_self: *SnippetManager) *gio.ListModel;
    pub const listAll = gtk_source_snippet_manager_list_all;

    /// List all the known groups within the snippet manager.
    ///
    /// The result should be freed with `glib.free`, and the individual strings are
    /// owned by `self` and should never be freed by the caller.
    extern fn gtk_source_snippet_manager_list_groups(p_self: *SnippetManager) [*][*:0]const u8;
    pub const listGroups = gtk_source_snippet_manager_list_groups;

    /// Queries the known snippets for those matching `group`, `language_id`, and/or
    /// `trigger_prefix`.
    ///
    /// If any of these are `NULL`, they will be ignored when filtering the available snippets.
    ///
    /// The `gio.ListModel` only contains information about the available snippets until
    /// `gio.ListModel.getItem` is called for a specific snippet. This helps reduce
    /// the number of `gobject.Object`'s that are created at runtime to those needed by
    /// the calling application.
    extern fn gtk_source_snippet_manager_list_matching(p_self: *SnippetManager, p_group: ?[*:0]const u8, p_language_id: ?[*:0]const u8, p_trigger_prefix: ?[*:0]const u8) *gio.ListModel;
    pub const listMatching = gtk_source_snippet_manager_list_matching;

    /// Sets the list of directories in which the `GtkSourceSnippetManager` looks for
    /// snippet files.
    ///
    /// If `dirs` is `NULL`, the search path is reset to default.
    ///
    /// At the moment this function can be called only before the
    /// snippet files are loaded for the first time. In practice
    /// to set a custom search path for a `GtkSourceSnippetManager`,
    /// you have to call this function right after creating it.
    extern fn gtk_source_snippet_manager_set_search_path(p_self: *SnippetManager, p_dirs: ?[*]const [*:0]const u8) void;
    pub const setSearchPath = gtk_source_snippet_manager_set_search_path;

    extern fn gtk_source_snippet_manager_get_type() usize;
    pub const getGObjectType = gtk_source_snippet_manager_get_type;

    extern fn g_object_ref(p_self: *gtksource.SnippetManager) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SnippetManager) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SnippetManager, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Represent white space characters with symbols.
///
/// `gtksource.SpaceDrawer` provides a way to visualize white spaces, by drawing
/// symbols.
///
/// Call `View.getSpaceDrawer` to get the `GtkSourceSpaceDrawer`
/// instance of a certain `View`.
///
/// By default, no white spaces are drawn because the
/// `SpaceDrawer.properties.enable_matrix` is `FALSE`.
///
/// To draw white spaces, `SpaceDrawer.setTypesForLocations` can
/// be called to set the `SpaceDrawer.properties.matrix` property (by default all
/// space types are enabled at all locations). Then call
/// `SpaceDrawer.setEnableMatrix`.
///
/// For a finer-grained method, there is also the `Tag`'s
/// `Tag.properties.draw_spaces` property.
///
/// # Example
///
/// To draw non-breaking spaces everywhere and draw all types of trailing spaces
/// except newlines:
/// ```c
/// gtk_source_space_drawer_set_types_for_locations (space_drawer,
///                                                  GTK_SOURCE_SPACE_LOCATION_ALL,
///                                                  GTK_SOURCE_SPACE_TYPE_NBSP);
///
/// gtk_source_space_drawer_set_types_for_locations (space_drawer,
///                                                  GTK_SOURCE_SPACE_LOCATION_TRAILING,
///                                                  GTK_SOURCE_SPACE_TYPE_ALL &
///                                                  ~GTK_SOURCE_SPACE_TYPE_NEWLINE);
///
/// gtk_source_space_drawer_set_enable_matrix (space_drawer, TRUE);
/// ```
///
/// # Use-case: draw unwanted white spaces
///
/// A possible use-case is to draw only unwanted white spaces. Examples:
///
/// - Draw all trailing spaces.
/// - If the indentation and alignment must be done with spaces, draw tabs.
///
/// And non-breaking spaces can always be drawn, everywhere, to distinguish them
/// from normal spaces.
pub const SpaceDrawer = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.SpaceDrawerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the `SpaceDrawer.properties.matrix` property is enabled.
        pub const enable_matrix = struct {
            pub const name = "enable-matrix";

            pub const Type = c_int;
        };

        /// The property is a `glib.Variant` property to specify where and
        /// what kind of white spaces to draw.
        ///
        /// The `glib.Variant` is of type `"au"`, an array of unsigned integers. Each
        /// integer is a combination of `SpaceTypeFlags`. There is one
        /// integer for each `SpaceLocationFlags`, in the same order as
        /// they are defined in the enum (`GTK_SOURCE_SPACE_LOCATION_NONE` and
        /// `GTK_SOURCE_SPACE_LOCATION_ALL` are not taken into account).
        ///
        /// If the array is shorter than the number of locations, then the value
        /// for the missing locations will be `GTK_SOURCE_SPACE_TYPE_NONE`.
        ///
        /// By default, `GTK_SOURCE_SPACE_TYPE_ALL` is set for all locations.4
        pub const matrix = struct {
            pub const name = "matrix";

            pub const Type = ?*glib.Variant;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtksource.SpaceDrawer` object.
    ///
    /// Useful for storing space drawing settings independently of a `View`.
    extern fn gtk_source_space_drawer_new() *gtksource.SpaceDrawer;
    pub const new = gtk_source_space_drawer_new;

    /// Binds the `SpaceDrawer.properties.matrix` property to a `gio.Settings` key.
    ///
    /// The `gio.Settings` key must be of the same type as the
    /// `SpaceDrawer.properties.matrix` property, that is, `"au"`.
    ///
    /// The `gio.Settings.bind` function cannot be used, because the default GIO
    /// mapping functions don't support `glib.Variant` properties (maybe it will be
    /// supported by a future GIO version, in which case this function can be
    /// deprecated).
    extern fn gtk_source_space_drawer_bind_matrix_setting(p_drawer: *SpaceDrawer, p_settings: *gio.Settings, p_key: [*:0]const u8, p_flags: gio.SettingsBindFlags) void;
    pub const bindMatrixSetting = gtk_source_space_drawer_bind_matrix_setting;

    extern fn gtk_source_space_drawer_get_enable_matrix(p_drawer: *SpaceDrawer) c_int;
    pub const getEnableMatrix = gtk_source_space_drawer_get_enable_matrix;

    /// Gets the value of the `SpaceDrawer.properties.matrix` property, as a `glib.Variant`.
    ///
    /// An empty array can be returned in case the matrix is a zero matrix.
    ///
    /// The `SpaceDrawer.getTypesForLocations` function may be more
    /// convenient to use.
    extern fn gtk_source_space_drawer_get_matrix(p_drawer: *SpaceDrawer) *glib.Variant;
    pub const getMatrix = gtk_source_space_drawer_get_matrix;

    /// If only one location is specified, this function returns what kind of
    /// white spaces are drawn at that location.
    ///
    /// The value is retrieved from the `SpaceDrawer.properties.matrix` property.
    ///
    /// If several locations are specified, this function returns the logical AND for
    /// those locations. Which means that if a certain kind of white space is present
    /// in the return value, then that kind of white space is drawn at all the
    /// specified `locations`.
    extern fn gtk_source_space_drawer_get_types_for_locations(p_drawer: *SpaceDrawer, p_locations: gtksource.SpaceLocationFlags) gtksource.SpaceTypeFlags;
    pub const getTypesForLocations = gtk_source_space_drawer_get_types_for_locations;

    /// Sets whether the `SpaceDrawer.properties.matrix` property is enabled.
    extern fn gtk_source_space_drawer_set_enable_matrix(p_drawer: *SpaceDrawer, p_enable_matrix: c_int) void;
    pub const setEnableMatrix = gtk_source_space_drawer_set_enable_matrix;

    /// Sets a new value to the `SpaceDrawer.properties.matrix` property, as a `glib.Variant`.
    ///
    /// If `matrix` is `NULL`, then an empty array is set.
    ///
    /// If `matrix` is floating, it is consumed.
    ///
    /// The `SpaceDrawer.setTypesForLocations` function may be more
    /// convenient to use.
    extern fn gtk_source_space_drawer_set_matrix(p_drawer: *SpaceDrawer, p_matrix: ?*glib.Variant) void;
    pub const setMatrix = gtk_source_space_drawer_set_matrix;

    /// Modifies the `SpaceDrawer.properties.matrix` property at the specified
    /// `locations`.
    extern fn gtk_source_space_drawer_set_types_for_locations(p_drawer: *SpaceDrawer, p_locations: gtksource.SpaceLocationFlags, p_types: gtksource.SpaceTypeFlags) void;
    pub const setTypesForLocations = gtk_source_space_drawer_set_types_for_locations;

    extern fn gtk_source_space_drawer_get_type() usize;
    pub const getGObjectType = gtk_source_space_drawer_get_type;

    extern fn g_object_ref(p_self: *gtksource.SpaceDrawer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.SpaceDrawer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SpaceDrawer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Represents a style.
///
/// The `GtkSourceStyle` structure is used to describe text attributes
/// which are set when given style is used.
pub const Style = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.StyleClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const background = struct {
            pub const name = "background";

            pub const Type = ?[*:0]u8;
        };

        pub const background_set = struct {
            pub const name = "background-set";

            pub const Type = c_int;
        };

        pub const bold = struct {
            pub const name = "bold";

            pub const Type = c_int;
        };

        pub const bold_set = struct {
            pub const name = "bold-set";

            pub const Type = c_int;
        };

        pub const foreground = struct {
            pub const name = "foreground";

            pub const Type = ?[*:0]u8;
        };

        pub const foreground_set = struct {
            pub const name = "foreground-set";

            pub const Type = c_int;
        };

        pub const italic = struct {
            pub const name = "italic";

            pub const Type = c_int;
        };

        pub const italic_set = struct {
            pub const name = "italic-set";

            pub const Type = c_int;
        };

        pub const line_background = struct {
            pub const name = "line-background";

            pub const Type = ?[*:0]u8;
        };

        pub const line_background_set = struct {
            pub const name = "line-background-set";

            pub const Type = c_int;
        };

        pub const pango_underline = struct {
            pub const name = "pango-underline";

            pub const Type = pango.Underline;
        };

        pub const scale = struct {
            pub const name = "scale";

            pub const Type = ?[*:0]u8;
        };

        pub const scale_set = struct {
            pub const name = "scale-set";

            pub const Type = c_int;
        };

        pub const strikethrough = struct {
            pub const name = "strikethrough";

            pub const Type = c_int;
        };

        pub const strikethrough_set = struct {
            pub const name = "strikethrough-set";

            pub const Type = c_int;
        };

        pub const underline_color = struct {
            pub const name = "underline-color";

            pub const Type = ?[*:0]u8;
        };

        pub const underline_color_set = struct {
            pub const name = "underline-color-set";

            pub const Type = c_int;
        };

        pub const underline_set = struct {
            pub const name = "underline-set";

            pub const Type = c_int;
        };

        pub const weight = struct {
            pub const name = "weight";

            pub const Type = pango.Weight;
        };

        pub const weight_set = struct {
            pub const name = "weight-set";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// This function modifies the `gtk.TextTag` properties that are related to the
    /// `gtksource.Style` properties. Other `gtk.TextTag` properties are left untouched.
    ///
    /// If `style` is non-`NULL`, applies `style` to `tag`.
    ///
    /// If `style` is `NULL`, the related *-set properties of `gtk.TextTag` are set to
    /// `FALSE`.
    extern fn gtk_source_style_apply(p_style: ?*const Style, p_tag: *gtk.TextTag) void;
    pub const apply = gtk_source_style_apply;

    /// Creates a copy of `style`, that is a new `gtksource.Style` instance which
    /// has the same attributes set.
    extern fn gtk_source_style_copy(p_style: *const Style) *gtksource.Style;
    pub const copy = gtk_source_style_copy;

    extern fn gtk_source_style_get_type() usize;
    pub const getGObjectType = gtk_source_style_get_type;

    extern fn g_object_ref(p_self: *gtksource.Style) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Style) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Style, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Controls the appearance of `View`.
///
/// `gtksource.StyleScheme` contains all the text styles to be used in
/// `View` and `Buffer`. For instance, it contains text styles
/// for syntax highlighting, it may contain foreground and background color for
/// non-highlighted text, color for the line numbers, current line highlighting,
/// bracket matching, etc.
///
/// Style schemes are stored in XML files. The format of a scheme file is
/// documented in the [style scheme reference](./style-reference.html).
///
/// The two style schemes with IDs "classic" and "tango" follow more closely the
/// GTK theme (for example for the background color).
pub const StyleScheme = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.StyleSchemeClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Style scheme description, a translatable string to present to the user.
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// Style scheme filename or `NULL`.
        pub const filename = struct {
            pub const name = "filename";

            pub const Type = ?[*:0]u8;
        };

        /// Style scheme id, a unique string used to identify the style scheme
        /// in `StyleSchemeManager`.
        pub const id = struct {
            pub const name = "id";

            pub const Type = ?[*:0]u8;
        };

        /// Style scheme name, a translatable string to present to the user.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn gtk_source_style_scheme_get_authors(p_scheme: *StyleScheme) ?[*]const [*:0]const u8;
    pub const getAuthors = gtk_source_style_scheme_get_authors;

    extern fn gtk_source_style_scheme_get_description(p_scheme: *StyleScheme) ?[*:0]const u8;
    pub const getDescription = gtk_source_style_scheme_get_description;

    extern fn gtk_source_style_scheme_get_filename(p_scheme: *StyleScheme) ?[*:0]const u8;
    pub const getFilename = gtk_source_style_scheme_get_filename;

    extern fn gtk_source_style_scheme_get_id(p_scheme: *StyleScheme) [*:0]const u8;
    pub const getId = gtk_source_style_scheme_get_id;

    /// Gets a metadata property from the style scheme.
    extern fn gtk_source_style_scheme_get_metadata(p_scheme: *StyleScheme, p_name: [*:0]const u8) ?[*:0]const u8;
    pub const getMetadata = gtk_source_style_scheme_get_metadata;

    extern fn gtk_source_style_scheme_get_name(p_scheme: *StyleScheme) [*:0]const u8;
    pub const getName = gtk_source_style_scheme_get_name;

    extern fn gtk_source_style_scheme_get_style(p_scheme: *StyleScheme, p_style_id: [*:0]const u8) ?*gtksource.Style;
    pub const getStyle = gtk_source_style_scheme_get_style;

    extern fn gtk_source_style_scheme_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleScheme) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleScheme) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleScheme, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A button to launch a style scheme selection dialog.
///
/// The `GtkSourceStyleSchemeChooserButton` is a button which displays
/// the currently selected style scheme and allows to open a style scheme
/// selection dialog to change the style scheme.
/// It is suitable widget for selecting a style scheme in a preference dialog.
///
/// In `GtkSourceStyleSchemeChooserButton`, a `StyleSchemeChooserWidget`
/// is used to provide a dialog for selecting style schemes.
pub const StyleSchemeChooserButton = extern struct {
    pub const Parent = gtk.Button;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Actionable, gtk.Buildable, gtk.ConstraintTarget, gtksource.StyleSchemeChooser };
    pub const Class = gtksource.StyleSchemeChooserButtonClass;
    f_parent_instance: gtk.Button,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtksource.StyleSchemeChooserButton`.
    extern fn gtk_source_style_scheme_chooser_button_new() *gtksource.StyleSchemeChooserButton;
    pub const new = gtk_source_style_scheme_chooser_button_new;

    extern fn gtk_source_style_scheme_chooser_button_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_chooser_button_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleSchemeChooserButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleSchemeChooserButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleSchemeChooserButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A widget for choosing style schemes.
///
/// The `GtkSourceStyleSchemeChooserWidget` widget lets the user select a
/// style scheme. By default, the chooser presents a predefined list
/// of style schemes.
///
/// To change the initially selected style scheme,
/// use `StyleSchemeChooser.setStyleScheme`.
/// To get the selected style scheme
/// use `StyleSchemeChooser.getStyleScheme`.
pub const StyleSchemeChooserWidget = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Buildable, gtk.ConstraintTarget, gtksource.StyleSchemeChooser };
    pub const Class = gtksource.StyleSchemeChooserWidgetClass;
    f_parent_instance: gtk.Widget,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtksource.StyleSchemeChooserWidget`.
    extern fn gtk_source_style_scheme_chooser_widget_new() *gtksource.StyleSchemeChooserWidget;
    pub const new = gtk_source_style_scheme_chooser_widget_new;

    extern fn gtk_source_style_scheme_chooser_widget_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_chooser_widget_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleSchemeChooserWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleSchemeChooserWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleSchemeChooserWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Provides access to `StyleScheme`s.
pub const StyleSchemeManager = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtksource.StyleSchemeManagerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const scheme_ids = struct {
            pub const name = "scheme-ids";

            pub const Type = ?[*][*:0]u8;
        };

        pub const search_path = struct {
            pub const name = "search-path";

            pub const Type = ?[*][*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Returns the default `gtksource.StyleSchemeManager` instance.
    extern fn gtk_source_style_scheme_manager_get_default() *gtksource.StyleSchemeManager;
    pub const getDefault = gtk_source_style_scheme_manager_get_default;

    /// Creates a new style manager.
    ///
    /// If you do not need more than one style manager
    /// then use `StyleSchemeManager.getDefault` instead.
    extern fn gtk_source_style_scheme_manager_new() *gtksource.StyleSchemeManager;
    pub const new = gtk_source_style_scheme_manager_new;

    /// Appends `path` to the list of directories where the `manager` looks for
    /// style scheme files.
    ///
    /// See `StyleSchemeManager.setSearchPath` for details.
    extern fn gtk_source_style_scheme_manager_append_search_path(p_manager: *StyleSchemeManager, p_path: [*:0]const u8) void;
    pub const appendSearchPath = gtk_source_style_scheme_manager_append_search_path;

    /// Mark any currently cached information about the available style schems
    /// as invalid.
    ///
    /// All the available style schemes will be reloaded next time the `manager` is accessed.
    extern fn gtk_source_style_scheme_manager_force_rescan(p_manager: *StyleSchemeManager) void;
    pub const forceRescan = gtk_source_style_scheme_manager_force_rescan;

    /// Looks up style scheme by id.
    extern fn gtk_source_style_scheme_manager_get_scheme(p_manager: *StyleSchemeManager, p_scheme_id: [*:0]const u8) ?*gtksource.StyleScheme;
    pub const getScheme = gtk_source_style_scheme_manager_get_scheme;

    /// Returns the ids of the available style schemes.
    extern fn gtk_source_style_scheme_manager_get_scheme_ids(p_manager: *StyleSchemeManager) ?[*]const [*:0]const u8;
    pub const getSchemeIds = gtk_source_style_scheme_manager_get_scheme_ids;

    /// Returns the current search path for the `manager`.
    ///
    /// See `StyleSchemeManager.setSearchPath` for details.
    extern fn gtk_source_style_scheme_manager_get_search_path(p_manager: *StyleSchemeManager) [*]const [*:0]const u8;
    pub const getSearchPath = gtk_source_style_scheme_manager_get_search_path;

    /// Prepends `path` to the list of directories where the `manager` looks
    /// for style scheme files.
    ///
    /// See `StyleSchemeManager.setSearchPath` for details.
    extern fn gtk_source_style_scheme_manager_prepend_search_path(p_manager: *StyleSchemeManager, p_path: [*:0]const u8) void;
    pub const prependSearchPath = gtk_source_style_scheme_manager_prepend_search_path;

    /// Sets the list of directories where the `manager` looks for
    /// style scheme files.
    ///
    /// If `path` is `NULL`, the search path is reset to default.
    extern fn gtk_source_style_scheme_manager_set_search_path(p_manager: *StyleSchemeManager, p_path: ?[*]const [*:0]const u8) void;
    pub const setSearchPath = gtk_source_style_scheme_manager_set_search_path;

    extern fn gtk_source_style_scheme_manager_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_manager_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleSchemeManager) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleSchemeManager) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleSchemeManager, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A preview widget for `StyleScheme`.
///
/// This widget provides a convenient `gtk.Widget` to preview a `StyleScheme`.
///
/// The `StyleSchemePreview.properties.selected` property can be used to manage
/// the selection state of a single preview widget.
pub const StyleSchemePreview = opaque {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ gtk.Accessible, gtk.Actionable, gtk.Buildable, gtk.ConstraintTarget };
    pub const Class = gtksource.StyleSchemePreviewClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const scheme = struct {
            pub const name = "scheme";

            pub const Type = ?*gtksource.StyleScheme;
        };

        pub const selected = struct {
            pub const name = "selected";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(StyleSchemePreview, p_instance))),
                    gobject.signalLookup("activate", StyleSchemePreview.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtksource.StyleSchemePreview` to preview the style scheme
    /// provided in `scheme`.
    extern fn gtk_source_style_scheme_preview_new(p_scheme: *gtksource.StyleScheme) *gtksource.StyleSchemePreview;
    pub const new = gtk_source_style_scheme_preview_new;

    /// Gets the `gtksource.StyleScheme` previewed by the widget.
    extern fn gtk_source_style_scheme_preview_get_scheme(p_self: *StyleSchemePreview) *gtksource.StyleScheme;
    pub const getScheme = gtk_source_style_scheme_preview_get_scheme;

    extern fn gtk_source_style_scheme_preview_get_selected(p_self: *StyleSchemePreview) c_int;
    pub const getSelected = gtk_source_style_scheme_preview_get_selected;

    extern fn gtk_source_style_scheme_preview_set_selected(p_self: *StyleSchemePreview, p_selected: c_int) void;
    pub const setSelected = gtk_source_style_scheme_preview_set_selected;

    extern fn gtk_source_style_scheme_preview_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_preview_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleSchemePreview) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleSchemePreview) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleSchemePreview, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A tag that can be applied to text in a `Buffer`.
///
/// `GtkSourceTag` is a subclass of `gtk.TextTag` that adds properties useful for
/// the GtkSourceView library.
///
/// If, for a certain tag, `gtk.TextTag` is sufficient, it's better that you create
/// a `gtk.TextTag`, not a `Tag`.
pub const Tag = extern struct {
    pub const Parent = gtk.TextTag;
    pub const Implements = [_]type{};
    pub const Class = gtksource.TagClass;
    f_parent_instance: gtk.TextTag,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether to draw white spaces.
        ///
        /// This property takes precedence over the value defined by the `SpaceDrawer`'s
        /// `SpaceDrawer.properties.matrix` property (only where the tag is applied).
        ///
        /// Setting this property also changes `Tag.properties.draw_spaces_set` to
        /// `TRUE`.
        pub const draw_spaces = struct {
            pub const name = "draw-spaces";

            pub const Type = c_int;
        };

        /// Whether the `Tag.properties.draw_spaces` property is set and must be
        /// taken into account.
        pub const draw_spaces_set = struct {
            pub const name = "draw-spaces-set";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a `GtkSourceTag`.
    ///
    /// Configure the tag using object arguments, i.e. using `gobject.Object.set`.
    ///
    /// For usual cases, `Buffer.createSourceTag` is more convenient to
    /// use.
    extern fn gtk_source_tag_new(p_name: ?[*:0]const u8) *gtksource.Tag;
    pub const new = gtk_source_tag_new;

    extern fn gtk_source_tag_get_type() usize;
    pub const getGObjectType = gtk_source_tag_get_type;

    extern fn g_object_ref(p_self: *gtksource.Tag) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Tag) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Tag, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Subclass of `gtk.TextView`.
///
/// `GtkSourceView` is the main class of the GtkSourceView library.
/// Use a `Buffer` to display text with a `GtkSourceView`.
///
/// This class provides:
///
///  - Show the line numbers;
///  - Show a right margin;
///  - Highlight the current line;
///  - Indentation settings;
///  - Configuration for the Home and End keyboard keys;
///  - Configure and show line marks;
///  - And a few other things.
///
/// An easy way to test all these features is to use the test-widget mini-program
/// provided in the GtkSourceView repository, in the tests/ directory.
///
/// # GtkSourceView as GtkBuildable
///
/// The GtkSourceView implementation of the `gtk.Buildable` interface exposes the
/// `View.properties.completion` object with the internal-child "completion".
///
/// An example of a UI definition fragment with GtkSourceView:
/// ```xml
/// <object class="GtkSourceView" id="source_view">
///   <property name="tab-width">4</property>
///   <property name="auto-indent">True</property>
///   <child internal-child="completion">
///     <object class="GtkSourceCompletion">
///       <property name="select-on-show">False</property>
///     </object>
///   </child>
/// </object>
/// ```
///
/// # Changing the Font
///
/// Gtk CSS provides the best way to change the font for a `GtkSourceView` in a
/// manner that allows for components like `Map` to scale the desired
/// font.
///
/// ```c
/// GtkCssProvider *provider = gtk_css_provider_new ();
/// gtk_css_provider_load_from_data (provider,
///                                  "textview { font-family: Monospace; font-size: 8pt; }",
///                                  -1,
///                                  NULL);
/// gtk_style_context_add_provider (gtk_widget_get_style_context (view),
///                                 GTK_STYLE_PROVIDER (provider),
///                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
/// g_object_unref (provider);
/// ```
///
/// If you need to adjust the font or size of font within a portion of the
/// document only, you should use a `gtk.TextTag` with the `gtk.TextTag.properties.family` or
/// `gtk.TextTag.properties.scale` set so that the font size may be scaled relative to
/// the default font set in CSS.
pub const View = extern struct {
    pub const Parent = gtk.TextView;
    pub const Implements = [_]type{ gtk.Accessible, gtk.AccessibleText, gtk.Buildable, gtk.ConstraintTarget, gtk.Scrollable };
    pub const Class = gtksource.ViewClass;
    f_parent_instance: gtk.TextView,

    pub const virtual_methods = struct {
        pub const line_mark_activated = struct {
            pub fn call(p_class: anytype, p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *const gtk.TextIter, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) void {
                return gobject.ext.as(View.Class, p_class).f_line_mark_activated.?(gobject.ext.as(View, p_view), p_iter, p_button, p_state, p_n_presses);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *const gtk.TextIter, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) callconv(.C) void) void {
                gobject.ext.as(View.Class, p_class).f_line_mark_activated = @ptrCast(p_implementation);
            }
        };

        pub const move_lines = struct {
            pub fn call(p_class: anytype, p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_down: c_int) void {
                return gobject.ext.as(View.Class, p_class).f_move_lines.?(gobject.ext.as(View, p_view), p_down);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_down: c_int) callconv(.C) void) void {
                gobject.ext.as(View.Class, p_class).f_move_lines = @ptrCast(p_implementation);
            }
        };

        pub const move_words = struct {
            pub fn call(p_class: anytype, p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: c_int) void {
                return gobject.ext.as(View.Class, p_class).f_move_words.?(gobject.ext.as(View, p_view), p_step);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: c_int) callconv(.C) void) void {
                gobject.ext.as(View.Class, p_class).f_move_words = @ptrCast(p_implementation);
            }
        };

        /// Inserts a new snippet at `location`
        ///
        /// If another snippet was already active, it will be paused and the new
        /// snippet will become active. Once the focus positions of `snippet` have
        /// been exhausted, editing will return to the previous snippet.
        pub const push_snippet = struct {
            pub fn call(p_class: anytype, p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_snippet: *gtksource.Snippet, p_location: ?*gtk.TextIter) void {
                return gobject.ext.as(View.Class, p_class).f_push_snippet.?(gobject.ext.as(View, p_view), p_snippet, p_location);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_snippet: *gtksource.Snippet, p_location: ?*gtk.TextIter) callconv(.C) void) void {
                gobject.ext.as(View.Class, p_class).f_push_snippet = @ptrCast(p_implementation);
            }
        };

        pub const show_completion = struct {
            pub fn call(p_class: anytype, p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(View.Class, p_class).f_show_completion.?(gobject.ext.as(View, p_view));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_view: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(View.Class, p_class).f_show_completion = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const auto_indent = struct {
            pub const name = "auto-indent";

            pub const Type = c_int;
        };

        /// Draw a specific background pattern on the view.
        pub const background_pattern = struct {
            pub const name = "background-pattern";

            pub const Type = gtksource.BackgroundPatternType;
        };

        /// The completion object associated with the view
        pub const completion = struct {
            pub const name = "completion";

            pub const Type = ?*gtksource.Completion;
        };

        /// The property denotes if snippets should be
        /// expanded when the user presses Tab after having typed a word
        /// matching the snippets found in `SnippetManager`.
        ///
        /// The user may tab through focus-positions of the snippet if any
        /// are available by pressing Tab repeatedly until the desired focus
        /// position is selected.
        pub const enable_snippets = struct {
            pub const name = "enable-snippets";

            pub const Type = c_int;
        };

        pub const highlight_current_line = struct {
            pub const name = "highlight-current-line";

            pub const Type = c_int;
        };

        pub const indent_on_tab = struct {
            pub const name = "indent-on-tab";

            pub const Type = c_int;
        };

        /// Width of an indentation step expressed in number of spaces.
        pub const indent_width = struct {
            pub const name = "indent-width";

            pub const Type = c_int;
        };

        /// The property is a `Indenter` to use to indent
        /// as the user types into the `View`.
        pub const indenter = struct {
            pub const name = "indenter";

            pub const Type = ?*gtksource.Indenter;
        };

        pub const insert_spaces_instead_of_tabs = struct {
            pub const name = "insert-spaces-instead-of-tabs";

            pub const Type = c_int;
        };

        /// Position of the right margin.
        pub const right_margin_position = struct {
            pub const name = "right-margin-position";

            pub const Type = c_uint;
        };

        /// Whether to display line mark pixbufs
        pub const show_line_marks = struct {
            pub const name = "show-line-marks";

            pub const Type = c_int;
        };

        /// Whether to display line numbers
        pub const show_line_numbers = struct {
            pub const name = "show-line-numbers";

            pub const Type = c_int;
        };

        /// Whether to display the right margin.
        pub const show_right_margin = struct {
            pub const name = "show-right-margin";

            pub const Type = c_int;
        };

        /// Whether smart Backspace should be used.
        pub const smart_backspace = struct {
            pub const name = "smart-backspace";

            pub const Type = c_int;
        };

        /// Set the behavior of the HOME and END keys.
        pub const smart_home_end = struct {
            pub const name = "smart-home-end";

            pub const Type = gtksource.SmartHomeEndType;
        };

        /// The `SpaceDrawer` object associated with the view.4
        pub const space_drawer = struct {
            pub const name = "space-drawer";

            pub const Type = ?*gtksource.SpaceDrawer;
        };

        /// Width of a tab character expressed in number of spaces.
        pub const tab_width = struct {
            pub const name = "tab-width";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// Keybinding signal to change case of the text at the current cursor position.
        pub const change_case = struct {
            pub const name = "change-case";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_case_type: gtksource.ChangeCaseType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("change-case", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Keybinding signal to edit a number at the current cursor position.
        pub const change_number = struct {
            pub const name = "change-number";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_count: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("change-number", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Keybinding signal to join the lines currently selected.
        pub const join_lines = struct {
            pub const name = "join-lines";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("join-lines", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a line mark has been activated (for instance when there
        /// was a button press in the line marks gutter).
        ///
        /// You can use `iter` to determine on which line the activation took place.
        pub const line_mark_activated = struct {
            pub const name = "line-mark-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_iter: *gtk.TextIter, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("line-mark-activated", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is a keybinding which gets emitted when the user initiates moving a line.
        ///
        /// The default binding key is Alt+Up/Down arrow. And moves the currently selected lines,
        /// or the current line up or down by one line.
        pub const move_lines = struct {
            pub const name = "move-lines";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_down: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("move-lines", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Keybinding signal to move the cursor to the matching bracket.
        pub const move_to_matching_bracket = struct {
            pub const name = "move-to-matching-bracket";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_extend_selection: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("move-to-matching-bracket", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is a keybinding which gets emitted when the user initiates moving a word.
        ///
        /// The default binding key is Alt+Left/Right Arrow and moves the current selection, or the current
        /// word by one word.
        pub const move_words = struct {
            pub const name = "move-words";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_count: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("move-words", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is emitted to insert a new snippet into the view.
        ///
        /// If another snippet was active, it will be paused until all focus positions of `snippet` have been exhausted.
        ///
        /// `location` will be updated to point at the end of the snippet.
        pub const push_snippet = struct {
            pub const name = "push-snippet";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_snippet: *gtksource.Snippet, p_location: *gtk.TextIter, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("push-snippet", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is a key binding signal which gets
        /// emitted when the user requests a completion, by pressing
        /// <keycombo><keycap>Control</keycap><keycap>space</keycap></keycombo>.
        ///
        /// This will create a `CompletionContext` with the activation
        /// type as `GTK_SOURCE_COMPLETION_ACTIVATION_USER_REQUESTED`.
        ///
        /// Applications should not connect to it, but may emit it with
        /// `gobject.signalEmitByName` if they need to activate the completion by
        /// another means, for example with another key binding or a menu entry.
        pub const show_completion = struct {
            pub const name = "show-completion";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("show-completion", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a the cursor was moved according to the smart home end setting.
        ///
        /// The signal is emitted after the cursor is moved, but
        /// during the `gtk.TextView.signals.move_cursor` action. This can be used to find
        /// out whether the cursor was moved by a normal home/end or by a smart
        /// home/end.
        pub const smart_home_end = struct {
            pub const name = "smart-home-end";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_iter: *gtk.TextIter, p_count: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(View, p_instance))),
                    gobject.signalLookup("smart-home-end", View.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `GtkSourceView`.
    ///
    /// By default, an empty `Buffer` will be lazily created and can be
    /// retrieved with `gtk.TextView.getBuffer`.
    ///
    /// If you want to specify your own buffer, either override the
    /// `gtk.TextView.virtual_methods.create_buffer` factory method, or use
    /// `View.newWithBuffer`.
    extern fn gtk_source_view_new() *gtksource.View;
    pub const new = gtk_source_view_new;

    /// Creates a new `gtksource.View` widget displaying the buffer `buffer`.
    ///
    /// One buffer can be shared among many widgets.
    extern fn gtk_source_view_new_with_buffer(p_buffer: *gtksource.Buffer) *gtksource.View;
    pub const newWithBuffer = gtk_source_view_new_with_buffer;

    /// Returns whether auto-indentation of text is enabled.
    extern fn gtk_source_view_get_auto_indent(p_view: *View) c_int;
    pub const getAutoIndent = gtk_source_view_get_auto_indent;

    /// Returns the `gtksource.BackgroundPatternType` specifying if and how
    /// the background pattern should be displayed for this `view`.
    extern fn gtk_source_view_get_background_pattern(p_view: *View) gtksource.BackgroundPatternType;
    pub const getBackgroundPattern = gtk_source_view_get_background_pattern;

    /// Gets the `Completion` associated with `view`.
    ///
    /// The returned object is guaranteed to be the same for the lifetime of `view`.
    /// Each `GtkSourceView` object has a different `Completion`.
    extern fn gtk_source_view_get_completion(p_view: *View) *gtksource.Completion;
    pub const getCompletion = gtk_source_view_get_completion;

    /// Gets the `View.properties.enable_snippets` property.
    ///
    /// If `TRUE`, matching snippets found in the `SnippetManager`
    /// may be expanded when the user presses Tab after a word in the `View`.
    extern fn gtk_source_view_get_enable_snippets(p_view: *View) c_int;
    pub const getEnableSnippets = gtk_source_view_get_enable_snippets;

    /// Returns the `Gutter` object associated with `window_type` for `view`.
    ///
    /// Only `GTK_TEXT_WINDOW_LEFT` and `GTK_TEXT_WINDOW_RIGHT` are supported,
    /// respectively corresponding to the left and right gutter. The line numbers
    /// and mark category icons are rendered in the left gutter.
    extern fn gtk_source_view_get_gutter(p_view: *View, p_window_type: gtk.TextWindowType) *gtksource.Gutter;
    pub const getGutter = gtk_source_view_get_gutter;

    /// Returns whether the current line is highlighted.
    extern fn gtk_source_view_get_highlight_current_line(p_view: *View) c_int;
    pub const getHighlightCurrentLine = gtk_source_view_get_highlight_current_line;

    /// Gets the `Hover` associated with `view`.
    ///
    /// The returned object is guaranteed to be the same for the lifetime of `view`.
    /// Each `View` object has a different `Hover`.
    extern fn gtk_source_view_get_hover(p_view: *View) *gtksource.Hover;
    pub const getHover = gtk_source_view_get_hover;

    /// Returns whether when the tab key is pressed the current selection
    /// should get indented instead of replaced with the `\t` character.
    extern fn gtk_source_view_get_indent_on_tab(p_view: *View) c_int;
    pub const getIndentOnTab = gtk_source_view_get_indent_on_tab;

    /// Returns the number of spaces to use for each step of indent.
    ///
    /// See `View.setIndentWidth` for details.
    extern fn gtk_source_view_get_indent_width(p_view: *View) c_int;
    pub const getIndentWidth = gtk_source_view_get_indent_width;

    /// Gets the `View.properties.indenter` property.
    extern fn gtk_source_view_get_indenter(p_view: *View) ?*gtksource.Indenter;
    pub const getIndenter = gtk_source_view_get_indenter;

    /// Returns whether when inserting a tabulator character it should
    /// be replaced by a group of space characters.
    extern fn gtk_source_view_get_insert_spaces_instead_of_tabs(p_view: *View) c_int;
    pub const getInsertSpacesInsteadOfTabs = gtk_source_view_get_insert_spaces_instead_of_tabs;

    /// Gets attributes and priority for the `category`.
    extern fn gtk_source_view_get_mark_attributes(p_view: *View, p_category: [*:0]const u8, p_priority: *c_int) *gtksource.MarkAttributes;
    pub const getMarkAttributes = gtk_source_view_get_mark_attributes;

    /// Gets the position of the right margin in the given `view`.
    extern fn gtk_source_view_get_right_margin_position(p_view: *View) c_uint;
    pub const getRightMarginPosition = gtk_source_view_get_right_margin_position;

    /// Returns whether line marks are displayed beside the text.
    extern fn gtk_source_view_get_show_line_marks(p_view: *View) c_int;
    pub const getShowLineMarks = gtk_source_view_get_show_line_marks;

    /// Returns whether line numbers are displayed beside the text.
    extern fn gtk_source_view_get_show_line_numbers(p_view: *View) c_int;
    pub const getShowLineNumbers = gtk_source_view_get_show_line_numbers;

    /// Returns whether a right margin is displayed.
    extern fn gtk_source_view_get_show_right_margin(p_view: *View) c_int;
    pub const getShowRightMargin = gtk_source_view_get_show_right_margin;

    /// Returns `TRUE` if pressing the Backspace key will try to delete spaces
    /// up to the previous tab stop.
    extern fn gtk_source_view_get_smart_backspace(p_view: *View) c_int;
    pub const getSmartBackspace = gtk_source_view_get_smart_backspace;

    /// Returns a `SmartHomeEndType` end value specifying
    /// how the cursor will move when HOME and END keys are pressed.
    extern fn gtk_source_view_get_smart_home_end(p_view: *View) gtksource.SmartHomeEndType;
    pub const getSmartHomeEnd = gtk_source_view_get_smart_home_end;

    /// Gets the `SpaceDrawer` associated with `view`.
    ///
    /// The returned object is guaranteed to be the same for the lifetime of `view`.
    /// Each `View` object has a different `SpaceDrawer`.
    extern fn gtk_source_view_get_space_drawer(p_view: *View) *gtksource.SpaceDrawer;
    pub const getSpaceDrawer = gtk_source_view_get_space_drawer;

    /// Returns the width of tabulation in characters.
    extern fn gtk_source_view_get_tab_width(p_view: *View) c_uint;
    pub const getTabWidth = gtk_source_view_get_tab_width;

    /// Determines the visual column at `iter` taking into consideration the
    /// `View.properties.tab_width` of `view`.
    extern fn gtk_source_view_get_visual_column(p_view: *View, p_iter: *const gtk.TextIter) c_uint;
    pub const getVisualColumn = gtk_source_view_get_visual_column;

    /// Inserts one indentation level at the beginning of the specified lines. The
    /// empty lines are not indented.
    extern fn gtk_source_view_indent_lines(p_view: *View, p_start: *gtk.TextIter, p_end: *gtk.TextIter) void;
    pub const indentLines = gtk_source_view_indent_lines;

    /// Inserts a new snippet at `location`
    ///
    /// If another snippet was already active, it will be paused and the new
    /// snippet will become active. Once the focus positions of `snippet` have
    /// been exhausted, editing will return to the previous snippet.
    extern fn gtk_source_view_push_snippet(p_view: *View, p_snippet: *gtksource.Snippet, p_location: ?*gtk.TextIter) void;
    pub const pushSnippet = gtk_source_view_push_snippet;

    /// If `TRUE` auto-indentation of text is enabled.
    ///
    /// When Enter is pressed to create a new line, the auto-indentation inserts the
    /// same indentation as the previous line. This is **not** a
    /// "smart indentation" where an indentation level is added or removed depending
    /// on the context.
    extern fn gtk_source_view_set_auto_indent(p_view: *View, p_enable: c_int) void;
    pub const setAutoIndent = gtk_source_view_set_auto_indent;

    /// Set if and how the background pattern should be displayed.
    extern fn gtk_source_view_set_background_pattern(p_view: *View, p_background_pattern: gtksource.BackgroundPatternType) void;
    pub const setBackgroundPattern = gtk_source_view_set_background_pattern;

    /// Sets the `View.properties.enable_snippets` property.
    ///
    /// If `enable_snippets` is `TRUE`, matching snippets found in the
    /// `SnippetManager` may be expanded when the user presses
    /// Tab after a word in the `View`.
    extern fn gtk_source_view_set_enable_snippets(p_view: *View, p_enable_snippets: c_int) void;
    pub const setEnableSnippets = gtk_source_view_set_enable_snippets;

    /// If `highlight` is `TRUE` the current line will be highlighted.
    extern fn gtk_source_view_set_highlight_current_line(p_view: *View, p_highlight: c_int) void;
    pub const setHighlightCurrentLine = gtk_source_view_set_highlight_current_line;

    /// If `TRUE`, when the tab key is pressed when several lines are selected, the
    /// selected lines are indented of one level instead of being replaced with a `\t`
    /// character. Shift+Tab unindents the selection.
    ///
    /// If the first or last line is not selected completely, it is also indented or
    /// unindented.
    ///
    /// When the selection doesn't span several lines, the tab key always replaces
    /// the selection with a normal `\t` character.
    extern fn gtk_source_view_set_indent_on_tab(p_view: *View, p_enable: c_int) void;
    pub const setIndentOnTab = gtk_source_view_set_indent_on_tab;

    /// Sets the number of spaces to use for each step of indent when the tab key is
    /// pressed.
    ///
    /// If `width` is -1, the value of the `View.properties.tab_width` property
    /// will be used.
    ///
    /// The `View.properties.indent_width` interacts with the
    /// `View.properties.insert_spaces_instead_of_tabs` property and
    /// `View.properties.tab_width`. An example will be clearer:
    ///
    /// If the `View.properties.indent_width` is 4 and `View.properties.tab_width` is 8 and
    /// `View.properties.insert_spaces_instead_of_tabs` is `FALSE`, then pressing the tab
    /// key at the beginning of a line will insert 4 spaces. So far so good. Pressing
    /// the tab key a second time will remove the 4 spaces and insert a `\t` character
    /// instead (since `View.properties.tab_width` is 8). On the other hand, if
    /// `View.properties.insert_spaces_instead_of_tabs` is `TRUE`, the second tab key
    /// pressed will insert 4 more spaces for a total of 8 spaces in the
    /// `gtk.TextBuffer`.
    ///
    /// The test-widget program (available in the GtkSourceView repository) may be
    /// useful to better understand the indentation settings (enable the space
    /// drawing!).
    extern fn gtk_source_view_set_indent_width(p_view: *View, p_width: c_int) void;
    pub const setIndentWidth = gtk_source_view_set_indent_width;

    /// Sets the indenter for `view` to `indenter`.
    ///
    /// Note that the indenter will not be used unless `gtksource.View.properties.auto`-indent
    /// has been set to `TRUE`.
    extern fn gtk_source_view_set_indenter(p_view: *View, p_indenter: ?*gtksource.Indenter) void;
    pub const setIndenter = gtk_source_view_set_indenter;

    /// If `TRUE` a tab key pressed is replaced by a group of space characters.
    ///
    /// Of course it is still possible to insert a real `\t` programmatically with the
    /// `gtk.TextBuffer` API.
    extern fn gtk_source_view_set_insert_spaces_instead_of_tabs(p_view: *View, p_enable: c_int) void;
    pub const setInsertSpacesInsteadOfTabs = gtk_source_view_set_insert_spaces_instead_of_tabs;

    /// Sets attributes and priority for the `category`.
    extern fn gtk_source_view_set_mark_attributes(p_view: *View, p_category: [*:0]const u8, p_attributes: *gtksource.MarkAttributes, p_priority: c_int) void;
    pub const setMarkAttributes = gtk_source_view_set_mark_attributes;

    /// Sets the position of the right margin in the given `view`.
    extern fn gtk_source_view_set_right_margin_position(p_view: *View, p_pos: c_uint) void;
    pub const setRightMarginPosition = gtk_source_view_set_right_margin_position;

    /// If `TRUE` line marks will be displayed beside the text.
    extern fn gtk_source_view_set_show_line_marks(p_view: *View, p_show: c_int) void;
    pub const setShowLineMarks = gtk_source_view_set_show_line_marks;

    /// If `TRUE` line numbers will be displayed beside the text.
    extern fn gtk_source_view_set_show_line_numbers(p_view: *View, p_show: c_int) void;
    pub const setShowLineNumbers = gtk_source_view_set_show_line_numbers;

    /// If `TRUE` a right margin is displayed.
    extern fn gtk_source_view_set_show_right_margin(p_view: *View, p_show: c_int) void;
    pub const setShowRightMargin = gtk_source_view_set_show_right_margin;

    /// When set to `TRUE`, pressing the Backspace key will try to delete spaces
    /// up to the previous tab stop.
    extern fn gtk_source_view_set_smart_backspace(p_view: *View, p_smart_backspace: c_int) void;
    pub const setSmartBackspace = gtk_source_view_set_smart_backspace;

    /// Set the desired movement of the cursor when HOME and END keys
    /// are pressed.
    extern fn gtk_source_view_set_smart_home_end(p_view: *View, p_smart_home_end: gtksource.SmartHomeEndType) void;
    pub const setSmartHomeEnd = gtk_source_view_set_smart_home_end;

    /// Sets the width of tabulation in characters.
    ///
    /// The `gtk.TextBuffer` still contains `\t` characters,
    /// but they can take a different visual width in a `View` widget.
    extern fn gtk_source_view_set_tab_width(p_view: *View, p_width: c_uint) void;
    pub const setTabWidth = gtk_source_view_set_tab_width;

    /// Removes one indentation level at the beginning of the
    /// specified lines.
    extern fn gtk_source_view_unindent_lines(p_view: *View, p_start: *gtk.TextIter, p_end: *gtk.TextIter) void;
    pub const unindentLines = gtk_source_view_unindent_lines;

    extern fn gtk_source_view_get_type() usize;
    pub const getGObjectType = gtk_source_view_get_type;

    extern fn g_object_ref(p_self: *gtksource.View) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.View) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *View, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Vim emulation.
///
/// The `GtkSourceVimIMContext` is a `gtk.IMContext` implementation that can
/// be used to provide Vim-like editing controls within a `View`.
///
/// The `GtkSourceViMIMContext` will process incoming `gdk.KeyEvent` as the
/// user types. It should be used in conjunction with a `gtk.EventControllerKey`.
///
/// Various features supported by `GtkSourceVimIMContext` include:
///
///  - Normal, Insert, Replace, Visual, and Visual Line modes
///  - Support for an integrated command bar and current command preview
///  - Search and replace
///  - Motions and Text Objects
///  - History replay
///  - Jumplists within the current file
///  - Registers including the system and primary clipboards
///  - Creation and motion to marks
///  - Some commonly used Vim commands
///
/// It is recommended that applications display the contents of
/// `VimIMContext.properties.command_bar_text` and
/// `VimIMContext.properties.command_text` to the user as they represent the
/// command-bar and current command preview found in Vim.
///
/// `GtkSourceVimIMContext` attempts to work with additional `gtk.IMContext`
/// implementations such as IBus by querying the `gtk.TextView` before processing
/// the command in states which support it (notably Insert and Replace modes).
///
/// ```c
/// GtkEventController *key;
/// GtkIMContext *im_context;
/// GtkWidget *view;
///
/// view = gtk_source_view_new ();
/// im_context = gtk_source_vim_im_context_new ();
/// key = gtk_event_controller_key_new ();
///
/// gtk_event_controller_key_set_im_context (GTK_EVENT_CONTROLLER_KEY (key), im_context);
/// gtk_event_controller_set_propagation_phase (key, GTK_PHASE_CAPTURE);
/// gtk_widget_add_controller (view, key);
/// gtk_im_context_set_client_widget (im_context, view);
///
/// g_object_bind_property (im_context, "command-bar-text", command_bar_label, "label", 0);
/// g_object_bind_property (im_context, "command-text", command_label, "label", 0);
/// ```
pub const VimIMContext = opaque {
    pub const Parent = gtk.IMContext;
    pub const Implements = [_]type{};
    pub const Class = gtksource.VimIMContextClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const command_bar_text = struct {
            pub const name = "command-bar-text";

            pub const Type = ?[*:0]u8;
        };

        pub const command_text = struct {
            pub const name = "command-text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Requests the application open the file found at `path`.
        ///
        /// If `path` is `NULL`, then the current file should be reloaded from storage.
        ///
        /// This may be executed in relation to the user running the
        /// `:edit` or `:e` commands.
        pub const edit = struct {
            pub const name = "edit";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_view: *gtksource.View, p_path: ?[*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(VimIMContext, p_instance))),
                    gobject.signalLookup("edit", VimIMContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The signal is emitted when a command should be
        /// executed. This might be something like `:wq` or `:e <path>`.
        ///
        /// If the application chooses to implement this, it should return
        /// `TRUE` from this signal to indicate the command has been handled.
        pub const execute_command = struct {
            pub const name = "execute-command";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_command: [*:0]u8, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(VimIMContext, p_instance))),
                    gobject.signalLookup("execute-command", VimIMContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Requests that the application format the text between
        /// `begin` and `end`.
        pub const format_text = struct {
            pub const name = "format-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_begin: *gtk.TextIter, p_end: *gtk.TextIter, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(VimIMContext, p_instance))),
                    gobject.signalLookup("format-text", VimIMContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Requests the application save the file.
        ///
        /// If a filename was provided, it will be available to the signal handler as `path`.
        /// This may be executed in relation to the user running the `:write` or `:w` commands.
        pub const write = struct {
            pub const name = "write";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_view: *gtksource.View, p_path: ?[*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(VimIMContext, p_instance))),
                    gobject.signalLookup("write", VimIMContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    extern fn gtk_source_vim_im_context_new() *gtksource.VimIMContext;
    pub const new = gtk_source_vim_im_context_new;

    /// Executes `command` as if it was typed into the command bar by the
    /// user except that this does not emit the
    /// `VimIMContext.signals.execute_command` signal.
    extern fn gtk_source_vim_im_context_execute_command(p_self: *VimIMContext, p_command: [*:0]const u8) void;
    pub const executeCommand = gtk_source_vim_im_context_execute_command;

    /// Gets the current command-bar text as it is entered by the user.
    extern fn gtk_source_vim_im_context_get_command_bar_text(p_self: *VimIMContext) [*:0]const u8;
    pub const getCommandBarText = gtk_source_vim_im_context_get_command_bar_text;

    /// Gets the current command text as it is entered by the user.
    extern fn gtk_source_vim_im_context_get_command_text(p_self: *VimIMContext) [*:0]const u8;
    pub const getCommandText = gtk_source_vim_im_context_get_command_text;

    extern fn gtk_source_vim_im_context_get_type() usize;
    pub const getGObjectType = gtk_source_vim_im_context_get_type;

    extern fn g_object_ref(p_self: *gtksource.VimIMContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.VimIMContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *VimIMContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interface for completion proposals.
///
/// This interface is used to denote that an object is capable of being
/// a completion proposal for `Completion`.
///
/// Currently, no method or functions are required but additional methods
/// may be added in the future. Proposals created by
/// `gtksource.CompletionProvider` can use `gobject.IMPLEMENTINTERFACE` to
/// implement this with `NULL` for the interface init function.
pub const CompletionProposal = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gtksource.CompletionProposalInterface;
    pub const virtual_methods = struct {
        /// Gets the typed-text for the proposal, if supported by the implementation.
        ///
        /// Implementing this virtual-function is optional, but can be useful to allow
        /// external tooling to compare results.
        pub const get_typed_text = struct {
            pub fn call(p_class: anytype, p_proposal: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) ?[*:0]u8 {
                return gobject.ext.as(CompletionProposal.Iface, p_class).f_get_typed_text.?(gobject.ext.as(CompletionProposal, p_proposal));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_proposal: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(CompletionProposal.Iface, p_class).f_get_typed_text = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the typed-text for the proposal, if supported by the implementation.
    ///
    /// Implementing this virtual-function is optional, but can be useful to allow
    /// external tooling to compare results.
    extern fn gtk_source_completion_proposal_get_typed_text(p_proposal: *CompletionProposal) ?[*:0]u8;
    pub const getTypedText = gtk_source_completion_proposal_get_typed_text;

    extern fn gtk_source_completion_proposal_get_type() usize;
    pub const getGObjectType = gtk_source_completion_proposal_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionProposal) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionProposal) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionProposal, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Completion provider interface.
///
/// You must implement this interface to provide proposals to `Completion`.
///
/// In most cases, implementations of this interface will want to use
/// `CompletionProvider.virtual_methods.populate_async` to asynchronously populate the results
/// to avoid blocking the main loop.
pub const CompletionProvider = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gtksource.CompletionProviderInterface;
    pub const virtual_methods = struct {
        /// This function requests `proposal` to be activated by the
        /// `gtksource.CompletionProvider`.
        ///
        /// What the provider does to activate the proposal is specific to that
        /// provider. Many providers may choose to insert a `gtksource.Snippet` with
        /// edit points the user may cycle through.
        ///
        /// See also: `Snippet`, `SnippetChunk`, `View.pushSnippet`
        pub const activate = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) void {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_activate.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_proposal);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) callconv(.C) void) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// This function requests that the `gtksource.CompletionProvider` prepares
        /// `cell` to display the contents of `proposal`.
        ///
        /// Based on `cells` column type, you may want to display different information.
        ///
        /// This allows for columns of information among completion proposals
        /// resulting in better alignment of similar content (icons, return types,
        /// method names, and parameter lists).
        pub const display = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_cell: *gtksource.CompletionCell) void {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_display.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_proposal, p_cell);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_cell: *gtksource.CompletionCell) callconv(.C) void) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_display = @ptrCast(p_implementation);
            }
        };

        /// This function should return the priority of `self` in `context`.
        ///
        /// The priority is used to sort groups of completion proposals by
        /// provider so that higher priority providers results are shown
        /// above lower priority providers.
        ///
        /// Higher value indicates higher priority.
        pub const get_priority = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext) c_int {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_get_priority.?(gobject.ext.as(CompletionProvider, p_self), p_context);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext) callconv(.C) c_int) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_get_priority = @ptrCast(p_implementation);
            }
        };

        /// Gets the title of the completion provider, if any.
        ///
        /// Currently, titles are not displayed in the completion results, but may be
        /// at some point in the future when non-`NULL`.
        pub const get_title = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) ?[*:0]u8 {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_get_title.?(gobject.ext.as(CompletionProvider, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_get_title = @ptrCast(p_implementation);
            }
        };

        /// This function is used to determine if a character inserted into the text
        /// editor should cause a new completion request to be triggered.
        ///
        /// An example would be period '.' which might indicate that the user wants
        /// to complete method or field names of an object.
        ///
        /// This method will only trigger when text is inserted into the `gtk.TextBuffer`
        /// while the completion list is visible and a proposal is selected. Incremental
        /// key-presses (like shift, control, or alt) are not triggerable.
        pub const is_trigger = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *const gtk.TextIter, p_ch: u32) c_int {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_is_trigger.?(gobject.ext.as(CompletionProvider, p_self), p_iter, p_ch);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_iter: *const gtk.TextIter, p_ch: u32) callconv(.C) c_int) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_is_trigger = @ptrCast(p_implementation);
            }
        };

        /// This function is used to determine if a key typed by the user should
        /// activate `proposal` (resulting in committing the text to the editor).
        ///
        /// This is useful when using languages where convention may lead to less
        /// typing by the user. One example may be the use of "." or "-" to expand
        /// a field access in the C programming language.
        pub const key_activates = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_keyval: c_uint, p_state: gdk.ModifierType) c_int {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_key_activates.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_proposal, p_keyval, p_state);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_keyval: c_uint, p_state: gdk.ModifierType) callconv(.C) c_int) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_key_activates = @ptrCast(p_implementation);
            }
        };

        /// Providers should return a list of alternates to `proposal` or `NULL` if
        /// there are no alternates available.
        ///
        /// This can be used by the completion view to allow the user to move laterally
        /// through similar proposals, such as overrides of methods by the same name.
        pub const list_alternates = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) ?*glib.PtrArray {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_list_alternates.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_proposal);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) callconv(.C) ?*glib.PtrArray) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_list_alternates = @ptrCast(p_implementation);
            }
        };

        pub const populate = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_error: ?*?*glib.Error) ?*gio.ListModel {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_populate.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_error: ?*?*glib.Error) callconv(.C) ?*gio.ListModel) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_populate = @ptrCast(p_implementation);
            }
        };

        /// Asynchronously requests that the provider populates the completion
        /// results for `context`.
        ///
        /// For providers that would like to populate a `gio.ListModel` while those
        /// results are displayed to the user,
        /// `CompletionContext.setProposalsForProvider` may be used
        /// to reduce latency until the user sees results.
        pub const populate_async = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_populate_async.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_populate_async = @ptrCast(p_implementation);
            }
        };

        /// Completes an asynchronous operation to populate a completion provider.
        pub const populate_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.ListModel {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_populate_finish.?(gobject.ext.as(CompletionProvider, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.ListModel) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_populate_finish = @ptrCast(p_implementation);
            }
        };

        /// This function can be used to filter results previously provided to
        /// the `CompletionContext` by the `gtksource.CompletionProvider`.
        ///
        /// This can happen as the user types additional text onto the word so
        /// that previously matched items may be removed from the list instead of
        /// generating new `gio.ListModel` of results.
        pub const refilter = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_model: *gio.ListModel) void {
                return gobject.ext.as(CompletionProvider.Iface, p_class).f_refilter.?(gobject.ext.as(CompletionProvider, p_self), p_context, p_model);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.CompletionContext, p_model: *gio.ListModel) callconv(.C) void) void {
                gobject.ext.as(CompletionProvider.Iface, p_class).f_refilter = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// This function requests `proposal` to be activated by the
    /// `gtksource.CompletionProvider`.
    ///
    /// What the provider does to activate the proposal is specific to that
    /// provider. Many providers may choose to insert a `gtksource.Snippet` with
    /// edit points the user may cycle through.
    ///
    /// See also: `Snippet`, `SnippetChunk`, `View.pushSnippet`
    extern fn gtk_source_completion_provider_activate(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) void;
    pub const activate = gtk_source_completion_provider_activate;

    /// This function requests that the `gtksource.CompletionProvider` prepares
    /// `cell` to display the contents of `proposal`.
    ///
    /// Based on `cells` column type, you may want to display different information.
    ///
    /// This allows for columns of information among completion proposals
    /// resulting in better alignment of similar content (icons, return types,
    /// method names, and parameter lists).
    extern fn gtk_source_completion_provider_display(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_cell: *gtksource.CompletionCell) void;
    pub const display = gtk_source_completion_provider_display;

    /// This function should return the priority of `self` in `context`.
    ///
    /// The priority is used to sort groups of completion proposals by
    /// provider so that higher priority providers results are shown
    /// above lower priority providers.
    ///
    /// Higher value indicates higher priority.
    extern fn gtk_source_completion_provider_get_priority(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext) c_int;
    pub const getPriority = gtk_source_completion_provider_get_priority;

    /// Gets the title of the completion provider, if any.
    ///
    /// Currently, titles are not displayed in the completion results, but may be
    /// at some point in the future when non-`NULL`.
    extern fn gtk_source_completion_provider_get_title(p_self: *CompletionProvider) ?[*:0]u8;
    pub const getTitle = gtk_source_completion_provider_get_title;

    /// This function is used to determine if a character inserted into the text
    /// editor should cause a new completion request to be triggered.
    ///
    /// An example would be period '.' which might indicate that the user wants
    /// to complete method or field names of an object.
    ///
    /// This method will only trigger when text is inserted into the `gtk.TextBuffer`
    /// while the completion list is visible and a proposal is selected. Incremental
    /// key-presses (like shift, control, or alt) are not triggerable.
    extern fn gtk_source_completion_provider_is_trigger(p_self: *CompletionProvider, p_iter: *const gtk.TextIter, p_ch: u32) c_int;
    pub const isTrigger = gtk_source_completion_provider_is_trigger;

    /// This function is used to determine if a key typed by the user should
    /// activate `proposal` (resulting in committing the text to the editor).
    ///
    /// This is useful when using languages where convention may lead to less
    /// typing by the user. One example may be the use of "." or "-" to expand
    /// a field access in the C programming language.
    extern fn gtk_source_completion_provider_key_activates(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_keyval: c_uint, p_state: gdk.ModifierType) c_int;
    pub const keyActivates = gtk_source_completion_provider_key_activates;

    /// Providers should return a list of alternates to `proposal` or `NULL` if
    /// there are no alternates available.
    ///
    /// This can be used by the completion view to allow the user to move laterally
    /// through similar proposals, such as overrides of methods by the same name.
    extern fn gtk_source_completion_provider_list_alternates(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) ?*glib.PtrArray;
    pub const listAlternates = gtk_source_completion_provider_list_alternates;

    /// Asynchronously requests that the provider populates the completion
    /// results for `context`.
    ///
    /// For providers that would like to populate a `gio.ListModel` while those
    /// results are displayed to the user,
    /// `CompletionContext.setProposalsForProvider` may be used
    /// to reduce latency until the user sees results.
    extern fn gtk_source_completion_provider_populate_async(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const populateAsync = gtk_source_completion_provider_populate_async;

    /// Completes an asynchronous operation to populate a completion provider.
    extern fn gtk_source_completion_provider_populate_finish(p_self: *CompletionProvider, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.ListModel;
    pub const populateFinish = gtk_source_completion_provider_populate_finish;

    /// This function can be used to filter results previously provided to
    /// the `CompletionContext` by the `gtksource.CompletionProvider`.
    ///
    /// This can happen as the user types additional text onto the word so
    /// that previously matched items may be removed from the list instead of
    /// generating new `gio.ListModel` of results.
    extern fn gtk_source_completion_provider_refilter(p_self: *CompletionProvider, p_context: *gtksource.CompletionContext, p_model: *gio.ListModel) void;
    pub const refilter = gtk_source_completion_provider_refilter;

    extern fn gtk_source_completion_provider_get_type() usize;
    pub const getGObjectType = gtk_source_completion_provider_get_type;

    extern fn g_object_ref(p_self: *gtksource.CompletionProvider) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.CompletionProvider) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CompletionProvider, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interface to populate interactive tooltips.
///
/// `GtkSourceHoverProvider` is an interface that should be implemented to extend
/// the contents of a `HoverDisplay`. This is typical in editors that
/// interact external tooling such as those utilizing Language Server Protocol.
///
/// If you can populate the `HoverDisplay` synchronously, use
/// `HoverProvider.virtual_methods.populate`. Otherwise, interface implementations that
/// may take additional time should use `HoverProvider.virtual_methods.populate_async`
/// to avoid blocking the main loop.
pub const HoverProvider = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gtksource.HoverProviderInterface;
    pub const virtual_methods = struct {
        pub const populate = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(HoverProvider.Iface, p_class).f_populate.?(gobject.ext.as(HoverProvider, p_self), p_context, p_display, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(HoverProvider.Iface, p_class).f_populate = @ptrCast(p_implementation);
            }
        };

        pub const populate_async = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(HoverProvider.Iface, p_class).f_populate_async.?(gobject.ext.as(HoverProvider, p_self), p_context, p_display, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(HoverProvider.Iface, p_class).f_populate_async = @ptrCast(p_implementation);
            }
        };

        pub const populate_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(HoverProvider.Iface, p_class).f_populate_finish.?(gobject.ext.as(HoverProvider, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(HoverProvider.Iface, p_class).f_populate_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_source_hover_provider_populate_async(p_self: *HoverProvider, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const populateAsync = gtk_source_hover_provider_populate_async;

    extern fn gtk_source_hover_provider_populate_finish(p_self: *HoverProvider, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const populateFinish = gtk_source_hover_provider_populate_finish;

    extern fn gtk_source_hover_provider_get_type() usize;
    pub const getGObjectType = gtk_source_hover_provider_get_type;

    extern fn g_object_ref(p_self: *gtksource.HoverProvider) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.HoverProvider) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HoverProvider, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Auto-indentation interface.
///
/// By default, `View` can auto-indent as you type when
/// `View.properties.auto_indent` is enabled. The indentation simply copies the
/// previous lines indentation.
///
/// This can be changed by implementing `GtkSourceIndenter` and setting the
/// `View.properties.indenter` property.
///
/// Implementors of this interface should implement both
/// `Indenter.virtual_methods.is_trigger` and `Indenter.virtual_methods.indent`.
///
/// `Indenter.virtual_methods.is_trigger` is called upon key-press to
/// determine of the key press should trigger an indentation.  The default
/// implementation of the interface checks to see if the key was
/// `gdk.KEY_Return` or `gdk.KEY_KP_Enter` without `GDK_SHIFT_MASK` set.
///
/// `Indenter.virtual_methods.indent` is called after text has been
/// inserted into `Buffer` when
/// `Indenter.virtual_methods.is_trigger` returned `TRUE`. The `gtk.TextIter`
/// is placed directly after the inserted character or characters.
///
/// It may be beneficial to move the insertion mark using
/// `gtk.TextBuffer.selectRange` depending on how the indenter changes
/// the indentation.
///
/// All changes are encapsulated within a single user action so that the
/// user may undo them using standard undo/redo accelerators.
pub const Indenter = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gtksource.IndenterInterface;
    pub const virtual_methods = struct {
        /// This function should be implemented to alter the indentation of text
        /// within the view.
        ///
        /// `view` is provided so that the indenter may retrieve settings such as indentation and tab widths.
        ///
        /// `iter` is the location where the indentation was requested. This typically
        /// is after having just inserted a newline (\n) character but can be other
        /// situations such as a manually requested indentation or reformatting.
        ///
        /// See `indenter.is_trigger` for how to trigger indentation on
        /// various characters inserted into the buffer.
        ///
        /// The implementor of this function is expected to keep `iter` valid across
        /// calls to the function and should contain the location of the insert mark
        /// after calling this function.
        ///
        /// The default implementation for this virtual function will copy the
        /// indentation of the previous line.
        pub const indent = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_view: *gtksource.View, p_iter: *gtk.TextIter) void {
                return gobject.ext.as(Indenter.Iface, p_class).f_indent.?(gobject.ext.as(Indenter, p_self), p_view, p_iter);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_view: *gtksource.View, p_iter: *gtk.TextIter) callconv(.C) void) void {
                gobject.ext.as(Indenter.Iface, p_class).f_indent = @ptrCast(p_implementation);
            }
        };

        /// This function is used to determine if a key pressed should cause the
        /// indenter to automatically indent.
        ///
        /// The default implementation of this virtual method will check to see
        /// if `keyval` is `gdk.KEY_Return` or `gdk.KEY_KP_Enter` and `state` does
        /// not have `GDK_SHIFT_MASK` set. This is to allow the user to avoid
        /// indentation when Shift+Return is pressed. Other indenters may want
        /// to copy this behavior to provide a consistent experience to users.
        pub const is_trigger = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_view: *gtksource.View, p_location: *const gtk.TextIter, p_state: gdk.ModifierType, p_keyval: c_uint) c_int {
                return gobject.ext.as(Indenter.Iface, p_class).f_is_trigger.?(gobject.ext.as(Indenter, p_self), p_view, p_location, p_state, p_keyval);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_view: *gtksource.View, p_location: *const gtk.TextIter, p_state: gdk.ModifierType, p_keyval: c_uint) callconv(.C) c_int) void {
                gobject.ext.as(Indenter.Iface, p_class).f_is_trigger = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// This function should be implemented to alter the indentation of text
    /// within the view.
    ///
    /// `view` is provided so that the indenter may retrieve settings such as indentation and tab widths.
    ///
    /// `iter` is the location where the indentation was requested. This typically
    /// is after having just inserted a newline (\n) character but can be other
    /// situations such as a manually requested indentation or reformatting.
    ///
    /// See `indenter.is_trigger` for how to trigger indentation on
    /// various characters inserted into the buffer.
    ///
    /// The implementor of this function is expected to keep `iter` valid across
    /// calls to the function and should contain the location of the insert mark
    /// after calling this function.
    ///
    /// The default implementation for this virtual function will copy the
    /// indentation of the previous line.
    extern fn gtk_source_indenter_indent(p_self: *Indenter, p_view: *gtksource.View, p_iter: *gtk.TextIter) void;
    pub const indent = gtk_source_indenter_indent;

    /// This function is used to determine if a key pressed should cause the
    /// indenter to automatically indent.
    ///
    /// The default implementation of this virtual method will check to see
    /// if `keyval` is `gdk.KEY_Return` or `gdk.KEY_KP_Enter` and `state` does
    /// not have `GDK_SHIFT_MASK` set. This is to allow the user to avoid
    /// indentation when Shift+Return is pressed. Other indenters may want
    /// to copy this behavior to provide a consistent experience to users.
    extern fn gtk_source_indenter_is_trigger(p_self: *Indenter, p_view: *gtksource.View, p_location: *const gtk.TextIter, p_state: gdk.ModifierType, p_keyval: c_uint) c_int;
    pub const isTrigger = gtk_source_indenter_is_trigger;

    extern fn gtk_source_indenter_get_type() usize;
    pub const getGObjectType = gtk_source_indenter_get_type;

    extern fn g_object_ref(p_self: *gtksource.Indenter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.Indenter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Indenter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interface implemented by widgets for choosing style schemes.
///
/// `GtkSourceStyleSchemeChooser` is an interface that is implemented by widgets
/// for choosing style schemes.
///
/// In GtkSourceView, the main widgets that implement this interface are
/// `StyleSchemeChooserWidget` and `StyleSchemeChooserButton`.
pub const StyleSchemeChooser = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gtksource.StyleSchemeChooserInterface;
    pub const virtual_methods = struct {
        /// Gets the currently-selected scheme.
        pub const get_style_scheme = struct {
            pub fn call(p_class: anytype, p_chooser: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gtksource.StyleScheme {
                return gobject.ext.as(StyleSchemeChooser.Iface, p_class).f_get_style_scheme.?(gobject.ext.as(StyleSchemeChooser, p_chooser));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_chooser: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gtksource.StyleScheme) void {
                gobject.ext.as(StyleSchemeChooser.Iface, p_class).f_get_style_scheme = @ptrCast(p_implementation);
            }
        };

        /// Sets the scheme.
        pub const set_style_scheme = struct {
            pub fn call(p_class: anytype, p_chooser: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_scheme: *gtksource.StyleScheme) void {
                return gobject.ext.as(StyleSchemeChooser.Iface, p_class).f_set_style_scheme.?(gobject.ext.as(StyleSchemeChooser, p_chooser), p_scheme);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_chooser: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_scheme: *gtksource.StyleScheme) callconv(.C) void) void {
                gobject.ext.as(StyleSchemeChooser.Iface, p_class).f_set_style_scheme = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Contains the currently selected style scheme.
        ///
        /// The property can be set to change the current selection programmatically.
        pub const style_scheme = struct {
            pub const name = "style-scheme";

            pub const Type = ?*gtksource.StyleScheme;
        };
    };

    pub const signals = struct {};

    /// Gets the currently-selected scheme.
    extern fn gtk_source_style_scheme_chooser_get_style_scheme(p_chooser: *StyleSchemeChooser) *gtksource.StyleScheme;
    pub const getStyleScheme = gtk_source_style_scheme_chooser_get_style_scheme;

    /// Sets the scheme.
    extern fn gtk_source_style_scheme_chooser_set_style_scheme(p_chooser: *StyleSchemeChooser, p_scheme: *gtksource.StyleScheme) void;
    pub const setStyleScheme = gtk_source_style_scheme_chooser_set_style_scheme;

    extern fn gtk_source_style_scheme_chooser_get_type() usize;
    pub const getGObjectType = gtk_source_style_scheme_chooser_get_type;

    extern fn g_object_ref(p_self: *gtksource.StyleSchemeChooser) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtksource.StyleSchemeChooser) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *StyleSchemeChooser, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BufferClass = extern struct {
    pub const Instance = gtksource.Buffer;

    f_parent_class: gtk.TextBufferClass,
    f_bracket_matched: ?*const fn (p_buffer: *gtksource.Buffer, p_iter: *gtk.TextIter, p_state: gtksource.BracketMatchType) callconv(.C) void,
    f__reserved: [20]*anyopaque,

    pub fn as(p_instance: *BufferClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionCellClass = extern struct {
    pub const Instance = gtksource.CompletionCell;

    f_parent_class: gtk.WidgetClass,

    pub fn as(p_instance: *CompletionCellClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionClass = extern struct {
    pub const Instance = gtksource.Completion;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CompletionClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionContextClass = extern struct {
    pub const Instance = gtksource.CompletionContext;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CompletionContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionProposalInterface = extern struct {
    pub const Instance = gtksource.CompletionProposal;

    f_parent_iface: gobject.TypeInterface,
    f_get_typed_text: ?*const fn (p_proposal: *gtksource.CompletionProposal) callconv(.C) ?[*:0]u8,

    pub fn as(p_instance: *CompletionProposalInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionProviderInterface = extern struct {
    pub const Instance = gtksource.CompletionProvider;

    f_parent_iface: gobject.TypeInterface,
    f_get_title: ?*const fn (p_self: *gtksource.CompletionProvider) callconv(.C) ?[*:0]u8,
    f_get_priority: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext) callconv(.C) c_int,
    f_is_trigger: ?*const fn (p_self: *gtksource.CompletionProvider, p_iter: *const gtk.TextIter, p_ch: u32) callconv(.C) c_int,
    f_key_activates: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_keyval: c_uint, p_state: gdk.ModifierType) callconv(.C) c_int,
    f_populate: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_error: ?*?*glib.Error) callconv(.C) ?*gio.ListModel,
    f_populate_async: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    f_populate_finish: ?*const fn (p_self: *gtksource.CompletionProvider, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.ListModel,
    f_refilter: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_model: *gio.ListModel) callconv(.C) void,
    f_display: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal, p_cell: *gtksource.CompletionCell) callconv(.C) void,
    f_activate: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) callconv(.C) void,
    f_list_alternates: ?*const fn (p_self: *gtksource.CompletionProvider, p_context: *gtksource.CompletionContext, p_proposal: *gtksource.CompletionProposal) callconv(.C) ?*glib.PtrArray,

    pub fn as(p_instance: *CompletionProviderInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionSnippetsClass = extern struct {
    pub const Instance = gtksource.CompletionSnippets;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *CompletionSnippetsClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CompletionWordsClass = extern struct {
    pub const Instance = gtksource.CompletionWords;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *CompletionWordsClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Character encoding.
///
/// The `gtksource.Encoding` boxed type represents a character encoding. It is used
/// for example by `gtksource.File`. Note that the text in GTK widgets is always
/// encoded in UTF-8.
pub const Encoding = opaque {
    /// Gets all encodings.
    extern fn gtk_source_encoding_get_all() *glib.SList;
    pub const getAll = gtk_source_encoding_get_all;

    /// Gets the `gtksource.Encoding` for the current locale.
    ///
    /// See also `glib.getCharset`.
    extern fn gtk_source_encoding_get_current() *const gtksource.Encoding;
    pub const getCurrent = gtk_source_encoding_get_current;

    /// Gets the list of default candidate encodings to try when loading a file.
    ///
    /// See `FileLoader.setCandidateEncodings`.
    ///
    /// This function returns a different list depending on the current locale (i.e.
    /// language, country and default encoding). The UTF-8 encoding and the current
    /// locale encoding are guaranteed to be present in the returned list.
    extern fn gtk_source_encoding_get_default_candidates() *glib.SList;
    pub const getDefaultCandidates = gtk_source_encoding_get_default_candidates;

    /// Gets a `gtksource.Encoding` from a character set such as "UTF-8" or
    /// "ISO-8859-1".
    extern fn gtk_source_encoding_get_from_charset(p_charset: [*:0]const u8) ?*const gtksource.Encoding;
    pub const getFromCharset = gtk_source_encoding_get_from_charset;

    extern fn gtk_source_encoding_get_utf8() *const gtksource.Encoding;
    pub const getUtf8 = gtk_source_encoding_get_utf8;

    /// Used by language bindings.
    extern fn gtk_source_encoding_copy(p_enc: *const Encoding) *gtksource.Encoding;
    pub const copy = gtk_source_encoding_copy;

    /// Used by language bindings.
    extern fn gtk_source_encoding_free(p_enc: *Encoding) void;
    pub const free = gtk_source_encoding_free;

    /// Gets the character set of the `gtksource.Encoding`, such as "UTF-8" or
    /// "ISO-8859-1".
    extern fn gtk_source_encoding_get_charset(p_enc: *const Encoding) [*:0]const u8;
    pub const getCharset = gtk_source_encoding_get_charset;

    /// Gets the name of the `gtksource.Encoding` such as "Unicode" or "Western".
    extern fn gtk_source_encoding_get_name(p_enc: *const Encoding) [*:0]const u8;
    pub const getName = gtk_source_encoding_get_name;

    extern fn gtk_source_encoding_to_string(p_enc: *const Encoding) [*:0]u8;
    pub const toString = gtk_source_encoding_to_string;

    extern fn gtk_source_encoding_get_type() usize;
    pub const getGObjectType = gtk_source_encoding_get_type;
};

pub const FileClass = extern struct {
    pub const Instance = gtksource.File;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *FileClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FileLoaderClass = extern struct {
    pub const Instance = gtksource.FileLoader;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *FileLoaderClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FileSaverClass = extern struct {
    pub const Instance = gtksource.FileSaver;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *FileSaverClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const GutterClass = extern struct {
    pub const Instance = gtksource.Gutter;

    f_parent_class: gtk.WidgetClass,

    pub fn as(p_instance: *GutterClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const GutterLinesClass = extern struct {
    pub const Instance = gtksource.GutterLines;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *GutterLinesClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const GutterRendererClass = extern struct {
    pub const Instance = gtksource.GutterRenderer;

    f_parent_class: gtk.WidgetClass,
    f_query_data: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_lines: *gtksource.GutterLines, p_line: c_uint) callconv(.C) void,
    f_begin: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_lines: *gtksource.GutterLines) callconv(.C) void,
    f_snapshot_line: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_snapshot: *gtk.Snapshot, p_lines: *gtksource.GutterLines, p_line: c_uint) callconv(.C) void,
    f_end: ?*const fn (p_renderer: *gtksource.GutterRenderer) callconv(.C) void,
    f_change_view: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_old_view: ?*gtksource.View) callconv(.C) void,
    f_change_buffer: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_old_buffer: ?*gtksource.Buffer) callconv(.C) void,
    f_query_activatable: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle) callconv(.C) c_int,
    f_activate: ?*const fn (p_renderer: *gtksource.GutterRenderer, p_iter: *gtk.TextIter, p_area: *gdk.Rectangle, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) callconv(.C) void,
    f__reserved: [20]*anyopaque,

    pub fn as(p_instance: *GutterRendererClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const GutterRendererPixbufClass = extern struct {
    pub const Instance = gtksource.GutterRendererPixbuf;

    f_parent_class: gtksource.GutterRendererClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *GutterRendererPixbufClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const GutterRendererTextClass = extern struct {
    pub const Instance = gtksource.GutterRendererText;

    f_parent_class: gtksource.GutterRendererClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *GutterRendererTextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HoverClass = extern struct {
    pub const Instance = gtksource.Hover;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *HoverClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HoverContextClass = extern struct {
    pub const Instance = gtksource.HoverContext;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *HoverContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HoverDisplayClass = extern struct {
    pub const Instance = gtksource.HoverDisplay;

    f_parent_class: gtk.WidgetClass,

    pub fn as(p_instance: *HoverDisplayClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HoverProviderInterface = extern struct {
    pub const Instance = gtksource.HoverProvider;

    f_parent_iface: gobject.TypeInterface,
    f_populate: ?*const fn (p_self: *gtksource.HoverProvider, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_error: ?*?*glib.Error) callconv(.C) c_int,
    f_populate_async: ?*const fn (p_self: *gtksource.HoverProvider, p_context: *gtksource.HoverContext, p_display: *gtksource.HoverDisplay, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    f_populate_finish: ?*const fn (p_self: *gtksource.HoverProvider, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int,

    pub fn as(p_instance: *HoverProviderInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The virtual function table for `gtksource.Indenter`.
pub const IndenterInterface = extern struct {
    pub const Instance = gtksource.Indenter;

    f_parent_iface: gobject.TypeInterface,
    /// the virtual function pointer for `gtksource.Indenter.isTrigger`
    f_is_trigger: ?*const fn (p_self: *gtksource.Indenter, p_view: *gtksource.View, p_location: *const gtk.TextIter, p_state: gdk.ModifierType, p_keyval: c_uint) callconv(.C) c_int,
    /// the virtual function pointer for `gtksource.Indenter.indent`
    f_indent: ?*const fn (p_self: *gtksource.Indenter, p_view: *gtksource.View, p_iter: *gtk.TextIter) callconv(.C) void,

    pub fn as(p_instance: *IndenterInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const LanguageClass = extern struct {
    pub const Instance = gtksource.Language;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *LanguageClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const LanguageManagerClass = extern struct {
    pub const Instance = gtksource.LanguageManager;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *LanguageManagerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const MapClass = extern struct {
    pub const Instance = gtksource.Map;

    f_parent_class: gtksource.ViewClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *MapClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const MarkAttributesClass = extern struct {
    pub const Instance = gtksource.MarkAttributes;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *MarkAttributesClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const MarkClass = extern struct {
    pub const Instance = gtksource.Mark;

    f_parent_class: gtk.TextMarkClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *MarkClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PrintCompositorClass = extern struct {
    pub const Instance = gtksource.PrintCompositor;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *PrintCompositorClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const RegionClass = extern struct {
    pub const Instance = gtksource.Region;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *RegionClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An opaque datatype.
///
/// Ignore all its fields and initialize the iter with `Region.getStartRegionIter`.
pub const RegionIter = extern struct {
    f_dummy1: ?*anyopaque,
    f_dummy2: u32,
    f_dummy3: ?*anyopaque,

    /// Gets the subregion at this iterator.
    extern fn gtk_source_region_iter_get_subregion(p_iter: *RegionIter, p_start: ?*gtk.TextIter, p_end: ?*gtk.TextIter) c_int;
    pub const getSubregion = gtk_source_region_iter_get_subregion;

    extern fn gtk_source_region_iter_is_end(p_iter: *RegionIter) c_int;
    pub const isEnd = gtk_source_region_iter_is_end;

    /// Moves `iter` to the next subregion.
    extern fn gtk_source_region_iter_next(p_iter: *RegionIter) c_int;
    pub const next = gtk_source_region_iter_next;
};

pub const SearchContextClass = extern struct {
    pub const Instance = gtksource.SearchContext;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SearchContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SearchSettingsClass = extern struct {
    pub const Instance = gtksource.SearchSettings;

    f_parent_class: gobject.ObjectClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *SearchSettingsClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SnippetChunkClass = extern struct {
    pub const Instance = gtksource.SnippetChunk;

    f_parent_class: gobject.InitiallyUnownedClass,

    pub fn as(p_instance: *SnippetChunkClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SnippetClass = extern struct {
    pub const Instance = gtksource.Snippet;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SnippetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SnippetContextClass = extern struct {
    pub const Instance = gtksource.SnippetContext;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SnippetContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SnippetManagerClass = extern struct {
    pub const Instance = gtksource.SnippetManager;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SnippetManagerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SpaceDrawerClass = extern struct {
    pub const Instance = gtksource.SpaceDrawer;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SpaceDrawerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleClass = extern struct {
    pub const Instance = gtksource.Style;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *StyleClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemeChooserButtonClass = extern struct {
    pub const Instance = gtksource.StyleSchemeChooserButton;

    f_parent: gtk.ButtonClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *StyleSchemeChooserButtonClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemeChooserInterface = extern struct {
    pub const Instance = gtksource.StyleSchemeChooser;

    f_base_interface: gobject.TypeInterface,
    f_get_style_scheme: ?*const fn (p_chooser: *gtksource.StyleSchemeChooser) callconv(.C) *gtksource.StyleScheme,
    f_set_style_scheme: ?*const fn (p_chooser: *gtksource.StyleSchemeChooser, p_scheme: *gtksource.StyleScheme) callconv(.C) void,
    f__reserved: [12]*anyopaque,

    pub fn as(p_instance: *StyleSchemeChooserInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemeChooserWidgetClass = extern struct {
    pub const Instance = gtksource.StyleSchemeChooserWidget;

    f_parent: gtk.WidgetClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *StyleSchemeChooserWidgetClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemeClass = extern struct {
    pub const Instance = gtksource.StyleScheme;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *StyleSchemeClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemeManagerClass = extern struct {
    pub const Instance = gtksource.StyleSchemeManager;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *StyleSchemeManagerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const StyleSchemePreviewClass = extern struct {
    pub const Instance = gtksource.StyleSchemePreview;

    f_parent_class: gtk.WidgetClass,

    pub fn as(p_instance: *StyleSchemePreviewClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const TagClass = extern struct {
    pub const Instance = gtksource.Tag;

    f_parent_class: gtk.TextTagClass,
    f__reserved: [10]*anyopaque,

    pub fn as(p_instance: *TagClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ViewClass = extern struct {
    pub const Instance = gtksource.View;

    f_parent_class: gtk.TextViewClass,
    f_line_mark_activated: ?*const fn (p_view: *gtksource.View, p_iter: *const gtk.TextIter, p_button: c_uint, p_state: gdk.ModifierType, p_n_presses: c_int) callconv(.C) void,
    f_show_completion: ?*const fn (p_view: *gtksource.View) callconv(.C) void,
    f_move_lines: ?*const fn (p_view: *gtksource.View, p_down: c_int) callconv(.C) void,
    f_move_words: ?*const fn (p_view: *gtksource.View, p_step: c_int) callconv(.C) void,
    f_push_snippet: ?*const fn (p_view: *gtksource.View, p_snippet: *gtksource.Snippet, p_location: ?*gtk.TextIter) callconv(.C) void,
    f__reserved: [20]*anyopaque,

    pub fn as(p_instance: *ViewClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const VimIMContextClass = extern struct {
    pub const Instance = gtksource.VimIMContext;

    f_parent_class: gtk.IMContextClass,

    pub fn as(p_instance: *VimIMContextClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BackgroundPatternType = enum(c_int) {
    none = 0,
    grid = 1,
    _,

    extern fn gtk_source_background_pattern_type_get_type() usize;
    pub const getGObjectType = gtk_source_background_pattern_type_get_type;
};

pub const BracketMatchType = enum(c_int) {
    none = 0,
    out_of_range = 1,
    not_found = 2,
    found = 3,
    _,

    extern fn gtk_source_bracket_match_type_get_type() usize;
    pub const getGObjectType = gtk_source_bracket_match_type_get_type;
};

pub const ChangeCaseType = enum(c_int) {
    lower = 0,
    upper = 1,
    toggle = 2,
    title = 3,
    _,

    extern fn gtk_source_change_case_type_get_type() usize;
    pub const getGObjectType = gtk_source_change_case_type_get_type;
};

pub const CompletionActivation = enum(c_int) {
    none = 0,
    interactive = 1,
    user_requested = 2,
    _,

    extern fn gtk_source_completion_activation_get_type() usize;
    pub const getGObjectType = gtk_source_completion_activation_get_type;
};

pub const CompletionColumn = enum(c_int) {
    icon = 0,
    before = 1,
    typed_text = 2,
    after = 3,
    comment = 4,
    details = 5,
    _,

    extern fn gtk_source_completion_column_get_type() usize;
    pub const getGObjectType = gtk_source_completion_column_get_type;
};

pub const CompressionType = enum(c_int) {
    none = 0,
    gzip = 1,
    _,

    extern fn gtk_source_compression_type_get_type() usize;
    pub const getGObjectType = gtk_source_compression_type_get_type;
};

/// An error code used with the `GTK_SOURCE_FILE_LOADER_ERROR` domain.
pub const FileLoaderError = enum(c_int) {
    too_big = 0,
    encoding_auto_detection_failed = 1,
    conversion_fallback = 2,
    _,

    extern fn gtk_source_file_loader_error_quark() glib.Quark;
    pub const quark = gtk_source_file_loader_error_quark;

    extern fn gtk_source_file_loader_error_get_type() usize;
    pub const getGObjectType = gtk_source_file_loader_error_get_type;
};

/// An error code used with the `GTK_SOURCE_FILE_SAVER_ERROR` domain.
pub const FileSaverError = enum(c_int) {
    invalid_chars = 0,
    externally_modified = 1,
    _,

    extern fn gtk_source_file_saver_error_quark() glib.Quark;
    pub const quark = gtk_source_file_saver_error_quark;

    extern fn gtk_source_file_saver_error_get_type() usize;
    pub const getGObjectType = gtk_source_file_saver_error_get_type;
};

/// The alignment mode of the renderer, when a cell spans multiple lines (due to
/// text wrapping).
pub const GutterRendererAlignmentMode = enum(c_int) {
    cell = 0,
    first = 1,
    last = 2,
    _,

    extern fn gtk_source_gutter_renderer_alignment_mode_get_type() usize;
    pub const getGObjectType = gtk_source_gutter_renderer_alignment_mode_get_type;
};

pub const NewlineType = enum(c_int) {
    lf = 0,
    cr = 1,
    cr_lf = 2,
    _,

    extern fn gtk_source_newline_type_get_type() usize;
    pub const getGObjectType = gtk_source_newline_type_get_type;
};

pub const SmartHomeEndType = enum(c_int) {
    disabled = 0,
    before = 1,
    after = 2,
    always = 3,
    _,

    extern fn gtk_source_smart_home_end_type_get_type() usize;
    pub const getGObjectType = gtk_source_smart_home_end_type_get_type;
};

pub const ViewGutterPosition = enum(c_int) {
    lines = -30,
    marks = -20,
    _,

    extern fn gtk_source_view_gutter_position_get_type() usize;
    pub const getGObjectType = gtk_source_view_gutter_position_get_type;
};

/// Flags to define the behavior of a `FileSaverFlags`.
pub const FileSaverFlags = packed struct(c_uint) {
    ignore_invalid_chars: bool = false,
    ignore_modification_time: bool = false,
    create_backup: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: FileSaverFlags = @bitCast(@as(c_uint, 0));
    const flags_ignore_invalid_chars: FileSaverFlags = @bitCast(@as(c_uint, 1));
    const flags_ignore_modification_time: FileSaverFlags = @bitCast(@as(c_uint, 2));
    const flags_create_backup: FileSaverFlags = @bitCast(@as(c_uint, 4));
    extern fn gtk_source_file_saver_flags_get_type() usize;
    pub const getGObjectType = gtk_source_file_saver_flags_get_type;
};

pub const SortFlags = packed struct(c_uint) {
    case_sensitive: bool = false,
    reverse_order: bool = false,
    remove_duplicates: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SortFlags = @bitCast(@as(c_uint, 0));
    const flags_case_sensitive: SortFlags = @bitCast(@as(c_uint, 1));
    const flags_reverse_order: SortFlags = @bitCast(@as(c_uint, 2));
    const flags_remove_duplicates: SortFlags = @bitCast(@as(c_uint, 4));
    extern fn gtk_source_sort_flags_get_type() usize;
    pub const getGObjectType = gtk_source_sort_flags_get_type;
};

/// `gtksource.SpaceLocationFlags` contains flags for white space locations.
///
/// If a line contains only white spaces (no text), the white spaces match both
/// `GTK_SOURCE_SPACE_LOCATION_LEADING` and `GTK_SOURCE_SPACE_LOCATION_TRAILING`.
pub const SpaceLocationFlags = packed struct(c_uint) {
    leading: bool = false,
    inside_text: bool = false,
    trailing: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SpaceLocationFlags = @bitCast(@as(c_uint, 0));
    const flags_leading: SpaceLocationFlags = @bitCast(@as(c_uint, 1));
    const flags_inside_text: SpaceLocationFlags = @bitCast(@as(c_uint, 2));
    const flags_trailing: SpaceLocationFlags = @bitCast(@as(c_uint, 4));
    const flags_all: SpaceLocationFlags = @bitCast(@as(c_uint, 7));
    extern fn gtk_source_space_location_flags_get_type() usize;
    pub const getGObjectType = gtk_source_space_location_flags_get_type;
};

/// `gtksource.SpaceTypeFlags` contains flags for white space types.
pub const SpaceTypeFlags = packed struct(c_uint) {
    space: bool = false,
    tab: bool = false,
    newline: bool = false,
    nbsp: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SpaceTypeFlags = @bitCast(@as(c_uint, 0));
    const flags_space: SpaceTypeFlags = @bitCast(@as(c_uint, 1));
    const flags_tab: SpaceTypeFlags = @bitCast(@as(c_uint, 2));
    const flags_newline: SpaceTypeFlags = @bitCast(@as(c_uint, 4));
    const flags_nbsp: SpaceTypeFlags = @bitCast(@as(c_uint, 8));
    const flags_all: SpaceTypeFlags = @bitCast(@as(c_uint, 15));
    extern fn gtk_source_space_type_flags_get_type() usize;
    pub const getGObjectType = gtk_source_space_type_flags_get_type;
};

/// Like GTK_SOURCE_CHECK_VERSION, but the check for gtk_source_check_version is
/// at runtime instead of compile time. This is useful for compiling
/// against older versions of GtkSourceView, but using features from newer
/// versions.
extern fn gtk_source_check_version(p_major: c_uint, p_minor: c_uint, p_micro: c_uint) c_int;
pub const checkVersion = gtk_source_check_version;

/// Free the resources allocated by GtkSourceView. For example it unrefs the
/// singleton objects.
///
/// It is not mandatory to call this function, it's just to be friendlier to
/// memory debugging tools. This function is meant to be called at the end of
/// `main`. It can be called several times.
extern fn gtk_source_finalize() void;
pub const finalize = gtk_source_finalize;

/// Returns the major version number of the GtkSourceView library.
/// (e.g. in GtkSourceView version 3.20.0 this is 3.)
///
/// This function is in the library, so it represents the GtkSourceView library
/// your code is running against. Contrast with the `GTK_SOURCE_MAJOR_VERSION`
/// macro, which represents the major version of the GtkSourceView headers you
/// have included when compiling your code.
extern fn gtk_source_get_major_version() c_uint;
pub const getMajorVersion = gtk_source_get_major_version;

/// Returns the micro version number of the GtkSourceView library.
/// (e.g. in GtkSourceView version 3.20.0 this is 0.)
///
/// This function is in the library, so it represents the GtkSourceView library
/// your code is running against. Contrast with the `GTK_SOURCE_MICRO_VERSION`
/// macro, which represents the micro version of the GtkSourceView headers you
/// have included when compiling your code.
extern fn gtk_source_get_micro_version() c_uint;
pub const getMicroVersion = gtk_source_get_micro_version;

/// Returns the minor version number of the GtkSourceView library.
/// (e.g. in GtkSourceView version 3.20.0 this is 20.)
///
/// This function is in the library, so it represents the GtkSourceView library
/// your code is running against. Contrast with the `GTK_SOURCE_MINOR_VERSION`
/// macro, which represents the minor version of the GtkSourceView headers you
/// have included when compiling your code.
extern fn gtk_source_get_minor_version() c_uint;
pub const getMinorVersion = gtk_source_get_minor_version;

/// Initializes the GtkSourceView library (e.g. for the internationalization).
///
/// This function can be called several times, but is meant to be called at the
/// beginning of `main`, before any other GtkSourceView function call.
extern fn gtk_source_init() void;
pub const init = gtk_source_init;

/// Simplified version of `schedulerAddFull`.
extern fn gtk_source_scheduler_add(p_callback: gtksource.SchedulerCallback, p_user_data: ?*anyopaque) usize;
pub const schedulerAdd = gtk_source_scheduler_add;

/// Adds a new callback that will be executed as time permits on the main thread.
///
/// This is useful when you need to do a lot of background work but want to do
/// it incrementally.
///
/// `callback` will be provided a deadline that it should complete it's work by
/// (or near) and can be checked using `glib.getMonotonicTime` for comparison.
///
/// Use `schedulerRemove` to remove the handler.
extern fn gtk_source_scheduler_add_full(p_callback: gtksource.SchedulerCallback, p_user_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) usize;
pub const schedulerAddFull = gtk_source_scheduler_add_full;

/// Removes a scheduler callback previously registered with
/// `schedulerAdd` or `schedulerAddFull`.
extern fn gtk_source_scheduler_remove(p_handler_id: usize) void;
pub const schedulerRemove = gtk_source_scheduler_remove;

/// Use this function to escape the following characters: `\n`, `\r`, `\t` and `\`.
///
/// For a regular expression search, use `glib.regexEscapeString` instead.
///
/// One possible use case is to take the `gtk.TextBuffer`'s selection and put it in a
/// search entry. The selection can contain tabulations, newlines, etc. So it's
/// better to escape those special characters to better fit in the search entry.
///
/// See also: `utilsUnescapeSearchText`.
///
/// <warning>
/// Warning: the escape and unescape functions are not reciprocal! For example,
/// escape (unescape (\)) = \\. So avoid cycles such as: search entry -> unescape
/// -> search settings -> escape -> search entry. The original search entry text
/// may be modified.
/// </warning>
extern fn gtk_source_utils_escape_search_text(p_text: [*:0]const u8) [*:0]u8;
pub const utilsEscapeSearchText = gtk_source_utils_escape_search_text;

/// Use this function before `SearchSettings.setSearchText`, to
/// unescape the following sequences of characters: `\n`, `\r`, `\t` and `\\`.
/// The purpose is to easily write those characters in a search entry.
///
/// Note that unescaping the search text is not needed for regular expression
/// searches.
///
/// See also: `utilsEscapeSearchText`.
extern fn gtk_source_utils_unescape_search_text(p_text: [*:0]const u8) [*:0]u8;
pub const utilsUnescapeSearchText = gtk_source_utils_unescape_search_text;

/// Type definition for a function that will be called to create a
/// `gio.MountOperation`. This is useful for creating a `gtk.MountOperation`.
pub const MountOperationFactory = *const fn (p_file: *gtksource.File, p_userdata: ?*anyopaque) callconv(.C) *gio.MountOperation;

/// This function is called incrementally to process additional background work.
/// A deadline is provided which can be checked using `glib.getMonotonicTime` so
/// that additional work can be processed each frame.
///
/// This is useful for situations where you are incrementally performing
/// background work such as spell checking or semantic syntax highlighting.
pub const SchedulerCallback = *const fn (p_deadline: i64, p_user_data: ?*anyopaque) callconv(.C) c_int;

/// Like `gtksource.getMajorVersion`, but from the headers used at
/// application compile time, rather than from the library linked
/// against at application run time.
pub const MAJOR_VERSION = 5;
/// Like `gtksource.getMicroVersion`, but from the headers used at
/// application compile time, rather than from the library linked
/// against at application run time.
pub const MICRO_VERSION = 0;
/// Like `gtksource.getMinorVersion`, but from the headers used at
/// application compile time, rather than from the library linked
/// against at application run time.
pub const MINOR_VERSION = 14;
