/*-------------------------------------------------------------------------- 
  Filename        : sgdiag.c
  Abstract        : SCSI Generic Disk Diagnostics tool
  Operation System: Linux 2.2 or greater
  Author          : Andy Cress   <arcress@users.sourceforge.net>
  Copyright (c) Intel Corporation 2001-2002

  ----------- Description --------------------------------------------------
  sgdiag
Sequence of events for this utility:
  * List each device on the system with firmware versions.
  * User selects a device for examination (or automatic if using -a)
  * Verify that the disk is present and ready
  * Send the specified SCSI command to the devices selected
  Diagnostic function choices:
  c = Compose Command to send         r = Reset SCSI bus
  i = Special Ser# Inquiries          f = Format SCSI disk
  1 = Do bug 1 (sense_len = 18)       2 = Do bug 2 (INQ hang)
  3 = Do bug 3 (fmt w short timeout)  
  Enter Selection ('q' to quit) : 
sgdiag    [-a -e -x]   
Options:
  -a  	Automatically send command for all drives.
  -e  	Do not write to any files. Usually a log file is created
    	and written to.
  -x    Outputs extra debug messages
  ----------- Change History -----------------------------------------------
  05/18/00 v0.50 ARCress  created 
  06/15/01 v0.90 ARCress  added do_format option
  07/09/01 v0.91 ARCress  added timeout param to scsi_format()
  07/18/01 v0.92 ARCress  added set SG_DEBUG
  08/14/01 v0.93 ARCress  fixed -a (fauto) option
  08/31/01 v0.97 ARCress  added 's' and 'w' functions
  09/27/01 v0.98 ARCress  fixed sninquiry code for other functions.
  10/01/01 v0.99 ARCress  longer format timeout for 36G disks to 50 min
  10/30/01 v1.0  ARCress  cleanup includes & indent -kr 
  11/08/01 v1.0  ARCress  increased size of devstat2 from 80 to 120 for 
                          Seagate overrun with 18G & 36G disks.
  11/14/01 v1.0  ARCress  read the CDB bytes in hex in do_sendcdb
  05/08/02 v1.3  ARCress  tune sg_debug level down to 5 by default
  05/14/02 v1.4  ARCress  implemented do_senddiag function
  07/25/02 v1.5  ARCress  recover from sense errors in get_scsi_info
  08/15/02 v1.6  ARCress  move common routines to sgcommon
		          added more usage text
  09/03/02 v1.7  ARCress  streamline display for errors in get_scsi_info
  10/11/02 v1.8  ARCress  make the format timeout a parameter (-t)
  01/03/03 v1.9  ARCress  added stop_unit function
  01/20/03 v1.10 ARCress  added extra warning for option f & w.
  03/10/03 v1.11 ARCress  added display of Emulated devices (sgcommon)
  07/22/04 v1.12 ARCress  updated default fmt_timeout from 50min to 150min.
  05/11/05 v1.13 ARCress  included patch for larger devlist from Nate Dailey 
  11/30/07 v1.19 ARCress  fixed error with -f option 
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002, Intel Corporation
All rights reserved.
Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:
  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <time.h>
#include <ctype.h>
#include <fcntl.h>
#include <scsi/sg.h>
#include "sgsub.h"
#include "sgcommon.h"

#define EBUFF_LEN   80
#define MAX_ERRORS   4
#define TRY_RECOVER  1

/* External Function prototypes */
extern void setbuf(FILE * stream, char *buf);
extern int errno;
extern char fdebugsub;		/* used in sgsub.c */
extern FILE *dbgout;		/* used in sgsub.c */
extern char HeaderStr[1];	/* from sgcommon.c */

/* Global data definitions */
char *progver  = "1.66";		/* program version */
char *progname = "sgdiag";	/* program name */
char logfile[] = "/var/log/sgdiag.log";	/* log filename */
char model[20] = "";		/* default model string */
char fsetfile = 0;
char device[80] = "/dev/sda";	/* UNIX device name (w path) */
int  sdelay = 10;		/* num sec to delay before TUR */
FILE *fdlog = NULL;		/* log file descriptor */
FILE *fdmsg;			/* file descriptor for messages  */
char fauto = 0;			/* =1 to automatically do all disks */
char filesok = 1;		/* =1 if ok to write to files */
char fdebug = 0;		/* =1 for debug messages */
char finteractive = 0;		/* =1 for interactive mode */
char flogopen = 0;		/* =1 if log file is open */
int  do_numeric = 1;		/* NUMERIC_SCAN_DEF; */
char fonedev = 0;		/* =1 to automatically run one disk */
int  ionedev = 0;
uchar sense_buffer[80];		/* buffer for sense data */
char  output[132];		/* message output buffer */
char  output2[80];		/* extra message output buffer */
uchar devstat[80];		/* used for inquiry & device status */
uchar devstat2[120];		/* used for Seagate inquiry2, etc.  */
uchar disk_buffer[0x200];	/* 512 bytes, used for misc scsi data */
uchar modebuf[300];		/* ~240 bytes, used for mode pages */
int idev = 0;
int ndev = 0;
int sgdbglvl = 5;		/* set sg_debug_level to 5 or 10 while here */
int fmt_timeout = 900000;	/* format timeout, 900000 jitters = 150 min */
				/* figure max 1 min / GB */
DEVLIST_TYPE devlist[MAX_DEVLIST_DEVS];
char erase_msg[] = "\nWARNING: This will erase all data on the selected device.\nPress Ctrl-C now if you wish to abort.\n";

/* Local subroutines */
int do_sendcdb(int idx);
int do_reset(int idx, int type);
int do_format(int idx, int noglist);
int do_sninquiry(int idx);
int do_senddiag(int idx);
int beforegd(int idx, int *pnumrdy);

int main(int argc, char **argv)
{
    int i, c, idx;
    char response = 0;
    short sts, rc;
    ulong ltime1;
    char fname[64];
    char ebuff[EBUFF_LEN];
    int sg_fd;
    int flags;
    int res, k;
    int iautodev;
    int eacces_err = 0;
    int num_errors = 0;
    char fgetall = 0;
    char fdobug = 0;
    char func = 0;
    DEVFS_LIST devfs_components = {0,0,0,0,0};
    fdmsg = stdout;		/* set default fdmsg */
    
    response = 'd';  /* default, used for fauto,fonedev*/

    while ((c = getopt(argc, argv, "aefi:lnrst:xI?")) != EOF)
	switch (c) {
	case 'a':
	    fauto = 1;		/* automatically do all disks */
	    break;
	case 'I':
	    finteractive = 1;
	    break;
	case 'i':
	    fonedev = 1;
            ionedev = atoi(optarg);
	    break;
	case 'e':
	    filesok = 0;
	    break;
	case 'r':
	    response = 'q';     /* only read list, then quit */
	    break;
	case 's':
	    response = 's';     /* do start unit */
	    break;
	case 'f':
	    response = 'f';     /* do format */
	    break;
	case 'l':
	    fgetall = 1;	/* not used */
	    break;
	case 'n':
	    do_numeric = 0;
	    break;
        case 't':
            i = atoi(optarg);       /* format timeout in minutes */
	    if (i >= 10)     /* don't allow timeout < 10 min */
	       fmt_timeout = i * 6000; /* 100 jitters/sec, 60 sec/min */
            break;
	case 'x':
	    fdebug = 1;
	    break;
	case '?':
	default:
	    printf("Usage: %s [-aefnsxI -i N -t min]\n", progname);
	    printf("   -a  Automatically send diag command for all drives.\n");
	    printf("   -e  Do not write to any files. Usually a log file is\n");
	    printf("       created and written to. \n");
	    printf("   -f  do Format of the drive specified by -i\n");
	    printf("   -iN autorun for the disk at Index N\n");
	    printf("   -n  turn off Numeric device names, use alpha instead\n");
	    printf("   -s  do StartUnit of the drive specified by -i\n");
	    printf("   -t  Timeout for format, in minutes\n");
	    printf("   -x  Outputs eXtra debug messages\n");
	    printf("   -I  Interactive mode, prompt for input.\n");
	    exit(1);
	}
    /* only run this as superuser */
    i = geteuid();
    if (i > 1) {
	printf("Not superuser (%d)\n", i);
	exit(1);
    }
    if (fdebug) sgdbglvl = 10;
    else        sgdbglvl = 5;
    /* open log file */
    if (filesok)
	fdlog = fopen(logfile, "a+");
    if (fdlog == NULL) {
	flogopen = 0;
	filesok = 0;
	fdlog = stderr;
    } else {			/* logfile open successful */
	flogopen = 1;
	time((time_t *) & ltime1);
	fprintf(fdlog, "\n--- %s v%s log started at %s", progname, progver,
		ctime((long *) &ltime1));	// ctime outputs a '\n' at end
    }
    fdebugsub = fdebug;
    dbgout = fdlog;
    iautodev = 0;
    /* loop to display disk choices and get defects */
    while (1) {
	if ((iautodev == 0) || finteractive) {
	   /* display header */
	   printf("\n");
	   printf("                %s utility v%s for SCSI disks\n", progname,
	       progver);
	   printf("              ******************************************\n");
	   if (flogopen)
	       printf("Log file %s is open, debug=%d\n", logfile,fdebug);
	   printf(HeaderStr);
	}
	/* get SCSI Device Info */
	idev = 0;
	flags = O_RDWR;		/* could use OPEN_FLAG if read-only. */
	num_errors = 0;
	for (k = 0; (k < MAX_SG_DEVS) && (idev < MAX_DEVLIST_DEVS) &&
                    (num_errors < MAX_ERRORS); ++k) {
	    make_dev_name(fname, k, do_numeric, &devfs_components);
	    if (devfs_components.all_leaves_searched != 0) {
		devfs_components.all_leaves_searched=0;
		break;
	    }
	    sg_fd = open(fname, flags | O_NONBLOCK);
	    if (sg_fd < 0) {  
		if (EBUSY == errno) {
		    printf("%s: device busy (O_EXCL lock), skipping\n",
			   fname);
		    continue;
		} else if ((ENODEV == errno) || (ENOENT == errno) ||
			     (ENXIO == errno)) {
		    /* device is not present, flag error & go on to next */
		    ++num_errors;
		    continue;
		} else {   /* open failed with other error */
		    if (EACCES == errno)
			eacces_err = 1;
		    sprintf(ebuff, "Error opening %s ", fname);
		    perror(ebuff);
		    ++num_errors;
		    continue;
		}
	    }
	    if ((iautodev == 0) || finteractive) 
		sts = get_scsi_info(sg_fd, idev, fname,0);
	    else 
		sts = 0; 
	    if (sts) {   /* opened ok, but inquiry error */
		showlog( "device %s failed with sts = %d\n",
			fname, sts);
	    } 
	    {
		if (sg_fd > 0) {
		    res = close(sg_fd);
		    if (res < 0)
			fprintf(fdmsg, "Error closing %s, errno = %d\n",
				fname, errno);
		    else
			devlist[idev].sgfd = 0;	/* set in get_scsi_info() */
		}
		idev++;
	    }
	}			/*end loop */
	ndev = idev;
	if (ndev == 0 && errno == ENODEV) {	/* CONFIG_CHR_DEV_SG != y ? */
	    sprintf(output,
		    "Cant open any SCSI devices.\nMake sure CONFIG_CHR_DEV_SG is set in /usr/src/linux/.config\n");
	    showit(output);
	    quit(1);
	}
        else if (ndev == 0) {
            sprintf(output,"Cant open any sg SCSI devices, errno = %d\n",errno);
	    if (errno == ENOENT) 
	       strcat(output,"Try option -n to change /dev/sg* device names\n");
            showit(output);
        }
	if (finteractive) {
	    printf("\n");
	printf("c = Compose Command to send  1 = Do bug 1 (sense_len = 18)\n");
	printf("r = Reset SCSI bus           2 = Do bug 2 (INQ hang)\n");
	printf("i = Special Ser# Inquiries   3 = Do bug 3 (fmt w short timeout)\n");
	printf("f = Format SCSI disk         w = Wipe SCSI disk (fmt wo glist)\n");
	printf("s = Start Unit               d = Send Diagnostic self-test\n");
	printf("t = sTop Unit                \n");
	    printf("Enter Selection ('q' to quit) : ");
	    response = get_func();
	} else { /* if (fauto || fonedev) */
            /* response defaults to 'd' above */
            /* if -f, response='f' */
	}
	printf("response=%c\n", response);
        fdobug = 0;
	switch (response) {
	case 'Q':
	case 'q':
	    quit(0);
	    break;
	case 'C':
	case 'c':
	    func = 'c';
	    break;
	case 'D':
	case 'd':
	    func = 'd';
	    break;
	case 'R':
	case 'r':
	    func = 'r';
	    break;
	case 'I':
	case 'i':
	    func = 'i';
	    break;
	case 'F':
	case 'f':
	    func = 'f';
	    printf(erase_msg);
	    break;
	case 'W':
	case 'w':
	    func = 'w';
	    printf(erase_msg);
	    break;
	case 'S':
	case 's':
	    func = 's';
	    break;
	case 'T':
	case 't':
	    func = 't';
	    break;
	case '1':
	    func = '1';
	    fdobug = 1;
	    break;
	case '2':
	    func = '2';
	    fdobug = 2;
	    break;
	case '3':
	    func = '3';
	    fdobug = 3;
	    break;
	case '4':
	    func = '4';
	    fdobug = 4;
	    break;
	default:
	    printf("Invalid input [%c], enter c,d,f,i,r,s,t,w,1,2,3,q.\n",
		   response);
	    continue;
	}			/*end switch */
	if (fauto)
	    i = iautodev;	// get next dev
	else if (fonedev)
	    i = ionedev;	
	else if (finteractive)
	    i = get_idev();	/* get the device index that the user picks */
	else 
	    i = iautodev;	// get next dev
	if (i == 0xff) {
	    if (finteractive) 
		printf("Index out of range, try again.\n");
            else {  /* if (fauto || fonedev) * then none */
                printf("Device not found.\n");
                quit(1);        /* exit */
            }
	} else {		/*index ok */
	    int numrdy;
	    idx = i;		/* do first disk */
	    if (fauto)
		iautodev = idx;	/* save disk index */
	    if (func == 'r')	/* do just a SCSI bus reset */
		rc = do_reset(idx, 2);
	    else if (func == 'f')
		rc = do_format(idx, 0);	/* normal, with glist */
	    else if (func == 'w')
		rc = do_format(idx, 1);	/* without glist */
	    else if (func == 'i') {	/* special inquiry */
		rc = do_sninquiry(idx);
		func = 0;
	    } else if (func == 'd') {
		sprintf(output,"[%d] Send Diagnostic self-test ...\n",idx);
		showit(output);
		devlist[idx].sgfd = open(devlist[idx].fname, O_RDWR);
		if (devlist[idx].sgfd <= 0) {
		   printf("Cannot open %s, errno = %d\n", 
		          devlist[idx].fname, errno);
		} else {
		   rc = do_senddiag(idx);
		   if (rc) {
		     sprintf(output,"[%d] Send Diagnostic error %d\n",idx,rc);
		     showit(output);
		   } else {		/* good results */
		     sprintf(output,"[%d] Send Diagnostic successful\n",idx);
		     showit(output);
		   }
		   close(devlist[idx].sgfd);
		}
	    } else if (func == 's') {	/* start unit */
		char *fname;
		int sgfd;
		/* open the device */
		fname = devlist[idx].fname;
		printf("Issuing start_unit for %s\n", fname);
		sgfd = open(fname, O_RDWR);	/* open blocking */
		if (sgfd < 0) {
		    printf("%s: open error, errno = %d\n", fname, errno);
		    rc = errno;
		} else {	/* opened ok */
		    rc = start_unit(sgfd);
		    printf("start_unit complete, status = %d\n", rc);
		    closefd();
		}
	    } else if (func == 't') {	/* stop unit */
		char *fname;
		int sgfd;
		/* open the device */
		fname = devlist[idx].fname;
		printf("Issuing stop_unit for %s\n", fname);
		sgfd = open(fname, O_RDWR);	/* open blocking */
		if (sgfd < 0) {
		    printf("%s: open error, errno = %d\n", fname, errno);
		    rc = errno;
		} else {	/* opened ok */
		    rc = stop_unit(sgfd);
		    printf("stop_unit complete, status = %d\n", rc);
		    closefd();
		    quit(0);  /* exit now, so we don't try to start it again */
		}
	    } else if (fdobug > 0) {  /* func = 1, 2, 3 */
		printf("Do Bug %d ...\n", fdobug);
		if (fdobug == 1) {	/*try with sense len = 18 */
		    uchar cmdbuf[140];	/*out:HDR+6=42, in:HDR+96=132 */
		    uchar inqCDB[6] = { 0x12, 0, 0, 0, 96, 0 };
		    struct sg_hdr1 *hp1;
		    int rsplen;
		    rsplen = HDR + 2 + 96;
		    memset(cmdbuf, 0, HDR);	/* zeros out all defaults */
		    memcpy(cmdbuf + HDR + 2, inqCDB, 6);
		    hp1 = (struct sg_hdr1 *) cmdbuf;
		    hp1->reply_len = rsplen;
		    rc = beforegd(idx, &numrdy);
		    rc = send_scsicdb(devlist[idx].sgfd, cmdbuf, 44, cmdbuf,
				     rsplen);
		    printf("send_scsicdb returned, status = %d\n", rc);
		    closefd();
		}
		if (fdobug == 2) {	/*try inquiry len = 80 */
		    uchar cmdbuf[120];	/*out:HDR+6=42, in:HDR+80=116 */
		    uchar inqCDB[6] = { 0x12, 0, 0, 0, 80, 0 };
		    struct sg_header *hp2;
		    int rsplen;
		    rsplen = HDR + 80;
		    memset(cmdbuf, 0, HDR);	/* zeros out all defaults */
		    memcpy(cmdbuf + HDR, inqCDB, 6);
		    hp2 = (struct sg_header *) cmdbuf;
		    hp2->reply_len = rsplen;
		    rc = beforegd(idx, &numrdy);
		    rc =
			send_scsicdb(devlist[idx].sgfd, cmdbuf, 42, cmdbuf,
				     rsplen);
		    printf("send_scsicdb returned, status = %d\n", rc);
		    closefd();
		}
		if (fdobug == 3) {
		    char *fname;
		    int sgfd;
		    /* open the device */
		    fname = devlist[idx].fname;
		    sgfd = open(fname, O_RDWR);	/* open blocking */
		    if (sgfd < 0) {
			printf("%s: open error, errno = %d\n", fname,
			       errno);
			rc = errno;
		    } else {	/* opened ok */
			rc = scsi_format(sgfd, 0xA5, 6000, 0);	/* timeout too short */
			/* This will cause the device to be set offline. */
			closefd();
		    }
		}
		if (fdobug == 4) {
		}
	    } /*endif fdobug */
	    else {	/* func == 'c', build & send the scsi command */
		rc = beforegd(idx, &numrdy);
		if (numrdy == 0)
		    idx = ndev;	/* skip the loop if none ready */
		/* Send command for each matching disk */
		for (i = idx; i < ndev; i++) {
		    if (devlist[i].fdoit) {
			rc = do_sendcdb(i);
			printf("do_sendcdb returned, status = %d\n", rc);
			// if (fdebug && rc != 0) quit(rc);   
		    }
		    if (finteractive || fonedev)
			i = ndev;	/* only do one if interactive */
		    else if (fdebug)
			printf("next i = %d\n", i + 1);
		}		/*endfor each disk */
		closefd();
	    }			/*endif doit */
            if (fonedev) {	/* then done */
		quit(0);	/* normal exit, successful */
	    } else if (finteractive) {	/* interactive, go to menu again */
		do_pause();
	    } else {    /* if (fauto) or default */
	        iautodev++;
                if (iautodev >= ndev) quit(0);
		if (devlist[iautodev].fdoit == 0) continue;
	    }
	}			/*index ok */
    }				/*end while(1) */
}				/*end main() */

int do_reset(int idx, int type)
{
    int sts = 0;
    char *fname;
    int sgfd;
    char *tstr[4] = { "Nothing", "Device", "Bus", "Host" };
    /*
     * Type values for scsi_reset(fd, type):
     * 0:   SG_SCSI_RESET_NOTHING;  SCSI Reset Nothing (check status),
     * 1:   SG_SCSI_RESET_DEVICE;   SCSI Device Reset,
     * 2:   SG_SCSI_RESET_BUS;      SCSI Bus Reset ,
     * 3:   SG_SCSI_RESET_HOST;     SCSI Host Reset ,
     */
    /* open the device */
    fname = devlist[idx].fname;
    //  openflags = O_RDWR; /* | O_NONBLOCK; */
    sgfd = open(fname, O_RDWR);	/* open blocking */
    if (sgfd < 0) {
	printf("%s: open error, errno = %d\n", fname, errno);
	return (errno);
    }
    devlist[idx].sgfd = sgfd;
    sts = set_sg_debug(sgfd, sgdbglvl);
    if (sts) {
        sprintf(output, "[%d] cant set debug level %d, sts = %d\n",
                    idx, sgdbglvl, sts);    
	showit(output);
    }
    /* do the reset */
    sprintf(output, "[%d] SCSI %s reset initiated.\n", idx, tstr[type]);
    showit(output);
    sts = scsi_reset(sgfd, type);
    if (sts) {			/* error */
	sprintf(output, "[%d] Error %d in scsi_reset\n", idx, sts);
    } /*endif error */
    else {			/* good results */
	sprintf(output, "[%d] Reset complete\n", idx);
    }				/*endif good */
    showit(output);
    /* close the device */
    closefd();
    return (sts);
}				/*end do_reset() */

int do_sninquiry(int idx)
{
    char *fname;
    int sgfd;
    int sts;
    char rbuf[128];
    int rlen;
    uchar rginq[] = { 0x80 };	// , 0x83;
    uchar rgmode[] = { 0x80, 0xC7 };
    int i, j;
    unsigned char page;
    /* open the device */
    fname = devlist[idx].fname;
    sgfd = open(fname, O_RDWR);	/* open blocking */
    if (sgfd < 0) {
	printf("%s: open error, errno = %d\n", fname, errno);
	return (errno);
    }
    devlist[idx].sgfd = sgfd;
    sts = set_sg_debug(sgfd, sgdbglvl);
    if (sts) {
	sprintf(output, "[%d] cant set debug flag, sts = %d", idx, sts);
	showit(output);
    }
    rlen = 50;
    sts = scsi_inquiry(sgfd, rbuf, rlen);
    if (sts != 0)
	printf("scsi_inquiry error %d\n", sts);
    else {
	rbuf[rlen] = 0;
	printf("scsi_inquiry:	%s\n", &rbuf[8]);
    }
    for (i = 0; i < sizeof(rginq); i++) {
	j = rginq[i];
	rlen = 50;
	memset(rbuf, 0, rlen);
	sts = sn_inquiry(sgfd, j, rbuf, rlen);
	if (sts != 0)
	    printf("sn_inquiry/%x error %d\n", j, sts);
	else {
	    rlen = rbuf[3] + 4;
	    rbuf[rlen] = 0;
	    printf("sn_inquiry/%x:	%s\n", j, &rbuf[4]);
	    if (j == 0x83)
		dump_buf(stdout, rbuf, rlen, NULL,0);
	    else if (fdebug)
		dump_buf(stdout, rbuf, 8, NULL,0);
	}
    }
    for (i = 0; i < sizeof(rgmode); i++) {
	page = rgmode[i];
	memset(modebuf, 0, 36);
	sts = mode_sense(sgfd, page, modebuf);
	if (sts != 0)
	    printf("sn_modesense/%x error %d\n", page, sts);
	else {
	    int plen;
	    plen = modebuf[3] + 4;
	    printf("sn_modesense/%x data ", page);
	    dump_buf(stdout, modebuf, plen, NULL,0);	/* size = 0x0a + 4 + 8 + 2 = 24 */
	}
    }
    /* close the device */
    closefd();
    return (sts);
}				/*end do_sninquiry() */

int do_format(int idx, int noglist)
{
    int sts = 0;
    int patt;
    char *fname;
    int sgfd;
    ulong ltime, ltime1, ltime2;
    int k, a, q, rc, i;
    int timeout, maxsec;
    char *gtag;
    /* open the device */
    fname = devlist[idx].fname;
    sgfd = open(fname, O_RDWR);	/* open blocking */
    if (sgfd < 0) {
	printf("%s: open error, errno = %d\n", fname, errno);
	return (errno);
    }
    devlist[idx].sgfd = sgfd;
    sts = set_sg_debug(sgfd, sgdbglvl);
    if (sts) {
	sprintf(output, "[%d] cant set debug flag, sts = %d", idx, sts);
	showit(output);
    }
    if (noglist == 1)
	gtag = " without glist";
    else
	gtag = "";
    /* do the format */
    time(&ltime1);
    sprintf(output, "[%d] scsi_format initiated%s, time = %s",
	    idx, gtag, ctime(&ltime1));
    showit(output);
    patt = 0xF6;		/* SCSI format pattern (0xF6 or 0xA5) */
    timeout = fmt_timeout;	/* 100 jitters/sec, default = 300000 jitters */
    sts = scsi_format(sgfd, patt, timeout, noglist);
    if (sts) {			/* error */
	sprintf(output, "[%d] Error %d in scsi_format, ", idx, sts);
    } /*endif error */
    else {			/* good results */
	sprintf(output, "[%d] scsi_format complete, ", idx);
    }				/*endif good */
    showit(output);
    /* loop for format in progress sense error */
    sts = test_unit_ready(sgfd, 0);	/* try again, dont show errors */
    maxsec = (timeout / 100) / 5;	/* max time to wait (5 sec each) */
    for (i = 0; i < maxsec && sts != 0; i++) {
	if (sts == SCHECK_CND) {  /* check for certain key/asc/ascq values */
	    /* issue start_unit() if sense data indicates. */
	    rc = get_sense(sts, sense_buffer);
	    k = sense_buffer[2] & 0x0f;	/*Sense Key */
	    a = sense_buffer[12]; /*ASC*/ 
	    q = sense_buffer[13]; /*ASCQ*/ 
	    if (k == 2 && a == 4 && q == 4) {
		if (i == 0)
		    showit("sense = format in progress\n");
		/* 02-04-04 means format is in progress, so wait. */
		sleep(5);	/* wait 5 seconds */
		sts = test_unit_ready(sgfd, 0);	/* don't show errors. */
	    } /*endif 02-04-04 */
	    else
		sts = 0;
	} else
	    sts = 0;
    }
    time(&ltime2);
    ltime = ltime2 - ltime1;	// total time in seconds
    ltime1 = ltime / 3600;	// num hours
    ltime = ltime % 3600;	// remainder
    sprintf(output, "elapsed time %02ld:%02ld:%02ld\n",
	    ltime1, ltime / 60, ltime % 60);
    showit(output);

#ifdef TRY_RECOVER
    /* Handle the 06/29/02 errors */
    sts = scsi_inquiry(sgfd, devstat, 80);	/* do an inquiry */
    sts = test_unit_ready(sgfd, 0);	/* clear out 06/29/02 error, ignore */
    sts = test_unit_ready(sgfd, 1);	/* try again, show errors */
    /* Try to restart the disk, if needed. */
    if (sts == SCHECK_CND) {	/* check for certain key/asc/ascq values */
	/* issue start_unit() if sense data indicates. */
	rc = get_sense(sts, sense_buffer);
	k = sense_buffer[2] & 0x0f;	/*Sense Key */
	a = sense_buffer[12]; /*ASC*/ 
	q = sense_buffer[13]; /*ASCQ*/ 
	if (k == 2 && a == 4 && q == 2) {
	    /* 02-04-02 means it needs a start command, so issue it. */
	    sts = start_unit(sgfd);
	    if (fdebug) {
		/*DEBUG*/ printf("\nStart Unit: sts=%d, ", sts);
	    }
	}			/*endif 02-04-02 */
    } /*endif sts */
    else {
	k = 0;
	a = 0;
	q = 0;
    }
    if (sts != 0) {		/* still error, so try to reset */
	sts = scsi_reset(sgfd, 3);
	printf("SCSI Reset issued, sts = %d\n", sts);
	sts = test_unit_ready(sgfd, 0);	/* ignore 06/29/02 error */
    }
    sts = test_unit_ready(sgfd, 1);	/* try again, show errors */
#endif

    /* close the device */
    closefd();
    return (sts);
}				/*end do_format() */

int do_senddiag(int idx)
{
    int sts;
    uchar rbuf[128];
    uchar *cdb;
    int devfd;
    cdb = rbuf + HDR;
    cdb[0] = 0x1D;  /* send diagnostic */
    cdb[1] = 0x04;  /* self test on */
    cdb[2] = 0;
    cdb[3] = 0;
    cdb[4] = 0;
    cdb[5] = 0;
    devfd = devlist[idx].sgfd;
    if (devfd == 0) return -1;
    sts = send_scsicdb(devfd, rbuf, HDR+6, rbuf, HDR);
    return(sts);
}

int do_sendcdb(int idx)
{
    int sts;
    int devfd;
    int isave;
    uchar ch, dv;
    char abuf[80];
    char rbuf[512];
    int rlen, wlen, maxlen;
    int j, rcdl;
    char *bufp;
    uchar *cdb;
    int clen, cval, dlen, rlen1;
    isave = idx;
    rcdl = 0;
    rlen = sizeof(rbuf);
    wlen = sizeof(rbuf);
    cdb = rbuf + HDR;		/* use rbuf[HDR] stack buffer for cdb */
    if (finteractive) {		/* get user input */
	/* Get the CDB information from the user input */
	clen = get_ival("Enter the CDB length (6, 10, etc)",0);
	if (clen <= 0)
	    return (SUNKN_ERR);
	else if (clen < 6)
	    clen = 6;
	for (j = 0; j < clen; j++) {
	    sprintf(abuf, "Enter CDB byte %d", j);
	    cval = get_ival(abuf,1);
	    cdb[j] = cval;
	}
	cdb += clen;
	rlen1 = get_ival("Enter the Response Data length",0);
	dlen = get_ival("Enter the Output Data length",0);
	if (dlen < 0)
	    dlen = 0;
	if (dlen > (sizeof(rbuf) - HDR - clen))
	    dlen = sizeof(rbuf) - HDR - clen;
	for (j = 0; j < dlen; j++) {
	    sprintf(abuf, "Enter Data byte %d", j);
	    cval = get_ival(abuf,1);
	    cdb[j] = cval;
	}
    } else {		/* not user input, do log sense */
	clen = 10;
	cdb[0] = 0x4d;		/* LOG_SENSE command */
	cdb[1] = 0x11;
	cdb[2] = 0x22;
	cdb[3] = 0x33;
	cdb[4] = 0x44;
	cdb[5] = 0x55;
	cdb[6] = 0x66;
	cdb[7] = 0x77;
	cdb[8] = 0x88;
	cdb[9] = 0x99;
	rlen1 = 10;
	dlen = 0;
    }	
    wlen = HDR + clen + dlen;
    rlen = HDR + rlen1;
    maxlen = wlen;
    if (maxlen < rlen)
	maxlen = rlen;
    bufp = malloc(maxlen);
    if (bufp == NULL)
	bufp = disk_buffer;
    memset(bufp, 0xAA, maxlen);
    memcpy(bufp, rbuf, wlen);
    /* Send SCSI CDB for each disk */
    {
	rcdl = 0;
	if (finteractive || fonedev)
	    idx = isave;	/* only do selected disk */
	devfd = devlist[idx].sgfd;
	ch = devlist[idx].chn;
	dv = devlist[idx].tgt;
	if (devlist[idx].fdoit == 1) {
	    sprintf(output, "Sending command to disk %d\n", idx);
	    showit(output);
	    dump_log( bufp, HDR, "Output header", 0);
	    dump_log( bufp + HDR, wlen - HDR, "SCSI CDB", 0);
	    sts = send_scsicdb(devfd, bufp, wlen, bufp, rlen);
	    if (sts) {
		rcdl = sts;
		sprintf(output, "[%d] Error %d in send_scsicdb\n", idx,
			sts);
		showit(output);
		if (fdebug || !fauto)
		    return (rcdl);
	    } else {		/* good results */
		sprintf(output, "[%d] Command complete\n", idx);
		showit(output);
		dump_buf(fdmsg, bufp, rlen, "Receive data",0);
		if (fdmsg != fdlog)
		   dump_log( bufp, rlen, "Receive data",0);
	    }			/*endif good */
	    if (fauto && rcdl != 0)
		return (rcdl);	/*skip to next one if error */
	    sprintf(output,
		    "[%d] send_scsicdb command complete, status = 0x%x\n",
		    idx, rcdl);
	    showit(output);
	    if ((rcdl & 0x0FFF) == 2) {
		sprintf(output, "   Sense data: %02x %02x %02x\n",
			sense_buffer[2] & 0x0f, sense_buffer[12],
			sense_buffer[13]);
		showit(output);
	    }
	    if (fdebug && rcdl != 0) {
		return (rcdl);	/* return (only if debug on) */
	    }
	    if (finteractive || fonedev)
		idx = ndev;	/* end loop */
	}			/*endif fdoit */
    }				/*endfor each disk */
    return (rcdl);
}				/* end do_sendcdb() */

int beforegd(int idx, int *pnumrdy)
{
    struct SCSI_INQUIRY *scsi_inq;
    int sts;
    int sgfd;
    int isave, nrdy;
    char *pl;
    int rcdl;
    int openflags;
    char *fname;
    isave = idx;
    nrdy = 0;
    sts = 0;
    rcdl = 0;
    openflags = O_RDWR;		/* | O_NONBLOCK; */
    for (idx = 0; idx < ndev; idx++) {
	if (finteractive || fonedev)
	    idx = isave;	/* only do selected disk */
	fname = devlist[idx].fname;
	sgfd = open(fname, openflags);	/* open blocking */
	if (sgfd < 0) {
	    printf("%s: open error, errno = %d\n", fname, errno);
	    return (errno);
	}
	devlist[idx].sgfd = sgfd;
	sts = set_sg_debug(sgfd, sgdbglvl);
	if (sts) {
	    sprintf(output, "[%d] cant set debug flag, sts = %d", idx,
		    sts);
	    showit(output);
	}
	/* Inquiry data was filled in during get_scsi_info */
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
	if ((scsi_inq->dev_type & 0x1F) == 00) {	/* if type = disk */
	    devlist[idx].fdoit = 1;
	    sprintf(output, "Device [%d] is ready for command\n", idx);
	    showit(output);
	    nrdy++;		/* number ready to send command */
	} else
	    devlist[idx].fdoit = 0;	/* not ok  */
	if (finteractive || fonedev)
	    idx = ndev;		/* end loop */
    }				/* end for devlist TUR loop */
    *pnumrdy = nrdy;		/* set return value */
    if (nrdy == 0) {
	sprintf(output, "There are no ready devices.\n");
        strcat(output,"Try 'modprobe sg' if not already loaded\n"); 
	showit(output);
	return (sts);
    } else {
	if (nrdy > 1)
	    pl = "s";
	else
	    pl = "";
	sprintf(output, "Sending SCSI command for %d disk%s.\n", nrdy, pl);
	showit(output);
    }
    setbuf(fdmsg, NULL);	/* set for unbuffered writes */
    return (rcdl);
}				/*end beforegd() */

/* end sgdiag.c */
