/* pdvmkpkg.c
   $Id: pdvmkpkg.c,v 1.10 2001/09/30 01:46:22 gwiley Exp $
   Glen Wiley, gwiley@ieee.org
  
	--------------------
	this program will create a self extracting/self executing delivery
	vehicle including the ability to uncompress and untar the payload
	all using a single file

	if zlib is available on the build machine then support for 
	compression can be built in (configure should detect this)

	the resulting pdv package is just the pdv utility with the payload
	appended and some meta-data regarding the payload appended to that
	--------------------

   Copyright (c)1999,2000,2001 Glen Wiley
  
	Permission is hereby granted, free of charge, to any person
	obtaining a copy of this software and associated documentation
	files (the "Software"), to deal in the Software without
	restriction, including without limitation the rights to use, copy,
	modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions: The above
	copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.  THE SOFTWARE
	IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
	NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
	WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

*/

#if HAVE_CONFIG_H
#include "config.h"
#endif
#include <errno.h>
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif
#if HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_ZLIB_H
#include <zlib.h>
#endif
#include "pdv.h"
#include "version.h"

/* basename of this program - used in error messages */
char *g_prg = NULL;

/* print usage message */
void usage(void);

/* print summary of package data */
void summary(const char *fn, struct payload_st *pld);

/*---------------------------------------- main */
int
main(int argc, char *argv[])
{
#if PDV_USE_ZLIB
	char *optchars = "vz?hTCa:e:c:H:o:p:s:";
#else
	char *optchars = "v?hTCa:e:c:H:o:p:s:";
#endif
	char *fn_pkg    = NULL;
	char *fn_pdv    = NULL;
	char *fn_pld    = NULL;
	char *fn_hlpmsg = NULL;
	char *fn_agrmsg = NULL;
	char *fn_spec   = NULL;
	char *badfield;
	int  retval     = 0;
	int  promptusr  = 0;
	int  nopts      = 0;
	int  result;
	char opt;
	struct stat       payldstat;
	struct payload_st pld_data;
	struct pdvspec_st spec;

	pld_data.fn_out        = NULL;
	pld_data.cmd           = NULL;
	pld_data.hlpmsg        = NULL;
	pld_data.agrmsg        = NULL;
	pld_data.iscompressed  = -1;   /* -1 here indicates not specified */
	pld_data.isatar        = -1;   /* later, 0 or 1 will indicate mode */
	pld_data.iszipped      = -1;   /* to support interactive prompts */
	pld_data.payloadstart  = 0L;
	pld_data.metadatastart = 0L;

	/* get our basename for the sake of error reporting */

	g_prg = strrchr(argv[0], '/');
	if(g_prg == NULL)
		g_prg = argv[0];
	else
		g_prg++;

	/* get command line options */

	if(argc == 1)
	{
		usage();
		exit(1);
	}

	while((opt=getopt(argc, argv, optchars)) != EOF)
	{
		switch(opt)
		{
			case 'a':
				fn_agrmsg = optarg;
				nopts++;
				break;

			case 'C':
				pld_data.iscompressed = 1;
				nopts++;
				break;

			case 'T':
				pld_data.isatar = 1;
				nopts++;
				break;

			case 'c':
				fn_pkg = optarg;
				nopts++;
				break;

			case 'e':
				pld_data.cmd = optarg;
				nopts++;
				break;

			case 'H':
				fn_hlpmsg = optarg;
				nopts++;
				break;

			case 'o':
				pld_data.fn_out = optarg;
				nopts++;
				break;

			case 'p':
				fn_pdv = optarg;
				nopts++;
				break;

			case 's':
				fn_spec = optarg;
				break;

			case '?':
			case 'h':
				usage();
				exit(1);
				break;

			case 'v':
				printf("pdvmkpkg, %s\n", g_versionstr);
				exit(0);
				break;

#if PLD_USE_ZLIB
			case 'z':
				pld_data.iszipped = 1;
				break;
#endif

			default:
				usage();
				fatalError(EINVAL, "");
		} /* switch(opt) */
	} /* while((opt=getopt(argc, argv, optchars)) != EOF) */

	/* the remaining arguments are all file names (right now only 1 is used */

	if(optind < argc)
	{
		nopts++;
		fn_pld = argv[optind];
	}

	/* if all we got on the command line was a spec file, then the user
	   wants us to generate the package from it
	*/

	if(nopts == 0)
	{
		result = readspec(fn_spec, &spec, &badfield);
		if(result != 0)
			fatalError(result, "reading spec file %s, field %s", fn_spec
			 , (badfield ? badfield : "unknown"));
		else
		{
			result = spectopayload(&spec, &pld_data);
			if(result != 0)
				fatalError(result, "error converting spec to payload struct");

			fn_pkg    = spec.fn_pkg;
			fn_hlpmsg = spec.fn_helpmsg;
			fn_agrmsg = spec.fn_agrmsg;
			fn_pdv    = spec.fn_pdvstub;
			fn_pld    = spec.fn_payld;
		}
	}

	if(fn_pkg == NULL)
	{
		printf("no package name specified - use '-c <package_file_name>'\n");
		return 1;
	}

	/* locate the pdv stub used as a prefix */

	fn_pdv = findpdv(argv[0], fn_pdv, 1);

	if(fn_pdv == NULL || access(fn_pdv, R_OK) != 0)
		fatalError(errno, "unable to access a pdv stub");

/*	if(checkpdvver(fn_pdv, VERSION) != 0)
		fatalError(errno, "pdv stub version does not match");
*/
	/* if no ouput filename specified then use the payload file name */

	if(pld_data.fn_out == NULL)
	{
		pld_data.fn_out = strdup(fn_pld);
		if(pld_data.fn_out == NULL)
			fatalError(errno, "malloc (strdup) failed");
	}

	/* if we are not interactive and these were not specified then the
	   user must not want them */

	if(pld_data.iscompressed == -1)
		pld_data.iscompressed = 0;

	if(pld_data.isatar == -1)
		pld_data.isatar = 0;

	if(pld_data.iszipped == -1)
		pld_data.iszipped = 0;

	/* verify access to payload and get the file mode */

	if(stat(fn_pld, &payldstat) != 0 || (payldstat.st_mode | S_IREAD) == 0)
	{
		fatalError((fn_pld == NULL ? ENOENT : errno)
		 , "unable to access payload %s", (fn_pld ? fn_pld : "(nil)"));
	}

	pld_data.payldmode = payldstat.st_mode;

	if(pld_data.isatar && pld_data.fn_out == NULL)
		fatalError(EINVAL, "an output filename must be specified unless the "
		 "payload is a tar file");

	/* read help message file as needed */

	if(fn_hlpmsg)
	{
		result = readMsgFile(fn_hlpmsg, &(pld_data.hlpmsg));
		if(result != 0)
			fatalError(errno, "error reading help message file %s", fn_pkg);
	}

	/* read agreement message file as needed */

	if(fn_agrmsg)
	{
		result = readMsgFile(fn_agrmsg, &(pld_data.agrmsg));
		if(result != 0)
			fatalError(errno, "error reading agreement message file %s", fn_pkg);
	}

	/* if some command line options accompanied the spec file then we
	   need to go ahead and create the spec file with all of those
		options */

	if(nopts > 0 && fn_spec != NULL)
	{
		spec.fn_spec    = fn_spec;
		spec.fn_helpmsg = fn_hlpmsg;
		spec.fn_agrmsg  = fn_agrmsg;
		spec.fn_output  = pld_data.fn_out;
		spec.fn_payld   = fn_pld;
		spec.fn_pdvstub = fn_pdv;
		spec.fn_exec    = pld_data.cmd;
		spec.fn_pkg     = fn_pkg;

		if(pld_data.iscompressed)
			spec.compress = strdup("y");
		else
			spec.compress = strdup("n");

		if(pld_data.isatar)
			spec.tar = strdup("y");
		else
			spec.tar = strdup("n");

		result = writespec(&spec);
		if(result != 0)
			fprintf(stderr, "warning: error writing spec file %s, %s\n"
			 , fn_spec, strerror(result));
	}

	newpkgfile(fn_pdv, fn_pkg, fn_pld, &pld_data, 1);

	summary(fn_pkg, &pld_data);

	return retval;
} /* main */

/*---------------------------------------- usage */
void
usage(void)
{
	fprintf(stderr, 
	 "\npdvmkpkg, %s\n\n"
#if PDV_USE_ZLIB
	 "USAGE: %s [-ihvCTz] [-s <specfile>] [-e <payload_execfile>] [-H <help_message_file>] [-a <agreement_file>] [-o <payload_outputfile>] [-p <pdv_file>] -c <pkg_file_to_create> <payload_files/dirs>\n"
#else
	 "USAGE: %s [-ihvCT] [-s <specfile>] [-e <payload_execfile>] [-H <help_message_file>] [-a <agreement_file>] [-o <payload_outputfile>] [-p <pdv_file>] -c <pkg_file_to_create> <payload_file>\n"
#endif
	 "\n  or\n\nUSAGE: %s -s <specfile>\n"
	 "\n"
	 "-c <pkg_file_to_create> pdv package file to create\n"
	 "-C                      payload should be filtered through uncompress\n"
	 "-e <payload_execfile>   file to exec on completion\n"
    "-h                      display this help message\n"
	 "-H <help_message_file>  use help message in this file in response to\n"
	 "                        the user invoking the package with -h on the\n"
	 "                        command line\n"
	 "-a <agreement_file>     display message in this file prior to extracting\n"
	 "                        and require the user to respond with 'yes' to\n"
	 "                        extract payload\n"
	 "-o <payload_outputfile> name of file to extract payload as when user\n"
	 "                        executes the package\n"
#if PDV_USE_ZLIB
	 "                        (not used if payload is a tar file or if internal\n"
	 "                        compression was used)\n"
#else
	 "                        (not used if payload is a tar file)\n"
#endif
	 "-p <pdv_file>           name of pdv stub to use, found via path if not \n"
	 "                        specified (pdv)\n"
	 "-T                      payload should be filtered through tar\n"
	 "-v                      print version and exit\n"
#if PDV_USE_ZLIB
	 "-z                      use zlib compression routines to package all\n"
	 "                        of the payload files (and directories)\n"
	 "<payload_files/dirs>    files and/or directories to be used as payload \n"
	 "                        in the package, separate names with spaces\n"
#else
	 "<payload_file>          the file to be used as payload in the packaged\n"
#endif
	 "-s <specfile>           if a specfile is the only command line option\n"
	 "                        specified then the settings are taken from \n"
	 "                        the specfile - if any other options are specified\n"
	 "                        then the specfile will be created using the\n"
	 "                        command line arguments.  This is way of reliably\n"
	 "                        reproducing a package repeatedly. NOTE: specfiles\n"
	 "                        are also used by the X11 utilities.\n"
	 "\n"
	 "example: the following command would create a file named setup.bin\n"
	 "  that could be sent to a user.  When the user executes setup.bin it\n"
	 "  will uncompress and untar fooapp.tar.Z and then execute fooapp/Install\n"
	 "  (which was presumedly generated from the tar file)\n"
	 "\n"
	 "     pdvmkpkg -CT -e fooapp/Install -c setup.bin fooapp.tar.Z\n"
#if PDV_USE_ZLIB
	 "\n"
	 "or, to use internal compression via zlib you might specify the\n"
	 "constituent parts of of the payload:\n"
	 "\n"
	 "     pdvmkpkg -z -e fooapp/Install -c setup.bin fooapp README.fooapp\n"
	 "\n"
	 "this would accomplish the same thing (assuming the fooapp dir and\n"
	 "the separate README.fooapp file were what was contained in the \n"
	 "compressed tar in the first example.\n"
#endif
	 "\n"
	 "for more information or the latest release see %s\n\n"
	 , g_versionstr, g_prg, g_prg, g_homepage);

	return;
} /* usage */

/*---------------------------------------- summary
  print summary of package data */
void
summary(const char *fn, struct payload_st *pld)
{
	printf("\n---------------------\n");
	printf("package data summary:\n\n");
	printf("package file name: %s\n", fn);
	printf("will %suncompress payload\n", (pld->iscompressed ? "" : "not "));
	printf("will %suntar payload\n", (pld->isatar ? "" : "not "));
	if(!pld->isatar)
		printf("payload extracts as: %s\n", pld->fn_out);
	if(pld->cmd)
		printf("will execute \"%s\"\n", pld->cmd);
	if(pld->hlpmsg)
		printf("user help is embedded\n");
	if(pld->agrmsg)
		printf("agreement message is embedded\n");
	printf("\n");

	return;
} /* summary */

/* pdvmkpkg.c */
