/**
 * @file Data structure which represents non-empty arrays
 */
import { Monad1 } from './Monad';
import { Comonad1 } from './Comonad';
import { FunctorWithIndex1 } from './FunctorWithIndex';
import { TraversableWithIndex1 } from './TraversableWithIndex';
import { FoldableWithIndex1 } from './FoldableWithIndex';
import { Ord } from './Ord';
import { Semigroup } from './Semigroup';
import { Option } from './Option';
import { Setoid } from './Setoid';
import { Predicate, Refinement } from './function';
declare module './HKT' {
    interface URI2HKT<A> {
        NonEmptyArray2v: NonEmptyArray<A>;
    }
}
export declare const URI = "NonEmptyArray2v";
export declare type URI = typeof URI;
/**
 * @since 1.15.0
 */
export interface NonEmptyArray<A> extends Array<A> {
    0: A;
    map: <B>(f: (a: A, index: number, nea: NonEmptyArray<A>) => B) => NonEmptyArray<B>;
    concat: (as: Array<A>) => NonEmptyArray<A>;
    reverse: () => NonEmptyArray<A>;
}
/**
 * @since 1.15.0
 */
export declare function make<A>(head: A, tail: Array<A>): NonEmptyArray<A>;
/**
 * @since 1.15.0
 */
export declare function head<A>(nea: NonEmptyArray<A>): A;
/**
 * @since 1.15.0
 */
export declare function tail<A>(nea: NonEmptyArray<A>): Array<A>;
/**
 * @since 1.15.0
 */
export declare function min<A>(ord: Ord<A>): (nea: NonEmptyArray<A>) => A;
/**
 * @since 1.15.0
 */
export declare function max<A>(ord: Ord<A>): (nea: NonEmptyArray<A>) => A;
/**
 * Builds a `NonEmptyArray` from an `Array` returning `none` if `as` is an empty array
 *
 * @since 1.15.0
 */
export declare function fromArray<A>(as: Array<A>): Option<NonEmptyArray<A>>;
/**
 * Builds a `NonEmptyArray` from a provably (compile time) non empty `Array`.
 *
 * @since 1.15.0
 */
export declare function fromNonEmptyArray<A>(as: Array<A> & {
    0: A;
}): NonEmptyArray<A>;
/**
 * Builds a `Semigroup` instance for `NonEmptyArray`
 *
 * @since 1.15.0
 */
export declare const getSemigroup: <A = never>() => Semigroup<NonEmptyArray<A>>;
/**
 * @example
 * import { fromNonEmptyArray, getSetoid, make } from 'fp-ts/lib/NonEmptyArray2v'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * const S = getSetoid(setoidNumber)
 * assert.strictEqual(S.equals(make(1, [2]), fromNonEmptyArray([1, 2])), true)
 * assert.strictEqual(S.equals(make(1, [2]), fromNonEmptyArray([1, 3])), false)
 *
 * @since 1.15.0
 */
export declare function getSetoid<A>(S: Setoid<A>): Setoid<NonEmptyArray<A>>;
/**
 * Group equal, consecutive elements of an array into non empty arrays.
 *
 * @example
 * import { make, group } from 'fp-ts/lib/NonEmptyArray2v'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(group(ordNumber)([1, 2, 1, 1]), [
 *   make(1, []),
 *   make(2, []),
 *   make(1, [1])
 * ])
 *
 * @since 1.15.0
 */
export declare const group: <A>(S: Setoid<A>) => (as: A[]) => NonEmptyArray<A>[];
/**
 * Sort and then group the elements of an array into non empty arrays.
 *
 * @example
 * import { make, groupSort } from 'fp-ts/lib/NonEmptyArray2v'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(groupSort(ordNumber)([1, 2, 1, 1]), [make(1, [1, 1]), make(2, [])])
 *
 * @since 1.15.0
 */
export declare const groupSort: <A>(O: Ord<A>) => (as: A[]) => NonEmptyArray<A>[];
/**
 * Splits an array into sub-non-empty-arrays stored in an object, based on the result of calling a `string`-returning
 * function on each element, and grouping the results according to values returned
 *
 * @example
 * import { make, groupBy } from 'fp-ts/lib/NonEmptyArray2v'
 *
 * assert.deepStrictEqual(groupBy(['foo', 'bar', 'foobar'], a => String(a.length)), {
 *   '3': make('foo', ['bar']),
 *   '6': make('foobar', [])
 * })
 *
 * @since 1.15.0
 */
export declare const groupBy: <A>(as: A[], f: (a: A) => string) => {
    [key: string]: NonEmptyArray<A>;
};
/**
 * @since 1.15.0
 */
export declare function last<A>(nea: NonEmptyArray<A>): A;
/**
 * @since 1.15.0
 */
export declare function sort<A>(O: Ord<A>): (nea: NonEmptyArray<A>) => NonEmptyArray<A>;
/**
 * @since 1.15.0
 */
export declare function findFirst<A, B extends A>(nea: NonEmptyArray<A>, refinement: Refinement<A, B>): Option<B>;
export declare function findFirst<A>(nea: NonEmptyArray<A>, predicate: Predicate<A>): Option<A>;
/**
 * @since 1.15.0
 */
export declare function findLast<A, B extends A>(nea: NonEmptyArray<A>, refinement: Refinement<A, B>): Option<B>;
export declare function findLast<A>(nea: NonEmptyArray<A>, predicate: Predicate<A>): Option<A>;
/**
 * @since 1.15.0
 */
export declare function findIndex<A>(nea: NonEmptyArray<A>, predicate: Predicate<A>): Option<number>;
/**
 * @since 1.15.0
 */
export declare function findLastIndex<A>(nea: NonEmptyArray<A>, predicate: Predicate<A>): Option<number>;
/**
 * @since 1.15.0
 */
export declare function insertAt<A>(i: number, a: A, nea: NonEmptyArray<A>): Option<NonEmptyArray<A>>;
/**
 * @since 1.15.0
 */
export declare function updateAt<A>(i: number, a: A, nea: NonEmptyArray<A>): Option<NonEmptyArray<A>>;
/**
 * @since 1.15.0
 */
export declare function filter<A, B extends A>(nea: NonEmptyArray<A>, refinement: Refinement<A, B>): Option<NonEmptyArray<A>>;
export declare function filter<A>(nea: NonEmptyArray<A>, predicate: Predicate<A>): Option<NonEmptyArray<A>>;
/**
 * @since 1.15.0
 */
export declare function filterWithIndex<A>(nea: NonEmptyArray<A>, predicate: (i: number, a: A) => boolean): Option<NonEmptyArray<A>>;
/**
 * @since 1.15.0
 */
export declare const nonEmptyArray: Monad1<URI> & Comonad1<URI> & TraversableWithIndex1<URI, number> & FunctorWithIndex1<URI, number> & FoldableWithIndex1<URI, number>;
