/*
 * Utilities to parse the 'appinfo' file structure
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"

#include <alps/apInfo.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>

/* Location of ALPS configuration file */
#define ALPS_CONFIGURATION_PATH	"/etc/sysconfig/alps"	/*  FIXME autoconf rule for this path */

/* Configuration variable specifying ALPS shared directory */
#define ALPS_CONF_SHARED	"ALPS_SHARED_DIR_PATH"
#define ALPS_CONF_SHARED_LEN	(sizeof(ALPS_CONF_SHARED) - 1)
/* ALPS appinfo file to access */
#define ALPS_APPINFO_PATH	"appinfo"

/* Maximum number of tries to read the NFS-shared file */
#define ALPS_APPINFO_MAX_TRIES	3

/* Base-2 logarithm of val > 0 */
static uint8_t lb(uint64_t val)
{
	uint8_t r = 0;

	while (val >>= 1)
		r++;
	return r;
}

/*
 *--------------------------------------------------------------
 * cmdDetail_t
 *--------------------------------------------------------------
 * int   flags		- command-specific flags (often 0)
 * int   width		- number of PEs for this command
 * int   depth		- processors per PE
 * int   fixedPerNode	- user set per-node PE count
 * int   memory		- per PE memory limit in megabytes
 * alps_archType_t arch	- architecture type
 * char  cmd[32]	- a.out name ("BASIL" for reservation)
 *
 * int   nodeCnt	- number of nodes allocated
 *
 * uint16_t pesPerSeg	-S value
 * uint16_t nodeSegCnt	-sn value
 * uint32_t segBits	-sl <0-3> - each bit is a segment number
 *
 * uint16_t rvalue	-r value - 0 or 1
 */

/**
 * appinfo_arch  -  translate alps_archType_t into string
 * @arch: alps_archType_t value
 */
const char *appinfo_arch(uint8_t arch)
{
	static const char *ai_arch[] = {
		"(invalid architecture)",
		"BW",
		"XT"
	};
	return ai_arch[arch >= sizeof(ai_arch)/sizeof(ai_arch[0]) ? 0 : arch];
}

/**
 * basil_get_appinfo_path  - determine configured path to 'appifo' at runtime.
 * Return allocated result, abort on configuration error.
 */
char *basil_get_appinfo_path(void)
{
	char alps_line[1024], *p, *path = NULL, *result = NULL;
	FILE *alps_conf;

	if (access(ALPS_CONFIGURATION_PATH, F_OK) < 0)
		fatal("no configuration file '%s'", ALPS_CONFIGURATION_PATH);

	alps_conf = fopen(ALPS_CONFIGURATION_PATH, "r");
	if (alps_conf == NULL)
		fatal("can not open '%s'", ALPS_CONFIGURATION_PATH);

	while (fgets(alps_line, sizeof(alps_line), alps_conf)) {
		if ((p = strchr(alps_line, '#')))	/* strip comments */
			*p = '\0';
		for (p = alps_line; isspace(*p); )	/* leading whitespace */
			p++;

		if (strncmp(p, ALPS_CONF_SHARED, ALPS_CONF_SHARED_LEN) == 0) {
			/* extract RHS */
			path = p + ALPS_CONF_SHARED_LEN;
			while (*path == '=' || *path == '"' || isspace(*path))
				path++;
			for (p = path; *p && !isspace(*p) && *p != '"'; )
				p++;
			*p = '\0';
			break;
		}
	}
	fclose(alps_conf);

	if (path == NULL)
		fatal("can not determine %s", ALPS_CONF_SHARED);
	if (access(path, F_OK) < 0)
		fatal("can not access %s='%s'", ALPS_CONF_SHARED, path);

	result = malloc(strlen(path) + strlen(ALPS_APPINFO_PATH) + 2);
	if (result == NULL)
		fatal("can not allocate");

	sprintf(result, "%s/%s", path, ALPS_APPINFO_PATH);
	return result;
}

/*
 *--------------------------------------------------------------
 * appInfoHdr_t (header of file)
 *--------------------------------------------------------------
 * size_t headerSz	- sizeof(appInfoHdr_t)
 * size_t apInfoSz	- sizeof(appInfo_t)
 * size_t cmdDetSz	- sizeof(cmdDetail_t)
 * size_t plistSz	- sizeof(placeList_t)
 * time_t created	- mtime (time file contents were written)
 * size_t apStart	- offset of first appInfo_t entry
 * int    apNum		- number of appInfo_t entries
 */

/**
 * basil_get_appinfo  -  return 'appinfo' as in-memory structure
 * Return buffer containing current appinfo (must be freed), or NULL.
 */
uint8_t *basil_get_appinfo(void)
{
	int fd = -1, tries;
	uint8_t *ai_buf	 = NULL;
	char	*ai_path = basil_get_appinfo_path();

	/* Not necessarily an error: shared directory may have been purged */
	if (access(ai_path, F_OK) < 0) {
		error("no appinfo file '%s'", ai_path);
		goto done;
	}

	for (tries = 0; ++tries < ALPS_APPINFO_MAX_TRIES; usleep(tries*50000)) {
		struct stat ai_stat;

		fd = open(ai_path, O_RDONLY);
		if (fd  == -1) {
			continue;
		} else if (fstat(fd, &ai_stat) < 0) {
			error("can not stat %s", ai_path);
		} else if (ai_stat.st_size == 0) {
			error("%s is empty", ai_path);
			goto done;
		} else if (ai_stat.st_size < sizeof(appInfoHdr_t)) {
			error("%s is too short", ai_path);
			goto done;
		} else {
			long n = ai_stat.st_size;

			ai_buf = malloc(n);
			if (ai_buf == NULL) {
				error("can not allocate appinfo buf of %ld", n);
				goto done;
			}
			do {
				n = read(fd, ai_buf, ai_stat.st_size);
			} while (n < 0 && errno == EINTR);
			if (n == ai_stat.st_size)
				break;
		}
		close(fd);
		fd = -1;
		free(ai_buf);
		ai_buf = NULL;
	}
	if (ai_buf == NULL && tries == ALPS_APPINFO_MAX_TRIES)
		error("failed to open '%s'", ai_path);
done:
	if (fd != -1)
		close(fd);
	free(ai_path);
	return ai_buf;
}

/*
 *--------------------------------------------------------------
 * appInfo_t
 *---------------------- GENERAL: ------------------------------
 * uint      resId	- ALPS reservation ID
 * uint64_t  apid	- ALPS application ID
 * uint64_t  pagg	- ALPS cookie (SGI pagg or shell SID)
 * uint64_t  flags	- RECFLAG_* flags
 *                        If RECFLAG_USERNL is set in 'flags',
 *                        this nid list may not be discarded by
 *                        recovery for confirm retry.
 * uid_t     uid	- user UID
 * int64_t   account	- same as uid or 0
 * gid_t     gid	- user GID
 *
 * int	     reqType	- most recent request type ALPS_RES_* values
 * int       fanout	- control tree fanout width
 *---------------------- OFFSETS: ------------------------------
 * size_t    cmdDetail	- offset of first cmdDetail_t entry
 * int       numCmds	- entries in cmdDetail
 *
 * size_t    places	- offset of first placeList_t entry
 * int	     numPlaces	- entries in places
 *---------------------- TIMES: --------------------------------
 * time_t    timePlaced	- for aprun
 * time_t    timeSubmitted
 * time_t    timeChkptd	- time of latest successful checkpoint
 * int       conTime	- connect time for context switched apps
 * int       timeLim
 * int       slicePri	- time slicing priority
 *
 * ---------------- XT_GNI only: -------------------------------
 * uint16_t  pTag	- 8-bit NTT-unique value used by drivers
 * uint32_t  cookie	- per-system unique value used by libs
 * uint16_t  nttGran	- NTT granularity (1-32; 0=no NTT)
 * uint32_t  numNtt	- number of NTT entries
 * uint32_t  hugeTLBPS	- MRT huge page size (KB)
 *---------------------- CLE 3.x only: -------------------------
 * pid_t     aprunPid	- PID of aprun (0 if nothing is running)
 * int       aprunNid	- NID of aprun's login node
 * time_t    swInTime	- time of last context switch in
 * time_t    swOutTime	- time of last context switch out
 *--------------------------------------------------------------
 */

/**
 * appinfo_by_resid  -  look up appinfo entry by reservation ID
 * @ai_buf: start of appinfo buffer
 * @resid:  reservation ID to search for
 * Returns pointer to appinfo entry, or NULL if not found.
 */
const appInfo_t *appinfo_by_resid(const uint8_t *ai_buf, uint32_t resid)
{
	const appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	size_t offset = ai_hdr->apStart;
	int i;

	for (i = 0; i < ai_hdr->apNum; i++) {
		appInfo_t *ai_info = (appInfo_t *)(ai_buf + offset);

		if (ai_info->resId == resid)
			return ai_info;

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
	return NULL;
}

/**
 * appinfo_by_apid  -  look up appinfo entry by application ID
 * @ai_buf: start of appinfo buffer
 * @apid:   application ID to search for
 * Returns pointer to appinfo entry, or NULL if not found.
 */
const appInfo_t *appinfo_by_apid(const uint8_t *ai_buf, uint64_t apid)
{
	const appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	size_t offset = ai_hdr->apStart;
	int i;

	for (i = 0; i < ai_hdr->apNum; i++) {
		appInfo_t *ai_info = (appInfo_t *)(ai_buf + offset);

		if (ai_info->apid == apid)
			return ai_info;

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
	return NULL;
}

/**
 * basil_get_apids_by_resv  -  return allocated APIDs for reservation ID
 * @ai_buf: start of appinfo buffer area
 * @resid:  reservation ID to look for (use 0 to get all Apids)
 * Return 0-terminated array of apids, NULL if no result and on error
 */
uint64_t *basil_get_apids_by_resv(const uint8_t *ai_buf, uint32_t resid)
{
	appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	uint64_t *apids = NULL;
	size_t offset;
	int i, n;

	for (i = 0, n = 1, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t *ai_info = (appInfo_t *)(ai_buf + offset);

		if (ai_info->timePlaced && (!resid || ai_info->resId == resid)) {
			apids = realloc(apids, (n + 1) * sizeof(*apids));
			if (apids == NULL) {
				error("failed to allocate Apid entry");
				break;
			}
			apids[n-1] = ai_info->apid;
			apids[n++] = 0;
		}
		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
	return apids;
}

/**
 * num_placed_applications  -  count the number of placed APIDs
 * @ai_buf:  start of appinfo buffer area
 * @resv_id: resId > 0 to select, 0 to count all placed applications
 */
uint32_t num_placed_applications(const uint8_t *ai_buf, uint32_t resv_id)
{
	appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	size_t offset;
	uint32_t i, cnt = 0;

	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t *ai_info = (appInfo_t *)(ai_buf + offset);

		if (ai_info->timePlaced)
			cnt += resv_id == 0 || ai_info->resId == resv_id;

		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
	return cnt;
}

/**
 * is_any_node_claimed  -  check if at least one node is being used
 * @buf_start: start of appinfo buffer area
 * @ai_info:   current appinfo entry
 * Returns true if at least one node has been claimed.
 * NB: the algorithm is not sufficient to distinguish node state
 *     between 'conf' and 'conf,claim' on CLE 2.x - use instead
 *     basil_rsvn_get_num_apps(rsvn).
 */
bool is_any_node_claimed(const uint8_t *buf_start, appInfo_t *ai_info)
{
	placeList_t *entry = (placeList_t *)(buf_start + ai_info->places);
	int j = 0, idx = -1, nid = -1, cores = 0;
	uint32_t mask = 0;

	for (j = 0; j < ai_info->numPlaces; j++, entry++) {
 		/*
		 * Each placement entry corresponds to one core (CPU) of a node.
		 * If all have the same bitmask of all 1's, and if no bit is set
		 * in any of the bitmasks, the node has not been claimed.
		 */
		if (entry->cmdIx != idx || entry->nid != nid) {
			if (cores != lb(mask + 1))
				return true;
			idx   = entry->cmdIx;
			nid   = entry->nid;
			mask  = entry->procMask;
			cores = 0;
		} else if (entry->procMask != mask) {
			return true;
		}
		cores++;
	}
	return false;
}

/*
 *--------------------------------------------------------------
 * placeList_t
 *--------------------------------------------------------------
 * int    cmdIx		- cmdInfo_t entry this PE belongs to
 * int    nid		- NID this PE is assigned to
 *--------------------------------------------------------------
 * the next entry differs between CLE 2 and 3:
 * short  X2PeMap	- CLE 2 variant: XT segment / X2 PE map
 * short  nodeMap	- CLE 3 variant: XT segment map
 *--------------------------------------------------------------
 * uint32_t  procMask	- bitmap of CPUs used (CLE 2 used 'int')
 */

/**
 * appinfo_nidlist  -  Extract nodelist from appinfo entry
 * @buf_start: start of appinfo buffer area
 * @ai_info:   current appinfo entry
 */
struct nodespec *appinfo_nidlist(const uint8_t *buf_start, appInfo_t *ai_info)
{
	placeList_t *entry = (placeList_t *)(buf_start + ai_info->places);
	struct nodespec *ns_head = NULL;
	int j;

	for (j = 0; j < ai_info->numPlaces; j++, entry++)
		if (ns_add_node(&ns_head, entry->nid))
			fatal("can not add node to list");
	return ns_head;
}
