//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cWeapon.hpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================
#ifndef cWeapon_hpp
#define cWeapon_hpp
//==============================================================================
// Includes
#include <string>
#include "Types.hpp"
#include "cVector2f.hpp"
#include "cObject.hpp"
#include "cMixer.hpp"
#include "cOwnable.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace std;
namespace ShootingStar {
//------------------------------------------------------------------------------
// Forward declarations
class cParticleEmitter;
//==============================================================================
	
//==============================================================================
//! Weapon
//------------------------------------------------------------------------------
class cWeapon : public cObject, public cOwnable
{
	public:
		enum WeaponState
		{
			State_Reloading,
			State_Ready
		};
	
	// Constructor & Destructor
	public:
		//! Constructor
		cWeapon (int clipSize, Uint32 fireDelay, Uint32 reloadTime);
		//! Destructor
		virtual ~cWeapon (void);
	
	// Public methods
	public:
		//! Set ammo
		void SetAmmo (int currentClip, int numberOfClips)
		{
			if ( currentClip <= mClipSize )
				mCurrentClip = currentClip;
			else
				mCurrentClip = mClipSize;
			mNumberOfClips = numberOfClips;
		};
		//! Return ammo
		void GetAmmo (int &currentClip, int &numberOfClips) const
		{
			currentClip = mCurrentClip;
			numberOfClips = mNumberOfClips;
		};
		int GetClipSize (void) const { return mClipSize; };
		int GetNumberOfClips (void) const { return mNumberOfClips; };
		void AddClips (int count) { mNumberOfClips += count; };
		
		virtual void PullTrigger (void) 
		{ 
			mFire = true; 
			if ( mCurrentClip > 0 && mState == State_Ready )
				BeginFiring ();
			if( mCurrentClip <= 0 )
				cMixer::GetInstance ().PlaySound (mSound);
		};
		virtual void ReleaseTrigger (void) 
		{ 
			mFire = false;
			if ( mState == State_Ready )
				EndFiring ();
		};
		bool IsFiring (void) { return mFire; };
		
		//! Update
		virtual void Update (Uint32 deltaTime);
		
		WeaponState GetState (void) const { return mState; };
		float GetReloadStatus (void) const
		{
			if ( mReloadTime == 0 )
				return 0.0f;
			return float (mReloadTimeLeft) / float (mReloadTime);
		};
		float GetClipStatus (void) const
		{
			if ( mClipSize == 0 )
				return 0.0f;
			return float (mCurrentClip) / float (mClipSize);
		};
		
		void Reset (void) { mState = State_Ready; };
		
		virtual string GetName (void) const = 0;
		
		void SetActive (bool value = true) 
		{ 
			mActive = value; 
			OnSetActive (); 			
			if ( !IsActive () )
				ReleaseTrigger ();
		};
		bool IsActive (void) const { return mActive; };
		
		void SetShooter (cParticleEmitter *pShooter) { mpShooter = pShooter; };
		cParticleEmitter *GetShooter (void) { return mpShooter; };
		
		virtual void OnSetActive (void) { };
		virtual void BeginFiring (void) { };
		virtual void EndFiring (void) { };
		
		virtual void BeginReload (void) { };
		virtual void EndReload (void) { };
		
	// Protected methods
	protected:
		//! Fire
		virtual void Fire (const cVector2f &position, float angle) { };
		
		void SpawnSuuliekki (void);
	
	// Member variables
	private:
		cParticleEmitter *mpShooter;
		WeaponState mState;
		bool mActive;
		bool mFire;
		int mClipSize;
		int mCurrentClip;
		int mNumberOfClips;
		Uint32 mFireDelay;
		Uint32 mFireDelayLeft;
		Uint32 mReloadTime;
		Uint32 mReloadTimeLeft;
		Uint32 mSound;
		int mChannel;
};
//==============================================================================

//==============================================================================
}		// End of the ShootingStar namespace
#endif // cWeapon_hpp
//------------------------------------------------------------------------------
// EOF
//==============================================================================
