/* icd.c

    icdprog - an open source PIC programmer for use with the Microchip ICD(1)
    Copyright (C) 2001-2004  Geir Thomassen, Andrew Ferry.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <stdarg.h>
#include "icd.h"
#include "icdprog.h"
#include "pic16f87x.h"
#include "progress_bar.h"
#include "serial.h"


void icd_hw_reset()
{
	rts_clear();
	dtr_clear();   /* reset */
	udelay(10000);
	dtr_set();     /* remove reset */
	udelay(10000);
}

#define MAX_CMD_LEN 100

int icd_cmd(int cmd)
{
	char command[MAX_CMD_LEN];
	unsigned char resp[3];
	int n_read;
	
	
(void) sprintf(command,"$%04X\r", cmd);

	icd_write(command, strlen(command));

	n_read=icd_read(resp);

	rts_clear();
	udelay(1);
	rts_set();
	
	if(n_read != 1) {
		return -1;
	}

	n_read=icd_read(resp+1);

	rts_clear();
	udelay(1);
	rts_set();

	if(n_read != 1) {
		return -1;
	}

	return (resp[0] << 8 ) |  resp[1];
}

int icd_baudrate_init()
{
	int tries=3;
	char ch;

	while(tries) {
		icd_write("U",1);

		udelay(10000);

		if(icd_read(&ch) > 0) {
			rts_clear();
			udelay(10);
			rts_set();
			if(ch=='u') {
				return 0;  /* Success, we have set the baud rate on the icd hardware */
			} 
		} 
		tries--;
	}

	return -1;
}


int icd_firmware_version()
{
	unsigned int ver1,ver2;
	if((ver1 = icd_cmd(ICD_GET_VER_MAJOR)) == -1)
		return -1;
		
	if((ver2 = icd_cmd(ICD_GET_VER_MINOR)) == -1)
		return -1;

	printf("ICD firmware version: %X.%02X.%02X\n", ver1>>8, ver1&0xFF, ver2);
	
	return 0;
}

int icd_voltages()
{
	int vcc,vpp;

	if(icd_cmd(ICD_EN_VPP) == -1) {
		return -1;
	}
	
	if((vcc=icd_cmd(ICD_GET_VCC)) == -1) {
		return -1;
	}
	
	if((vpp=icd_cmd(ICD_GET_VPP)) == -1) {
		return -1;
	}  
	
	if(icd_cmd(ICD_DIS_VPP) == -1) {
		return -1;
	}

	printf("Vcc = %4.2fV, ", ((double)vcc) / 40.0);
	printf("Vpp = %5.2fV\n", ((double)(vpp & 0xFF)) / 11.25); /* What the heck does the high byte contain ? */
	
	return 0;
}


controller icd_controller_type()
{
	unsigned int dev_id, type,rev;

	controller t;
	
	dev_id=icd_cmd(ICD_GET_CONTROLLER_TYPE);
	type = (dev_id>>5) & 0x1FF;
	rev = type & 0x1F;

	t.type=0;
	t.flash_max=0;
	t.eeprom_max=0;
	
	if(dev_id == 0x3FFF) {
		printf("no target\n");
		return t;
	} else {
		switch(type) {
		case 0x68:
			printf("Controller: PIC 16F870 rev %d\n",rev);
			t.flash_max=2047;
			t.eeprom_max=63;
			t.type=870;
			break;
		case 0x69:
			printf("Controller: PIC 16F871 rev %d\n",rev);
			t.flash_max=2047;
			t.eeprom_max=63;
			t.type=871;
			break;
		case 0x47:
			printf("Controller: PIC 16F872 rev %d\n",rev);
			t.flash_max=2047;
			t.eeprom_max=63;
			t.type=872;
			break;
		case 0x4B:
			printf("Controller: PIC 16F873 rev %d\n",rev);
			t.flash_max=4095;
			t.eeprom_max=127;
			t.type=873;
			break;
		case 0x49:
			printf("Controller: PIC 16F874 rev %d\n",rev);
			t.flash_max=4095;
			t.eeprom_max=127;
			t.type=874;
			break;
		case 0x4F:
			printf("Controller: PIC 16F876 rev %d\n",rev);
			t.flash_max=8191;
			t.eeprom_max=255;
			t.type=876;
			break;
		case 0x4D:
			printf("Controller: PIC 16F877 rev %d\n",rev);
			t.flash_max=8191;
			t.eeprom_max=255;			
			t.type= 877;
			break;

		default:
			printf("Unknown controller, device id = %02X\n",dev_id);
			t.type=-1;
			break;
		}
	}
	return t;
}


int icd_erase()
{
	if(icd_cmd(0x7000) == -1)   /* Enable Vpp */
		return -1;

	if(icd_cmd(0x7007) != 0x3FFF) {
		return -1;
	}

	return 0;
}

int icd_prog_loc(int *mem, int addr, int mask, int mode)
{

	int resp;
	
	if (mode == PROGRAM) {
		resp=icd_cmd(ICD_WRITE | mem[addr]) & mask;
		if (resp != mem[addr]){
			progress_end();
			printf("programming error - value:0x%04X response:0x%04X \n",mem[addr],resp);
			return -1;
		}
		return 0;
	}
	
	if (mode == VERIFY) {
		resp=icd_cmd(ICD_READ) & mask;
		if (resp != mem[addr]){
			progress_end();
			printf("verify error - value:0x%04X response:0x%04X \n",mem[addr],resp);
			return -1;
		}
	
		return 0;
	}	
	
	if (mode == READ) {
		resp=icd_cmd(ICD_READ) & mask;
	
		if (resp != 0x3FFF) {	
			mem[addr]=resp;
		} 
		else  
			
			mem[addr]= UNINITIALIZED;
		return 0;
	}
	printf("Bug - unrecognized mode in icd_prog_loc()");
	return -1;
}

int icd_prog (int *mem,int prog_cmd,int start,int finish,int mode)
{
	int addr,last_addr=-10;
	
int total_words=0;
	int mask=0xFFFF;

	for(addr=start;addr<=finish;addr++)
		if(mem[addr] != UNINITIALIZED)
			total_words++;

	if(prog_cmd==ICD_PROG_EEPROM)
		mask=0xFF;
		
	icd_cmd(ICD_EN_VPP);

	progress_init(total_words);

	
	for(addr=start;addr <= finish;addr++) {
		if(mem[addr] != UNINITIALIZED) {
			progress();
			if(addr != last_addr+1) {

				if(icd_cmd(prog_cmd) != prog_cmd) {
					progress_end();
					printf("ICD Command error (Program)\n");
					return -1;
				}

				if(icd_cmd(ICD_SET_ADDR | addr) !=  (ICD_SET_ADDR | addr)) {
					progress_end();
					printf("ICD Command Error (Set Address)\n");
					return -1;
				}
			}

			if (icd_prog_loc(mem, addr, mask, mode) == -1)
				return -1;
	
			last_addr = addr;
		}
	}
	
	icd_cmd(ICD_DIS_VPP);
	progress();
	progress_end();
	return 0;
}

int icd_prog_ext(int *mem, int start, int finish, int mode) 
{

	int addr;


	icd_cmd(ICD_EN_VPP);
	
	progress_init(finish-ID_LOC_ADDR+1);

	
	if(icd_cmd(ICD_PROG_FLASH) != ICD_PROG_FLASH) {
		progress_end();
		printf("ICD Command error (Program Flash)\n");
		return -1;
	}
	
	if(icd_cmd(ICD_EXT_ADDR) != 0x0000) {
		progress_end();
		printf("ICD Command error (Ext Address)\n");
		return -1;
	}

	for(addr = ID_LOC_ADDR; addr < start; addr++) {
		if(icd_cmd(ICD_INC_ADDR) != ICD_INC_ADDR) {  
			progress_end();
			printf("ICD Command error (Increment address)\n");
			return -1;
		}
		progress();
	}
	
	for(addr = start; addr <= finish; addr++) {	
		if(mem[addr] == UNINITIALIZED) {
			if(icd_cmd(ICD_INC_ADDR) != ICD_INC_ADDR) {
				progress_end();
				printf("ICD Command error (Increment Address)\n");
				return -1;
			}
		} 
		else {     
		    
if (icd_prog_loc(mem, addr, 0xFFFF, mode)  == -1)
			return -1;
		}
		progress();
	}
	icd_cmd(ICD_DIS_VPP);

	progress_end();
	return 0;
}


int icd_init(char *port)
{

	if(serial_open(port)) {
		return -1;
	}

	icd_hw_reset();

	rts_set();

	if(icd_baudrate_init()) {
		fprintf(stderr,"Can't establish contact with the ICD\n\n");
		fprintf(stderr,"Please check:\n");
		fprintf(stderr," * Is the ICD connected to the correct serial port (%s)?\n", port);
		fprintf(stderr," * Is the power to the ICD switched on? (the red power LED should be on)\n");
		fprintf(stderr," * Is the cable to the ICD OK? (it should be a straight RS-232 cabele, a\n");
		fprintf(stderr,"   null modem cable will *not* work)\n");
		return -1;
	}

	if(icd_cmd(0x6300) == -1) {    /* I really don't know what this is, but MPLAB does
					     this. The program works ok without this though ..*/
		printf("Warning: The mysterious $6300 command failed\n");
	}

	
	if(icd_firmware_version() != 0) {
		printf("Warning, can't read icd firmware version\n");
	}

	if(icd_voltages() != 0) {
	        fprintf(stderr,"Error reading target board voltages\n");
		exit(2);
	}
	
      	return 0;
}


void icd_close()
{
	rts_clear();
	serial_close();
}
