/* serial.c

    icdprog - an open source PIC programmer for use with the Microchip ICD(1)
    Copyright (C) 2001-2004  Geir Thomassen.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <termios.h>
#include <linux/serial.h>
#include <time.h>
#include <string.h>
#include "serial.h"
#include "icdprog.h"

int icd_fd;  /* file descriptor for serial port */
struct serial_struct serinfo_orig;
  /* store for original serinfo */


void udelay(unsigned usec)
{
	/* wait for usec microseconds or more ... */

	struct timespec time;

	time.tv_sec  = usec / 1000000;
	time.tv_nsec = ( usec % 1000000) * 1000;

	nanosleep(&time,NULL);
}

void dtr_set()
{
	int flag = TIOCM_DTR;

	if(ioctl(icd_fd, TIOCMBIS, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void dtr_clear()
{
	int flag  = TIOCM_DTR;

	if(ioctl(icd_fd, TIOCMBIC, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void rts_set()
{
	int flag = TIOCM_RTS;

	if(ioctl(icd_fd, TIOCMBIS, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void rts_clear()
{
	int flag  = TIOCM_RTS;

	if(ioctl(icd_fd, TIOCMBIC, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}


int icd_read(char *ch)
{
	unsigned n_read;

 	n_read=read(icd_fd,ch,1);

	return n_read;
}

int icd_write(void *data, unsigned n_bytes)
{
	ssize_t n_written;

	if((n_written=write(icd_fd, data, n_bytes)) == -1) {
		perror("write");
		exit(-1);
	}

	if(n_written != n_bytes) {
		printf("WTF is going on ?\n");
	}

	return 0;
}

int low_latency(int fd)
{
         struct serial_struct serinfo;


         serinfo.reserved_char[0] = 0;


         if (ioctl(fd, TIOCGSERIAL, &serinfo) < 0) {
                 perror("Cannot get serial info");
                 close(fd);
                 return -1;
         }


         serinfo.flags |= ASYNC_LOW_LATENCY;

         if (ioctl(fd, TIOCSSERIAL, &serinfo) < 0) {
                 perror("Cannot set serial info");
                 return -1;
         }

         return 0;
}

void restore_serinfo(int fd)
{

         if (ioctl(fd, TIOCSSERIAL, &serinfo_orig) < 0) {
                 perror("Cannot restore serial info");

         }

}


int serial_open(char *port)
{
	struct termios oldtio, newtio;

	if((icd_fd=open(port, O_NOCTTY | O_RDWR | O_SYNC)) == -1) {
		perror("Error opening device");
		return -1;
	}

	tcgetattr(icd_fd, &oldtio);

	memset(&newtio,0, sizeof(newtio));
	newtio.c_cflag = BAUD_RATE | CS8 | CLOCAL | CREAD;
	newtio.c_iflag = IGNPAR;
	newtio.c_oflag =  0;
	newtio.c_lflag =  0;

	newtio.c_cc[VTIME] = 20;
        newtio.c_cc[VMIN] = 0;

	tcflush(icd_fd, TCIFLUSH);
	tcsetattr(icd_fd, TCSANOW, &newtio);

	if (low_latency(icd_fd) != 0)
		return -1;
	    
	return 0;
}

void serial_close()
{
	restore_serinfo(icd_fd);
	close(icd_fd);
}

