﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/chime-sdk-messaging/model/ChannelPrivacy.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class ListChannelsRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API ListChannelsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListChannels"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  AWS_CHIMESDKMESSAGING_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_CHIMESDKMESSAGING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the <code>AppInstance</code>.</p>
   */
  inline const Aws::String& GetAppInstanceArn() const { return m_appInstanceArn; }
  inline bool AppInstanceArnHasBeenSet() const { return m_appInstanceArnHasBeenSet; }
  template <typename AppInstanceArnT = Aws::String>
  void SetAppInstanceArn(AppInstanceArnT&& value) {
    m_appInstanceArnHasBeenSet = true;
    m_appInstanceArn = std::forward<AppInstanceArnT>(value);
  }
  template <typename AppInstanceArnT = Aws::String>
  ListChannelsRequest& WithAppInstanceArn(AppInstanceArnT&& value) {
    SetAppInstanceArn(std::forward<AppInstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The privacy setting. <code>PUBLIC</code> retrieves all the public channels.
   * <code>PRIVATE</code> retrieves private channels. Only an
   * <code>AppInstanceAdmin</code> can retrieve private channels. </p>
   */
  inline ChannelPrivacy GetPrivacy() const { return m_privacy; }
  inline bool PrivacyHasBeenSet() const { return m_privacyHasBeenSet; }
  inline void SetPrivacy(ChannelPrivacy value) {
    m_privacyHasBeenSet = true;
    m_privacy = value;
  }
  inline ListChannelsRequest& WithPrivacy(ChannelPrivacy value) {
    SetPrivacy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of channels that you want to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListChannelsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token passed by previous API calls until all requested channels are
   * returned.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListChannelsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the <code>AppInstanceUser</code> or <code>AppInstanceBot</code>
   * that makes the API call.</p>
   */
  inline const Aws::String& GetChimeBearer() const { return m_chimeBearer; }
  inline bool ChimeBearerHasBeenSet() const { return m_chimeBearerHasBeenSet; }
  template <typename ChimeBearerT = Aws::String>
  void SetChimeBearer(ChimeBearerT&& value) {
    m_chimeBearerHasBeenSet = true;
    m_chimeBearer = std::forward<ChimeBearerT>(value);
  }
  template <typename ChimeBearerT = Aws::String>
  ListChannelsRequest& WithChimeBearer(ChimeBearerT&& value) {
    SetChimeBearer(std::forward<ChimeBearerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appInstanceArn;
  bool m_appInstanceArnHasBeenSet = false;

  ChannelPrivacy m_privacy{ChannelPrivacy::NOT_SET};
  bool m_privacyHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_chimeBearer;
  bool m_chimeBearerHasBeenSet = false;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
