/*
 * misc_utils.cpp
 */
#include "config.h"

#include <stdbool.h>
#include <stdio.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#include <stdarg.h>

#ifdef HAVE_UNISTD_H
# include <sys/types.h>
# include <unistd.h>
# include <ctype.h>
#endif

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#else
# define dirent direct
# ifdef HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# ifdef HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# ifdef HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#ifdef HAVE_STRING_H
# if !defined STDC_HEADERS && defined HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif

#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <libgen.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib.h>

#include "misc_utils.h"

#include <array>
#include <sstream>  // for ostringstream
#include <iomanip>  // for setw/setfill
#include <iostream> // for cout/endl
#include <map>
#include <memory>
#include <string>

int execute_using_shell(const std::string &command) 
{
  return execlp(const_cast<char*>(config.shell_for_execution.c_str()), 
                const_cast<char*>(config.shell_for_execution.c_str()), 
                "-c", 
                const_cast<char*>(command.c_str()), 
                NULL);
}

int parse_rx_format_string(std::string &target, const std::string &format, int track_no,
                           const std::string &artist, const std::string &album,
                           const std::string &year, const std::string &song)
{
    int s, d = 0;
    char ch;
    std::string tmp;
    std::string track_no_str;

    if(track_no >= 0)
    {
        std::ostringstream tmp_track;
        tmp_track << std::setw(2) << std::setfill('0') << track_no + 1;
        track_no_str = tmp_track.str();
    }

    target = "";
    
    for(s = d = 0; s<format.length(); ++s)
      {
        // determine space character and check for error
        if(format[ s ] == '%')
          {
            ch = format[ s + 1 ];
            
            if(ch == '\0')
              {
                return -1;
              }
            else if(ch == '%' || ch == '#' || ch == 'a' || ch == 'v' || ch == 'y' || ch == 's')
              {
                s += 1;
              }
            else if(ch == ' ')
              {
                ch = format[ s + 2 ];
                
                if(ch == '\0')
                  {
                    return -1;
                  }
                
                if(ch == '%' || ch == '#' || ch == 'a' || ch == 'v' || ch == 'y' || ch == 's')
                  {
                    s += 2;
                  }
              }
            else
              {
                return -1;
              }
            
            // determine what to copy
            tmp = "";
            
            switch(format[ s ])
              {
              case '%' :
                tmp = "%";
                break;
              case '#' :
                tmp = track_no_str;
                break;
              case 'a' :
                tmp = artist;
                break;
              case 'v' :
                tmp = album;
                break;
              case 'y' :
                tmp = year;
                break;
              case 's' :
                tmp = song;
                break;
              }
            
            target += tmp;            
          }
        else
          {
            target += format[ s ];
          }
      }    
    return 0;
}

char *time_to_readable(time_t sec)
{
    static char buf[ 10 ];

    std::ostringstream tmp;
    tmp << std::setw(2) << (int) sec / 60 << ":" << std::setw(2) << std::setfill('0') << (int) sec % 60;
    buf[tmp.str().copy(buf, 9)] = '\0';
    return buf;
}

char *length_to_readable(unsigned length)
{
    time_t sec;

    sec = (float) length / CD_SECTORS_PER_SEC;
    return time_to_readable(sec);
}

char *expand_tilde(char *path)
{
    static char buf[ MAX_FILE_PATH_LENGTH + MAX_FILE_NAME_LENGTH ];

    if(path[ 0 ] != '~')
    {
        return path;
    }

    strcpy(buf, getenv("HOME"));
    strcpy(buf + strlen(buf), path + 1);
    return buf;
}

char *construct_file_name(const char *path, const char *name)
{
    int offset;
    static char buf[ MAX_FILE_PATH_LENGTH + MAX_FILE_NAME_LENGTH ];

    strcpy(buf, path);
    offset = strlen(buf) - 1;

    if(buf[ offset ] != '/')
    {
        buf[ ++offset ] = '/';
    }

    offset++;

    strcpy(buf + offset, name);
    return buf;
}

std::string file_name_without_path(const std::string& src)
{
    return src.substr(src.rfind('/')+1, src.length()-src.rfind('/')-1);
}

std::string file_path_without_name(const std::string& src)
{
    return src.substr(0, src.rfind('/'));
}

void auto_append_extension(char *src, int type)
{
    int offset;

    offset = strlen(src) - 4;
    /* aasdfasdfasd.wav */
    /*             ^    */

    if(type == WAV)
    {
        if(strcmp(".wav", src + offset) != 0)
        {
            offset += 4;
            strcpy(src + offset, ".wav");
            return;
        }
    }
    else if(type == OGG)
    {
        if(strcmp(".ogg", src + offset) != 0)
        {
            offset += 4;
            strcpy(src + offset, ".ogg");
            return;
        }
    }
    else if(type == FLAC)
    {
        offset--;

        if(strcmp(".flac", src + offset) != 0)
        {
            offset += 5;
            strcpy(src + offset, ".flac");
            return;
        }
    }
    else if(type == OPUS)
    {
        offset--;

        if(strcmp(".opus", src + offset) != 0)
        {
            offset += 5;
            strcpy(src + offset, ".opus");
            return;
        }
    }
    else if(type == MP2)
    {
        if(strcmp(".mp2", src + offset) != 0)
        {
            offset += 4;
            strcpy(src + offset, ".mp2");
            return;
        }
    }
    else if(type == MUSE)
    {
        if(strcmp(".mpc", src + offset) != 0)
        {
            offset += 4;
            strcpy(src + offset, ".mpc");
            return;
        }
    }
    else
    {
        if(strcmp(".mp3", src + offset) != 0)
        {
            offset += 4;
            strcpy(src + offset, ".mp3");
            return;
        }
    }
}

std::string get_default_track_title(const int track)
{
    const std::string name_format = config.encoded_file_name_format;
    int read_offset = -1;
    std::ostringstream ss;    
    while(name_format[ ++read_offset ] != '\0')
    {
      if(name_format[ read_offset ] == '%')
        ss << track+1;
      else
        ss << name_format[ read_offset ];
    }
    return ss.str();
}

char *get_string_piece(FILE *file, int delim)
{
    /* gets one complete row from 'file' and save it in 'buffer'.
    buffer's memory will be freed and allocated to fit the stringsize
    automatically. */

  char *buffer1 = (char *) malloc(1),
    *buffer2 = (char *) malloc(1);
  
  std::array<char, 1024> tmp;
  
  char **active=0, **inactive=0;
  int i = 0;

    strcpy(buffer1, "");
    strcpy(buffer2, "");
    tmp[0] = '\0'; // tmp -> ""
    
    do
    {
        /*switch the frames*/
        if(inactive == &buffer1)
        {
            active = &buffer1;
            inactive = &buffer2;
        }
        else
        {
            active = &buffer2;
            inactive = &buffer1;
        }

        /*get the next part, and handle EOF*/
        if(fgets(tmp.data(), 1024, file) == NULL)
        {
            /* ok, so we reached the end of the
               file w/o finding the delimiter */
            free(*active);
            return NULL;
        }

        free(*active);
        *active = (char *) malloc((++i) * 1024);
        sprintf(*active, "%s%s", *inactive, tmp.data());

    }
    while(strchr(*active, delim) == NULL);

    free(*inactive);
    return *active;
}

char *get_ascii_file(FILE *file)
{
    /* gets one complete row from 'file' and save it in 'buffer'.
    buffer's memory will be freed and allocated to fit the stringsize
    automatically. */

    char *buffer1 = (char *) malloc(1),
      *buffer2 = (char *) malloc(1),
      *tmp = (char *) malloc(1024);
    char **active=0, **inactive=0;
    int i = 0;

    *buffer1 = *buffer2 = *tmp = 0;

    do
    {
        /*switch the frames*/
        if(inactive == &buffer1)
        {
            active = &buffer1;
            inactive = &buffer2;
        }
        else
        {
            active = &buffer2;
            inactive = &buffer1;
        }

        /*get the next part, and handle EOF*/
        if(fgets(tmp, 1024, file) == NULL)
        {
            free(*active);
            return *inactive;
        }

	*active = (char*)realloc(*active, ++i*1024);
        sprintf(*active, "%s%s", *inactive, tmp);
    }
    while(1);
}

void strip_trailing_space(char **string)
{
    int i = strlen(*string) - 1;

    if (string == NULL || *string == NULL)
        return;
}

FILE *socket_init(const std::string& server, short int port)
{
    struct hostent *host;
    struct sockaddr_in socket_address;
    int hsocket;

    /* see if some default values have been set */
    if(server.empty())
    {
        return NULL;
    }

    host = gethostbyname(server.c_str());

    if(host == NULL)
    {
        return NULL;    /* we don't know the host */
    }

    /* set socket address */
    memset(&socket_address, 0, sizeof(socket_address));
    memcpy((char *) & socket_address.sin_addr, host->h_addr, host->h_length);
    socket_address.sin_family = host->h_addrtype;
    socket_address.sin_port = htons(port);

    /* get the actual socket handle */
    hsocket = socket(host->h_addrtype, SOCK_STREAM, 0);

    if(hsocket < 0)           /* we couldn't grab the socket */
    {
        return NULL;
    }

    if(connect(hsocket, (struct sockaddr *) & socket_address,
               sizeof(socket_address)) < 0)
    {
        return NULL;
    }

    return fdopen(hsocket, "r+");            /* we made it */
}

std::string int2str(int integer)
{
    std::ostringstream tmp;
    tmp << integer;
    return tmp.str();
}

long check_free_space(const std::string& dir)
{
    FILE *file;
    long len;
    char cmd[MAX_FILE_NAME_LENGTH];

    snprintf(cmd, MAX_FILE_NAME_LENGTH - 1, "df -P '%s' 2>/dev/null | awk '{print $4}' | tail -1", dir.c_str());
    file = popen(cmd, "r");
    const int ignore_return = fscanf(file, "%ld", &len); /* now read the length */
    pclose(file);
    return len;
}

int check_dir(const std::string &dir)
{
    int rc;
    struct stat ds;

    rc = access(dir.c_str(), F_OK);

    if(rc != 0)
    {
        return MISC_DOES_NOT_EXISTS;
    }

    rc = stat(dir.c_str(), &ds);

    if(rc != 0)
    {
        return MISC_NOT_DIR;
    }

    if(!S_ISDIR(ds.st_mode))
    {
        return MISC_NOT_DIR;
    }

    rc = access(dir.c_str(), X_OK | W_OK | R_OK);

    if(rc != 0)
    {
        return MISC_NOT_WRITABLE;
    }

    return MISC_OK;
}

int create_dir_2 (const std::string& path) 
{
  int rc = mkdir(path.c_str(), 0777);
  if ((rc!=0) and (errno==ENOENT)) 
    {
      //missing parent, create it and try again
      char *tmp = strdup(path.c_str()); 	/* preserve our string */
      char *parent = dirname(tmp);
      create_dir_2(parent);
      free(tmp);
      rc = mkdir(path.c_str(), 0777);
    }
  return rc;
}

int create_dir(const std::string& path)
{
    std::string tmp = path; /* preserve our string */
    std::string parent = dirname(const_cast<char*>(tmp.c_str()));
    int rc = check_dir(parent);

    if(strcmp(parent.c_str(), "/") == 0)
    {
        return -1; 	/* stop endless loop */
    }

    if(rc == MISC_NOT_DIR || rc == MISC_NOT_WRITABLE)
    {
        return -1;
    }
    else if(rc == MISC_DOES_NOT_EXISTS)
    {
        /* recursively create */
        if(create_dir(parent) != 0)
        {
            fprintf(stderr, "failed to create %s\n", parent.c_str());
            return -1;
        }
    }

    rc = mkdir(path.c_str(), 0755);
    return rc;
}

int is_found(const std::string& plugin)
{
    FILE *pf;
    char buffer[MAX_COMMAND_LENGTH];
    char cmd[MAX_COMMAND_LENGTH];

    snprintf(cmd, sizeof(cmd) -1, "%s 2>&1", plugin.c_str());

    pf = popen(cmd, "r");

    if(pf == NULL)
        return 0;

    const char*  ignore_return = fgets(buffer, sizeof(buffer) - 1, pf);
    pclose(pf);

    if(strncmp(buffer, "sh:", 3) != 0)
        return 1;
    return 0;
}

void string_copy(const std::string& in, char *out, const int size)
{
    const int copied = in.copy(out, size - 1);
    if (copied != in.length())
    {
        perror(_("Bug in string copy operation. Input buffer too small!"));
        exit(1);
    }
    out[copied] = '\0';
}

void replace_string_in_place(std::string& subject, const std::string& search, const std::string& replace)
{
    size_t pos = 0;
    while((pos = subject.find(search, pos)) != std::string::npos) {
        subject.replace(pos, search.length(), replace);
        pos += replace.length();
    }
}


#include <algorithm> 
#include <functional> 
#include <cctype>
#include <locale>

// trim from start
std::string &ltrim(std::string &s, const std::string what) {
  std::function<bool(char)> has = [what](char c){ return what.find(c) != std::string::npos; };
  s.erase(s.begin(), std::find_if(s.begin(), s.end(), std::not1(has)));
  return s;
}

// trim from end
std::string &rtrim(std::string &s, const std::string what) {
  std::function<bool(char)> has = [what](char c){ return what.find(c) != std::string::npos; };
  s.erase(std::find_if(s.rbegin(), s.rend(), std::not1(has)).base(), s.end());
  return s;
}

// trim from both ends
std::string &trim(std::string &s, const std::string what) {
  return ltrim(rtrim(s, what), what);
}

static int encoding_ext_vec_index;
static std::vector<CommonEncoderType> encoding_ext_vec;
static std::string wd, ed;
static std::string wfext;

std::vector<CommonEncoderType> get_encoding_types()
{
    return encoding_ext_vec;
}

int get_type_index_for_encoding_file()
{
    return encoding_ext_vec_index;
}

void set_type_index_for_encoding_file(int saved_type_index)
{
    encoding_ext_vec_index = saved_type_index;
}

CommonEncoderType get_type_for_encoding_file()
{
    if (encoding_ext_vec_index >= encoding_ext_vec.size())
    {
        encoding_ext_vec_index = 0;
        //printf("\n+++ get_type_for_encoding_file - RESET type to WAV and index: %i\n", encoding_ext_vec_index);
        return WAV;
    }

    CommonEncoderType *fileext = &encoding_ext_vec[encoding_ext_vec_index];
    encoding_ext_vec_index++;
    //printf("+++ get_type_for_encoding_file - NEXT type index: %i - current type: %i\n", encoding_ext_vec_index, *fileext);

    return *fileext;
}

void load_enabled_encoder_types()
{
    encoding_ext_vec_index = 0;
    encoding_ext_vec.clear();

    if(config.encoder.mp3.enabled && has_plugin_available(MP3))
        encoding_ext_vec.push_back(MP3);
    if(config.encoder.ogg.enabled && has_plugin_available(OGG))
        encoding_ext_vec.push_back(OGG);
    if(config.encoder.flac.enabled && has_plugin_available(FLAC))
        encoding_ext_vec.push_back(FLAC);
    if(config.encoder.mp2.enabled && has_plugin_available(MP2))
        encoding_ext_vec.push_back(MP2);
    if(config.encoder.musepack.enabled && has_plugin_available(MUSE))
        encoding_ext_vec.push_back(MUSE);
    if(config.encoder.opus.enabled && has_plugin_available(OPUS))
        encoding_ext_vec.push_back(OPUS);
}

int create_filenames_from_format(_main_data *main_data)
{
    int i;
    int rc2;
    static std::string df;

    rtrim(config.wav_path, "/ ");
    rtrim(config.mp3_path, "/ ");

    wd = config.wav_path + "/";
    ed = config.mp3_path + "/";

    if(config.cddb_config.make_directories && !config.cddb_config.dir_format_string.empty())
    {
        rc2 = parse_rx_format_string(
                df,
                config.cddb_config.dir_format_string, -1,
                main_data->disc_artist, main_data->disc_title, main_data->disc_year, "");

        if(rc2 < 0)
        {
            err_handler(RX_PARSING_ERR, "Check if the directory format string contains format characters other than %a %# %v %y or %s.");
            return 0;
        }

        //remove_non_unix_characters(df);

#ifdef DEBUG
        std::cout << "\ncreate_filenames_from_format df=\'" << df << "\' " <<std::endl;
#endif
        
        if(config.cddb_config.convert_spaces == true)
            std::replace( df.begin(), df.end(), ' ', '_');

        if(df.length() > 0)
        {
            wd += df + "/";
            ed += df + "/";

#ifdef DEBUG
            std::cout << "create_dirs: wd=\'" << wd << "\' " << std::endl;
            std::cout << "             ed=\'" << ed << "\' " << std::endl;
#endif

            create_dir(wd);
            create_dir(ed);
        }
    }

    wfext = (config.auto_append_extension == true) ? ".wav" : "";

    for (i = 0; i < main_data->num_tracks(); i++)
    {
        main_data->track[i].encoding_files.clear();
        main_data->track[i].current_type = NO_TYPE;

        for (const auto it_type : encoding_ext_vec)
        {
            struct EncodingFiles enc_file;
            enc_file.type = it_type;
            enc_file.encoded_file_exist = false;
            main_data->track[i].encoding_files.push_back(enc_file);
        }
    }

    return 1;
}

int create_file_names_for_track(_main_data *main_data, const int track,
                                std::string& wfp, std::string& efp, CommonEncoderType type)
{
    static std::string buffer;
    int rc;
    std::string conv_str;
    std::string track_artist;
    std::string ecfext;
    std::ostringstream tmp;

    // if (track < 0)
    // {
    //     err_handler(WAV_PATH_CREATE_ERROR, _("Invalid track number"));
    //     return 0;
    // }

    if(!main_data->track[ track ].artist.empty())
        track_artist = main_data->track[ track ].artist;
    else
        track_artist = main_data->disc_artist;

    rc = parse_rx_format_string(buffer,
                                config.cddb_config.format_string, track,
                                track_artist, main_data->disc_title, main_data->disc_year,
                                main_data->track[ track ].title);
    if(rc < 0)
    {
        err_handler(RX_PARSING_ERR, _("Check if the filename format string contains format characters other than %a %# %v %y or %s."));
        return 0;
    }

    if(buffer.empty())
        buffer = main_data->track[ track ].title;

    conv_str = g_locale_from_utf8(buffer.c_str(), -1, NULL, NULL, NULL);
    
    //remove_non_unix_characters(conv_str);
    std::replace( conv_str.begin(), conv_str.end(), '/', '-');

    if(config.cddb_config.convert_spaces == true)
        std::replace( conv_str.begin(), conv_str.end(), ' ', '_');

    tmp << wd << conv_str << wfext;
    wfp = tmp.str();

    // Encoding...
    ecfext = "";
    main_data->track[track].current_type = type;

    if (type > WAV)
    {
        if (config.auto_append_extension == true)
            ecfext = encoder_type_config_map.at(type).extension;

        auto it = std::find_if(main_data->track[track].encoding_files.begin(), main_data->track[track].encoding_files.end(),
              [&type](const EncodingFiles &f) { return f.type == type; });
        main_data->track[track].current_file = std::distance(main_data->track[track].encoding_files.begin(), it);
    }
    tmp.str("");
    tmp << ed << conv_str << ecfext;
    efp = tmp.str();

#ifdef DEBUG
    std::cout << "create_file_names_for_track: wfp=\'" << wfp << "\'" << std::endl;
    std::cout << "                             type=\'" << type << "\'" << std::endl;
    std::cout << "                             efp=\'" << efp << "\'" << std::endl;
    printf("\t\tcurrent_type: %i\n", main_data->track[track].current_type);
#endif
    
    return 1;
}

/*
** get_track_title - Copy track artist & title for specified track from
**                   main_data to a specified location in freedb format.
*/
void get_track_title(std::string& dest, _main_data *main_data, int tno)
{
    const MainData::Track& tk_p = main_data->track[tno];

    if(!tk_p.artist.empty())
    {
        dest = tk_p.artist + " / " + tk_p.title;
    }
    else
    {
        dest  = tk_p.title;
    }
}

/*
** put_track_title - Put track artist & title in freedb format into
**                   main_data for specified track number.
** NOTE: A track artist can be specified
*/
void put_track_title(const std::string& src, _main_data *main_data, int tno)
{
    if (main_data->track.size()<=tno) {
      std::cout << "put_track_title: different number of tracks from CDDB compared to CD!" << std::endl;
      exit(-1); // this should never happen...
    }
    
    MainData::Track& tk_p = main_data->track[tno];
    std::string sp = src;
    
    /* Strip leading blanks */
    trim(sp, " ");

    /* Split off track artist, if specified */
    if( sp.rfind('/') != std::string::npos )
    {
      tk_p.artist = sp.substr(0, sp.rfind('/'));
      trim(tk_p.artist, " ");
    }
    else
    {
        tk_p.artist = "";
    }

    tk_p.title = sp.substr(sp.rfind('/')+1, sp.length()-sp.rfind('/')-1);
    trim(tk_p.title);
}

/*
 * Set plugins location:
 * for debugging purposes and for Debian alternative plugin location
 */
void set_plugins_path()
{
    char plugins_path[MAX_COMMAND_LENGTH];
    char *path = getenv("PATH");

#ifdef DEBUG
    getcwd(plugins_path, sizeof(plugins_path));
    strcat(plugins_path, "/plugins:");
#else
    strcpy(plugins_path, "/usr/lib/ripperx:");
#endif

    char *found = strstr(path, plugins_path);

    if (found == NULL)        // Only add the path if it isn't already present
    {
        strcat(plugins_path, getenv("PATH"));
        setenv("PATH", plugins_path, 1);
    }
}

EncoderPlugins *get_encoder_plugins()
{
    return encoder_plugins;
}

void load_plugins_state()
{
    for (int i = 0; i < encoder_plugins_count; i++)
        encoder_plugins[i].available = is_found(encoder_plugins[i].encoder) ? 1 : 0;
}

int has_plugin_available(CommonEncoderType type)
{
    for (int i = 0; i < encoder_plugins_count; i++)
    {
        if (encoder_plugins[i].encoding_type != type)
            continue;
        if (encoder_plugins[i].available)
            return 1;
    }
    return 0;
}
