/* lexical scanner
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "cstringf.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "regalloc.h"
#include "reswords.h"
#include "compdriver.h"
#include "parser.h"
#include "treetools.h"
#include "showprog.h"

charp_t rtl[] = {
    "-- rtl : jal-internal run-time library",
    "",
    "const target_clock  = 0",
    "const target_fuses  = 0",
    "const target_cpu    = 0",
    "const pic_12        = 1",
    "const pic_14        = 2",
    "const pic_16        = 3",
    "const sx_12         = 4",
    "const target_chip   = 0",
    "const pic_16c84     = 1",
    "const pic_16f84     = 2",
    "const pic_16f877    = 3",
    "const sx18          = 4",
    "const sx28          = 5",
    "const pic_12c509a   = 6",
    "const pic_12c509    = 6",
    "const pic_12c508    = 7",
    "const pic_12ce674   = 8",
    "const pic_16f628    = 9",
    "const pic_18f242    = 10",
    "const pic_18f252    = 11",
    "const pic_18f442    = 12",
    "const pic_18f452    = 13",
    "const pic_12f629    = 14",
    "const pic_12f675    = 15",
    "const pic_16f88     = 16",
    "const pic_16f876    = 17", /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    "const pic_16f873    = 18", /* Added 16f876 & 16f873 by Javi 2003-03-01 */

    "",
    "const bit on   = true   -- 'build-in' aliases",
    "const bit off  = false",
    "const bit high = true",
    "const bit low  = false",
    "const bit yes  = true",
    "const bit no   = false",
    "",
    "",
    "-- build in monadic and diadic bit",
    "operator !  ( bit in a ) return bit is end operator",
    "operator &  ( bit in a, bit in  b ) return bit is end operator",
    "operator |  ( bit in a, bit in  b ) return bit is end operator",
    "operator ^  ( bit in a, bit in  b ) return bit is end operator",
    "operator == ( bit in a, bit in  b ) return bit is end operator",
    "operator != ( bit in a, bit in  b ) return bit is end operator",
    "",
    "-- build-in monadic and diadic byte   ",
    "operator -  ( byte in a ) return byte is end operator   ",
    "operator !  ( byte in a ) return byte is end operator   ",
    "operator +  ( byte in a, byte in  b ) return byte is end operator   ",
    "operator -  ( byte in a, byte in  b ) return byte is end operator   ",
    "operator &  ( byte in a, byte in  b ) return byte is end operator   ",
    "operator |  ( byte in a, byte in  b ) return byte is end operator   ",
    "operator ^  ( byte in a, byte in  b ) return byte is end operator   ",
    "operator >  ( byte in a, byte in  b ) return bit is end operator   ",
    "operator >= ( byte in a, byte in  b ) return bit is end operator   ",
    "operator <  ( byte in a, byte in  b ) return bit is end operator   ",
    "operator <= ( byte in a, byte in  b ) return bit is end operator   ",
    "operator == ( byte in a, byte in  b ) return bit is end operator   ",
    "operator != ( byte in a, byte in  b ) return bit is end operator   ",
    "",
    "",
    "-- build-in monadic and diadic word   ",
    "operator -  ( word in a ) return word is end operator   ",
    "operator !  ( word in a ) return word is end operator   ",
    "operator +  ( word in a, word in  b ) return word is end operator   ",
    "operator -  ( word in a, word in  b ) return word is end operator   ",
    "operator &  ( word in a, word in  b ) return word is end operator   ",
    "operator |  ( word in a, word in  b ) return word is end operator   ",
    "operator ^  ( word in a, word in  b ) return word is end operator   ",
    "operator >  ( word in a, word in  b ) return bit is end operator   ",
    "operator >= ( word in a, word in  b ) return bit is end operator   ",
    "operator <  ( word in a, word in  b ) return bit is end operator   ",
    "operator <= ( word in a, word in  b ) return bit is end operator   ",
    "operator == ( word in a, word in  b ) return bit is end operator   ",
    "operator != ( word in a, word in  b ) return bit is end operator   ",
    "",
    "",

    NULL
};
charp_t rtl_trailer[] = {
    "assembler",
    "  local idle_loop",
    "  idle_loop: page goto idle_loop",
    "end assembler",
    "",
    NULL
};

charp_t rtl_trailer_16[] = {
    "assembler",
    "  local idle_loop",
    "  idle_loop: goto idle_loop",
    "end assembler",
    "",
    NULL
};

/* intial context, used for post-scanning error reporting */
scanner_context_t scanner_initial_context = {
    NULL,
    "",
    0,
    0,
    "",
    true,
    false,
    NULL,
    NULL,
    "",
    NULL,
    ""
};
scanner_context_t *scanner_context = &scanner_initial_context;
string scanner_progress;

void print_src(char *s)
{
    if (src_file != NULL) {
        fprintf(src_file, "%s\n", s);
    }
}

/* advance one char in the input */
void scanner_advance_char(void)
{
    scanner_context_t *p = scanner_context;
    string s;
    stack_guard;
    if (p->eoi)
        return;

    /* add the next char to the token */
    p->token[p->loc->token_size++] = p->line[p->pos++];
    p->token[p->loc->token_size] = '\0';

    if (verbose_scanner) {
        log((m, "c='%c' t='%s' l='%s'", p->token[p->loc->token_size - 1], p->token, p->line));
    }

    /* when no next char available on the same line */
    /* proceed to get the next line */
    if (p->pos <= p->len)
        return;

    if (verbose_scanner) {
        log((m, "n=%d p=%d l=%d ef=%d ei=%d", p->loc->line_nr, p->pos, p->len, p->eof, p->eoi));
    }

    /* this file exhausted? */
    if (p->eof) {

        if (p->f != NULL) {
            print_src(scanner_context->src_line);
            print_src("-- ======================================");
            sprintf(s, "-- end %s", scanner_context->file_name);
            print_src(s);
            print_src("-- ======================================");
            fclose(p->f);
        }

        scanner_context = p->previous;
        sprintf(scanner_progress, "parsing %s line", scanner_context->file_name);
        if (scanner_context->previous == NULL) {
            scanner_context->eof = true;
            scanner_context->eoi = true;
        }
        return;
    }

    /* get next line */
    if (p->internal != NULL) {
        if (*p->internal == NULL) {
            p->line = "";
            p->eof = true;
        } else {
            p->line = *(p->internal);
            (p->internal)++;
        }
    } else {
        string s;
        print_src(p->src_line);
        get_line(p->f, s, &p->eof);
        p->line = new_string(s);
        sprintf(p->src_line, s);
    }
    p->loc->line_nr++;
    p->loc->line = p->line;
    p->len = string_length(p->line);
    p->pos = 0;
    scanner_total_lines++;
    scanner_total_chars += p->len;
    progress(scanner_progress, scanner_total_lines);
}

/* return the next char in the input */
char scanner_next_char(void)
{
    scanner_context_t *p = scanner_context;
    stack_guard;
    if (verbose_scanner) {
        log((m, "scanner_next_char='%c'", p->line[p->pos]));
    }
    if (p->eoi)
        return ' ';
    if (p->pos >= p->len)
        return ' ';
    return p->line[p->pos];
}

boolean scanner_at_end_of_line(void)
{
    scanner_context_t *p = scanner_context;
    stack_guard;
    return p->pos >= p->len;
}

/* advance one token in the input */
void scanner_next_token(void)
{

    /* create a new context record */
    loc_t p = allocate(sizeof(loc_r));
    *p = *scanner_context->loc;
    scanner_context->loc = p;
    stack_guard;

    /* skip whitespace */
    if (verbose_scanner) {
        log((m, "start skipping whitespace"));
    }
    while (is_separator(scanner_next_char())) {
        if (scanner_context->eoi)
            return;
        scanner_advance_char();
    }

    /* save token start */
    if (verbose_scanner) {
        log((m, "start new token"));
    }
    assert_pointer(NULL, scanner_context->loc);
    scanner_context->loc->token_start = scanner_context->pos + 1;
    scanner_context->loc->token_size = 0;

    /* get the token */
    if (is_id_start(scanner_next_char())) {

        /* get all consequtive letters and numbers etc. */
        scanner_advance_char();
        while (is_id_next(scanner_next_char())) {
            scanner_advance_char();
        }

    } else if (is_number_start(scanner_next_char())) {

        /* get all consequtive numbers or letters etc. 
           after the first number */
        scanner_advance_char();
        while (is_number_next(scanner_next_char())) {
            scanner_advance_char();
        }

    } else if (scanner_next_char() == '"') {

        /* get all consequtive characters 
           untill and including a '"'. */
        scanner_advance_char();
        for (;;) {
            if (scanner_context->pos == scanner_context->len) {
                break;
            }
            if ('"' == scanner_next_char()) {
                scanner_advance_char();
                break;
            }
            scanner_advance_char();
        }


    } else if (is_special(scanner_next_char())) {

        /* handle special-character operators */
        scanner_advance_char();
        while (is_special(scanner_next_char())
               || is_number(scanner_next_char())
               || is_letter(scanner_next_char())
            ) {
            scanner_advance_char();
        }

    } else {

        /* just return the one char token */
        scanner_advance_char();

    }

    /* when the token is a comment: skip the remainder of the line */
    if (string_match(scanner_context->token, s_comment)
        || string_match(scanner_context->token, s_semicolon)
        ) {
        while (!scanner_at_end_of_line()) {
            scanner_advance_char();
        }
        scanner_next_token();
    }

    if (verbose_scanner) {
        log((m, "token='%s'", scanner_context->token));
    }
}

void scanner_open(charp_t * internal, char *file_name, boolean save)
{
    scanner_context_t *new = allocate(sizeof(scanner_context_t));
    string s;
    stack_guard;

    if (verbose_scanner) {
        log((m, "scanner open i=%d n=%s", internal, file_name));
    }

    /* internal or external source? */
    if (internal == NULL) {
        new->f = fopen(file_name, "r");
        jal_assert(NULL, (new->f != NULL));
        sprintf(scanner_progress, "parsing %s line", file_name);
        string_copy(new->file_name, file_name);
        chop_extension(new->file_name);
        chop_path(new->file_name);
    } else {
        sprintf(new->file_name, "");
        sprintf(scanner_progress, "parsing jal-rtl line");
        new->f = NULL; 
    }

    /* initialize */
    new->internal = internal;
    new->loc = allocate(sizeof(loc_r));
    new->pos = 0;
    new->len = 0;
    new->eof = false;
    new->eoi = false;
    new->loc->line_nr = 0;
    new->loc->file_name = new_string(file_name);
    new->loc->token_start = 0;
    new->loc->token_size = 0;
    new->loc->line = NULL;
    new->line = "";
    new->src_line[0] = '\0';

    /* handle include line for src file */
    if (internal == NULL) {
        sprintf(s, "-- %s", scanner_context->src_line);
        print_src(s);
        print_src("-- ======================================");
        sprintf(s, "-- start %s", file_name);
        print_src(s);
        print_src("-- ======================================");
        scanner_context->src_line[0] = '\0';
    }

    /* put new source into the chain of sources */
    new->previous = scanner_context;
    scanner_context = new;

    /* get the first line and first token */
    scanner_advance_char();
    scanner_next_token();

    scanner_total_files++;
}

/* report whether the token matches s */
boolean scanner_is_like(char *s)
{
    boolean match = string_match(s, scanner_context->token);
    stack_guard;
    if (verbose_scanner) {
        log((m, "input like token=%s s=%s match=%d", scanner_context->token, s, match));
    }
    return match;
}

/* report whether the token matches s, advance token if match */
boolean scanner_is_like_advance(char *s)
{
    stack_guard;
    if (scanner_is_like(s)) {
        scanner_next_token();
        return true;
    }
    return false;
}

/* cassert that the token matches s */
void scanner_check(char *s)
{
    stack_guard;
    if (!scanner_is_like(s)) {
        /* t0026 */
        cfatal((m, "expected '%s' but found '%s'", s, scanner_context->token));
    }
}

/* cassert that the token matches s and advance */
void scanner_check_advance(char *s)
{
    stack_guard;
    scanner_check(s);
    scanner_next_token();
}

/* report whether the token is an identifier */
boolean scanner_is_identifier(void)
{
    stack_guard;
    return is_id_start(scanner_context->token[0]);
}

/* cassert that the token is an identifier */
void scanner_check_identifier(void)
{
    stack_guard;
    if (!scanner_is_identifier()) {
        /* t0027 */
        cfatal((m, "identifier expected"));
    }
}

/* cassert that the token is an identifier and advance */
void scanner_check_identifier_advance(void)
{
    stack_guard;
    scanner_check_identifier();
    scanner_next_token();
}

/* return current location */
loc_t scanner_here(void)
{
    stack_guard;
    return scanner_context->loc;
}
