﻿#region License
// Copyright (c) 2007 James Newton-King
//
// Permission is hereby granted, free of charge, to any person
// obtaining a copy of this software and associated documentation
// files (the "Software"), to deal in the Software without
// restriction, including without limitation the rights to use,
// copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following
// conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
// OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
// HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
// WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
#endregion

using System;
using System.Collections.Generic;
#if NET20
using Newtonsoft.Json.Utilities.LinqBridge;
#endif
using System.Threading;
using Newtonsoft.Json.Serialization;

namespace Newtonsoft.Json.Utilities
{
  internal class ThreadSafeStore<TKey, TValue>
  {
    private readonly object _lock = new object();
    private Dictionary<TKey, TValue> _store;
    private readonly Func<TKey, TValue> _creator;

    public ThreadSafeStore(Func<TKey, TValue> creator)
    {
      if (creator == null)
        throw new ArgumentNullException("creator");

      _creator = creator;
      _store = new Dictionary<TKey, TValue>();
    }

    public TValue Get(TKey key)
    {
      TValue value;
      if (!_store.TryGetValue(key, out value))
        return AddValue(key);

      return value;
    }

    private TValue AddValue(TKey key)
    {
      TValue value = _creator(key);

      lock (_lock)
      {
        if (_store == null)
        {
          _store = new Dictionary<TKey, TValue>();
          _store[key] = value;
        }
        else
        {
          // double check locking
          TValue checkValue;
          if (_store.TryGetValue(key, out checkValue))
            return checkValue;

          Dictionary<TKey, TValue> newStore = new Dictionary<TKey, TValue>(_store);
          newStore[key] = value;

#if !NETFX_CORE
          Thread.MemoryBarrier();
#endif
          _store = newStore;
        }

        return value;
      }
    }
  }
}