/*
 * main.c - oTo main
 *
    Copyright (C) 2001  Yao Zhang

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * yaoz@users.sourceforge.net
 */
#include <getopt.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ot.h"

void ot_dump(char *file, int in_hex)
{
    int i;
    FontDirectory *fontDirectory;
    NameTable *nameTable;
    CMapTable *cmapTable;

    fontDirectory = new_font_directory();
    read_font_directory(file, fontDirectory);
    if (fontDirectory->offsetSubtable.numTables <= 0) {
        return;
    }

    printf("File: %s\n", file);
    printf("\n");

    printf("Offset Subtable: \n");
    print_offset_subtable(&fontDirectory->offsetSubtable);
    printf("\n");
    printf("Table Directory: \n");
    for (i = 0; i < fontDirectory->offsetSubtable.numTables; i++) {
        print_table_directory_entry(&fontDirectory->tableDirectory[i]);
    }
    printf("\n");

    nameTable  = new_name_table();
    read_name_table(file, fontDirectory, nameTable);
    printf("Naming Table: \n");
    printf("Name Table Header: \n");
    print_name_table_header(&nameTable->nameTableHeader);
    printf("Name Records: \n");
    for (i = 0; i < nameTable->nameTableHeader.count; i++) {
        print_name_record_entry(&nameTable->nameArray[i].nameRecord);
        print_name_string(&nameTable->nameArray[i], in_hex);
    }
    printf("\n");
    delete_name_table(nameTable);

    cmapTable = new_cmap_table();
    read_cmap_table(file, fontDirectory, cmapTable);
    printf("CMap Table: \n");
    printf("CMap Index: \n");
    print_cmap_index(&cmapTable->cmapIndex);
    printf("CMap Encoding Subtables: \n");
    for (i = 0; i < cmapTable->cmapIndex.numberSubtables; i++) {
        print_cmap_encoding_and_subtable(&cmapTable->cmapArray[i]);
    }
    printf("\n");
    delete_cmap_table(cmapTable);

    delete_font_directory(fontDirectory);
}

void ot_conv(char *in, char *out, char *naming_spec)
{
    register uint8_t *p;
    uint8_t s[16];
    int i, numTables;
    FontDirectory *inDirectory, *outDirectory;
    int offset;
    FILE *fp;
    CMapTable *cmapTable;
    int head_offset;

    cmapTable = NULL;
    head_offset = 0;

    inDirectory = new_font_directory();
    outDirectory = new_font_directory();

    read_font_directory(in, inDirectory);
    if (inDirectory->offsetSubtable.numTables <= 0) {
        return;
    }
    read_font_directory(in, outDirectory);

    numTables = inDirectory->offsetSubtable.numTables;

    /* First phase: calculate check sums, etc. */
    i = floor(log(numTables)/log(2));
    outDirectory->offsetSubtable.searchRange = 16*(1<<i);
    outDirectory->offsetSubtable.entrySelector = i;
    outDirectory->offsetSubtable.rangeShift = 16*numTables - 16*(1<<i);

    offset = 4+4*2 + numTables*(4*4);
    for (i = 0; i < numTables; i++) {
        TableDirectoryEntry *inDirectoryEntry;
        TableDirectoryEntry *outDirectoryEntry;

        inDirectoryEntry = &inDirectory->tableDirectory[i];
        outDirectoryEntry = &outDirectory->tableDirectory[i];

        if (strncmp((const char *)inDirectoryEntry->tag, CMAP_TAG, 4) == 0) {
            uint32_t length;
            uint8_t *buffer;

            cmapTable = new_cmap_table();
            read_cmap_table(in, inDirectory, cmapTable);
            add_unicode_cmap(cmapTable,
                             &buffer, &length);

            outDirectoryEntry->checkSum = calc_table_checksum(buffer, length);
            outDirectoryEntry->offset = offset;
            outDirectoryEntry->length = length;

            free(buffer);

            offset += ((length+3)/4)*4;
        } else if (strncmp((const char *)inDirectoryEntry->tag, HEAD_TAG, 4) == 0) {
            uint32_t length;
            uint8_t *buffer;

            length = inDirectoryEntry->length;
            buffer = (uint8_t *)calloc(length, 1);
            read_table(buffer, length,
                       in, inDirectoryEntry->offset);
            p = buffer+2*4;
            SET4B(p, 0);	/* Set checkSumAdjustment to 0 */

            outDirectoryEntry->checkSum = calc_table_checksum(buffer, length);
            outDirectoryEntry->offset = offset;
            outDirectoryEntry->length = length;

            head_offset = offset;

            free(buffer);

            offset += ((length+3)/4)*4;
        } else if (strncmp((const char *)inDirectoryEntry->tag, NAME_TAG, 4) == 0) {
            NameTable *nameTable;
            uint32_t length;
            uint8_t *buffer;

            nameTable = new_name_table();
            read_name_table(in, inDirectory, nameTable);
            if (naming_spec == NULL) {
                add_utf8_names(nameTable);
            } else {
                replace_names(nameTable, naming_spec);
            }
            length = nameTable->length;
            buffer = nameTable->buffer;

            outDirectoryEntry->checkSum = calc_table_checksum(buffer, length);
            outDirectoryEntry->offset = offset;
            outDirectoryEntry->length = length;

            delete_name_table(nameTable);

            offset += ((length+3)/4)*4;
        } else {
            uint32_t length;
            uint8_t *buffer;

            length = inDirectoryEntry->length;
            buffer = (uint8_t *)calloc(length, 1);
            read_table(buffer, length,
                       in, inDirectoryEntry->offset);

            outDirectoryEntry->checkSum = calc_table_checksum(buffer, length);
            outDirectoryEntry->offset = offset;
            outDirectoryEntry->length = length;

            free(buffer);

            offset += ((length+3)/4)*4;
        }
    }

    /* Second phase: write to font file. */
    fp = fopen(out, "w");

    /* Output Offset Subtable */
    p = s;
    SET4B(p, outDirectory->offsetSubtable.version);
    fwrite(s, 4, 1, fp);
    p = s;
    SET2B(p, outDirectory->offsetSubtable.numTables);
    fwrite(s, 2, 1, fp);
    p = s;
    SET2B(p, outDirectory->offsetSubtable.searchRange);
    fwrite(s, 2, 1, fp);
    p = s;
    SET2B(p, outDirectory->offsetSubtable.entrySelector);
    fwrite(s, 2, 1, fp);
    p = s;
    SET2B(p, outDirectory->offsetSubtable.rangeShift);
    fwrite(s, 2, 1, fp);

    /* Output Table Directory Entries */
    for (i = 0; i < numTables; i++) {
        TableDirectoryEntry *outDirectoryEntry;

        outDirectoryEntry = &outDirectory->tableDirectory[i];
        fwrite(outDirectoryEntry->tag, 4, 1, fp);
        p = s;
        SET4B(p, outDirectoryEntry->checkSum);
        fwrite(s, 4, 1, fp);
        p = s;
        SET4B(p, outDirectoryEntry->offset);
        fwrite(s, 4, 1, fp);
        p = s;
        SET4B(p, outDirectoryEntry->length);
        fwrite(s, 4, 1, fp);
    }

    /* Output Tables */
    for (i = 0; i < numTables; i++) {
        TableDirectoryEntry *inDirectoryEntry;
        TableDirectoryEntry *outDirectoryEntry;

        inDirectoryEntry = &inDirectory->tableDirectory[i];
        outDirectoryEntry = &outDirectory->tableDirectory[i];

        if (strncmp((const char *)inDirectoryEntry->tag, CMAP_TAG, 4) == 0) {
            uint32_t length;
            uint8_t *buffer;

            if (cmapTable == NULL) {
                cmapTable = new_cmap_table();
                read_cmap_table(in, inDirectory, cmapTable);
            }
            add_unicode_cmap(cmapTable,
                             &buffer, &length);

            fwrite(buffer, 1, length, fp);

            free(buffer);

            if ((length%4) != 0) {
                memset(s, 0, 4);
                fwrite(s, 1, 4-(length%4), fp);
            }
        } else if (strncmp((const char *)inDirectoryEntry->tag, HEAD_TAG, 4) == 0) {
            uint32_t length;
            uint8_t *buffer;

            length = inDirectoryEntry->length;
            buffer = (uint8_t *)calloc(length, 1);
            read_table(buffer, length,
                       in, inDirectoryEntry->offset);
            p = buffer+2*4;
            SET4B(p, 0);	/* Set checkSumAdjustment to 0 */

            fwrite(buffer, 1, length, fp);

            free(buffer);

            if ((length%4) != 0) {
                memset(s, 0, 4);
                fwrite(s, 1, 4-(length%4), fp);
            }
        } else if (strncmp((const char *)inDirectoryEntry->tag, NAME_TAG, 4) == 0) {
            NameTable *nameTable;
            uint32_t length;
            uint8_t *buffer;

            nameTable = new_name_table();
            read_name_table(in, inDirectory, nameTable);
            if (naming_spec == NULL) {
                add_utf8_names(nameTable);
            } else {
                replace_names(nameTable, naming_spec);
            }
            length = nameTable->length;
            buffer = nameTable->buffer;

            fwrite(buffer, 1, length, fp);

            delete_name_table(nameTable);

            if ((length%4) != 0) {
                memset(s, 0, 4);
                fwrite(s, 1, 4-(length%4), fp);
            }
        } else {
            uint32_t length;
            uint8_t *buffer;

            length = inDirectoryEntry->length;
            buffer = (uint8_t *)calloc(length, 1);
            read_table(buffer, length,
                       in, inDirectoryEntry->offset);

            fwrite(buffer, 1, length, fp);

            free(buffer);

            if ((length%4) != 0) {
                memset(s, 0, 4);
                fwrite(s, 1, 4-(length%4), fp);
            }
        }
    }

    fclose(fp);

    delete_font_directory(inDirectory);
    delete_font_directory(outDirectory);

    if (cmapTable != NULL) {
        delete_cmap_table(cmapTable);
    }

    /* Calculate check sum of the whole font file. */
    if (head_offset > 0) {
        uint32_t length;
        uint8_t *buffer, s[4];
        uint32_t checkSum;

        fp = fopen(out, "r+");

        fseek(fp, 0, SEEK_END);
        length = ftell(fp);
        buffer = (uint8_t *)calloc(length, 1);
        fseek(fp, 0, SEEK_SET);
        fread(buffer, 1, length, fp);
        checkSum = calc_table_checksum(buffer, length);
        free(buffer);

        /* Write out 'head' table's checkSumAdjustment */
        checkSum -= 0xB1B0AFBA;
        p = s;
        SET4B(p, checkSum);
        fseek(fp, head_offset+2*4, SEEK_SET);
        fwrite(s, 1, 4, fp);

        fclose(fp);
    }
}

int main(int argc, char *argv[])
{
    int in_hex = 0;
    char *naming_spec = NULL;

    while (1) {
        int c;
        static struct option longopts[] = {
            {"names", 1, NULL, 'n'},
            {"hex", 0, NULL, 'x'},
            {NULL, 0, NULL, 0}
        };
        int longindex;

        c = getopt_long(argc, argv, "n:x",
                        longopts, &longindex);
        if (c == -1) {
            break;
        }

        switch (c) {
        case 'n':
            naming_spec = optarg;
            break;
        case 'x':
            in_hex = 1;
            break;
        case '?':
            break;
        default:
            printf("?? getopt returned character code 0x%X ??\n", c);
            break;
        }
    }

    switch (argc-optind) {
    case 1:
        ot_dump(argv[optind], in_hex);
        break;
    case 2:
        ot_conv(argv[optind], argv[optind+1], naming_spec);
        break;
    default:
        printf("Usage: oto [OPTION...] in-ttf-file [out-ttf-file]\n");
        printf("List/translate TrueType font names and/or cmap.\n");
        printf("\n");
        printf("  -n, --names=FILE        get naming table from file\n");
        printf("  -x, --hex               print name strings in hex\n");
        return 1;
    }

    return 0;
}
