/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include <usb.h>

#include "razertool.h"
#include "razerlib.h"
#include "razerfirmware.h"

static void do_help(const char *path)
{
	fprintf(stderr,
		"WARNING: USE OF THIS PROGRAM MAY DAMAGE YOUR MOUSE\n"
		"\n"
		"razerflash " VERSION ", Copyright (C) 2006  Christopher Lais\n"
		"razerflash comes with ABSOLUTELY NO WARRANTY; refer to the\n"
		"GNU GPL version 2 or later for details.\n"
		"\n"
		"WARNING: USE OF THIS PROGRAM MAY DAMAGE YOUR MOUSE\n"
		"\n"
		"To flash your Razer Copperhead(TM) using razerflash, plug\n"
		"it in while holding down the reset button, then run:\n"
		"\n"
		"%s -F <file.s19>\n"
		"\n"
		"DO NOT UNPLUG THE DEVICE OR POWER OFF THE COMPUTER WHILE\n"
		"FLASHING.  BATTERY BACKUP IS STRONGLY RECOMMENDED.\n"
		"\n"
		"After flashing, unplug it and plug it in normally.\n"
		"\n"
		"WARNING: USE OF THIS PROGRAM MAY DAMAGE YOUR MOUSE\n",
		path
	);
}

static void do_load_firmware(const char *filename, razer_firmware_t *fw)
{
	int fd;
	int status;

	fd = open(filename, O_RDONLY);
	if (!fd) {
		perror("ERROR: open()");
		exit(-1);
	}

	status = razer_firmware_parse_srec(fd, fw);
	if (status) {
		fprintf(stderr, "Invalid firmware\n");
		exit(-1);
	}

	close(fd);
}

static const char *progress_action_to_str(int action)
{
	switch (action) {
	  case RAZER_PROGRESS_READING: return "Reading";
	  case RAZER_PROGRESS_VERIFYING: return "Verifying";
	  case RAZER_PROGRESS_VERIFY_ERROR: return "Verification failed";
	  case RAZER_PROGRESS_ERASING: return "Erasing";
	  case RAZER_PROGRESS_PROGRAMMING: return "Programming";
	}
	return "";
}

static void razerflash_progress(int action, int start_block, int end_block, double fraction)
{
	static int last_action = RAZER_PROGRESS_DONE;

	if (action == RAZER_PROGRESS_VERIFY_ERROR) {
		fprintf(stderr, "\n%s: %04x-%04x\n", progress_action_to_str(action), start_block, end_block);
		return;
	}

	if (action != last_action) {
		if (last_action != RAZER_PROGRESS_DONE)
			fprintf(stderr, "\r%s: done!              \n", progress_action_to_str(last_action));
		last_action = action;
	}

	if (action == RAZER_PROGRESS_DONE)
		return;

	fprintf(stderr, "\r%s: %04x-%04x [%6.2f%%]", progress_action_to_str(action), start_block, end_block, 100 * fraction);
}

enum {
	RF_SHOW_HELP,
	RF_DUMP_FIRMWARE,
	RF_VERIFY_FIRMWARE,
	RF_FLASH_FIRMWARE,
	RF_READ_LASER,
	RF_WRITE_LASER,
};

#define YES_STRING	"Gouge my eyes"

int main(int argc, char *argv[])
{
	struct usb_device *dev;
	struct usb_dev_handle *udev;
	int status;
	int stat;
	int action = RF_SHOW_HELP;
	int laserval;

	razer_firmware_t fw;

	usb_init();

	if (argc == 1+1 && !strcmp(argv[1], "-d")) {
		action = RF_DUMP_FIRMWARE;
		memset(fw.mem, 0xff, sizeof(fw.mem));
	}

	if (argc == 1+2 && !strcmp(argv[1], "-d")) {
		do_load_firmware(argv[2], &fw);
		action = RF_DUMP_FIRMWARE;
	}

	if (argc == 1+2 && !strcmp(argv[1], "-v")) {
		do_load_firmware(argv[2], &fw);
		action = RF_VERIFY_FIRMWARE;
	}

	if (argc == 1+2 && !strcmp(argv[1], "-F")) {
		do_load_firmware(argv[2], &fw);
		action = RF_FLASH_FIRMWARE;
	}

	if (argc == 1+1 && !strcmp(argv[1], "-l")) {
		action = RF_READ_LASER;
	}

	if (argc == 1+2 && !strcmp(argv[1], "-L")) {
		char *err = NULL;
		char yes[64];

		/* Lower is higher power. */
		laserval = strtoul(argv[2], &err, 16);

		if (!*argv[2] || strlen(argv[2]) != 2 || (err && *err) || laserval < 0x00 || laserval > 0xff) {
			fprintf(stderr, "Invalid laser value: %s\n", argv[2]);
			return -1;
		}

		if (!isatty(STDOUT_FILENO)) {
			fprintf(stderr, "STDOUT must be a tty\n");
			return -1;
		}

		fprintf(stderr,
			"Razer Copperhead First Edition: 3d\n"
			"Razer Copperhead (non-first):   cc\n"
			"\n"
			"WARNING: Changing the laser power is EXTREMELY DANGEROUS.\n"
			"         It will destroy your mouse *AND* your eyes.\n"
			"         DO NOT LOOK AT THE LASER.  It is invisible to the\n"
			"         human eye even at the highest power, and will\n"
			"         *PERMANENTLY DAMAGE YOUR EYES*.\n"
			"\n"
			"    *CHANGING THE LASER POWER WILL VOID YOUR WARRANTY*\n"
			"\n"
			"    Bit 7:    Bin 2A/3A\n"
			"    Bits 6-0: Power - 0x00 is 100%%, 0x7f is 33.85%%\n"
			"\n"
			"Type \"" YES_STRING "\" to set laser power to %02x (Bin %s, Power %6.2f%%) anyway:\n",
			laserval,
			(laserval & 0x80) ? "3A" : "2A",
			100.0 - (100.0 * (laserval & 0x7f) / 192.0)
		);

		if (read(STDOUT_FILENO, yes, strlen(YES_STRING)+2) != (strlen(YES_STRING)+1) || strcmp(yes, YES_STRING "\n")) {
			fprintf(stderr, "Aborted.\n");
			return -1;
		}

		fprintf(stderr, "Continuing anyway, as you requested.\n");

		action = RF_WRITE_LASER;
	}

	if (action == RF_SHOW_HELP) {
		do_help(argv[0]);
		return -1;
	}

	dev = razer_find(NULL);
	if (dev) {
		razer_t *r = razer_open(dev);
		if (r) {
			if (r->firmware_mode) {
				fprintf(stderr, "Setting firmware mode...\n");
				r->firmware_mode(r);
				sleep(2);
			} else {
				fprintf(stderr,
					"%s detected.\n"
					"Please plug the mouse in with the profile button depressed.\n",
					r->name
				);
				razer_close(r);
				return -1;
			}
		}
	}

	/* TODO: Open by device, etc */
	dev = razer_find_flash(NULL);
	if (!dev) {
		do_help(argv[0]);
		fprintf(stderr, "\nERROR: Unable to find Freescale JW32 ICP\n");
		return -1;
	}

	udev = usb_open(dev);
	if (!udev) {
		perror("usb_open()");
		return -1;
	}

	status = razer_firmware_status(udev, &stat, RAZERLIB_DEFAULT_TIMEOUT);
	if (status) {
		fprintf(stderr, "razer_firmware_status(): %s\n", razer_strerror(-status));
		if (status == -EPERM)
			fprintf(stderr, "(got root?)\n");
		return -1;
	}

	switch (action) {
	  case RF_DUMP_FIRMWARE:
		fprintf(stderr,
			"WARNING: This is *SLOW* and outputs to stdout, so ^C and redirect\n"
			"stdout to a file if you haven't already done so.\n"
		);
		status = razer_firmware_read_flash_slow(udev, &fw, razerflash_progress);
		if (status) {
			fprintf(stderr, "\nrazer_firmware_read_flash_slow(): %s\n", razer_strerror(-status));
			return -1;
		}
		razer_firmware_dump_srecord(&fw);
		break;
	  case RF_VERIFY_FIRMWARE:
		status = razer_firmware_verify_flash(udev, &fw, razerflash_progress);
		if (status > 0) {
			fprintf(stderr, "\nFlash verification: failed at %04x\n", status);
			fprintf(stderr,
				"This is usually normal; Modified profiles or verifying with\n"
				"a different firmware can cause verification failures.\n"
			);
			return 1;
		}
		if (status) {
			fprintf(stderr, "\nrazer_firmware_verify_flash(): %s\n", razer_strerror(-status));
			return -1;
		}
		fprintf(stderr, "Flash verification: passed\n");
		break;
	  case RF_FLASH_FIRMWARE:
		fprintf(stderr, "FLASHING DEVICE - DO NOT UNPLUG OR POWER OFF\n");
		status = razer_firmware_write_flash(udev, &fw, razerflash_progress);
		if (status) {
			fprintf(stderr, "\nrazer_firmware_write_flash(): %s\n", razer_strerror(-status));
			fprintf(stderr, "Flash failed.  Sorry.\n");
			return -1;
		}
		fprintf(stderr, "Flash successful!\n");
		break;
	  case RF_READ_LASER: {
		int laser;
		status = razer_firmware_read_laser_power(udev, &laser, NULL);
		if (status < 0) {
			fprintf(stderr, "razer_firmware_read_laser_power(): %s\n", razer_strerror(-status));
			return -1;
		}
		printf(
			"%02x (Bin %s, Power %6.2f%%)\n",
			laser,
			(laser & 0x80) ? "3A" : "2A",
			100.0 - (100.0 * (laser & 0x7f) / 192.0)
		);
		break;
	  }
	  case RF_WRITE_LASER: {
		fprintf(stderr, "Setting laser power (be patient)\n");
		status = razer_firmware_write_laser_power(udev, laserval, razerflash_progress);
		if (status) {
			fprintf(stderr, "razer_firmware_read_laser_power(): %s\n", razer_strerror(-status));
			return -1;
		}
		fprintf(stderr, "Laser value programmed.\n");
		break;
	  }
	}

	usb_close(udev);

	return 0;
}
