package addmask

import (
	"errors"
	"fmt"
	"io"

	"gitlab.com/ajwalker/phrasestream"
)

// Store is an interface for fetching and adding phrases so that they can be
// later shared with future instantiations.
type Store interface {
	List() ([]string, error)
	Add(string) error
}

// AddMask implements behaviour similar to GitHub's `::add-mask::` action
// runner functionality. It wraps output streams, and allows dynamic masking
// of secrets, so that future instances of that secret in the stream is masked.
type AddMask struct {
	store   Store
	control []*phrasestream.Phrasestream
	maskers []*phrasestream.Phrasestream
}

// New returns a new AddMask.
func New(store Store, outputs ...io.Writer) (*AddMask, error) {
	m := &AddMask{store: store}

	for _, output := range outputs {
		masker := phrasestream.New(output)
		m.control = append(m.control, phrasestream.New(masker))
		m.maskers = append(m.maskers, masker)
	}

	for _, control := range m.control {
		control.Add("::add-mask::", m.handle, phrasestream.StopCRLF)
	}

	if store != nil {
		existing, err := store.List()
		if err != nil {
			return nil, err
		}

		for _, mask := range existing {
			m.addMask(mask)
		}
	}

	return m, nil
}

// Get fetches a wrapped writer by index. The order is the same as writers
// provided in New().
func (m *AddMask) Get(idx int) io.Writer {
	if idx < len(m.control) {
		return m.control[idx]
	}

	return nil
}

// Closes flushes internal buffers.
func (m *AddMask) Close() error {
	var errs []error

	for _, control := range m.control {
		if err := control.Close(); err != nil {
			errs = append(errs, err)
		}
	}

	for _, masker := range m.maskers {
		if err := masker.Close(); err != nil {
			errs = append(errs, err)
		}
	}

	return errors.Join(errs...)
}

func (m *AddMask) handle(w io.Writer, phrase string, buf []byte) error {
	secret := string(buf)

	if m.store != nil {
		if err := m.store.Add(secret); err != nil {
			return fmt.Errorf("adding phrase to store: %w", err)
		}
	}

	m.addMask(secret)

	_, _ = w.Write([]byte("::add-mask::" + secret))

	return nil
}

func (m *AddMask) addMask(phrase string) {
	for _, masker := range m.maskers {
		for _, encoded := range encode(phrase) {
			masker.Add(encoded, phrasestream.Mask, nil)
		}
	}
}
