#ifndef BUFFER_H
#define BUFFER_H

#include <regex.h>

/* Notes */ /*{{{C}}}*//*{{{*/
/* 

This modules implements generic functions which work on a gap buffer of
character and character groups.  A gap buffer is a common way for to
store text in text editors and it works like this:

| text before gap |   ... gap ... | text after gap |

The start/end of the gap reflects the cursor position in the text.
Inserting is done by decreasing the gap, deleting by increasing it.
The stored text consists of characters, fold marks or UTF-8 sequences
and the gap only ever moves by single characters or character sequences.
The special character '\0' quotes a following fold mark or quote mark,
using the constants QUOTE, FOLDSTART and FOLDEND.  It is made sure not
to have more than one fold mark per line and to keep the fold structure
consistent when using the functions from this module.

My personal experience from hacking on the folding editor Origami is,
that a fancy data structure (double-linked n-tree) is very efficient, as
even with a deeply folded 3 MB file there is no slowdown.  This can not
be expected from using a gap buffer.  The reason why I use it never the
less, is that a fancy data structure makes it nearly impossible to have
correct and elegant primitives which work on the raw text.  In fact,
a significant slow down can be felt from lots of node-traversing which
happens when deleting bigger amounts of text.  This is the reason, why
most versions of Origami enforce using folds for everything.  While this
seemed elegant in the beginning, after a longer time I have found that
folds are no universal solution to everything.

As a result, this implementation is slower for basic operations (opening
and closing folds, screen display), faster for complex operations (block
operations, loading, saving) and tries to offer the best of both worlds
(traditional text editor and folding editor).

*/
/*}}}*/

#define NUL       ((char)1)
#define QUOTE     ((char)2) /* quote char after */
#define FOLDSTART ((char)4)
#define FOLDEND   ((char)8)

/* UTF-8 character, fold mark or quoted character */
struct MetaChar
{
  char c[4];
};

#define ASCII(Mchar)  ((Mchar).c[0])
#define MARK(Mchar)   ((Mchar).c[1])
#define QUOTED(Mchar) ((Mchar).c[2])

#define LEFT      1
#define RIGHT     2

#define BUFMARKS  16

enum Buffermode { CHANGED=1, READONLY=2, Buffermode_last=3 };

#define LANGUAGES 14

struct Language
{
  const char *name;            /* name of language              */
  const char *beg;             /* comment begin                 */
  const char *end;             /* comment end                   */
};

struct Buffer
{
  size_t size;                 /* size of buffer area           */
  char *buffer;                /* buffer area with gap          */
  char *gapstart;              /* first byte of gap             */
  char *gapend;                /* last byte of gap              */
  int cury;                    /* current line number (0 based) */
  char *name;                  /* name of file                  */
  enum Buffermode mode;        /* mode flags                    */
  struct Language *lang;       /* language                      */
  int curch;                   /* current MetaChar position     */
  int mark[BUFMARKS];          /* alternate gap start offsets   */
  char markused[BUFMARKS];     /* flag for usage of a mark      */
  int msz;                     /* number of marks               */
  int refcnt;                  /* nuber of references           */
};

/* Buffers use UTF-8.  This is global, because copying between
 * buffers would need character set information otherwise and
 * recode, which might not even be possible when e.g. going
 * from UTF-8 to ISO-8859-1.
 */
extern int bf_utf8;

extern const struct MetaChar bf_foldstart;
extern const struct MetaChar bf_foldend;
extern struct Language language[];

struct Buffer *bf_alloc(void);
void bf_free(struct Buffer *b);
int bf_empty(struct Buffer *b);
int bf_forward(struct Buffer *b);
int bf_backward(struct Buffer *b);
size_t bf_lcharlen(const char *s, const char *start);
size_t bf_rcharlen(const char *s);
int bf_insert(struct Buffer *b, const struct MetaChar *ch);
int bf_delete(struct Buffer *b);
int bf_insertstr(struct Buffer *b, const char *str, const char *end);
int bf_lchar(const struct Buffer *b, struct MetaChar *ch);
int bf_rchar(const struct Buffer *b, struct MetaChar *ch);
int bf_load(struct Buffer *b, const char *cmd, const char *filename, int filefd, const char *begin, const char *end);
int bf_save(struct Buffer *b, const char *file, const char *beg, const char *end, unsigned int *chars);
int bf_write(struct Buffer *b, const char *file, unsigned int *chars);
int bf_linestart(struct Buffer *b, int realbegin);
int bf_isfold(struct Buffer *s, int fold, int where);
int bf_lineend(struct Buffer *b, int realend);
int bf_tofoldstart(struct Buffer *b, int level, int *linesback);
int bf_tofoldend(struct Buffer *b, int level, int *linesforward);
void bf_begin(struct Buffer *b);
void bf_end(struct Buffer *b);
int bf_name(struct Buffer *b, const char *name);
struct Language *bf_lang(struct Buffer *b, struct Language *l);
int bf_gotoline(struct Buffer *b, int line);
int bf_strfsearch(struct Buffer *b, const char *needle, size_t needlesize);
int bf_regfsearch(struct Buffer *b, const regex_t *needle);
int bf_strbsearch(struct Buffer *b, const char *needle, size_t needlesize);
int bf_regbsearch(struct Buffer *b, const regex_t *needle);
void bf_tomark(struct Buffer *b, int to);
int bf_mark(struct Buffer *b, int mark);
void bf_unmark(struct Buffer *b, int mark);
long bf_uc(const struct MetaChar *ch);

#endif
