package webtestsapis

import (
	"context"
	"fmt"
	"net/http"

	"github.com/hashicorp/go-azure-helpers/resourcemanager/commonids"
	"github.com/hashicorp/go-azure-sdk/sdk/client"
	"github.com/hashicorp/go-azure-sdk/sdk/odata"
)

// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See NOTICE.txt in the project root for license information.

type WebTestsListOperationResponse struct {
	HttpResponse *http.Response
	OData        *odata.OData
	Model        *[]WebTest
}

type WebTestsListCompleteResult struct {
	LatestHttpResponse *http.Response
	Items              []WebTest
}

type WebTestsListCustomPager struct {
	NextLink *odata.Link `json:"nextLink"`
}

func (p *WebTestsListCustomPager) NextPageLink() *odata.Link {
	defer func() {
		p.NextLink = nil
	}()

	return p.NextLink
}

// WebTestsList ...
func (c WebTestsAPIsClient) WebTestsList(ctx context.Context, id commonids.SubscriptionId) (result WebTestsListOperationResponse, err error) {
	opts := client.RequestOptions{
		ContentType: "application/json; charset=utf-8",
		ExpectedStatusCodes: []int{
			http.StatusOK,
		},
		HttpMethod: http.MethodGet,
		Pager:      &WebTestsListCustomPager{},
		Path:       fmt.Sprintf("%s/providers/Microsoft.Insights/webTests", id.ID()),
	}

	req, err := c.Client.NewRequest(ctx, opts)
	if err != nil {
		return
	}

	var resp *client.Response
	resp, err = req.ExecutePaged(ctx)
	if resp != nil {
		result.OData = resp.OData
		result.HttpResponse = resp.Response
	}
	if err != nil {
		return
	}

	var values struct {
		Values *[]WebTest `json:"value"`
	}
	if err = resp.Unmarshal(&values); err != nil {
		return
	}

	result.Model = values.Values

	return
}

// WebTestsListComplete retrieves all the results into a single object
func (c WebTestsAPIsClient) WebTestsListComplete(ctx context.Context, id commonids.SubscriptionId) (WebTestsListCompleteResult, error) {
	return c.WebTestsListCompleteMatchingPredicate(ctx, id, WebTestOperationPredicate{})
}

// WebTestsListCompleteMatchingPredicate retrieves all the results and then applies the predicate
func (c WebTestsAPIsClient) WebTestsListCompleteMatchingPredicate(ctx context.Context, id commonids.SubscriptionId, predicate WebTestOperationPredicate) (result WebTestsListCompleteResult, err error) {
	items := make([]WebTest, 0)

	resp, err := c.WebTestsList(ctx, id)
	if err != nil {
		result.LatestHttpResponse = resp.HttpResponse
		err = fmt.Errorf("loading results: %+v", err)
		return
	}
	if resp.Model != nil {
		for _, v := range *resp.Model {
			if predicate.Matches(v) {
				items = append(items, v)
			}
		}
	}

	result = WebTestsListCompleteResult{
		LatestHttpResponse: resp.HttpResponse,
		Items:              items,
	}
	return
}
