# SPDX-License-Identifier: Apache-2.0
# Copyright 2012-2021 The Meson development team
# Copyright © 2021-2023 Intel Corporation

from __future__ import annotations

"""Posix specific implementations of mesonlib functionality."""

import fcntl
import typing as T

from .core import MesonException
from .platform import BuildDirLock as BuildDirLockBase

__all__ = ['BuildDirLock']

class BuildDirLock(BuildDirLockBase):

    def __enter__(self) -> None:
        self.lockfile = open(self.lockfilename, 'w', encoding='utf-8')
        try:
            fcntl.flock(self.lockfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
        except (BlockingIOError, PermissionError):
            self.lockfile.close()
            raise MesonException('Some other Meson process is already using this build directory. Exiting.')
        except OSError as e:
            self.lockfile.close()
            raise MesonException(f'Failed to lock the build directory: {e.strerror}')

    def __exit__(self, *args: T.Any) -> None:
        fcntl.flock(self.lockfile, fcntl.LOCK_UN)
        self.lockfile.close()
