/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#define  __NO_VERSION__

#include "nv-misc.h"
#include "os-interface.h"
#include "nv-linux.h"

#if defined(NV_LINUX_ACPI_EVENTS_SUPPORTED)

/*
 * This code allows RM to subscribe to events surrounding the power adapter.
 *
 */

/*
 * c file definitions of function that do not need external visibility
 *
 */
static int         nv_acpi_add         (struct acpi_device *device);
static int         nv_acpi_remove      (struct acpi_device *device, int type);
static void        nv_acpi_event       (acpi_handle, u32, void *);

#if defined(NV_ACPI_DEVICE_OPS_HAS_MATCH)
static int         nv_acpi_match       (struct acpi_device *, struct acpi_driver *);
#endif

#if defined(ACPI_VIDEO_HID) && defined(NV_ACPI_DEVICE_ID_HAS_DRIVER_DATA) 
static const struct acpi_device_id nv_video_device_ids[] = {
    { 
        .id          = ACPI_VIDEO_HID, 
        .driver_data = 0, 
    },
    { 
        .id          = "",
        .driver_data = 0, 
    },
};
#endif

static struct acpi_driver *nv_acpi_driver;

static const struct acpi_driver nv_acpi_driver_template = {
    .name = "NVIDIA ACPI Video Driver",
    .class = "video",
#if defined(ACPI_VIDEO_HID)
#if defined(NV_ACPI_DEVICE_ID_HAS_DRIVER_DATA)
    .ids = nv_video_device_ids,
#else
    .ids = ACPI_VIDEO_HID,
#endif
#endif
    .ops = {
        .add = nv_acpi_add,
        .remove = nv_acpi_remove,
#if defined(NV_ACPI_DEVICE_OPS_HAS_MATCH)
        .match = nv_acpi_match,
#endif
    },
};

int nv_acpi_init(void)
{
    /*
     * This function will register the RM with the Linux
     * ACPI subsystem.
     */
    int status;

    if (nv_acpi_driver != NULL)
        return -EBUSY;

    os_alloc_mem((void **)&nv_acpi_driver, sizeof(struct acpi_driver));
    if (nv_acpi_driver == NULL)
        return -ENOMEM;

    memcpy((void *)nv_acpi_driver, (void *)&nv_acpi_driver_template,
            sizeof(struct acpi_driver));

    status = acpi_bus_register_driver(nv_acpi_driver);
    if (status < 0)
    {
        nv_printf(NV_DBG_INFO,
            "NVRM: nv_acpi_init: acpi_bus_register_driver() failed (%d)!\n", status);
        os_free_mem(nv_acpi_driver);
        nv_acpi_driver = NULL;
    }

    return status;
}

int nv_acpi_uninit(void)
{
    if (nv_acpi_driver == NULL)
        return -ENXIO;

    acpi_bus_unregister_driver(nv_acpi_driver);
    os_free_mem(nv_acpi_driver);

    nv_acpi_driver = NULL;

    return 0;
}

static int nv_acpi_add(struct acpi_device *device)
{
    /*
     * This function will cause RM to initialize the things it needs for acpi interaction
     * on the display device.
     */
    int status = -1;
    RM_STATUS rmStatus = RM_ERROR;
    nv_acpi_t *pNvAcpiObject = NULL;
    union acpi_object control_argument_0 = { ACPI_TYPE_INTEGER };
    struct acpi_object_list control_argument_list = { 0, NULL };
    nv_stack_t *sp = NULL;
    struct list_head *node, *next;
    nv_acpi_integer_t device_id = 0;

    NV_KMEM_CACHE_ALLOC_STACK(sp);
    if (sp == NULL)
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: failed to allocate stack!\n");
        return -ENOMEM;
    }

    // allocate data structure we need
    rmStatus = os_alloc_mem((void **) &pNvAcpiObject, sizeof(nv_acpi_t));
    if (rmStatus != RM_OK)
    {
        NV_KMEM_CACHE_FREE_STACK(sp);
        nv_printf(NV_DBG_ERRORS,
            "NVRM: nv_acpi_add: failed to allocate ACPI device management data!\n");
        return -ENOMEM;
    }

    os_mem_set((void *)pNvAcpiObject, 0, sizeof(nv_acpi_t));

    device->driver_data = pNvAcpiObject;
    pNvAcpiObject->device = device;

    pNvAcpiObject->sp = sp;

    // grab handles to all the important nodes representing devices

    list_for_each_safe(node, next, &device->children) 
    {
        struct acpi_device *dev =
            list_entry(node, struct acpi_device, node);

        if (!dev)
            continue;

        status =
            acpi_evaluate_integer(dev->handle, "_ADR", NULL, &device_id);
        if (ACPI_FAILURE(status))
            /* Couldnt query device_id for this device */
            continue;

        device_id = (device_id & 0xffff);

        /* Check if the device_id is amongst the set of 
           possible device-ids for a CRT ...*/
        if ((device_id >= 0x0100) && (device_id <= 0x0107))
        {
            pNvAcpiObject->crt_handle = dev->handle;
        }
        /* the set of possible device-ids for a TV */
        else if ((device_id >= 0x0200) && (device_id <= 0x0207))
        {
            pNvAcpiObject->tv_handle = dev->handle;
        }
        else if ((device_id == 0x0110) ||  /* device id for internal LCD */
                 (device_id == 0x0118) ||  /* alternate ACPI ID for the 
                                                            internal LCD */
                 (device_id == 0x0400))    /* ACPI spec 3.0 specified 
                                             device id for a internal LCD*/
        {
            pNvAcpiObject->lcd_handle = dev->handle;
        }
        else if (((device_id >= 0x0111) && (device_id <= 0x0117)) || /* the set
                                        of possible device-ids for a DFP */
                  (device_id == 0x0120) || /* device id for non-LVDS DFP */
                  (device_id == 0x0300))   /* ACPI spec 3.0 specified 
                                              device id for non-LVDS DFP */
        {
            pNvAcpiObject->dvi_handle = dev->handle;
        }
    }

    // arg 0, bits 1:0, 0 = enable events
    control_argument_0.integer.type = ACPI_TYPE_INTEGER;
    control_argument_0.integer.value = 0x0;

    // listify it
    control_argument_list.count = 1;
    control_argument_list.pointer = &control_argument_0;

    // _DOS method takes 1 argument and returns nothing
    status = acpi_evaluate_object(device->handle, "_DOS", &control_argument_list, NULL);

    if (ACPI_FAILURE(status))
    {
        nv_printf(NV_DBG_INFO,
            "NVRM: nv_acpi_add: failed to enable display switch events (%d)!\n", status);
    }

    status = acpi_install_notify_handler(device->handle, ACPI_DEVICE_NOTIFY,
                    nv_acpi_event, pNvAcpiObject);

    if (ACPI_FAILURE(status))
    {
        nv_printf(NV_DBG_INFO,
            "NVRM: nv_acpi_add: failed to install event notification handler (%d)!\n", status);
    }
    else
    {
        try_module_get(THIS_MODULE);
        pNvAcpiObject->notify_handler_installed = 1;
    }

    return 0;
}

static int nv_acpi_remove(struct acpi_device *device, int type)
{
    /*
     * This function will cause RM to relinquish control of the VGA ACPI device.
     */
    acpi_status status;
    union acpi_object control_argument_0 = { ACPI_TYPE_INTEGER };
    struct acpi_object_list control_argument_list = { 0, NULL };
    nv_acpi_t *pNvAcpiObject = acpi_driver_data(device);

    // arg 0, bits 1:0, 1 = disable events
    control_argument_0.integer.type = ACPI_TYPE_INTEGER;
    control_argument_0.integer.value = 0x1;

    // listify it
    control_argument_list.count = 1;
    control_argument_list.pointer = &control_argument_0;

    // _DOS method takes 1 argument and returns nothing
    status = acpi_evaluate_object(device->handle, "_DOS", &control_argument_list, NULL);

    if (ACPI_FAILURE(status))
    {
        nv_printf(NV_DBG_INFO,
            "NVRM: nv_acpi_remove: failed to disable display switch events (%d)!\n", status);
    }

    if (pNvAcpiObject->notify_handler_installed)
    {
        // no status returned for this function
        acpi_os_wait_events_complete(NULL);

        // remove event notifier
        status = acpi_remove_notify_handler(device->handle, ACPI_DEVICE_NOTIFY, nv_acpi_event);
    }

    if (pNvAcpiObject->notify_handler_installed &&
        ACPI_FAILURE(status))
    {
        nv_printf(NV_DBG_INFO,
            "NVRM: nv_acpi_remove: failed to remove event notification handler (%d)!\n", status);
    }
    else
    {
        NV_KMEM_CACHE_FREE_STACK(pNvAcpiObject->sp);
        os_free_mem((void *)pNvAcpiObject);
        module_put(THIS_MODULE);
    }

    return status;
}

static void nv_acpi_event(acpi_handle handle, u32 event_type, void *data)
{
    /*
     * This function will handle acpi events from the linux kernel, used
     * to detect notifications from the VGA device.
     */
    nv_acpi_t *pNvAcpiObject = data;
    u32 event_val = 0;
    nv_acpi_integer_t state;
    int status = 0;


    if (event_type == NV_SYSTEM_ACPI_DISPLAY_DEVICE_CYCLE)
    {
        if (pNvAcpiObject->lcd_handle) 
        {
            status = acpi_evaluate_integer(pNvAcpiObject->lcd_handle,
                                           "_DGS", 
                                           NULL, 
                                           &state);
            if (ACPI_FAILURE(status)) 
            {
                nv_printf(NV_DBG_INFO, 
                "NVRM: nv_acpi_event: failed to query _DGS method for LCD\n");
            }
            else if (state)
                event_val |= NV_HOTKEY_STATUS_DISPLAY_ENABLE_LCD;
        }

        if (pNvAcpiObject->crt_handle) 
        {
            status = acpi_evaluate_integer(pNvAcpiObject->crt_handle,
                                           "_DGS",
                                           NULL,
                                           &state);
            if (ACPI_FAILURE(status)) 
            {
                nv_printf(NV_DBG_INFO, 
                "NVRM: nv_acpi_event: failed to query _DGS method for CRT\n");
            }
            else if (state)
                event_val |= NV_HOTKEY_STATUS_DISPLAY_ENABLE_CRT;
        }

        if (pNvAcpiObject->tv_handle) 
        {
            status = acpi_evaluate_integer(pNvAcpiObject->tv_handle,
                                           "_DGS",
                                           NULL,
                                           &state);
            if (ACPI_FAILURE(status)) 
            {
                nv_printf(NV_DBG_INFO,
                "NVRM: nv_acpi_event: failed to query _DGS method for TV\n");
            }
            else if (state)
                event_val |= NV_HOTKEY_STATUS_DISPLAY_ENABLE_TV;
        }

        if (pNvAcpiObject->dvi_handle)
        {
            status = acpi_evaluate_integer(pNvAcpiObject->dvi_handle,
                                           "_DGS",
                                           NULL,
                                           &state);
            if (ACPI_FAILURE(status)) 
            {
                nv_printf(NV_DBG_INFO,
                "NVRM: nv_acpi_event: failed to query _DGS method for DVI\n");
            }
            else if (state)     
                event_val |= NV_HOTKEY_STATUS_DISPLAY_ENABLE_DFP;
        }

        nv_printf(NV_DBG_INFO,
        "NVRM: nv_acpi_event: Event-type 0x%x, Event-val 0x%x\n", 
        event_type, event_val);

        rm_system_event(pNvAcpiObject->sp, event_type, event_val);
    }

    // no unsubscription or re-enable necessary. Once DOD has been set, we are go.
    // once we are subscribed to ACPI events, we don't have to re-subscribe unless
    // unsubscribe.
}

#if defined(NV_ACPI_DEVICE_OPS_HAS_MATCH)
static int nv_acpi_match(struct acpi_device *device, struct acpi_driver *driver)
{
    acpi_handle DOS_method_handler, DOD_method_handler;

    /*
     *
     * attempt to get handles to all of these objects
     * _DOS = enable/disable output switching method
     * _DOD = enumerate all devices attached to the display driver
     * 
     * If handles are available, this is the VGA object.
     *
     */

    if ((!acpi_get_handle(device->handle, "_DOS", &DOS_method_handler)) &&
        (!acpi_get_handle(device->handle, "_DOD", &DOD_method_handler)))
    {
        nv_printf(NV_DBG_INFO, "NVRM: nv_acpi_match: VGA ACPI device detected.\n");

        return 0;
    }
    else
        return (-ENODEV);
}
#endif


#else // NV_LINUX_ACPI_EVENTS_SUPPORTED

int nv_acpi_init(void)
{
    return 0;
}

int nv_acpi_uninit(void)
{
    return 0;
}

#endif
