/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2000-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#define  __NO_VERSION__
#include "nv-misc.h"
#include "os-interface.h"
#include "nv-linux.h"

/*
 * Registry variables.
 *
 * These are defined here and used throughout resman. The default values
 * are used unless the user overrides them when loading the kernel module
 * with insmod or by specifying any given option in the modprobe config
 * file (/etc/modules.conf).
 *
 * Registry entries interesting on Linux are defined in this source file.
 * Any given registry name (windows registry leaf names) are mapped to
 * variables with the following naming convention: NVreg_NAME, where NAME
 * is the registry name with space converted to underscores.
 *
 * Note: much of resman merely probes for the existence of a flag, actual
 *       values are often ignored.
 * 
 * TODO:
 *
 *   persistence
 *
 *   Linux does not currently offer a convenient way to store registry
 *   type data persistently from kernel modules. This functionlity has
 *   been requested by module developers and there has been an effort
 *   to implement it with an extension to the module parameter/modprobe
 *   mechanism, but this experimental implementation is not included in
 *   any current Linux kernel releases. While there are several options
 *   to add persistency ourselves, it would have to be done properly to
 *   work reliable. 
 *
 *   device dependence
 *
 *   The registry functions currently operate on a global namespace.
 *   This could be changed to work on a per-device basis.
 */

/* 
 * Option: VideoMemoryTypeOverride
 * 
 * Description:
 *
 * We normally detect memory type on TNT cards by scanning the embedded
 * BIOS. Unfortunately, we've seen some cases where a TNT card has been
 * flashed with the wrong bios. For example, an SDRAM based TNT has been
 * flashed with an SGRAM bios, and therefore claims to be an SGRAM TNT.
 * We've therefore provided an override here. Make sure to set the value
 * toe the type of memory used on your card.
 *
 * Possible Values:
 *
 * 1: SDRAM
 * 2: SGRAM
 *
 * Note that we can only do so much here. There are border cases where
 * even this fails. For example, if 2 TNT cards are in the same system,
 * one SGRAM, one SDRAM.
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_VideoMemoryTypeOverride = 1;
NV_MODULE_PARAMETER(NVreg_VideoMemoryTypeOverride);

/*
 * Option: EnableVia4x
 *
 * Description:
 *
 * We've had problems with some Via chipsets in 4x mode, we need force
 * them back down to 2x mode. If you'd like to experiment with retaining
 * 4x mode, you may try setting this value to 1 If that hangs the system,
 * you're stuck with 2x mode; there's nothing we can do about it.
 *
 * Possible Values:
 *
 * 0: disable AGP 4x on Via chipsets (default)
 * 1: enable  AGP 4x on Via chipsets
 */

static int NVreg_EnableVia4x = 0;
NV_MODULE_PARAMETER(NVreg_EnableVia4x);

/*
 * Option: EnableALiAGP
 *
 * Description:
 *
 * Some ALi chipsets (ALi1541, ALi1647) are known to cause severe system
 * stability problems with AGP enabled. To avoid lockups, we disable AGP
 * on systems with these chipsets by default. It appears that updating the
 * system BIOS and using recent versions of the kernel AGP Gart driver can
 * make such systems much more stable. If you own a system with one of the
 * aforementioned chipsets and had it working reasonably well previously,
 * or if you want to experiment with BIOS and AGPGART revisions, you can
 * re-enable AGP support by setting this option to 1.
 *
 * Possible Values:
 *
 *  0: disable AGP on Ali1541 and ALi1647 (default)
 *  1: enable  AGP on Ali1541 and ALi1647
 */

static int NVreg_EnableALiAGP = 0;
NV_MODULE_PARAMETER(NVreg_EnableALiAGP);

/* 
 * Option: ReqAGPRate
 *
 * Description:
 *
 * Normally, the driver will compare speed modes of the chipset & card,
 * picking the highest common rate. This key forces a maximum limit, to
 * limit the driver to lower speeds. The driver will not attempt a speed
 * beyond what the chipset and card claim they are capable of.
 *
 * Make sure you really know what you're doing before you enable this
 * override. By default, AGP drivers will enable the fastest AGP rate
 * your card and motherboard chipset are capable of. Then, in some cases,
 * our driver will force this rate down to work around bugs in both our
 * chipsets, and motherboard chipsets. Using this variable will override
 * our bug fixes. This may be desirable in some cases, but not most.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  This option expects a bitmask (15 = 8 | 4 | 2 | 1, etc.)
 *
 *  Note that on AGP 3.x chipsets, the only supported AGP rates are
 *  AGP 8x and AGP 4x (if set in ReqAGPRate, AGP 2x and 1x are
 *  ignored by the driver).
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_ReqAGPRate = 15;
NV_MODULE_PARAMETER(NVreg_ReqAGPRate);

/*  
 * Option: NvAGP
 *
 * Description: 
 *
 * This options controls which AGP GART driver is used when no explicit
 * request is made to change the default (X server).
 *
 * Possible Values:
 *
 *   0 = disable AGP support
 *   1 = use NVIDIA's builtin driver (if possible)
 *   2 = use the kernel's AGPGART driver (if possible)
 *   3 = use any available driver (try 1, then 2)
 *
 * Please note that NVIDIA's internal AGP GART driver will not be used
 * if AGPGART was either statically linked into your kernel or built as
 * a kernel module and loaded before the NVIDIA kernel module.
 */

static int NVreg_NvAGP = 3;
NV_MODULE_PARAMETER(NVreg_NvAGP);


/* 
 * Option: EnableAGPSBA
 *
 * Description:
 *
 * For stability reasons, the driver will not Side Band Addressing even if
 * both the host chipset and the AGP card support it. You may override this
 * behaviour with the following registry key.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Side Band Addressing (default on x86, see below)
 *  1 = enable  Side Band Addressing (if supported)
 */

/*
 * The default on x86 is to disable AGP side band addressing; if you want
 * to turn it on, change the registry key below.
 */
static int NVreg_EnableAGPSBA = 0;

NV_MODULE_PARAMETER(NVreg_EnableAGPSBA);

/*
 * Option: EnableAGPFW
 *
 * Description:
 *
 * Similar to Side Band Addressing, Fast Writes are disabled by default. If
 * you wish to enable them on systems that support them, you can do so with
 * this registry key. Please note that this may render your system unstable
 * with many AGP chipsets.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Fast Writes (default)
 *  1 = enable  Fast Writes (if supported)
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_EnableAGPFW = 0;
NV_MODULE_PARAMETER(NVreg_EnableAGPFW);

/*
 * Option: SoftEDIDs
 *
 * Description:
 *
 * The SoftEDIDs registry key enables dynamic generation of an appropriate
 * EDID for mobile LCD displays from data stored in the video BIOS. If this
 * is turned off, then on mobile systems, a hardcoded EDID will be chosen
 * from a table, based on the value of the Mobile registry key.
 *
 * Possible Values:
 *
 *  0 = disable dynamic EDID generation
 *  1 = enable  dynamic EDID generation (default)
 */

static int NVreg_SoftEDIDs = 1;
NV_MODULE_PARAMETER(NVreg_SoftEDIDs);

/* 
 * Option: Mobile
 *
 * Description:
 *
 * The Mobile registry key should only be needed on mobile systems if
 * SoftEDIDs is disabled (see above), in which case the mobile value
 * will be used to lookup the correct EDID for the mobile LCD.
 *
 * Possible Values:
 *
 *  ~0 = auto detect the correct value (default)
 *   1 = Dell notebooks
 *   2 = non-Compal Toshiba
 *   3 = all other notebooks
 *   4 = Compal/Toshiba
 *   5 = Gateway
 *
 * Make sure to specify the correct value for your notebook.
 */

static int NVreg_Mobile = ~0;
NV_MODULE_PARAMETER(NVreg_Mobile);

/*
 * Option: ModifyDeviceFiles
 *
 * Description:
 *
 * When this option is enabled, the NVIDIA driver will verify the validity
 * of the NVIDIA device files in /dev and attempt to dynamically modify
 * and/or (re-)create them, if necessary. If you don't wish for the NVIDIA
 * driver to touch the device files, you can use this registry key.
 *
 * Possible Values:
 *  0 = disable dynamic device file management
 *  1 = enable  dynamic device file management
 */

static int NVreg_ModifyDeviceFiles = 1;
NV_MODULE_PARAMETER(NVreg_ModifyDeviceFiles);


/*
 * Option: DeviceFileUID
 *
 * Description:
 *
 * This registry key specifies the UID assigned to the NVIDIA device files
 * created and/or modified by the NVIDIA driver when dynamic device file
 * management is enabled.
 *
 * The default UID is 0 ('root').
 */

static int NVreg_DeviceFileUID = 0;
NV_MODULE_PARAMETER(NVreg_DeviceFileUID);

/*
 * Option: DeviceFileGID
 *
 * Description:
 *
 * This registry key specifies the GID assigned to the NVIDIA device files
 * created and/or modified by the NVIDIA driver when dynamic device file
 * management is enabled.
 *
 * The default GID is 0 ('root').
 */

static int NVreg_DeviceFileGID = 0;
NV_MODULE_PARAMETER(NVreg_DeviceFileGID);

/*
 * Option: DeviceFileMode
 *
 * Description:
 *
 * This registry key specifies the device file mode assigned to the NVIDIA
 * device files created and/or modified by the NVIDIA driver when dynamic
 * device file management is enabled.
 *
 * The default mode is 0666 (octal, rw-rw-rw-).
 */

static int NVreg_DeviceFileMode = 0666;
NV_MODULE_PARAMETER(NVreg_DeviceFileMode);


static int NVreg_ResmanDebugLevel = ~0;
NV_MODULE_PARAMETER(NVreg_ResmanDebugLevel);

static int NVreg_FlatPanelMode = 0;
NV_MODULE_PARAMETER(NVreg_FlatPanelMode);

static int NVreg_DevicesConnected = 0;
NV_MODULE_PARAMETER(NVreg_DevicesConnected);

static int NVreg_RmLogonRC = 1;
NV_MODULE_PARAMETER(NVreg_RmLogonRC);

/*
 * Option: RemapLimit
 *
 * Description:
 *
 * This registry key specifies the maximum amount of memory allowed to 
 * be remapped through the IOMMU/SWIOTLB. On 64-bit platforms, getting
 * 32-bit physical addresses is difficult and requires remapping 64-bit
 * physical addresses through either the IOMMU or SWIOTLB (depending on
 * the CPU). These apertures are of limited size (usually default to 64
 * MegaBytes). Exhausting these apertures can either immediately panic 
 * the kernel or lead to failures in other kernel subsystems, depending
 * on the running kernel's behavior.
 *
 * The NVIDIA driver now attempts to avoid exhausting the these pools of
 * memory to avoid these stability problems. Unfortunately, there is no 
 * way to determine the size of these pools at runtime, so the NVIDIA
 * driver has to hard code this limit.
 *
 * This registry key allows the end user to manually tweak this value if
 * necessary. Specifically, if the IOMMU or SWIOTLB aperture is larger
 * than 64 MegaBytes, the end user can adjust this value to take advantage
 * of the larger pool.
 *
 * Note that the default value of this limit is 60 MegaBytes, which leaves
 * 4 MegaBytes available for the rest of the system. If the end user 
 * adjusts this value, it is recommended to leave 4 MegaBytes as well.
 * For example, if the end user wants to adjust this value to account for
 * a 128 MegaByte pool, it is suggested to set this value to 124 MegaBytes.
 *
 */

static int NVreg_RemapLimit = 0;
NV_MODULE_PARAMETER(NVreg_RemapLimit);

/*
 * Option: UpdateMemoryTypes
 *
 * Description:
 *
 * Many kernels have broken implementations of the change_page_attr()
 * kernel interface that may cause cache aliasing problems. Linux/x86-64
 * kernels between 2.6.0 and 2.6.10 may prompt kernel BUG()s due to
 * improper accounting in the interface's large page management code, for
 * example. For this reason, the NVIDIA Linux driver is very careful about
 * not using the change_page_attr() kernel interface on these kernels.
 *
 * Due to the serious nature of the problems that can arise from bugs in
 * the change_page_attr(), set_pages_{uc,wb}() and other kernel interfaces
 * used to modify memory types, the NVIDIA driver implements a manual
 * registry key override to allow forcibly enabling or disabling use of
 * these APIs.
 *
 * Possible values:
 *
 * ~0 = use the NVIDIA driver's default logic (default)
 *  1 = enable use of change_page_attr(), etc.
 *  0 = disable use of change_page_attr(), etc.
 *
 * By default, the NVIDIA driver will attempt to auto-detect if it can
 * safely use the change_page_attr() and other kernel interfaces to modify
 * the memory types of kernel mappings.
 */

static int NVreg_UpdateMemoryTypes = ~0;
NV_MODULE_PARAMETER(NVreg_UpdateMemoryTypes);

/*
 * Option: DetectPrimaryVga
 *
 * Description:
 *
 * The NVIDIA kernel module used to rely on the X server to specify whether
 * a graphics device is the primary device or if needs to have its VBIOS
 * posted. This mechanism has been deprecated in favor of new logic in the
 * kernel module that can determine whether a given device is the primary or
 * a secondary, and post the VBIOS based on that logic.
 *
 * Possible values:
 *
 *  0 = rely on the X server to identify the primary device
 *  1 = rely on the new primary VGA device detection logic (default)
 */

static int NVreg_DetectPrimaryVga = 1;
NV_MODULE_PARAMETER(NVreg_DetectPrimaryVga);

/*
 * Option: RegistryDwords
 *
 * Description:
 *
 * This option accepts a semicolon-separated list of key=value pairs. Each
 * key name is checked agains the table of static options; if a match is
 * found, the static option value is overridden, but invalid options remain
 * invalid. Pairs that do not match an entry in the static option table
 * are passed on to the RM directly.
 *
 * Format:
 *
 *  NVreg_RegistryDwords="<key=value>;<key=value>;..."
 */

char *NVreg_RegistryDwords = NULL;
NV_MODULE_STRING_PARAMETER(NVreg_RegistryDwords);


// 1 - Force sourcing vbios from ROM
// 0 - business as usual
// Temporary WAR for bug 169275

static int NVreg_VbiosFromROM = 0;
NV_MODULE_PARAMETER(NVreg_VbiosFromROM);

// 1 - Panel brightness control enabled
// 0 - Panel brightness control disabled

static int NVreg_EnableBrightnessControl = 0;
NV_MODULE_PARAMETER(NVreg_EnableBrightnessControl);

// Panel brightness PWM frequency

static int NVreg_PanelPWMFrequency = 1018;
NV_MODULE_PARAMETER(NVreg_PanelPWMFrequency);

// Panel brightness limits

static int NVreg_PanelBrightnessLimits = 0x0000ff00;
NV_MODULE_PARAMETER(NVreg_PanelBrightnessLimits);

/*
 * Option: RMEdgeIntrCheck
 *
 * Description:
 *
 * Enable/disable check for edge-triggered interrupts.
 * Please see the common problems section of the readme for more
 * details on edge-triggered interrupt problems.
 *
 * Possible values:
 * 1 - enable edge-triggered interrupt check (default)
 * 0 - disable edge-triggered interrupt check
 */

static int NVreg_RMEdgeIntrCheck = 1;
NV_MODULE_PARAMETER(NVreg_RMEdgeIntrCheck);

/*
 * Option: UsePageAttributeTable
 *
 * Description:
 *
 * Enable/disable use of the page attribute table (PAT) available in
 * modern x86/x86-64 processors to set the effective memory type of memory
 * mappings to write-combining (WC). If disabled, the driver will fall
 * back to using MTRRs, if possible.
 *
 * If enabled, an x86 processor with PAT support is present and the host
 * system's Linux kernel did not configure one of the PAT entries to
 * indicate the WC memory type, the driver will change the second entry in
 * the PAT from its default (write-through (WT)) to WC at module load
 * time. If the kernel did update one of the PAT entries, the driver will
 * not modify the PAT.
 *
 * In both cases, the driver will honor attempts to map memory with the WC
 * memory type by selecting the appropriate PAT entry using the correct
 * set of PTE flags.
 *
 * Possible values:
 *
 * ~0 = use the NVIDIA driver's default logic (default)
 *  1 = enable use of the PAT for WC mappings.
 *  0 = disable use of the PAT for WC mappings.
 */

static int NVreg_UsePageAttributeTable = ~0;
NV_MODULE_PARAMETER(NVreg_UsePageAttributeTable);

/*
 * Option: MapRegistersEarly
 *
 * Description:
 *
 * When this option is enabled, the NVIDIA kernel module will attempt to
 * map the device registers of NVIDIA GPUs at probe(), rather than at
 * open() time. This is useful for debugging purposes, only.
 *
 * Possible Values:
 *
 *   0 = do not map GPU registers early (default)
 *   1 = map GPU registers early
 */

static int NVreg_MapRegistersEarly = 0;
NV_MODULE_PARAMETER(NVreg_MapRegistersEarly);

/*
 * You can enable any of the registry options disabled by default by
 * editing their respective entries in the table below. The last field
 * determines if the option is considered valid - in order for the
 * changes to take effect, you need to recompile and reload the NVIDIA
 * kernel module.
 */

nv_parm_t nv_parms[] = {
    { "NVreg",  "VideoMemoryTypeOverride",  &NVreg_VideoMemoryTypeOverride,  0 },
    { "NVreg",  "EnableVia4x",              &NVreg_EnableVia4x,              1 },
    { "NVreg",  "EnableALiAGP",             &NVreg_EnableALiAGP,             1 },
    { "NVreg",  "NvAGP",                    &NVreg_NvAGP,                    1 },
    { "NVreg",  "ReqAGPRate",               &NVreg_ReqAGPRate,               0 },
    { "NVreg",  "EnableAGPSBA",             &NVreg_EnableAGPSBA,             1 },
    { "NVreg",  "EnableAGPFW",              &NVreg_EnableAGPFW,              1 },
    { "NVreg",  "SoftEDIDs",                &NVreg_SoftEDIDs,                1 },
    { "NVreg",  "Mobile",                   &NVreg_Mobile,                   1 },
    { "NVreg",  "ResmanDebugLevel",         &NVreg_ResmanDebugLevel,         1 },
    { "NVreg",  "FlatPanelMode",            &NVreg_FlatPanelMode,            1 },
    { "NVreg",  "DevicesConnected",         &NVreg_DevicesConnected,         1 },
    { "NVreg",  "RmLogonRC",                &NVreg_RmLogonRC,                1 },
    { "NVreg",  "VbiosFromROM",             &NVreg_VbiosFromROM,             1 },
    { "NVreg",  "ModifyDeviceFiles",        &NVreg_ModifyDeviceFiles,        1 },
    { "NVreg",  "DeviceFileUID",            &NVreg_DeviceFileUID,            1 },
    { "NVreg",  "DeviceFileGID",            &NVreg_DeviceFileGID,            1 },
    { "NVreg",  "DeviceFileMode",           &NVreg_DeviceFileMode,           1 },
    { "NVreg",  "RemapLimit",               &NVreg_RemapLimit,               1 },
    { "NVreg",  "UpdateMemoryTypes",        &NVreg_UpdateMemoryTypes,        1 },
    { "NVreg",  "DetectPrimaryVga",         &NVreg_DetectPrimaryVga,         1 },
    { "NVreg",  "EnableBrightnessControl",  &NVreg_EnableBrightnessControl,  1 },
    { "NVreg",  "PanelPWMFrequency",        &NVreg_PanelPWMFrequency,        1 },
    { "NVreg",  "PanelBrightnessLimits",    &NVreg_PanelBrightnessLimits,    1 },
    { "NVreg",  "RMEdgeIntrCheck",          &NVreg_RMEdgeIntrCheck,          1 },
    { "NVreg",  "UsePageAttributeTable",    &NVreg_UsePageAttributeTable,    1 },
    { "NVreg",  "MapRegistersEarly",        &NVreg_MapRegistersEarly,        1 },
    {  NULL,     NULL,                      NULL,                            0 }
};

static void parse_option_string(void)
{
    unsigned int i, len;
    nv_parm_t *entry;
    char *option_string = NULL;
    char *ptr, *mod;
    char *token;
    char *name, *value;
    U032 data;

    if (NVreg_RegistryDwords != NULL)
    {
        len = strlen(NVreg_RegistryDwords) + 1;

        if (os_alloc_mem((void **)&option_string, len) != RM_OK)
            return;

        ptr = NVreg_RegistryDwords;
        mod = option_string;

        while (*ptr != '\0')
        {
            if (!isspace(*ptr)) *mod++ = *ptr;
            ptr++;
        }
        *mod = '\0';

        ptr = option_string;

        while ((token = strsep(&ptr, ";")) != NULL)
        {
            if (!(name = strsep(&token, "=")) || !strlen(name))
                continue;
            if (!(value = strsep(&token, "=")) || !strlen(value))
                continue;
            if (strsep(&token, "=") != NULL)
                continue;

            data = (U032)simple_strtoul(value, NULL, 0);

            for (i = 0; (entry = &nv_parms[i])->name != NULL; i++)
            {
                if (strcmp(entry->name, name) == 0)
                    break;
            }

            if (!entry->name)
                rm_write_registry_dword(NULL, "NVreg", name, data);
            else
                *entry->data = data;
        }

        os_free_mem(option_string);
    }
}

void NV_API_CALL os_registry_init(void)
{
    nv_parm_t *entry;
    unsigned int i;

    parse_option_string();

    for (i = 0; (entry = &nv_parms[i])->name != NULL; i++)
    {
        if (!entry->valid) continue;
        rm_write_registry_dword(NULL, entry->node, entry->name, *entry->data);
    }
}
