/*
 *  control.c -  Fan control daemon for MacBook
 *
 *  Copyright (C) 2010  Mikael Strom <mikael@sesamiq.com>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>
#include "config.h"

//------------------------------------------------------------------------------

#define SENS_PATH	"/sys/devices/platform/applesmc.768"
#define FAN1_MIN	"/sys/devices/platform/applesmc.768/fan1_min"
#define FAN2_MIN	"/sys/devices/platform/applesmc.768/fan2_min"
#define FAN1_MAN	"/sys/devices/platform/applesmc.768/fan1_manual"
#define FAN2_MAN	"/sys/devices/platform/applesmc.768/fan2_manual"

struct
{
	char *key;
	char *desc;
}
sensor_desc[] =
{
	{"TB0T", "Battery TS_MAX Temp"},
	{"TB1T", "Battery TS1 Temp"},
	{"TB2T", "Battery TS2 Temp"},
	{"TB3T", "Battery Temp"},
	{"TC0D", "CPU 0 Die Temp"},
	{"TC0P", "CPU 0 Proximity Temp"},
	{"TG0D", "GPU Die - Digital"},
	{"TG0P", "GPU 0 Proximity Temp"},
	{"TG0T", "GPU 0 Die - Analog Temp"},
	{"TG0H", "Left Heat Pipe/Fin Stack Proximity Temp"},
	{"TG1H", "Left Heat Pipe/Fin Stack Proximity Temp"},
	{"TN0P", "MCP Proximity"},
	{"TN0D", "MCP Die"},
	{"Th2H", "Right Fin Stack Proximity Temp"},
	{"Tm0P", "Battery Charger Proximity Temp"},
	{"Ts0P", "Palm Rest Temp"}
};
#define N_DESC (sizeof(sensor_desc) / sizeof(sensor_desc[0]))

#define SENSKEY_MAXLEN	16

struct sensor
{
	int id;
	int excluded;
	char name[SENSKEY_MAXLEN];
	float value;
};

//------------------------------------------------------------------------------

int sensor_count = 0;
int fan_count = 0;
float temp_avg = 0;
int fan_speed;

struct sensor *sensors = NULL;
struct sensor *sensor_TC0P = NULL;
struct sensor *sensor_TG0P = NULL;

#define CTL_NONE	0	// sensor control fan flags
#define CTL_AVG		1
#define CTL_TC0P	2
#define CTL_TG0P	3

int fan_ctl = 0;		// which sensor controls fan

//------------------------------------------------------------------------------

void read_sensors()
{
	int i;
	for(i = 0; i < sensor_count; ++i)
	{
		if(! sensors[i].excluded)
		{
			// read temp value

			char fname[512];
			sprintf(fname, "%s/temp%d_input", SENS_PATH, sensors[i].id);

			FILE *fp = fopen(fname, "r");
			if(fp == NULL)
			{
				printf("Error: Can't open %s\n", fname);
				fflush(stdout);
			}
			else
			{
				char val_buf[16];
				int n = fread(val_buf, 1, sizeof(val_buf), fp);
				if(n < 1)
				{
					printf("Error: Can't read  %s\n", fname);
				}
				else
				{
					sensors[i].value = (float)atoi(val_buf) / 1000.0;
				}
				fclose(fp);
			}
		}
	}

	// calc average

	temp_avg = 0.0;
	int active_sensors = 0;

	for(i = 0; i < sensor_count; ++i)
	{
		if(! sensors[i].excluded)
		{
			temp_avg += sensors[i].value;
			++active_sensors;
		}
	}

	temp_avg = temp_avg / active_sensors;
}

//------------------------------------------------------------------------------

void calc_fan()
{
	fan_speed = fan_min;
	fan_ctl = CTL_NONE;

	// calc fan speed on average

	float fan_window = fan_max - fan_min;
	float temp_avg_window = temp_avg_ceiling - temp_avg_floor;
	float normalized_temp = (temp_avg - temp_avg_floor) / temp_avg_window;
	float fan_avg_speed = (normalized_temp * fan_window);
	if(fan_avg_speed > fan_speed)
	{
		fan_speed = fan_avg_speed;
		fan_ctl = CTL_AVG;
	}

	// calc fan speed for TC0P

	if(sensor_TC0P != NULL)
	{
		float temp_window = temp_TC0P_ceiling - temp_TC0P_floor;
		float normalized_temp = (sensor_TC0P->value - temp_TC0P_floor) / temp_window;
		float fan_TC0P_speed = (normalized_temp * fan_window);
		if(fan_TC0P_speed > fan_speed)
		{
			fan_speed = fan_TC0P_speed;
			fan_ctl = CTL_TC0P;
		}
	}

	// calc fan speed for TG0P

	if(sensor_TG0P != NULL)
	{
		float temp_window = temp_TG0P_ceiling - temp_TG0P_floor;
		float normalized_temp = (sensor_TG0P->value - temp_TG0P_floor) / temp_window;
		float fan_TG0P_speed = (normalized_temp * fan_window);
		if(fan_TG0P_speed > fan_speed)
		{
			fan_speed = fan_TG0P_speed;
			fan_ctl = CTL_TG0P;
		}
	}

	// finally clamp

	fan_speed = min(fan_max, fan_speed);
}

//------------------------------------------------------------------------------

void set_fan()
{
	FILE *fp;

	char buf[16];

	// update fan 1

	fp = fopen(FAN1_MIN, "w");
	if(fp == NULL)
	{
		printf("Error: Can't open %s\n", FAN1_MIN);
	}
	else
	{
		sprintf(buf, "%d", fan_speed);
		fwrite(buf, 1, strlen(buf), fp);
		fclose(fp);
	}

	// set fan 1 manual to zero

	fp = fopen(FAN1_MAN, "w");
	if(fp == NULL)
	{
		printf("Error: Can't open %s\n", FAN1_MAN);
	}
	else
	{
		strcpy(buf, "0");
		fwrite(buf, 1, strlen(buf), fp);
		fclose(fp);
	}

	// update fan 2

	if(fan_count > 1)
	{
		fp = fopen(FAN2_MIN, "w");
		if(fp == NULL)
		{
			printf("Error: Can't open %s\n", FAN2_MIN);
		}
		else
		{
			sprintf(buf, "%d", fan_speed);
			fwrite(buf, 1, strlen(buf), fp);
			fclose(fp);
		}

		// set fan 2 manual to zero

		fp = fopen(FAN2_MAN, "w");
		if(fp == NULL)
		{
			printf("Error: Can't open %s\n", FAN2_MAN);
		}
		else
		{
			strcpy(buf, "0");
			fwrite(buf, 1, strlen(buf), fp);
			fclose(fp);
		}
	}

	fflush(stdout);
}

//------------------------------------------------------------------------------

void adjust()
{
	read_sensors();
	calc_fan();
	set_fan();
}

//------------------------------------------------------------------------------

void scan_sensors()
{
	int i;
	int j;
	struct stat buf;
	int result;

	sensor_TC0P = NULL;
	sensor_TG0P = NULL;

	// get number of fans

	result = stat(FAN1_MIN, &buf);
	if(result != 0)
	{
		printf("No fans detected, terminating!\n");
		exit(-1);
	}
	else
	{
		fan_count = 1;
	}

	result = stat(FAN2_MIN, &buf);
	if(result != 0)
	{
		printf("Found 1 fan.\n");
	}
	else
	{
		fan_count = 2;
		printf("Found 2 fans.\n");
	}

	// count number of sensors

	int count = 0;	
	while(count < 100)	// more than 100 sensors is an error!
	{
		char fname[512];

		// sensor numbering start at 1
		sprintf(fname, "%s/temp%d_input", SENS_PATH, count + 1);
		result = stat(fname, &buf);

		if(result == 0)
		{
			++count;
		}
		else
		{
			break;		// done
		}
	}

	sensor_count = count;

	if(sensor_count > 0)
	{
		// Get sensor id, labels and descriptions, check exclude list

		if(sensors != NULL)
		{
			free(sensors);
		}

		sensors = malloc(sizeof(struct sensor) * sensor_count);
		assert(sensors != NULL);

		printf("Found %d sensors:\n", sensor_count);

		for(i = 0; i < sensor_count; ++i)
		{
			char fname[512];

			// set id and check exclude list
			sensors[i].id = i + 1;
			sensors[i].excluded = 0;
			
			for(j = 0; j < MAX_EXCLUDE && exclude[j] != 0; ++j)
			{
				if(exclude[j] == sensors[i].id)
				{
					sensors[i].excluded = 1;
					break;
				}
			}

			// read label
			sprintf(fname, "%s/temp%d_label", SENS_PATH, sensors[i].id);

			sensors[i].name[0] = 0; // set zero length

			FILE *fp = fopen(fname, "r");
			if(fp == NULL)
			{
				printf("Error: Can't open %s\n", fname);
			}
			else
			{
				char key_buf[SENSKEY_MAXLEN];
				memset(key_buf, 0, SENSKEY_MAXLEN);

				int n = fread(key_buf, 1, SENSKEY_MAXLEN - 1, fp);
				if(n < 1)
				{
					printf("Error: Can't read  %s\n", fname);
				}
				else
				{
					char *p_endl = strrchr(key_buf, '\n');
					if(p_endl)
					{
						*p_endl = 0; 	// remove '\n'
					}
					strncpy(sensors[i].name, key_buf, SENSKEY_MAXLEN);
				}
				fclose(fp);
			}
		}

		for(i = 0; i < sensor_count; ++i)		// for each label found
		{
			if(! sensors[i].excluded)
			{
				// try to find TC0P and TG0P
				// if found, assign sensor_TC0P and sensor_TG0P for later use

				if(strcmp(sensors[i].name, "TC0P") == 0)
				{
					sensor_TC0P = &sensors[i];
				}
				else if(strcmp(sensors[i].name, "TG0P") == 0)
				{
					sensor_TG0P = &sensors[i];
				}
			}

			// print out sensor information. 

			printf("\t%2d: ", sensors[i].id);

			int found = 0;
			for(j = 0; j < N_DESC && ! found; ++j)		// find in descriptions table
			{
				if(strcmp(sensors[i].name, sensor_desc[j].key) == 0)
				{
					found = 1;
					printf("%s - %s", sensor_desc[j].key, sensor_desc[j].desc);
				}
			}
			if(! found)
			{
				printf("%s - ?", sensors[i].name);
			}

			printf(" %s\n", sensors[i].excluded ? "   ***EXCLUDED***" : "");
		}
	}
	else
	{
		printf("No sensors detected, terminating!\n");
		exit(-1);
	}

	fflush(stdout);
}

//------------------------------------------------------------------------------

void logger()
{
	int i;

	if(log_level > 0)
	{
		printf("Speed: %d, %sAVG: %.1fC" ,
				fan_speed,
				fan_ctl == CTL_AVG ? "*" : " ",
				temp_avg);

		if(sensor_TC0P != NULL)
		{
			printf(", %sTC0P: %.1fC" ,
					fan_ctl == CTL_TC0P ? "*" : " ",
					sensor_TC0P->value);
		}

		if(sensor_TG0P != NULL)
		{
			printf(", %sTG0P: %.1fC" ,
					fan_ctl == CTL_TG0P ? "*" : " ",
					sensor_TG0P->value);
		}

		if(log_level > 1)
		{
			printf(", Sensors: ");
			for(i = 0; i < sensor_count; ++i)
			{
				if(! sensors[i].excluded)
				{
					printf("%s:%.0f ", sensors[i].name, sensors[i].value);
				}
			}
		}

		printf("\n");
		fflush(stdout);
	}
}

//------------------------------------------------------------------------------

