/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutk/clutk.h>
#include "nl-notify.h"

#include "nl-config.h"
#include "nl-defines.h"
#include "nl-texture-frame.h"

G_DEFINE_TYPE (NlNotify, nl_notify, CLUTTER_TYPE_GROUP);

#define NL_NOTIFY_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_NOTIFY, \
  NlNotifyPrivate))

struct _NlNotifyPrivate
{
  ClutterActor *bg;
  ClutterActor *win;
  ClutterActor *button;

  gint pid;
  gint tag;
};

static gboolean
_timeout (NlNotify *notify)
{
  NlConfig *cfg = nl_config_get_default ();

  clutter_ungrab_pointer ();

  if (cfg->low_graphics)
    {
      clutter_actor_set_opacity (CLUTTER_ACTOR (notify), 0);
      clutter_actor_hide (CLUTTER_ACTOR (notify));
    }
  else
    {
      clutter_actor_animate (CLUTTER_ACTOR (notify), CLUTTER_EASE_OUT_SINE,
                             MID_TIME, "opacity", 0, NULL);
    }
  return FALSE;
}

void
nl_notify_popup (NlNotify    *notify,
                 const gchar *label,
                 CtkImage    *image,
                 gint         pid)
{
  NlNotifyPrivate *priv;
  NlConfig  *cfg = nl_config_get_default ();
  gchar     *text;
  CtkButton *button;
  gfloat     nat_width;

  g_return_if_fail (NL_IS_NOTIFY (notify));
  g_return_if_fail (CTK_IS_IMAGE (image));
  priv = notify->priv;

  priv->pid = pid;
  button = CTK_BUTTON (priv->button);

  clutter_actor_raise_top (CLUTTER_ACTOR (notify));
  clutter_actor_show (CLUTTER_ACTOR (notify));

  ctk_button_set_image (button, image);

  text = g_strdup_printf ("<big><b>%s</b></big>", label);
  clutter_text_set_markup (CLUTTER_TEXT (ctk_button_get_text (button)), text);
  g_free (text);

  clutter_actor_get_preferred_width (priv->button,
                                     cfg->monitor_width,
                                     &nat_width,
                                     &nat_width);

  nat_width *= 1.2;
  clutter_actor_set_width (priv->win, nat_width);
  clutter_actor_set_x (priv->win, (cfg->monitor_width-nat_width)/2.0);
    
  if (cfg->low_graphics)
    clutter_actor_set_opacity (CLUTTER_ACTOR (notify), 255);
  else
    clutter_actor_animate (CLUTTER_ACTOR (notify), CLUTTER_EASE_OUT_SINE,
                           MID_TIME, "opacity", 255, NULL);

  clutter_grab_pointer (CLUTTER_ACTOR (notify));

  priv->tag = g_timeout_add (3000, (GSourceFunc)_timeout, notify);
}

static void
allocate (ClutterActor          *actor,
          const ClutterActorBox *box,
          ClutterAllocationFlags flags)
{
  NlNotifyPrivate *priv = NL_NOTIFY (actor)->priv;
  ClutterActorBox child_box;
  gfloat nat_width;
  gfloat nat_height;
  gfloat width, height;

  clutter_actor_get_preferred_height (priv->win,
                                      box->x2 - box->x1,
                                      &nat_height,
                                      &nat_height);
  clutter_actor_get_preferred_width (priv->button,
                                     box->y2 - box->y1,
                                     &nat_width,
                                     &nat_width);
  nat_width += 18;

  width = box->x2 - box->x1;
  height = box->y2 - box->y1;

  child_box.x1 = (gint)((width/2) - (nat_width/2));
  child_box.x2 = (gint)(child_box.x1 + nat_width);
  child_box.y1 = (gint)((height/2) - (nat_height/2));
  child_box.y2 = (gint)(child_box.y1 + nat_height);

  clutter_actor_allocate (priv->win, &child_box, flags);

  clutter_actor_allocate (priv->bg, box, flags);
}

/* GObject stuff */
static void
nl_notify_class_init (NlNotifyClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);

  act_class->allocate = allocate;

  g_type_class_add_private (obj_class, sizeof (NlNotifyPrivate));
}

static void
nl_notify_init (NlNotify *notify)
{
  NlNotifyPrivate *priv;
  NlConfig        *cfg = nl_config_get_default ();
  ClutterColor     black = { 0x00, 0x00, 0x00, 0x99 };
  gfloat           width_factor;
  ClutterActor    *temp, *label;
  CtkPadding       padding = { 15, 15, 15, 15 };

  priv = notify->priv = NL_NOTIFY_GET_PRIVATE (notify);

  if (cfg->is_portrait)
    width_factor = 0.6;
  else
    width_factor = 0.4;

  clutter_actor_set_reactive (CLUTTER_ACTOR (notify), FALSE);
  clutter_actor_set_opacity (CLUTTER_ACTOR (notify), 0);

  priv->bg = clutter_rectangle_new_with_color (&black);
  clutter_container_add_actor (CLUTTER_CONTAINER (notify), priv->bg);
  clutter_actor_set_size (priv->bg, cfg->monitor_width, cfg->monitor_height);
  clutter_actor_set_position (priv->bg, 0, 0);
  clutter_actor_show (priv->bg);

  priv->win = ctk_vbox_new (12);
  ctk_actor_set_padding (CTK_ACTOR (priv->win), &padding);
  clutter_container_add_actor (CLUTTER_CONTAINER (notify), priv->win);
  clutter_actor_show (priv->win);

  temp = clutter_texture_new_from_file (PKGDATADIR"/iconview_normal.png",NULL);
  ctk_actor_set_background (CTK_ACTOR (priv->win),
                            nl_texture_frame_new (CLUTTER_TEXTURE (temp),
                                                  25, 25, 25, 25));

  label = ctk_text_new (_("Opening..."));
  clutter_text_set_line_wrap (CLUTTER_TEXT (label), TRUE);
  clutter_text_set_line_wrap_mode (CLUTTER_TEXT (label),
                                   PANGO_WRAP_WORD_CHAR);
  ctk_box_pack (CTK_BOX (priv->win), label, FALSE, FALSE);
  ctk_text_set_alignment (CTK_TEXT (label), PANGO_ALIGN_CENTER);
  clutter_actor_show (label);

  priv->button = ctk_button_new (CTK_ORIENTATION_HORIZONTAL);
  clutter_actor_set_reactive (priv->button, FALSE);
  ctk_box_pack (CTK_BOX (priv->win), priv->button, FALSE, TRUE);
  clutter_actor_show (priv->button);

  padding.top = padding.bottom = padding.left = 0.0;
  ctk_actor_set_padding (CTK_ACTOR (priv->button), &padding);
}

NlNotify *
nl_notify_get_default (void)

{
  static NlNotify *notify = NULL;

  if (!notify)
    notify = g_object_new (NL_TYPE_NOTIFY,
                           NULL);

  return notify;
}
