/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MAlignmentObject.h"

#include <datatype/DNASequence.h>

namespace GB2 {

GObject* MAlignmentObject::clone() const {
	MAlignmentObject* cln = new MAlignmentObject(msa, getGHintsMap());
    cln->setIndexInfo(getIndexInfo());
    return cln;
}


void MAlignmentObject::insertGap(int seqNum, int pos, int nGaps) {
    int aliLen = msa.getLength();
    int nSeqs = msa.getNumSequences();
    assert(seqNum >=0 && seqNum <= nSeqs);
    assert(pos >=0 && pos <= aliLen);
    assert(nGaps > 0);
    assert(msa.isNormalized());
    assert(!isStateLocked());
    
    MAlignment maBefore = msa; //TODO: high mem use probabilty -> limit and use empty maBefore?

    int nGapsAtEnd = 0;
    MAlignmentItem& affItem = msa.alignedSeqs[seqNum];
    for (; nGapsAtEnd < nGaps; nGapsAtEnd++) {
        char c = affItem.sequence[aliLen - 1 - nGapsAtEnd];
        if (c != MAlignment_GapChar) {
            break;
        }
    }
    //now trim nGapsAtEnd in affected item
    if (nGapsAtEnd > 0) {
        affItem.sequence.resize(aliLen - nGapsAtEnd);
    }

    //insert nGaps to the selected sequence
    QByteArray gapAff(nGaps, MAlignment_GapChar);
    affItem.sequence.insert(pos, gapAff);

    //insert nGaps - nGapsAtEnd to other sequences
    if (nGaps - nGapsAtEnd > 0) {
        QByteArray gapOthers(nGaps - nGapsAtEnd , MAlignment_GapChar);
        for (int i=0; i < nSeqs; i++) {
            MAlignmentItem& item = msa.alignedSeqs[i];
            if (i != seqNum) {
                item.sequence.append(gapOthers);
            }
        }
    }

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::insertGap(int pos, int nGaps) {
    assert(pos >= 0 && pos <= msa.getLength());
    assert(nGaps > 0);
    assert(msa.isNormalized());
    assert(!isStateLocked());

    MAlignment maBefore = msa;
    QByteArray gap(nGaps, MAlignment_GapChar);
    for (int i=0, n = msa.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = msa.alignedSeqs[i];
        item.sequence.insert(pos, gap);
    }

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);
}

int MAlignmentObject::deleteGap(int seqNum, int pos, int maxGaps) {
    assert(seqNum >=0 && seqNum <= msa.getNumSequences());
    assert(maxGaps > 0 && pos + maxGaps <= msa.getLength());
    assert(pos >=0 && pos+maxGaps <= msa.getLength());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    MAlignment maBefore = msa;

    int nDeleted = 0;
    MAlignmentItem& delItem = msa.alignedSeqs[seqNum];
    for(int i = pos, n = pos + maxGaps; i<n; i++) {
        char c = delItem.sequence[i];
        if (c != MAlignment_GapChar) {
            break;
        }
        nDeleted++;
    }
    if (nDeleted == 0) {
        return 0;
    }
    delItem.sequence.remove(pos, nDeleted);
    delItem.sequence.append(QByteArray(nDeleted, MAlignment_GapChar));

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);

    return nDeleted;
}

int MAlignmentObject::deleteGap(int pos, int maxGaps) {
    assert(pos >=0 && pos <= msa.getLength());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    MAlignment maBefore = msa;
    //find min gaps in all sequences starting from pos
    int minGaps = maxGaps;
    foreach(const MAlignmentItem& item, msa.alignedSeqs) {
        int nGaps = 0;
        for (int i=pos, n = pos+maxGaps; i < n; i++, nGaps++) {
            if (item.sequence[i]!=MAlignment_GapChar) { 
                break;
            }
        }
        minGaps = qMin(minGaps, nGaps);
        if (minGaps == 0) {
            break;
        }
    }
    if (minGaps == 0) {
        return  0;
    }
    int nDeleted = minGaps;
    for (int i=0, n = msa.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = msa.alignedSeqs[i];
        item.sequence.remove(pos, nDeleted);
    }

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);

    return nDeleted;
}

void MAlignmentObject::addSequence(const DNASequence& seq, int seqIdx) {
    assert(!isStateLocked());
    assert(msa.isNormalized());
    assert(seq.alphabet == msa.alphabet);
    assert(seq.length() == msa.getLength());
    
    MAlignment maBefore = msa;

    MAlignmentItem mai(seq.getName(), seq.seq);
    if (seqIdx < 0 || seqIdx >= msa.getLength()) {
        msa.alignedSeqs.append(mai);
    } else {
        msa.alignedSeqs.insert(seqIdx, mai);
    }

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceContentChanged = false;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::removeSequence(int seqNum) {
    assert(seqNum >= 0 && seqNum <= msa.getNumSequences());
    assert(msa.isNormalized());
    assert(!isStateLocked());

    MAlignment maBefore = msa;

    msa.alignedSeqs.removeAt(seqNum);

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceContentChanged = false;
    emit si_alignmentChanged(maBefore, mi);
    
}


void MAlignmentObject::setMAlignment(const MAlignment& newMa) {
    assert(!isStateLocked());
    assert(newMa.isNormalized());

    MAlignment maBefore = msa;

    msa = newMa;
    msa.setName( getGObjectName() );
    
    setModified(true);

    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::setGObjectName(const QString& newName) {
    msa.setName( newName );
    GObject::setGObjectName(newName);
}

}//namespace


