/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorNameList.h"
#include "MSAEditor.h"
#include "MSAEditorSequenceArea.h"

#include <gobjects/MAlignmentObject.h>
#include <datatype/MAlignment.h>
#include <util_gui/GUIUtils.h>

#include <QtGui/QApplication>
#include <QtGui/QClipboard>
#include <QtGui/QPainter>
#include <QtGui/QMouseEvent>

namespace GB2 {

MSAEditorNameList::MSAEditorNameList(MSAEditorUI* _ui, QScrollBar* _nhBar) : editor(_ui->editor), ui(_ui), nhBar(_nhBar) {
    setFocusPolicy(Qt::WheelFocus);
    cachedView = new QPixmap();
    completeRedraw = true;
    
    connect(editor, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    
    copyCurrentSequenceAction = new QAction(tr("copy_current_sequence"), this);
    connect(copyCurrentSequenceAction, SIGNAL(triggered()), SLOT(sl_copyCurrentSequence()));

    removeCurrentSequenceAction = new QAction("Remove current sequence", this);
    connect(removeCurrentSequenceAction, SIGNAL(triggered()), SLOT(sl_removeCurrentSequence()));
        
    connect(editor, SIGNAL(si_buildPopupMenu(GObjectView* , QMenu*)), SLOT(sl_buildContextMenu(GObjectView*, QMenu*)));
    connect(editor->getMSAObject(), SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
        SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));

    connect(editor->getMSAObject(), SIGNAL(si_lockedStateChanged()), SLOT(sl_lockedStateChanged()));

    connect(ui->seqArea, SIGNAL(si_startChanged(const QPoint& , const QPoint& )), SLOT(sl_startChanged(const QPoint& , const QPoint&)));
    connect(ui->seqArea, SIGNAL(si_cursorMoved(const QPoint& , const QPoint& )), SLOT(sl_cursorMoved(const QPoint& , const QPoint& )));
    connect(ui->seqArea, SIGNAL(si_scaleChanged()), SLOT(sl_scaleChanged()));
    
    nhBar->setEnabled(false);
    updateActions();
}

MSAEditorNameList::~MSAEditorNameList() {
    delete cachedView;
}

void MSAEditorNameList::updateActions() {
    copyCurrentSequenceAction->setEnabled(true);
    
    MAlignmentObject* maObj = editor->getMSAObject();
    removeCurrentSequenceAction->setEnabled(!maObj->isStateLocked() && maObj->getMAlignment().getNumSequences() > 1);
}

#define MARGIN_TEXT_LEFT 5
#define MARGIN_TEXT_TOP 2
#define MARGIN_TEXT_BOTTOM 2

void MSAEditorNameList::updateScrollBar() {
    nhBar->disconnect(this);

    QFont f = ui->seqArea->getFont();
    f.setItalic(true);
    QFontMetrics fm(f);
    int maxNameWidth = 0;
    foreach(const MAlignmentItem& mai, editor->getMSAObject()->getMAlignment().alignedSeqs) {
        maxNameWidth = qMax(fm.width(mai.name), maxNameWidth);
    }

    int availableWidth = width() - MARGIN_TEXT_LEFT;
    int nSteps = 1;
    int stepSize = fm.width('W');
    if (availableWidth < maxNameWidth) {
        int dw = maxNameWidth - availableWidth;
        nSteps += dw / stepSize + (dw % stepSize != 0 ? 1 : 0);
    }
    nhBar->setMinimum(0);
    nhBar->setMaximum(nSteps - 1);
    nhBar->setValue(0);

    nhBar->setEnabled(nSteps > 1);
    connect(nhBar, SIGNAL(valueChanged(int)), SLOT(sl_nameBarMoved(int)));
}


void MSAEditorNameList::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorNameList::sl_buildContextMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorNameList::buildMenu(QMenu* m) {    
    QMenu* copyMenu = GUIUtils::findSubMenu(m, MSAE_MENU_COPY);
    assert(copyMenu!=NULL);
    copyMenu->addAction(copyCurrentSequenceAction);

    QMenu* editMenu = GUIUtils::findSubMenu(m, MSAE_MENU_EDIT);
    assert(editMenu!=NULL);
    editMenu->addAction(removeCurrentSequenceAction);
}


void MSAEditorNameList::sl_copyCurrentSequence() {
    int n = ui->seqArea->getCursorPos().y();
    const MAlignmentItem& item = editor->getMSAObject()->getMAlignment().alignedSeqs[n];
    QApplication::clipboard()->setText(QString(item.sequence));
}

void MSAEditorNameList::sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo& mi) {
    if (mi.sequenceListChanged) {
        completeRedraw = true;
        updateScrollBar();
        update();
    }
}

void MSAEditorNameList::sl_nameBarMoved(int) {
    completeRedraw = true;
    update();
}

void MSAEditorNameList::sl_removeCurrentSequence() {
    int n = ui->seqArea->getCursorPos().y();
    MAlignmentObject* maObj = editor->getMSAObject();
    assert(!maObj->isStateLocked());
    MAlignment ma = maObj->getMAlignment();
    assert(ma.getNumSequences() > 1);
    ma.alignedSeqs.removeAt(n);
    maObj->setMAlignment(ma);
}

void MSAEditorNameList::sl_lockedStateChanged() {
    updateActions();
}

void MSAEditorNameList::resizeEvent(QResizeEvent* e) {
    completeRedraw = true;
    updateScrollBar();
    QWidget::resizeEvent(e);
}

void MSAEditorNameList::paintEvent(QPaintEvent*) {
    drawAll();
}

void MSAEditorNameList::keyPressEvent (QKeyEvent *e) {
    int key = e->key();
    switch(key) {
        case Qt::Key_Up:
            ui->seqArea->moveCursor(0, -1);    
            break;
        case Qt::Key_Down:
            ui->seqArea->moveCursor(0,  1);    
            break;
        case Qt::Key_Left:
            nhBar->triggerAction(QAbstractSlider::SliderSingleStepSub);
            break;
        case Qt::Key_Right:
            nhBar->triggerAction(QAbstractSlider::SliderSingleStepAdd);
            break;
        case Qt::Key_Home:
            ui->seqArea->setFirstVisibleSequence(0);
            ui->seqArea->setCursorPos(ui->seqArea->getCursorPos().x(), 0);
            break;
        case Qt::Key_End:
            {
                int s = editor->getNumSequences() - 1;
                ui->seqArea->setFirstVisibleSequence(s);
                ui->seqArea->setCursorPos(ui->seqArea->getCursorPos().x(), s);
            }
            break;
        case Qt::Key_PageUp:
            { 
                int nVis = ui->seqArea->getNumVisibleSequences(false);
                int fp = qMax(0, ui->seqArea->getFirstVisibleSequence() - nVis);
                int cp = qMax(0, ui->seqArea->getCursorPos().y() - nVis);
                ui->seqArea->setFirstVisibleSequence(fp);
                ui->seqArea->setCursorPos(ui->seqArea->getCursorPos().x(), cp);
            }
            break;
        case Qt::Key_PageDown:
            { 
                int nVis = ui->seqArea->getNumVisibleSequences(false);
                int nSeq = editor->getNumSequences();
                int fp = qMin(nSeq-1, ui->seqArea->getFirstVisibleSequence() + nVis);
                int cp = qMin(nSeq-1, ui->seqArea->getCursorPos().y() + nVis);
                ui->seqArea->setFirstVisibleSequence(fp);
                ui->seqArea->setCursorPos(ui->seqArea->getCursorPos().x(), cp);
            } 
            break;
    }
    QWidget::keyPressEvent(e);
}

void MSAEditorNameList::mousePressEvent(QMouseEvent *e) {
    int y = e->y();
    int s = ui->seqArea->getSequenceNumByY(y);
    if (s !=-1 ) {
        QPoint pos = ui->seqArea->getCursorPos();
        pos.setY(s);
        ui->seqArea->setCursorPos(pos);
    }
    QWidget::mousePressEvent(e);
}

void MSAEditorNameList::wheelEvent (QWheelEvent * we) {
    bool toMin = we->delta() > 0;
    ui->seqArea->getVBar()->triggerAction(toMin ? QAbstractSlider::SliderSingleStepSub : QAbstractSlider::SliderSingleStepAdd);
    QWidget::wheelEvent(we);
}


void MSAEditorNameList::sl_startChanged(const QPoint& p, const QPoint& prev) {
    if (p.y() == prev.y()) {
        return;
    }
    completeRedraw = true;
    update();
}

void MSAEditorNameList::sl_cursorMoved(const QPoint& p, const QPoint& prev) {
    if (p.y() == prev.y()) {
        return;
    }
    update();
}

void MSAEditorNameList::focusInEvent(QFocusEvent* fe) {
    QWidget::focusInEvent(fe);
    update();
}

void MSAEditorNameList::focusOutEvent(QFocusEvent* fe) {
    QWidget::focusOutEvent(fe);
    update();
}

void MSAEditorNameList::sl_scaleChanged() {
    completeRedraw = true; 
    updateScrollBar(); 
    update();
}

//////////////////////////////////////////////////////////////////////////
// draw methods
void MSAEditorNameList::drawAll() {
    QSize s = size();
    if (cachedView->size() != s) {
        assert(completeRedraw);
        delete cachedView;
        cachedView = new QPixmap(s);
    }
    if (completeRedraw) {
        QPainter pCached(cachedView);
        drawContent(pCached);
        completeRedraw = false;
    }
    QPainter p(this);
    p.drawPixmap(0, 0, *cachedView);
    drawSelection(p);
    //drawFocus(p);
}

void MSAEditorNameList::drawContent(QPainter& p) {
    p.fillRect(cachedView->rect(), Qt::white);
    int startSeq = ui->seqArea->getFirstVisibleSequence(); 
    int lastSeq = ui->seqArea->getLastVisibleSequence(true);
    for (int s = startSeq; s <= lastSeq; s++) {
        drawSequenceItem(p, s, false);
    }
}

void MSAEditorNameList::drawSequenceItem(QPainter& p, int s, bool selected) {
    QFont f = ui->seqArea->getFont();
    f.setItalic(true);
    QFontMetrics fm(f);
    p.setFont(f);

    int w = width();
    LRegion yRange = ui->seqArea->getSequenceYRange(s, true);
    QRect itemRect(0, yRange.startPos, w-1, yRange.len-1);
    int textX = MARGIN_TEXT_LEFT;
    int textW = w - MARGIN_TEXT_LEFT;
    int textY = yRange.startPos + MARGIN_TEXT_TOP;
    int textH = yRange.len - MARGIN_TEXT_TOP - MARGIN_TEXT_BOTTOM;

    const MAlignment& ma = editor->getMSAObject()->getMAlignment();
    const MAlignmentItem& mai = ma.alignedSeqs[s];
    QRect textRect(textX, textY, textW, textH);
    if (nhBar->isEnabled()) {
        int stepSize = fm.width('W');
        int dx = stepSize * nhBar->value();
        textRect = textRect.adjusted(-dx, 0, 0, 0);
    }

    if (selected) {
        p.setPen(QPen(Qt::gray, 1, Qt::DashLine));
        p.drawRect(itemRect);
    }
    p.setPen(Qt::black);
    p.drawText(textRect, Qt::AlignVCenter | Qt::AlignLeft, mai.name);
}

void MSAEditorNameList::drawSelection(QPainter& p) {
    int s = ui->seqArea->getCursorPos().y();
    drawSequenceItem(p, s, true);
}

void MSAEditorNameList::drawFocus(QPainter& p) {
    if (hasFocus()) {
        p.setPen(QPen(Qt::black, 1, Qt::DotLine));
        p.drawRect(0, 0, width()-1, height()-1);
    }
}

}//namespace
