/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <limits.h>

#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/ScriptRegistry.h>

#include <datatype/DNASequence.h>

#include <util_tasks/TaskSignalMapper.h>
#include <util_tasks/FailTask.h>

#include <workflow/ConfigurationEditor.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>

#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_library/BioDatatypes.h>

#include "RemoteRequestWorker.h"
#include "ScriptEngineContext.h"

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_PLUGIN_REMOTE_REQUEST);

const QString RemoteRequestWorkerFactory::ACTOR("remoterequest");

const static QString SEQ_PORT("seq");
const static QString OUT_PORT("out");

const static QString SCRIPT_ATTR( "1script" );
const static QString STRAND_ATTR( "2strand" );
const static QString MAXRESLEN_ATTR( "3maxreslen" );
const static QString MINRESLEN_ATTR( "4minreslen" );

QString RemoteRequestPromter::composeRichDoc() {
    BusPort * input = qobject_cast<BusPort *> ( target->getPort(SEQ_PORT) );
    Actor * seqProducer = input->getProducer( SEQ_PORT );
    QString seqProducerText = seqProducer ? tr("from <u>%1</u>,").arg(seqProducer->getLabel()) : QString();

    QString scriptName = getRequiredParam( SCRIPT_ATTR );
    RemoteRequestTaskSettings cfg;
    cfg.maxrl = getParameter( MAXRESLEN_ATTR ).toInt();
    cfg.minrl = getParameter( MINRESLEN_ATTR ).toInt();
    cfg.strand = SendSelectionStrand( getParameter(STRAND_ATTR).toInt() );
    QString strandName;
    switch( cfg.strand ) {
        case SendSelectionStrand_Both: strandName = RemoteRequestWorker::tr("both strands"); break;
        case SendSelectionStrand_Direct: strandName = RemoteRequestWorker::tr("direct strand"); break;
        case SendSelectionStrand_Complement: strandName = RemoteRequestWorker::tr("complement strand"); break;
    }
    
    QString doc = tr("Query each sequence %1 to remote database using adapter <u>%2</u>."
        "<br>Save results which are shorter than <u>%4</u> and longer than <u>%3</u>." 
        "<br>Prefer sending <u>%5</u> and translate the query according to remote database requirements.")
        .arg( seqProducerText )
        .arg( scriptName )
        .arg( cfg.minrl )
        .arg( cfg.maxrl )
        .arg( strandName );
    return doc;
}

void RemoteRequestWorker::init() {
    seqPort = ports.value(SEQ_PORT);
    outPort = ports.value(OUT_PORT);

    taskCfg.maxrl = actor->getParameter( MAXRESLEN_ATTR )->value.toInt(); 
    taskCfg.minrl = actor->getParameter( MINRESLEN_ATTR )->value.toInt();

    //init script
    ServiceRegistry * sr = AppContext::getServiceRegistry();
    ScriptRegistryService * srs = qobject_cast<ScriptRegistryService *>( sr->findServices(Service_ScriptRegistry)[0] );
    const QString scrName = actor->getParameter( SCRIPT_ATTR )->value.toString();
    taskCfg.script = srs->getScript( scrName, ScriptTypes::SCRIPT_TYPE_HTTP_ANNOTATOR );
    taskCfg.strand = SendSelectionStrand( actor->getParameter( STRAND_ATTR )->value.toInt() );
}

bool RemoteRequestWorker::isReady() {
    return seqPort->hasMessage();
}

namespace {
void logScriptException( const QScriptValue & exc, const QScriptEngine & engine ) {
    QString exc_msg = exc.isError() ? exc.property("message").toString() : exc.toString();
    log.error( RemoteRequestWorker::tr("Exception was thrown during the script initialization: ") + exc_msg );
    QStringList backtrace = engine.uncaughtExceptionBacktrace();
    if( !backtrace.isEmpty() ) {
        log.details( RemoteRequestWorker::tr("exception_backtrace:") );
        foreach( QString s, backtrace ) {
            log.details(s);
        }
    }
}
}

Task * RemoteRequestWorker::tick() {
    struct TickError {
        TickError(){}
        TickError( const QString & err ) : what(err) {}
        QString what;
    };
    TickError err;


    DNASequence inputSeq = seqPort->get().getData().value<DNASequence>();
    try {
        if( inputSeq.isNull() ) {
            throw TickError( tr("Null query supplied to RemoteRequestWorker: %1").arg(inputSeq.getName()) );
        }
        if( !taskCfg.script || !taskCfg.script->isReady() ) {
            //TODO: must be an invalid situation
            throw TickError( tr("Script is unregistered or invalid") );
        }
        //obtain some script parameters
        HttpAnnotatorAlphabet scrAl;   //alphabet which is needed for correct script work
        HttpAnnotatorStrand scrStrand; //strand in which script performs the search
        int maxQueryLength = 0;        //maximum length of query which script can accept
        {
            QScriptEngine engine;
            QScriptValue exception = engine.nullValue();
            taskCfg.script->init_engine( &engine, &exception );
            if( exception.isError() ) {
                logScriptException( exception, engine );
                throw TickError( tr("Exception was thrown during the script initialization") );
            }
            scrAl = ScriptHttpAnnotatorContext::getAlphabet( &engine );
            maxQueryLength = ScriptHttpAnnotatorContext::getMaxQueryLen( &engine );
            scrStrand = ScriptHttpAnnotatorContext::getStrand( &engine );
        }

        taskCfg.query = QByteArray( inputSeq.constData(), inputSeq.length() );
        bool isAmino = inputSeq.alphabet->isAmino();

        //init strand:
        //if the Script reports that it searches in both strands itself, only direct strand will be sent. 
        if( HttpAnnotatorStrand_Both == scrStrand ) {
            taskCfg.strand = SendSelectionStrand_Direct;
        } //otherwise, user-chosen value will be used. It is set in init().

        //init complement translation
        if( !isAmino && SendSelectionStrand_Direct != taskCfg.strand ) {
            QList<GB2::DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(inputSeq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                taskCfg.complT = compTTs.first(); 
            } else {
                if( SendSelectionStrand_Complement == taskCfg.strand ) {
                    throw TickError( tr("No Nucl->Compl translations found") );
                } 
                log.info( tr("No Nucl->Compl translations found. Only direct strand will be sent to remote database") );
                taskCfg.complT = 0;
                taskCfg.strand = SendSelectionStrand_Direct;
            }
        }
        //init amino translation 
        if( !isAmino ) {
            if( HttpAnnotatorAlphabet_Amino == scrAl ) {
                DNATranslationType dnaTranslType = (inputSeq.alphabet->getType() == DNAAlphabet_NUCL) ? 
                                                    DNATranslationType_NUCL_2_AMINO : DNATranslationType_RAW_2_AMINO;
                QList<GB2::DNATranslation*> TTs = AppContext::getDNATranslationRegistry()->lookupTranslation(inputSeq.alphabet, dnaTranslType);
                if( !TTs.isEmpty() ) {
                    taskCfg.aminoT = TTs.first();
                } else {
                    throw TickError( tr("No Nucl->Protein translations found. Nucleotide query cannot be searched in protein database") );
                }
            }
        } else if( HttpAnnotatorAlphabet_Amino != scrAl ) {
            throw TickError( tr("Protein sequence cannot be searched in nucleotide database") );
        }

        //truncate the query according to the maximum input accepted by script.
        //TODO: consider splitting the query and adding corresponding option to UI?
        int from = maxQueryLength * (taskCfg.aminoT ? 3 : 1);
        int howmuch = taskCfg.query.size() - from;
        taskCfg.query.remove( from, howmuch );
    } catch( const TickError & exc ) {
        if( failFast ) {
            return new FailTask( exc.what );
        } 

        outPort->put( Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()) );
        if( seqPort->isEnded() ) {
            outPort->setEnded();
        }
        log.error( exc.what );
        return 0;
    }

    Task * t = new RemoteRequestTask( taskCfg );
    connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_onTaskFinished(Task*)));
    return t;
}

void RemoteRequestWorker::cleanup() {
}

bool RemoteRequestWorker::isDone() {
    return seqPort->isEnded();
}

void RemoteRequestWorker::sl_onTaskFinished( Task * t ) {
    RemoteRequestTask * rrTask = qobject_cast<RemoteRequestTask *>( t );
    assert( rrTask );
    QList<SharedAnnotationData> resultedAnnotatinos = rrTask->getResultedAnnotations();
    if( outPort ) {
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >( resultedAnnotatinos );
        outPort->put( Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v) );
        if( seqPort->isEnded() ) {
            outPort->setEnded();
        }
    }
}

void RemoteRequestWorkerFactory::init() {
    QList<PortDescriptor*> portDescs; 
    QList<Attribute*> attribs;

    //Create input port descriptors 
    {
        Descriptor seqDesc( SEQ_PORT, RemoteRequestWorker::tr("Query sequence"), RemoteRequestWorker::tr("A sequence which will be queried to remote database.") );
        Descriptor outDesc( OUT_PORT, RemoteRequestWorker::tr("Annotations"), RemoteRequestWorker::tr("Annotations converted from results of query to remote database.") );

        portDescs << new PortDescriptor( seqDesc, BioDataTypes::DNA_SEQUENCE_TYPE(), true );
        portDescs << new PortDescriptor( outDesc, BioDataTypes::ANNOTATION_TABLE_TYPE(), false, true );
    }
    //Create attributes descriptors    
    {
        Descriptor scriptsDesc( SCRIPT_ATTR, RemoteRequestWorker::tr("Request type"), RemoteRequestWorker::tr("Remote database adapter.") );
        Descriptor strandDesc( STRAND_ATTR, RemoteRequestWorker::tr("Search in"), RemoteRequestWorker::tr("Which strands should be searched: direct, complement or both.") );
        Descriptor maxResLenDesc( MAXRESLEN_ATTR, RemoteRequestWorker::tr("Max result length"), RemoteRequestWorker::tr("Upper bound for length of results.") );
        Descriptor minResLenDesc( MINRESLEN_ATTR, RemoteRequestWorker::tr("Min result length"), RemoteRequestWorker::tr("Lower bound for length of results.") );

        attribs << new Attribute( scriptsDesc, CoreDataTypes::STRING_TYPE(), true );
        attribs << new Attribute( strandDesc, CoreDataTypes::NUM_TYPE(), false, 0 );
        attribs << new Attribute( maxResLenDesc, CoreDataTypes::NUM_TYPE(), false, INT_MAX ); //FIXME: default value
        attribs << new Attribute( minResLenDesc, CoreDataTypes::NUM_TYPE(), false, 0 );
    }

    Descriptor desc( RemoteRequestWorkerFactory::ACTOR, RemoteRequestWorker::tr("Remote request"), 
                     RemoteRequestWorker::tr("Annotates input sequences using requests to remote databases. Database adapters are provided by scripts registered in Script Registry.") );
    ActorPrototype * proto = new BusActorPrototype( desc, portDescs, attribs );

    //create delegates for attribute editing
    QMap<QString, PropertyDelegate *> delegates;    
    {
        ServiceRegistry * sr = AppContext::getServiceRegistry();
        ScriptRegistryService * srs = qobject_cast<ScriptRegistryService *>( sr->findServices(Service_ScriptRegistry)[0] );
        QList<Script *> scripts = srs->getScriptsByType( ScriptTypes::SCRIPT_TYPE_HTTP_ANNOTATOR );

        QVariantMap scriptsMap;
        foreach( Script * s, scripts ) {
            if( s->isReady() ) {
                scriptsMap[ s->getName() ] = s->getName();
            }
        }
        delegates[ SCRIPT_ATTR ] = new ComboBoxDelegate( scriptsMap );
    }
    {
        QVariantMap strandsMap;
        strandsMap[RemoteRequestWorker::tr("both strands")] = QVariant(SendSelectionStrand_Both);
        strandsMap[RemoteRequestWorker::tr("direct strand")] = QVariant(SendSelectionStrand_Direct);
        strandsMap[RemoteRequestWorker::tr("complement strand")] = QVariant(SendSelectionStrand_Complement);
        delegates[ STRAND_ATTR ] = new ComboBoxDelegate( strandsMap );
    }
    {
        QVariantMap eMap; eMap["minimum"] = (0); eMap["maximum"] = (INT_MAX);
        delegates[MAXRESLEN_ATTR] = new SpinBoxDelegate(eMap);
    }
    {
        QVariantMap eMap; eMap["minimum"] = (0); eMap["maximum"] = (INT_MAX);
        delegates[MINRESLEN_ATTR] = new SpinBoxDelegate(eMap);
    }
    proto->setEditor( new DelegateEditor(delegates) );

    proto->setIconPath( ":/remote_request/images/remote_db_request.png" );
    proto->setPrompter( new RemoteRequestPromter() );
    WorkflowEnv::getProtoRegistry()->registerProto( BioActorLibrary::CATEGORY_BASIC(), proto );

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById( LocalDomainFactory::ID );
    localDomain->registerEntry( new RemoteRequestWorkerFactory() );
}

Worker * RemoteRequestWorkerFactory::createWorker( Actor * a ) {
    return new RemoteRequestWorker(a);
}

} //ns LocalWorkflow
} //ns GB2
