/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_ACTOR_CFG_H_
#define _GB2_ACTOR_CFG_H_

#include "ProxyDelegate.h"
#include <workflow/WorkflowModel.h>
#include <workflow_support/WorkflowUtils.h>

#include <core_api/Log.h>

#include <QtCore/QAbstractTableModel>

namespace GB2 {

static LogCategory log(ULOG_CAT_WD);

class ActorCfgModel : public QAbstractTableModel {
private:
    Actor* subject;
    QList<Attribute*> attrs;
    QList<Iteration>& iterations;
    int itIdx;
    QVariantMap listValues;

public:
    ActorCfgModel(QObject *parent, QList<Iteration>& lst) : QAbstractTableModel(parent), subject(NULL), iterations(lst), itIdx(-1) {}

    void setActor(Actor* cfg) {
        listValues.clear();
        subject = cfg;
        if (cfg) {
            attrs = cfg->getParameters().values();
        } else {
            attrs.clear();
        }
        reset();
    }
    void selectIteration(int i) {
        listValues.clear();
        itIdx = i;
        reset();
    }
    void setIterations(QList<Iteration>& lst) {iterations = lst;reset();}

    int columnCount(const QModelIndex &) const { return 2; }

    int rowCount ( const QModelIndex & parent = QModelIndex() ) const {
        if (parent.isValid()) {
            return 0;
        }
        return attrs.isEmpty() || parent.isValid() ? 0 : attrs.size();
    }

    Qt::ItemFlags flags ( const QModelIndex & index ) const {
        if (index.column() == 0) {
            return Qt::ItemIsEnabled;
        }
        return Qt::ItemIsEditable | Qt::ItemIsEnabled | Qt::ItemIsSelectable;
    }

    QVariant headerData ( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const {
        if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
            switch(section) {
                case 0: return WorkflowEditor::tr("Name");
                case 1: return WorkflowEditor::tr("Value");
            }
        }
        return QVariant();
    }

    /*Used to supply item data to views and delegates. 
    Generally, models only need to supply data for Qt::DisplayRole and any application-specific user roles, 
    but it is also good practice to provide data for Qt::ToolTipRole, Qt::AccessibleTextRole, and Qt::AccessibleDescriptionRole.*/
    QVariant data ( const QModelIndex & index, int role = Qt::DisplayRole ) const {
        const Attribute *item = attrs.at(index.row());
        if (role == DescriptorRole) {
            return qVariantFromValue<Descriptor>(*item);
        }
        if (index.column() == 0) {
            switch (role)
            {
            case Qt::DisplayRole: return item->getDisplayName();
            case Qt::ToolTipRole: return item->getDocumentation();
            case Qt::FontRole:
                if (item->required) {
                    QFont fnt; fnt.setBold(true);
                    return QVariant(fnt);
                }
            default:
                return QVariant();
            
            }
        }
        if (role == ConfigurationEditor::ItemListValueRole) {
            return listValues.value(item->getId());
        }
        // parameter value
        QVariant val = item->value;
        bool isDefaultVal = true;
        if (itIdx >= 0) {
            int x = itIdx;
            if (x >= iterations.size()) {
                //fixme
                x = 0;
            }
            const Iteration& it = iterations.at(x);
            if (it.cfg.contains(subject->getId())) {
                const QVariantMap& params = it.cfg[subject->getId()];
                if (params.contains(item->getId())) {
                    val = params.value(item->getId());
                    isDefaultVal = false;
                }
            }
        }
        ConfigurationEditor* ed = subject->getEditor();
        PropertyDelegate* pd = ed ? ed->getDelegate(item->getId()) : NULL;
        switch (role)
        {
        case Qt::DisplayRole: 
        case Qt::ToolTipRole:
            return pd ? pd->getDisplayValue(val) : val;
        case Qt::ForegroundRole:
            return isDefaultVal ? QVariant(QColor(Qt::gray)) : QVariant();
        case DelegateRole:
            return qVariantFromValue<PropertyDelegate*>(pd);
        case Qt::EditRole:
        case ConfigurationEditor::ItemValueRole:
            return val;
        }
        return QVariant();
    }

    /*Used to modify the item of data associated with a specified model index. 
    To be able to accept user input, provided by user interface elements, this function must handle data associated with Qt::EditRole. 
    The implementation may also accept data associated with many different kinds of roles specified by Qt::ItemDataRole. 
    After changing the item of data, models must emit the dataChanged() signal to inform other components of the change.*/
    bool setData ( const QModelIndex & index, const QVariant & value, int role = Qt::EditRole ) {
        if (index.column() != 1) {
            return false;
        }
        Attribute* item = attrs[index.row()];
        switch (role)
        {
        case ConfigurationEditor::ItemListValueRole: listValues.insert(item->getId(), value); return true;
        case Qt::EditRole:
        case ConfigurationEditor::ItemValueRole:
            const QString& key = item->getId();
            if (itIdx >= 0) {
                int x = itIdx;
                if (x >= iterations.size()) {
                    //fixme
                    x = 0;
                }
                QVariantMap& cfg = iterations[x].cfg[subject->getId()];
                QVariant old = cfg.contains(key) ? cfg.value(key) : item->value;
                if (old != value) {
                    cfg.insert(key, value);
                    emit dataChanged(index, index);
                    log.trace("committed property change");
                }
            } else {
                if (item->value != value) {
                    subject->setParameter(key, value);
                    emit dataChanged(index, index);
                    log.trace("committed property change");
                }
            }
            return true;
        }
        return false;
    }

};

}//namespace
#endif
